/*
    xmms-rplay is an output plugin for xmms that talks to rplay
    Copyright (C) 2002-2003 lmoore@tump.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    thanks to esdout for guidance.
 */
#include <xmms/plugin.h>
#include <xmms/configfile.h>
#include <rplay.h>
#include <sys/time.h>
#include <time.h>
#include <pthread.h>
#include <xmms/util.h>
#include "rplayout.h"

typedef enum { NOOP, PAUSE, UNPAUSE } pause_op_t;

/* the control socket */
static int              ctrl_fd;

/* the flow socket */
static int              flow_fd;

/* the spool id that we're assigned. */
static int              spool_id;

/* the total number of bytes written to rplay. */
static int              bytes_written;

/* the bytes per second (cached value of sample_rate*sample_size*channels) */
static int              bps, sample_rate, sample_size, channels;

/* the time basis after a reset */
static int              time_basis;

static char             *audio_format;
static char             *audio_endianess;

/* the buffer we present to xmms. */
static char             *sound_buffer;
static int              buffer_size;

/* the read and write indexes of the single-producer/single-consumer buffer */
static int              r_idx, w_idx;

/* open has been called, but close has not. */
static int              going;

/* the thread responsible for writing the buffer to the rplay flow */
static pthread_t        buffer_thread;

/* the thread responsible for controling the flow (pausing, continuing,
 * stopping, getting position data, etc) */
static pthread_t        ctrl_thread;

/* a variable telling the ctrl_thread to pause or unpause the flow. */
static pause_op_t       do_pause;

/* the current pause state. */
static int              paused;

/* as far as rplayd is concerned the current sample, the total number of
 * samples to play, and the total number of bytes that have been written
 * to the sound device. */
static int              current_sample, total_samples, committed_bytes;
static int              do_l_volume, do_r_volume, r_volume, l_volume;

/* the buffer writing loop. */
static void *rp_loop(void *);

/* the control loop */
static void *ctrl_loop(void *);

/* return the number of bytes used in the buffer. */
static int rp_used() {
  int rv = w_idx - r_idx;

  if (rv < 0) {
    rv += buffer_size;
  }

  return rv;
}

void rp_get_volume(int *l, int *r) {
  *l = l_volume;
  *r = r_volume;
}

void rp_set_volume(int l, int r) {
  do_l_volume = l;
  do_r_volume = r;
}

int open_rplay_streams() {
  char  response[RPTP_MAX_LINE];
  int   n;

  /* open the control socket */
  ctrl_fd = rptp_open(rp_config.server, rp_config.port,
                      response, sizeof(response));
  if (ctrl_fd < 0) {
    rptp_perror("open ctrl_fd");
    return 0;
  }

  /* figure out the initial volume */
  l_volume = r_volume =
    (int)((atoi(rptp_parse(response, "volume")) / 256.0) * 100);

  /* open up the flow socket */
  flow_fd = rptp_open(rp_config.server, rp_config.port,
                      response, sizeof(response));
  if (flow_fd < 0) {
    rptp_perror("open");
    return 0;
  }

  /* start the flow */
  rptp_putline(flow_fd,
               "play input=flow list-name=\"xmms\" sound=\"xmms\" "
               "input-format=%s input-sample-rate=%d input-bits=%d "
               "input-channels=%d input-byte-order=%s",
               audio_format, sample_rate, sample_size,
               channels, audio_endianess);
  n = rptp_getline(flow_fd, response, sizeof(response));
  if (n < 0 || response[0] != RPTP_OK) {
    rptp_perror(response);
    rptp_close(ctrl_fd);
    rptp_close(flow_fd);
    return 0;
  }

  spool_id = atoi(1 + rptp_parse(response, "id"));

  /* put the sound data. */
  rptp_putline(flow_fd, "put id=#%d size=0", spool_id);
  n = rptp_getline(flow_fd, response, sizeof(response));
  if (n < 0 || response[0] != RPTP_OK) {
    rptp_perror(response);
    rptp_close(ctrl_fd);
    rptp_close(flow_fd);
    return 0;
  }

  /* create the buffer */
  r_idx = w_idx = 0;
  buffer_size = rp_config.buffer_size;
  sound_buffer = g_malloc0(buffer_size);

  /* away we go */
  going = TRUE;

  pthread_create(&buffer_thread, NULL, rp_loop, NULL);
  pthread_create(&ctrl_thread, NULL, ctrl_loop, NULL);

  return 1;
}

int rp_open_audio(AFormat fmt, int rate, int nch) {
  /* figure out what format to use. */
  switch (fmt) {
  case FMT_U8:
    audio_format = "ulinear8";
    sample_size = 8;
    audio_endianess = "little-endian";
    break;

  case FMT_S8:
    audio_format = "linear8";
    sample_size = 8;
    audio_endianess = "little-endian";
    break;

  case FMT_U16_LE:
#ifndef WORDS_BIGENDIAN
  case FMT_U16_NE:
#endif
    audio_format = "ulinear16";
    sample_size = 16;
    audio_endianess = "little-endian";
    break;

  case FMT_S16_LE:
#ifndef WORDS_BIGENDIAN
  case FMT_S16_NE:
#endif
    audio_format = "linear16";
    sample_size = 16;
    audio_endianess = "little-endian";
    break;

  case FMT_U16_BE:
#ifdef WORDS_BIGENDIAN
  case FMT_U16_NE:
#endif
    audio_format = "ulinear16";
    sample_size = 16;
    audio_endianess = "big-endian";
    break;

  case FMT_S16_BE:
#ifdef WORDS_BIGENDIAN
  case FMT_S16_NE:
#endif
    audio_format = "linear16";
    sample_size = 16;
    audio_endianess = "big-endian";
    break;

  default:
    return 0;
  }

  sample_rate = rate;
  channels = nch;

  bps = sample_rate * channels * (sample_size / 8);

  /* reset the reset-ables */
  do_l_volume = do_r_volume = -1;
  do_pause = NOOP;
  paused = FALSE;
  bytes_written = 0;
  time_basis = 0;

  return open_rplay_streams();
}

void rp_write_audio(void *ptr, int length) {
  int total_written = 0;
  bytes_written += length;
  while (length > 0) {
    int to_write = MIN(length, buffer_size - w_idx);
    memcpy(sound_buffer + w_idx, ptr + total_written, to_write);
    length -= to_write;
    total_written += to_write;
    w_idx = (w_idx + to_write) % buffer_size;
  }
}

void rp_close_audio() {
  fprintf(stderr, "called rp_close\n");
  if (!going) {
    return;
  }
  going = FALSE;
  pthread_join(buffer_thread, NULL);
  pthread_join(ctrl_thread, NULL);
  g_free(sound_buffer);
  rptp_close(flow_fd);
  rptp_close(ctrl_fd);
}

void rp_flush(int time) {
  time_basis = time / 1000 * sample_rate;
  bytes_written = (time / 1000) * bps;
  fprintf(stderr, "called rp_flush %d %d %d\n",
          time, time_basis, bytes_written);

  rp_close_audio();
  open_rplay_streams();
}

void rp_pause(short pause_op) {
  if (pause_op && !paused) {
    do_pause = PAUSE;
  } else if (!pause_op && paused) {
    do_pause = UNPAUSE;
  }
}

int rp_buffer_free() {
  return buffer_size - rp_used() - 1;
}

int rp_buffer_playing() {
  return going && current_sample < total_samples;
}

int rp_written_time() {
  return going ? (int)(bytes_written * 1000.0 / (float)bps) : 0;
}

int rp_output_time() {
  return going ? (int)(committed_bytes * 1000.0 / bps) : 0;
}

static void *rp_loop(void *user_data) {
  while (going) {
    int length;
    if ((length = rp_used()) > 0 && !paused) {
      while (length > 0) {
        int n;
        n = MIN(length, buffer_size - r_idx);
        if ((n = rptp_write(flow_fd, sound_buffer + r_idx, n)) == -1) {
          rptp_perror("rp_loop -- rptp_write");
          break;
        }
        r_idx = (r_idx + n) % buffer_size;
        length -= n;
      }
    } else {
      xmms_usleep(10000);
    }
  }

  return NULL;
}

void *
ctrl_loop(void *user_data) {
  char  response[RPTP_MAX_LINE];
  int   n;

  rptp_putline(ctrl_fd, "set notify=position,modify notify-rate=0.1");
  n = rptp_getline(ctrl_fd, response, sizeof(response));
  if (n < 0 || response[0] != RPTP_OK) {
    rptp_perror("notify");
    return NULL;
  }

  while (going) {
    fd_set              set;
    struct timeval      timeout;

    timeout.tv_sec = 0;
    timeout.tv_usec = 100000;

    FD_ZERO(&set);
    FD_SET(ctrl_fd, &set);

    if (select(ctrl_fd + 1, &set, NULL, NULL, &timeout) > 0 &&
        rptp_getline(ctrl_fd, response, sizeof(response)) >= 0) {
      if (response[0] == '@') {
        char *event = rptp_parse(response, "event");
        if (event) {
          if (strcmp(event, "position") == 0) {
            int my_spool_id = atoi(1 + rptp_parse(response, "id"));
            if (spool_id == my_spool_id) {
              current_sample = atoi(rptp_parse(response, "sample"));
              total_samples = atoi(rptp_parse(response, "samples"));
              committed_bytes = (time_basis + current_sample) *
                (sample_size/8) * channels;
            }
          } else if (strcmp(event, "modify") == 0) {
            int my_spool_id = atoi(1 + rptp_parse(response, "id"));
            if (my_spool_id == spool_id) {
              fprintf(stderr, "%s\n", response);
              l_volume = (int)((atoi(rptp_parse(response, "left-volume"))/
                                256.0)*100);
              r_volume = (int)((atoi(rptp_parse(response, "right-volume"))/
                                256.0)*100);
            }
          }
        }
      }
    }

    /* the answers to these putlines don't really matter, *and* they're
     * asynchronous anyway, so we'll hear the answers above */

    /* check if we need to modify the volume */
    if (do_l_volume >= 0) {
      rptp_putline(ctrl_fd, "modify left-volume=%d right-volume=%d id=#%d",
                   (int)((do_l_volume / 100.0) * 256),
                   (int)((do_r_volume / 100.0) * 256),
                   spool_id);
      do_l_volume = do_r_volume = -1;
    }

    /* check if we need to {un,}pause */
    if (do_pause == PAUSE) {
      rptp_putline(ctrl_fd, "pause #%d", spool_id);
      do_pause = NOOP;
      paused = TRUE;
    } else if (do_pause == UNPAUSE) {
      rptp_putline(ctrl_fd, "continue #%d", spool_id);
      do_pause = NOOP;
      paused = FALSE;
    }
  }

  /* stop the flow, if we don't do this, sometimes, weird things happen. */
  rptp_putline(ctrl_fd, "stop #%d", spool_id);
  rptp_getline(ctrl_fd, response, sizeof(response));

  return NULL;
}
