/*
 * Copyright (c) 2003 Benedikt Meurer (benedikt.meurer@unix-ag.uni-siegen.de)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif	/* !HAVE_CONFIG_H */

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <gtk/gtk.h>
#include <libxfce4util/i18n.h>
#include <libxfcegui4/dialogs.h>

#include "propertiesdlg.h"

static void properties_dlg_class_init(PropertiesDlgClass *);
static void properties_dlg_init(PropertiesDlg *);
static void properties_dlg_finalize(GObject *);

/* signal handlers */
static void postscript_cb(GtkToggleButton *, gpointer user_data);

static GtkWidgetClass *parent_class = NULL;


GtkType
properties_dlg_get_type(void)
{
	static GtkType properties_dlg_type = 0;

	if (!properties_dlg_type) {
		static const GTypeInfo properties_dlg_info = {
			sizeof(PropertiesDlgClass),
			NULL,
			NULL,
			(GClassInitFunc)properties_dlg_class_init,
			NULL,
			NULL,
			sizeof(PropertiesDlg),
			0,
			(GInstanceInitFunc)properties_dlg_init
		};

		properties_dlg_type = g_type_register_static(GTK_TYPE_DIALOG,
		    "PropertiesDlg", &properties_dlg_info, 0);
	}

	return(properties_dlg_type);
}

static void
properties_dlg_class_init(PropertiesDlgClass *klass)
{
	GObjectClass *gobject_class;

	gobject_class = G_OBJECT_CLASS(klass);
	gobject_class->finalize = properties_dlg_finalize;

	parent_class = gtk_type_class(gtk_dialog_get_type());
}

static void
properties_dlg_init(PropertiesDlg *dlg)
{
	GtkWidget *frame;
	GtkWidget *label;
	GtkWidget *table;

	gtk_dialog_add_buttons(GTK_DIALOG(dlg),
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		GTK_STOCK_OK, GTK_RESPONSE_OK,
		NULL);
	gtk_window_set_default_size(GTK_WINDOW(dlg), 200, 100);

	/* the "General"-frame */
	frame = gtk_frame_new(_("General"));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), frame, FALSE,
		TRUE, 2);

	table = gtk_table_new(2, 2, FALSE);
	gtk_container_add(GTK_CONTAINER(frame), table);

	label = gtk_label_new(_("Alias name:"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
		GTK_FILL, 0, 0, 0);

	dlg->alias = gtk_entry_new();
	gtk_table_attach(GTK_TABLE(table), dlg->alias, 1, 2, 0, 1,
		GTK_EXPAND | GTK_FILL, 0, 0, 0);

	label = gtk_label_new(_("System name:"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
		GTK_FILL, 0, 0, 0);

	dlg->name = gtk_entry_new();
	gtk_table_attach(GTK_TABLE(table), dlg->name, 1, 2, 1, 2,
		GTK_EXPAND | GTK_FILL, 0, 0, 0);

	/* the "Capabilities"-frame */
	frame = gtk_frame_new(_("Capabilities"));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), frame, FALSE,
		TRUE, 2);

	table = gtk_table_new(4, 2, FALSE);
	gtk_container_add(GTK_CONTAINER(frame), table);

	dlg->postscript = gtk_radio_button_new_with_label(NULL,
		_("Postscript printer"));
	g_signal_connect(G_OBJECT(dlg->postscript), "toggled",
		G_CALLBACK(postscript_cb), dlg);
	gtk_table_attach(GTK_TABLE(table), dlg->postscript, 0, 1, 0, 1,
		GTK_EXPAND | GTK_FILL, 0, 0, 0);

	dlg->lineprinter = gtk_radio_button_new_with_label_from_widget(
		GTK_RADIO_BUTTON(dlg->postscript), _("Line printer"));
	gtk_table_attach(GTK_TABLE(table), dlg->lineprinter, 0, 1, 1, 2,
		GTK_EXPAND | GTK_FILL, 0, 0, 0);

	dlg->image = gtk_image_new();
	gtk_table_attach(GTK_TABLE(table), dlg->image, 1, 2, 0, 2,
		GTK_FILL, GTK_FILL, 0, 0);
}

static void
properties_dlg_finalize(GObject *object)
{
	PropertiesDlg *dlg;

	dlg = PROPERTIES_DLG(object);

	if (dlg->pr)
		xfprinter_free(dlg->pr);

	G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
update_image(PropertiesDlg *dlg)
{
	GdkPixbuf *icon;

	icon = xfprinter_icon(dlg->pr, FALSE);
	gtk_image_set_from_pixbuf(GTK_IMAGE(dlg->image), icon);
	g_object_unref(G_OBJECT(icon));
}

GtkWidget *
properties_dlg_new(GtkWindow *parent)
{
	PropertiesDlg *dlg;

	dlg = PROPERTIES_DLG(g_object_new(properties_dlg_get_type(), NULL));
	dlg->pr = NULL;
	gtk_window_set_transient_for(GTK_WINDOW(dlg), parent);

	return(GTK_WIDGET(dlg));
}

void
properties_dlg_popup(PropertiesDlg *dlg, const gchar *title,const XfPrinter *pr)
{
	if (dlg->pr != NULL)
		xfprinter_free(dlg->pr);

	if (pr != NULL)
		dlg->pr = xfprinter_dup(pr);
	else
		dlg->pr = xfprinter_new(_("New printer"), "lp");

	if (title)
		gtk_window_set_title(GTK_WINDOW(dlg), title);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dlg->pr->postscript
			? dlg->postscript : dlg->lineprinter), TRUE);

	gtk_widget_set_sensitive(dlg->alias, (pr == NULL));

	gtk_entry_set_text(GTK_ENTRY(dlg->alias), dlg->pr->alias);
	gtk_entry_set_text(GTK_ENTRY(dlg->name), dlg->pr->name);

	update_image(dlg);
}

const XfPrinter *
properties_dlg_run(PropertiesDlg *dlg)
{
	const gchar *t;
	gint response;

	gtk_widget_show_all(GTK_WIDGET(dlg));

	for (;;) {
		response = gtk_dialog_run(GTK_DIALOG(dlg));

		if (response != GTK_RESPONSE_OK)
			break;

		t = gtk_entry_get_text(GTK_ENTRY(dlg->alias));
		if (t == NULL || strlen(t) < 1) {
			show_error(_("No valid alias name specified"));
			continue;
		}
		else
			xfprinter_set_alias(dlg->pr, t);

		t = gtk_entry_get_text(GTK_ENTRY(dlg->name));
		if (t == NULL || strlen(t) < 1) {
			show_error(_("No valid system name specified"));
			continue;
		}
		else
			xfprinter_set_name(dlg->pr, t);

		return(dlg->pr);
	}

	return(NULL);
}

void
properties_dlg_hide(PropertiesDlg *dlg)
{
	g_return_if_fail(dlg != NULL);

	gtk_widget_hide_all(GTK_WIDGET(dlg));
}

static void
postscript_cb(GtkToggleButton *btn, gpointer user_data)
{
	PropertiesDlg *dlg;

	dlg = PROPERTIES_DLG(user_data);

	dlg->pr->postscript = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
			dlg->postscript));

	update_image(dlg);
}

