/* ==================================================== ======== ======= *
 *
 *  uubrick.cpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uubrick.cpp	ubit:03.06.04"
#include <iostream>
#include "config.h"
#include <udefs.hpp>
#include <ubrick.hpp>
#include <uerror.hpp>
#include <ucond.hpp>
#include <ucall.hpp>
#include <ustr.hpp>
#include <uctrl.hpp>
#include <ubox.hpp>
#include <uboxImpl.hpp>
#include <uview.hpp>
#include <uevent.hpp>
#include <update.hpp>
#include <uappli.hpp>
using namespace std;

const char* UBrick::getUbitVersion() {return PACKAGE_VERSION;}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UBrick::error(const char *id, const char *msg)  {
  UError::error(/*cname(),*/ id, msg);
}

void UBrick::error(const char* id, const char *msg, long arg)  {
  UError::error(/*cname(),*/ id, msg, arg);
}

void UBrick::error(const char* id, const char *msg, const char* arg) {
  UError::error(id, msg, arg);
}

void UBrick::error(const char* id, const char *msg, const UStr& arg) {
  UError::error(id, msg, arg);
}

void UBrick::error(const char* id, const char *msg, const std::string& arg){
  UError::error(id, msg, arg);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

ULink::ULink(class UBrick* _child) : UGenLink() {
  b = _child; par = null; cond = null;
}

UBox* ULink::getBoxChild() const {
  if (!b) {
    UError::error("internal@ULink::getBoxChild", "null child");
    return null;
  }
  else return b->boxCast();
}

UWin* ULink::getWinChild() const {
  if (!b) {
    UError::error("internal@ULink::getWinChild", "null child");
    return null;
  }
  else return b->winCast();
}

// NB: null cond means always
bool ULink::verifies(const UContext *ctx, const class UCtrl *ctrl) const {
  return (cond == null || cond->verifies(ctx, ctrl));
}

UGenLink* UGenChain::reset() {
  UGenLink* l = firstlink;
  firstlink = null;
  lastlink = null;
  return l;
}

void UGenChain::append(UGenChain *chain) {
  //2e chaine vide : ne rien faire
  if (!chain || !chain->firstlink) return;

  if (lastlink) {
    lastlink->nextlink = chain->firstlink;
    lastlink = chain->lastlink;
  }
  else {
    firstlink = chain->firstlink;
    lastlink = chain->lastlink;
  }
}

void UGenChain::add(UGenLink *link) {
  if (lastlink) {
    lastlink->nextlink = link;
    lastlink = link;
  }
  else {
    lastlink = firstlink = link;
  }
}

void UGenChain::remove(UGenLink *link) {
  UGenLink *prevl = null;
  for (UGenLink *l = firstlink; l!=null; prevl = l, l = l->getNext()) 
    if (l == link) {
      removeAfter(prevl);
      return;
    }
}

// insere au debut si poslink==null sinon insere APRES poslink
// -- suppose que poslink soit *effectivement* dans la liste!

void UGenChain::insertAfter(UGenLink *addedlink, UGenLink *poslink) {
 
  if (!firstlink) {  // liste vide
    firstlink = lastlink = addedlink;
    addedlink->nextlink = null;   // securit
  }
  else if (poslink == null) {  // insert at the beginning 
    UGenLink *nextlink = firstlink;
    firstlink = addedlink;
    addedlink->nextlink = nextlink;
    if (!nextlink) lastlink = addedlink;
  }
  else {  // insert after poslink
    UGenLink *nextlink = poslink->getNext();
    poslink->nextlink = addedlink;
    addedlink->nextlink = nextlink;
    if (!nextlink) lastlink = addedlink;
  }
}

// enleve et renvoie le lien qui SUIT 'poslink' (n'enleve pas 'poslink'!)
// -- enleve et renvoie le 1er lien (s'il existe) si 'poslink'==null
// -- renvoie null et ne fait rien si aucun lien ne suit 'poslink'
//NB: important:
//suppose que poslink est *effectivement* dans la liste!

UGenLink* UGenChain::removeAfter(UGenLink *poslink) {
  UGenLink *l = null;

  if (poslink == null) { // enleve et renvoie le 1er lien (s'il existe)
    l = firstlink;
    if (l) { // l exsite
      firstlink = l->getNext();
      if (!firstlink) lastlink = null;    //plus rien apres firstlink
      l->nextlink = null; //securite
    }
  }

  else {
    l = poslink->getNext();
    if (l) {  // l exsite
      poslink->nextlink = l->getNext();
      if (!poslink->nextlink) lastlink = poslink;  //plus rien apres poslink
      l->nextlink = null; //securite
    }
  }

  return l;  // renvoyer l (eventuellement null)
}

ULink* UChain::search(const UBrick *b, const ULink *fromlink) {
  ULink *l = const_cast<ULink*>(fromlink ? fromlink : first());
  for (; l!=null; l = l->getNext()) 
    if (l->getChild() == b) return l;
  return null;  // not found
}

/* ==================================================== ======== ======= */

ULinkLink* ULLChain::removePointingTo(ULink *pointed) {
  ULinkLink *prevl = null;
  ULinkLink *l = first();

  for ( ; l != null; l = l->getNext()) {
    if (l->link() != pointed)
      prevl = l;
    else {
      if (prevl) prevl->nextlink = l->getNext();
      else firstlink = l->getNext();
      if (lastlink == l) lastlink = prevl;
      return l;
    }
  }
  return null;  // not found
}

void ULLChain::updateParents(const UUpdate &upmode) const {  
  // mettre a jour toutes les Grp qui pointent sur l'item
  for (ULinkLink *ll = first(); ll != null; ll = ll->getNext()) {
    UGroup *grp = ll->getParent();
    // box==null --> pas encore passe par initContext()
    if (grp) grp->update(upmode);
  }
}

void ULLChain::fireParents(UEvent& e, const UOn &on) const {  
  // mettre a jour toutes les Box qui pointent sur l'item
  for (ULinkLink *ll = first(); ll != null; ll = ll->getNext()) {
    UGroup *grp = ll->getParent();                       //???!!!!TrueParent??
    // box==null --> pas encore passe par initContext()
    if (grp) {
      e.setSource(grp);
      grp->fire(e, on);
    }
  }
}

void ULLChain::setModesOfParents(u_modes bmodes, u_modes cmodes, 
				 bool onoff) const {
  for (ULinkLink *ll = first(); ll != null; ll = ll->getNext()) {
    UGroup *box = ll->getParent();
    // box==null --> pas encore passe par initContext()
    if (box) {
      box->setBmodes(bmodes, onoff);
      box->setCmodes(cmodes, onoff);
    }
  }
}

void ULLChain::setModesOfParentViews(int vmodes, bool onoff) const {

  for (ULinkLink *ll = first(); ll != null; ll = ll->getNext()) {
    //UBox* box = ll->getParent() ? dynamic_cast<UBox*>(ll->getParent()) : null;
    UBox* box = ll->getParent() ? ll->getParent()->boxCast() : null;
  
    if (box) {
      for (ULinkLink *llk2 = box->getParentList().first();
	   llk2 != null; 
	   llk2 = llk2->getNext()){

	UBoxLink *link = static_cast<UBoxLink*>(llk2->link());
	for (UView* v = link->getViews(); v != null; v = v->getNext())
	  v->setVmodes(vmodes, onoff);
      }
    }
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UBrick::setBmodes(u_modes m, bool on) {
  if (on) bmodes = bmodes | m;
  else bmodes = bmodes & ~m;
}

bool UBrick::isAutoUpdate() const 
{return (bmodes & UMode::NO_AUTO_UPDATE) == 0;}

void UBrick::setAutoUpdate(bool on) 
{setBmodes(UMode::NO_AUTO_UPDATE, !on);}

/* ==================================================== ======== ======= */

void* UBrick::operator new(size_t sz) {
  UBrick *b = (UBrick*) ::operator new(sz);
  // trick for detecting that this object was created using 'new'
  // (cf. UBrick::UBrick)
  b->cache = (UChain*)b;
  return b;
}

/* ==================================================== ======== ======= */

void UBrick::operator delete(void* p) {
  if (!p) return;
  UBrick* b = (UBrick*)p;

  if (!b->isBmode(UMode::CAN_DEL)) {
    UError::error("UBrick::operator delete", 
		  "attempt to 'delete' an object that was not created by 'new' ; object:", (long)p);
  }
  
  else if (b->refcount > 0) {
    UError::error("UBrick::operator delete", 
		  "attempt to 'delete' an object that is pointed by a 'uptr' (Ubit smart pointer) ; object:", (long)p);
    // peut etre rellement detruit plus tard
    b->setBmodes(UMode::DESTRUCTED, false);
  }
  
  else {
    // this object will be deleted when this is safe to do so
    UAppli::safeDeleteRequest(p);
  }
}

/* ==================================================== ======== ======= */

// PBM: addRef() faux dans le cas des constantes types UColor::white, etc.
// car l'ordre
// d'init des vars globales est aleatoire. ceci dit ca n'a aucune importance
// car ces vars ne sont jamais detruitre en cours d'exec (le refcount sera
// faux mais on n'en a cure)

void UBrick::removeRef() {
  refcount--;
  if (refcount == 0 && isBmode(UMode::CAN_DEL) && !hasValidParent()) {
    delete this;  // detacher du graphe, detruire l'objet et ses enfants
  }
}

/* ==================================================== ======== ======= */

UBrick::UBrick(u_modes b_modes) {
  if (cache == (UChain*)this) {
    // object was created using 'new' ==> DEL and AUTODEL modes
    // bmodes = b_modes | (UMode::CAN_DEL|UMode::CAN_AUTODEL); !!
    bmodes = (b_modes | UMode::CAN_DEL);
  }
  else {
    //cout << "UBrick::UBrick:  NOT new"<<this <<"\n";
    bmodes = b_modes; 
  }
  cache = null;
  refcount = 0;
}

/* ==================================================== ======== ======= */
// ATTENTION:
// les appels aux fcts vituelles ne marchent pas normalement dans les
// destructeurs (l'appel est fait avec la classe du destructeur,
// pas avec la classe effective de this)
//
// ==> appeler destructs() DANS LE DESTRUCTEUR de la CLASSE de l'objet
//     pour que destructs() soit appele avec ce type     
//     adequate pour qu'il soit applique avec le type effectif de l'objet
// ==> CECI IMPOSE DE DEFINIR DES DESTRUCTEURS DANS LES SOUS-CLASSES

void UBrick::destructs() {
  // superclasses' cleanup() methods wont be called
  if (isBmode(UMode::DESTRUCTED)) return;

  // this object is now being destructed
  setBmodes(UMode::DESTRUCTING, true);
  
  // mod 4aug03 (n'etait jamais appele car dans UBrick::~UBrick)
  // doit etre avant sinon ne fera rien car CBs deja enleves
  if (isBmode(UMode::DESTRUCT_CB)) {
    UEvent e(UEvent::destruct, null, null, NULL);
    e.setAux(this);
    fire(e, UOn::destruct);
  }

  // !!!!!!!!!! BUG GRAVE !!!!!!!!!
  // removeAllAttr(true, false);

  if (parents.first()) removeFromParents(false);
  // faudrait aussi s'occuper des liens croises (par les ucall) !!!

  // must be at the end (callbacks wont be called otherwise)
  setBmodes(UMode::DESTRUCTED, true);
}

/* ==================================================== ======== ======= */
// !!selflink = the link that POINTS to 'this' obj!

void UBrick::addingTo(ULink *selflink, UGroup *parent) {
  selflink->setParent(parent);

  // Rendre parent sensitif aux events ad hoc
  // !!!ATTENTION: PROBLEME
  // setModes() pose probleme avec select/unselect car ce n'est pas un mode
  // qu'il faut propager au parent mais juste un referent pour le UOn
  // (pour l'instant c'est regle en ne rajoutant pas ces modes et 
  // en testant directement l'egalite des UOn dans match() et verifies()

  if (selflink->getCond()) selflink->getCond()->setParentModes(parent);

  ///... inutile pour les UCONST!!!;
  parents.add(new ULinkLink(selflink));
}

/* ==================================================== ======== ======= */
//Notes:
// 1. removingFrom() requires a destructor to be defined
// 2. selflink = the link that points to 'this' obj!

void UBrick::removingFrom(ULink *selflink, UGroup *parent) {

  // ne PAS enlever les modes du parent car il peut y avoir
  // plusieurs callbacks avec la meme condition
  // (et de toute facon ce n'est pas une erreur, juste un peu plus long)

  ///... inutile pour les UCONST!!!;
  ULinkLink *ll = parents.removePointingTo(selflink);
  if (ll) delete ll;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

int UBrick::getParentCount() const{
  int kk = 0;
  for (ULinkLink *l = parents.first(); l!=null; l = l->getNext()) 
    kk++;   //k++ mis ici pour eviter les bugs... des compilateurs!
  return kk;
}

UGroup** UBrick::getParents() const{
  int count = 0;
  return getParents(count);
}

// !!!!!!ATT: ambiguite sur la NATURE des parents :                
// on veut TOUS les parents ou seulement ceux lies aux 'children' lists ????

UGroup** UBrick::getParents(int& count) const {
  count = getParentCount();

  if (count <= 0) return null;
  else {
    UGroup **tab = new UGroup*[count+1];
    int kk = 0;
    for (ULinkLink *l = parents.first(); l!=null; l = l->getNext(), kk++){
      tab[kk] = l->link()->getParent();
    }
    tab[kk] = null;  // null terminated
    return tab;
  }
}

int UBrick::getParents(std::vector<UGroup*>& parvect) const {
  int count = getParentCount();

  if (count <= 0) return 0;
  else {
    parvect.reserve(count);
    int kk = 0;
    for (ULinkLink *l = parents.first(); l!=null; l = l->getNext(), kk++){
      parvect.push_back(l->link()->getParent());
    }
    return count;
  }
}

/* ==================================================== ======== ======= */

UGroup* UBrick::getParent(int pos) const {
  if (pos == -1) {  // find the last one
    return (parents.last() ? parents.last()->link()->getParent() : null);
  }
  else if (pos >= 0) {
    int kk = 0;
    for (ULinkLink *l = parents.first(); l!=null; l = l->getNext(), kk++){
      if (kk == pos) return l->link()->getParent();
    }
  }
  // < -1 or to large
  error("UBrick::getParent", UError::Warning_out_of_range, pos);
  return null;
}


bool UBrick::hasValidParent() const {
  for (ULinkLink *ll = parents.first(); ll != null; ll = ll->getNext()) {
    UGroup* par = ll->link()->getParent();
    if (!par->isCmode(UMode::FAKE_PARENT)
        // && !par->isBmode(UMode::DELETING)
        )
      return true;
  }
  return false;
}

bool UBrick::hasValidParent(bool& fake_or_deleted_par) const {
  bool valid_par = false;
  fake_or_deleted_par = false;

  for (ULinkLink *ll = parents.first(); ll != null; ll = ll->getNext()) {
    UGroup* par = ll->link()->getParent();
    if (par->isCmode(UMode::FAKE_PARENT)
        // || par->isBmode(UMode::DELETING)
        )
      fake_or_deleted_par = true;
    else
      valid_par = true;
  }

  return valid_par;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UBrick::removeFromParents(bool update_parents) {
  // nb: on enleve le 1er jusuq'a ce qu'il n'y en ait plus
  // - correct pour multi-occurences:
  // - l'ordre n'a pas d'importance puisqu'on les enleve tous (si un 
  //   child est inclus plusieurs fois dans un meme parent on ne sait pas 
  //   dans quel ordre parent->remove(child)  enleve ces occurences)

  ULinkLink *ll;
  while ((ll = parents.first())) {
    UGroup* par = ll->getParent();
    ULink* prevlink;

    if (par->getChildImpl(UBox::ATTR_LIST, this, 0, prevlink, null))
      par->removeImpl(UBox::ATTR_LIST, this, prevlink, 
		      UGroup::REMOVE_FROM_PARENTS, update_parents, null);

    else if (par->getChildImpl(UBox::ELEM_LIST, this, 0, prevlink, null))
      par->removeImpl(UBox::ELEM_LIST, this, prevlink, 
		      UGroup::REMOVE_FROM_PARENTS, update_parents, null);

    else {
      // !Cas d'erreur: child pointe vers parent alors que parent ne pointe
      //  pas vers child
      // ==> enlever directement ce lien de la liste des parents et le detruire
      error("UBrick::removeFromParents",
	    "child not found in child lists of object: ", cname());
      parents.removeAfter(null);   // null => debut de chaine
      delete ll; 
    }
  }
}

// returns true if argument is a direct parent (case indirect = false)
// or an ancestor (case indirect = true) of 'this' object

bool UBrick::isChildOf(UGroup *possible_ancestor, bool indirect) const{

  for (ULinkLink *l = parents.first(); l!=null; l = l->getNext()) {
    UGroup *par = l->link()->getParent();
    // par == null or this if no other parent (UFrame or similar)
    if (par && par != this) {
      if (par == possible_ancestor) return true; 
      else if (indirect) {
	if (par->isChildOf(possible_ancestor, true))
	  return true;
      }
    }
  }
  return false; // not found
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UBrick::fire(UEvent& e, const UOn &cond) const {

  // dont call callback if the object has already been destructed
  if (isBmode(UMode::DESTRUCTED)) return;
  
  if (cache) {
    ULink *l; UCall *c;
   
    for (l = cache->first(); l != null; l = l->getNext()) {
      if (l->match(cond) && (c = dynamic_cast<UCall*>(l->getChild()))) {
      //if (l->match(cond) && (c = l->getChild()->callCast())) {
        e.setCond(&cond); 
	c->call(e);
        // return now if the object has been destructed by c->call()
        if (isBmode(UMode::DESTRUCTED)) return;
      }
    }
  }
}

void UBrick::addToCache(UBrick& b) {
  if (!cache) cache = new UChain();
  cache->add(new ULink(&b));
}

void UBrick::addCall(ULink& l) {
  if (!cache) cache = new UChain();
  cache->add(&l);
}

void UBrick::addChangeCall(UBrick& b) {
  if (!cache) cache = new UChain();

  ULink *l = new ULink(&b);
  l->setCond(UOn::change);
  setBmodes(UMode::CHANGE_CB, true);
  cache->add(l);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UElem::onChange(UCall& c) {
  addChangeCall(c);
}

void UElem::changed(bool update_now) {
  if (update_now && (bmodes & UMode::NO_AUTO_UPDATE) == 0)
    update();

  // faux: parent pas notifie si pas de CB sur ustr !!!
  //  if (cache) {  !!!

  UEvent e(UEvent::change,  null, null, NULL);
  e.setAux(this);

  if (cache) fire(e, UOn::change);

  // ensuite on lance les callbacks des parents
  parents.fireParents(e, UOn::elemChange);
}


/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
