/* ==================================================== ======== ======= *
 *
 *  update.hh : graphics update
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _update_hh
#define	_update_hh
//pragma ident	"@(#)update.hh	ubit:03.02.0"

/** specifies how to update UGroup, UBox, UWin objects and subclasses.
 * This object is given as an argument of UGroup::update()
 * Examples: 
 * <pre>
 *       box.update(UUpdate::paint);   // repaints the box
 *       box.update(UUpdate::all);     // updates box layout then repaints
 *       box.update();                 // same as: box.update(UUpdate::all);
 * </pe>
 * Notes:
 *  - by default, no update is performed is the object is not shown
 *    (see: UCtrl::isShown()). This can be changed by calling the 
 *    layoutIfNotShown() method:
 * <pre>
 *       UUpdate upd = UUpdate::all;   // layout and paint
 *       upd.evenIfNotShown();
 *       box.update(upd);
 * </pre>
 *  - the paintDoubleBuffered() method requests double buffering painting:
 * <pre>
 *       UUpdate upd = UUpdate::paint; // just paint
 *       upd.setDoubleBuffering();
 *       box.update(upd);
 * </pre>
 *  - other UUpdate methods are related to the internal implementation.
 */
class UUpdate {
  friend class UBox;
  friend class UWin;
  friend class USoftwinImpl;
  friend class UHardwinImpl;
public:
  enum Mode {
    ALL,
    LAYOUT,
    PAINT,
    MOVE, 
    SCROLL, 
    SHOW, 
    HIDE, 
    TITLE
  };
  static const UUpdate layout;  ///< same as: UUpdate(UUpdate::LAYOUT)
  static const UUpdate paint;   ///< same as: UUpdate(UUpdate::PAINT)
  static const UUpdate all;     ///< same as: UUpdate(UUpdate::ALL) = LAYOUT then PAINT

  UUpdate(Mode = ALL);
  ///< constructor (see class UUpdate and its methods for details).

  void layoutIfNotShown(bool = true);
  ///< updates objects even if they are not shown (see UCtrl::isShown()).

  void paintDoubleBuffered(bool = true);
  ///< paints in double buffering mode.

  void paintElem(const UElem*);
  ///< [impl] paints this element (and whatever is located beneath or above it).

  void paintStr(const UStr*, int strpos1, int strpos2); 
  ///< [impl] paints a subpart of this string.

  void paintRegion(const URegion*);
  ///< [impl] paints this region.

  void paintTitle(const UStr*); 
  ///< [impl] paints this title.

  void setOffset(u_pos delta_x, u_pos delta_y);
  ///< [impl] used with the MOVE and SCROLL modes.

private:
  Mode ix;
  bool always, doublebuf;
  u_pos delta_x, delta_y;
  const URegion* region;
  const UElem* elem;
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
