/*   FILE: bistree.c --
 * AUTHOR: W. Michael Petullo <new@flyn.org>
 *   DATE: 10 Febuary 2000
 *   NOTE: From _Mastering Algorithms with C_ by Kyle Loudon.
 */

#include <stdlib.h>
#include <string.h>
#include <bistree.h>

/* ============================ rotate_left () ============================= */
static void rotate_left(bintree_node_t ** node)
{
    bintree_node_t *left, *grandchild;
    left = bintree_left(*node);
    if (((avl_node_t *) bintree_data(left))->factor == AVL_LFT_HEAVY) {
	bintree_left(*node) = bintree_right(left);
	bintree_right(left) = *node;
	((avl_node_t *) bintree_data(*node))->factor = AVL_BALANCED;
	((avl_node_t *) bintree_data(left))->factor = AVL_BALANCED;
	*node = left;
    } else {
	grandchild = bintree_right(left);
	bintree_right(left) = bintree_left(grandchild);
	bintree_left(grandchild) = left;
	bintree_left(*node) = bintree_right(grandchild);
	bintree_right(grandchild) = *node;
	switch (((avl_node_t *) bintree_data(grandchild))->factor) {
	case AVL_LFT_HEAVY:
	    ((avl_node_t *) bintree_data(*node))->factor = AVL_RGT_HEAVY;
	    ((avl_node_t *) bintree_data(left))->factor = AVL_BALANCED;
	    break;
	case AVL_BALANCED:
	    ((avl_node_t *) bintree_data(*node))->factor = AVL_BALANCED;
	    ((avl_node_t *) bintree_data(left))->factor = AVL_BALANCED;
	    break;
	case AVL_RGT_HEAVY:
	    ((avl_node_t *) bintree_data(*node))->factor = AVL_BALANCED;
	    ((avl_node_t *) bintree_data(left))->factor = AVL_LFT_HEAVY;
	    break;
	}
	((avl_node_t *) bintree_data(grandchild))->factor = AVL_BALANCED;
	*node = grandchild;
    }
    return;
}

/* ============================ rotate_right () ============================ */
static void rotate_right(bintree_node_t ** node)
{
    bintree_node_t *right, *grandchild;
    right = bintree_right(*node);
    if (((avl_node_t *) bintree_data(right))->factor == AVL_RGT_HEAVY) {
	bintree_right(*node) = bintree_left(right);
	bintree_left(right) = *node;
	((avl_node_t *) bintree_data(*node))->factor = AVL_BALANCED;
	((avl_node_t *) bintree_data(right))->factor = AVL_BALANCED;
	*node = right;
    } else {
	grandchild = bintree_left(right);
	bintree_left(right) = bintree_right(grandchild);
	bintree_right(grandchild) = right;
	bintree_right(*node) = bintree_left(grandchild);
	bintree_left(grandchild) = *node;
	switch (((avl_node_t *) bintree_data(grandchild))->factor) {
	case AVL_LFT_HEAVY:
	    ((avl_node_t *) bintree_data(*node))->factor = AVL_BALANCED;
	    ((avl_node_t *) bintree_data(right))->factor = AVL_RGT_HEAVY;
	    break;
	case AVL_BALANCED:
	    ((avl_node_t *) bintree_data(*node))->factor = AVL_BALANCED;
	    ((avl_node_t *) bintree_data(right))->factor = AVL_BALANCED;
	    break;
	case AVL_RGT_HEAVY:
	    ((avl_node_t *) bintree_data(*node))->factor = AVL_LFT_HEAVY;
	    ((avl_node_t *) bintree_data(right))->factor = AVL_BALANCED;
	    break;
	}
	((avl_node_t *) bintree_data(grandchild))->factor = AVL_BALANCED;
	*node = grandchild;
    }
    return;
}

static void destroy_right(bistree_t * tree, bintree_node_t * node);

/* ============================ destroy_left () ============================ */
static void destroy_left(bistree_t * tree, bintree_node_t * node)
{
    bintree_node_t **position;
    if (bintree_size(tree) == 0)
	return;
    if (node == NULL)
	position = &tree->root;
    else
	position = &node->left;
    if (*position != NULL) {
	destroy_left(tree, *position);
	destroy_right(tree, *position);
	if (tree->destroy != NULL) {
	    tree->destroy(((avl_node_t *) (*position)->data)->data);
	}
	free((*position)->data);
	free(*position);
	*position = NULL;
	tree->size--;
    }
    return;
}

/* ============================ destroy_right () =========================== */
static void destroy_right(bistree_t * tree, bintree_node_t * node)
{
    bintree_node_t **position;
    if (bintree_size(tree) == 0)
	return;
    if (node == NULL)
	position = &tree->root;
    else
	position = &node->right;
    if (*position != NULL) {
	destroy_left(tree, *position);
	destroy_right(tree, *position);
	if (tree->destroy != NULL) {
	    tree->destroy(((avl_node_t *) (*position)->data)->data);
	}
	free((*position)->data);
	free(*position);
	*position = NULL;
	tree->size--;
    }
    return;
}

/* ============================ insert () ================================== */
static int insert(bistree_t * tree, bintree_node_t ** node,
		  const void *data, int
		  *balanced)
{
    avl_node_t *avl_data;
    int cmpval, retval;
    if (bintree_is_eob(*node)) {
	if ((avl_data = (avl_node_t *) malloc(sizeof(avl_node_t))) == NULL)
	    return -1;
	avl_data->factor = AVL_BALANCED;
	avl_data->hidden = 0;
	avl_data->data = (void *) data;
	return bintree_ins_left(tree, *node, avl_data);
    } else {
	cmpval =
	    tree->compare(data,
			  ((avl_node_t *) bintree_data(*node))->data);
	if (cmpval < 0) {
	    if (bintree_is_eob(bintree_left(*node))) {
		if ((avl_data = (avl_node_t *) malloc(sizeof(avl_node_t)))
		    == NULL)
		    return -1;
		avl_data->factor = AVL_BALANCED;
		avl_data->hidden = 0;
		avl_data->data = (void *) data;
		if (bintree_ins_left(tree, *node, avl_data) != 0)
		    return -1;
		*balanced = 0;
	    } else {
		if (
		    (retval =
		     insert(tree, &bintree_left(*node), data,
			    balanced)) != 0) {
		    return retval;
		}
	    }
	    if (!(*balanced)) {
		switch (((avl_node_t *) bintree_data(*node))->factor) {
		case AVL_LFT_HEAVY:
		    rotate_left(node);
		    *balanced = 1;
		    break;
		case AVL_BALANCED:
		    ((avl_node_t *) bintree_data(*node))->factor = AVL_LFT_HEAVY;
		    break;
		case AVL_RGT_HEAVY:
		    ((avl_node_t *) bintree_data(*node))->factor = AVL_BALANCED;
		    *balanced = 1;
		}
	    }
	} else if (cmpval > 0) {
	    if (bintree_is_eob(bintree_right(*node))) {
		if ((avl_data = (avl_node_t *) malloc(sizeof(avl_node_t)))
		    == NULL)
		    return -1;
		avl_data->factor = AVL_BALANCED;
		avl_data->hidden = 0;
		avl_data->data = (void *) data;
		if (bintree_ins_right(tree, *node, avl_data) != 0)
		    return -1;
		*balanced = 0;
	    } else {
		if (
		    (retval =
		     insert(tree, &bintree_right(*node), data,
			    balanced)) != 0) {
		    return retval;
		}
	    }
	    if (!(*balanced)) {
		switch (((avl_node_t *) bintree_data(*node))->factor) {
		case AVL_LFT_HEAVY:
		    ((avl_node_t *) bintree_data(*node))->factor = AVL_BALANCED;
		    *balanced = 1;
		    break;
		case AVL_BALANCED:
		    ((avl_node_t *) bintree_data(*node))->factor = AVL_RGT_HEAVY;
		    break;
		case AVL_RGT_HEAVY:
		    rotate_right(node);
		    *balanced = 1;
		}
	    }
	} else {
	    if (tree->destroy != NULL) {
		tree->destroy(((avl_node_t *)
			       bintree_data(*node))->data);
	    }
	    ((avl_node_t *) bintree_data(*node))->data = (void *) data;
	    ((avl_node_t *) bintree_data(*node))->hidden = 0;
	    *balanced = 1;
	}
    }
    return 0;
}

/* ============================ hide () ==================================== */
static int hide(bistree_t * tree, bintree_node_t * node, const void *data)
{
    int cmpval, retval;
    if (bintree_is_eob(node)) {
	return -1;
    }
    cmpval =
	tree->compare(data, ((avl_node_t *) bintree_data(node))->data);
    if (cmpval < 0) {
	retval = hide(tree, bintree_left(node), data);
    } else if (cmpval > 0) {
	retval = hide(tree, bintree_right(node), data);
    } else {
	((avl_node_t *) bintree_data(node))->hidden = 1;
	retval = 0;
    }
    return retval;
}

/* ============================ lookup () ================================== */
static void *lookup(const bistree_t * tree, bintree_node_t * node,
		    void *data)
{
    int cmpval;
    void *retval;
    if (bintree_is_eob(node)) {
	return NULL;
    }
    cmpval =
	tree->compare(data, ((avl_node_t *) bintree_data(node))->data);
    if (cmpval < 0) {
	retval = lookup(tree, bintree_left(node), data);
    } else if (cmpval > 0) {
	retval = lookup(tree, bintree_right(node), data);
    } else {
	if (!((avl_node_t *) bintree_data(node))->hidden)
	    retval = ((avl_node_t *) bintree_data(node))->data;
	else {
	    return NULL;
	}
    }
    return retval;
}

/* ============================ bistree_init () ============================ */
void bistree_init(bistree_t * tree,
		  int (*compare) (const void *key1, const void *key2),
		  void (*destroy) (void *data))
{
    bintree_init(tree, destroy);
    tree->compare = compare;
    return;
}

/* ============================ bistree_destroy () ========================= */
void bistree_destroy(bistree_t * tree)
{
    destroy_left(tree, NULL);
    memset(tree, 0, sizeof(bistree_t));
    return;
}

/* ============================ bistree_insert () ========================== */
int bistree_insert(bistree_t * tree, const void *data)
{
    int balanced = 0;
    return insert(tree, &bintree_root(tree), data, &balanced);
}

/* ============================ bistree_remove () ========================== */
int bistree_remove(bistree_t * tree, const void *data)
{
    return hide(tree, bintree_root(tree), data);
}

/* ============================ bistree_lookup () ========================== */
void *bistree_lookup(const bistree_t * tree, void *data)
{
    return lookup(tree, bintree_root(tree), data);
}
