#!/usr/local/bin/perl
#
# This object represents a service that is running on a given machine.  It
# contains information such as the state of the service (red,green,etc...), the
# time the service was last updated, summary information about the state, and
# additional information helpful if there is a problem.
#
#   host          - string FQDN of the machine its associated with
#   name          - string name of the service
#   color         - computed color of the service (can be purple or blue)
#   rcolor        - reported color of the service (only red, yellow, green)
#   rtime         - reported last updated time of the service (time() format)
#   stime         - reported start of current status (time() format)
#   summary       - short summary providing additional info about the service
#   message       - more detailed information about the service
#
# + new()         - constructor (sets instance vars to arguments passed in)
# + gets/sets()   - magical set/get functions (autoloaded based on func name)
# + display()     - output format and view
#
# + has_problem() - 
#
# History:
# (1) Cleaned up (Ed July 31, 1997);

package Spong::Service;

# Constructor.  This is a lazy constructor in that it doesn't get all it's 
# instance vars set at the time of creation.  Vars are loaded as they are asked
# for, this gives us a performance gain.  This object is dependent on the 
# ServiceList object for loading the information about it from the database.

sub new {
   my( $proto, $host, $service ) = @_;
   my( $class ) = ref($proto) || $proto;
   my $self = {};

   # A hack to get around a local problem.

   if( $service eq "procs" ) { $service = "jobs"; }

   $self->{'name'}    = $service;
   $self->{'host'}    = $host;
   $self->{'color'}   = "";
   $self->{'rcolor'}  = "";
   $self->{'rtime'}   = "";
   $self->{'summary'} = "";
   $self->{'message'} = "";
   $self->{'stime'}   = "";

   bless ($self,$class);
   return $self;
}


# Get/Set methods, nothing fancy here...

sub host { my $var = 'host';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }

sub name { 
   my $var = 'name';

   # A hack to get around a local problem.

   if( defined $_[1] ) { 
      if( $_[1] eq "procs" ) {
	 $_[0]->{$var} = "jobs";
      } else {
	 $_[0]->{$var} = $_[1]; 
      } 
   }
   return $_[0]->{$var}; 
}

sub color { my $var = 'color';
   if( defined $_[1] ) { 
      $_[0]->{$var} = $_[1]; 
   } else {
      my $name = $_[0]->host();

      # If there is a problem reported, and we are in this machine's down time
      # then don't report it, instead mark it as blue.

      foreach( @{$main::HOSTS{$name}->{'down'}},
               @{$main::HOSTS{$name}->{'down:' . $_[0]->name() }} ) {
	 my( $day, $shour, $smin, $ehour, $emin ) = 
	    ( /^(.):(\d+):(\d+)-(\d+):(\d+)$/ );
	 my( $nmin, $nhour, $nday ) = (localtime(time()))[1,2,6];
	 
	 if( $day eq "*" || $nday eq $day ) {
	    my $ntotal = $nhour * 60 + $nmin;
	    my $stotal = $shour * 60 + $smin;
	    my $etotal = $ehour * 60 + $emin;

	    if( $ntotal >= $stotal && $ntotal <= $etotal ) {
	       return 'blue';
	    }
	 }
      }
   }

   return $_[0]->{$var}; 
}

sub rcolor { my $var = 'rcolor';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub rtime { my $var = 'rtime';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub stime { my $var = 'stime';
   if( $_[0]->{$var} eq "" ) { $_[0]->summary(); } return $_[0]->{$var}; }

# These functions handle the loading of of the service data from the 
# database files

sub filename {
  my( $self ) = shift;
  my( $file );

   # This is a local hack to get around the procs/jobs problems, first look
   # for the jobs service, if it is not there, look for procs...
   
   if( $self->name() eq "jobs" ) {
      $file = $self->host()."/services/".$self->name()."-".$self->rcolor();
      if( ! -f "$main::SPONGDB/$file" ) {
	 $file = $self->host()."/services/procs-".$self->rcolor();
      }
   } else {
      $file = $self->host()."/services/".$self->name()."-".$self->rcolor();
   }

   return $file;
}

sub load {
   my( $self ) = shift;
   my $file = $self->filename();
  
   open( FILE, "$main::SPONGDB/$file" );
   my $header = <FILE>; chomp $header;
   # If a timestamp is present, read it and set start time
   if ($header =~ /^timestamp (\d+) (\d+)/ ) {
      $self->{'stime'} = $1; 
      $header = <FILE>; chomp $header;
   }
   ($self->{'rtime'}, $self->{'summary'}) = ( $header =~ /^(\d+) (.*)$/ );
   while( <FILE> ) { $self->{'message'} .= $_; }
   close( FILE );
}

# These are both lazy loading functions.  If we don't have summary and message
# information already loaded, then go out and grab it from the database and
# return that information to the user.

sub summary {
   my $self = shift;

   if( ! $self->{'summary'} ) { $self->load(); }
   return $self->{'summary'};
}

sub message {
   my $self = shift;

   if( ! $self->{'message'} ) { $self->load(); }
   return $self->{'message'};
}


# This returns true or false depending on if there is a problem with the 
# service.  This allows you to define some smarts into what a problem is.

sub has_problem {
   my $self = shift;

   if( $self->color() eq "red" ) {
      return 1; 
   } else {
      return 0;
   }
}


# Display summary.  Does both text and html, doesn't make any calls to 
# sub-objects or other helper objects, just spits out the data that it has.
#
# brief:      Just show the color of the service (or a colored ball in html)
# standard:   Shows a line stating the name/color/updated/summary information
# full:       Shows standard information plus any messages the service has

sub display {
   my( $self, $type, $view ) = @_;

   if( $type eq "text" ) { return $self->display_text( $view ); }
   if( $type eq "html" ) { return $self->display_html( $view ); }
   if( $type eq "wml" ) { return $self->display_wml( $view ); }
}

sub display_text {
   my( $self, $format ) = @_;
   
   if( $format eq "really_brief" ) {
      if( $self->color() eq "green" )  { print ".  "; }
      if( $self->color() eq "purple" ) { print "?  "; }
      if( $self->color() eq "yellow" ) { print "Y  "; }
      if( $self->color() eq "red" )    { print "R  "; }
      if( $self->color() eq "blue" )   { print "B  "; }
      if( $self->color() eq "clear" )  { print "o  "; }
   } elsif( $format eq "brief" ) {
      print $self->color(), "\n";
   } elsif( $format eq "standard_table" ) {
      my( $d1, $m1, $y1 ) = (localtime( $self->rtime()))[3,4,5];
      my( $d2, $m2, $y2 ) = (localtime())[3,4,5];
      my $name = substr( $self->name(), 0, 7 );
      my $color = $self->color();
      my $summary = $self->summary();
      my $firstline = 1;

      print $name, " "x(8-length($name));
      print $color, " "x(9-length($color));

      if( $d1 == $d2 && $m1 == $m2 && $y1 == $y2 ) {
	 $date = POSIX::strftime( $main::TIMEFMTNOSEC, localtime($self->rtime()) );
      } else {
	 $date = POSIX::strftime( $main::DATEFMT, localtime($self->rtime()) );
      }
      print $date, " "x(11-length($date));

      do {
	 print " "x30 if ! $firstline++;
	 print substr( $summary, 0, 50 ), "\n";
	 $summary = substr( $summary, 51 );
      } while( length( $summary > 50 ) );
   } elsif( $format eq "standard" ) {
      print "Name    Color    Updated    Summary\n";
      print "------- -------- ---------- ", "-"x50, "\n";
      $self->display_text( "standard_table" );
   } elsif( $format eq "full" ) {
      $self->display_text( "standard" );
      print "\nStatus unchanged in " .
           $self->format_duration($self->stime(),$self->rtime) . "\n";
      print "-"x78, "\n";
      print $self->message(), "\n";
   }
}

sub display_html {
   my( $self, $format ) = @_;
   my $host  = $self->host;
   my $name  = $self->name();
   my $color = $self->color();
   my( $d1, $m1, $y1 ) = (localtime( $self->rtime()))[3,4,5];
   my( $d2, $m2, $y2 ) = (localtime())[3,4,5];

   if( $format eq "brief" ) {
      print "<a href=\"!!WWWSPONG!!/service/$host/$name\">\n";

      if( $main::WWW_USE_IMAGES == 1 ) {
	 print "<img src=\"!!WWWGIFS!!/$color.gif\" alt=$color border=0>";
      } else {
	 print "<table border=0 cellspacing=0 cellpadding=0><tr>";
	 print "<td width=20 bgcolor=\"" . $main::WWW_COLOR{$color} ;
	 print "\"><font color=\"" . $main::WWW_COLOR{$color} . "\">";
	 print "___</font></td></tr></table>\n";
      }
      print "</a>";
   } elsif( $format eq "standard_table" ) {
      print "<tr><td align=left valign=top nowrap>\n"; 
      print "<a href=\"!!WWWSPONG!!/service/$host/$name\">$name</a></td>\n";
      print "<td align=center valign=top>\n"; 

      if( $main::WWW_USE_IMAGES == 1 ) {
	 print "<a href=\"!!WWWSPONG!!/service/$host/$name\">\n";
	 print "<img src=\"!!WWWGIFS!!/$color.gif\" alt=$color border=0></a>";
      } else {
	 print "<table border=0 cellspacing=0 cellpadding=0><tr>";
	 print "<td width=20 bgcolor=\"" . $main::WWW_COLOR{$color} . "\">";
	 print "<a href=\"!!WWWSPONG!!/service/$host/$name\">";
	 print "<font color=\"" . $main::WWW_COLOR{$color} . "\">___</font>";
	 print "</a></td></tr></table>\n";
      }

      print "</td>\n"; 
      print "<td align=center valign=top nowrap>";

      if( $d1 == $d2 && $m1 == $m2 && $y1 == $y2 ) {
	 print POSIX::strftime( $main::TIMEFMTNOSEC, localtime($self->rtime()) ), "  ";
      } else {
	 print POSIX::strftime( $main::DATEFMT, localtime($self->rtime()) ), "  ";
      }

      print "</td>\n";
      print "<td align=left valign=top>", $self->summary(), "</td></tr>\n";
   } elsif( $format eq "standard" ) {
      print "<table width=100% border=1 cellspacing=2 cellpadding=2><tr>\n";
      print "<td width=60 align=center><b>Service</b></td>\n";
      print "<td width=1%>&nbsp</td>\n";
      print "<td width=60 align=center><b>Updated</b></td>\n";
      print "<td width=100% align=center><b>Summary</b></td></tr>\n";

      $self->display_html( "standard_table" );

      print "</table>\n";
   } elsif( $format eq "full" ) {
      print "<font size=+2><b>", $self->host(), "/", $self->name();
      print "</b></font>\n";

      # This breaks the generic interactive vs non-interactive model
      $self->add_action_bar();

      print "<table width=100% cellspacing=0 cellpadding=0 border=0>";
      print "<tr><td bgcolor=\"" . $main::WWW_COLOR{$color} . "\">&nbsp;</td>";
      print "</tr></table><p>";

      print "<b>Updated:</b> ";
      print POSIX::strftime( "$main::TIMEFMTNOSEC, $main::DATEFMT", localtime($self->rtime()) );

      print "<br><b>Duration:</b> ";
      print $self->format_duration($self->stime(),$self->rtime());

      print "<br><b>Summary:</b> ", $self->summary(), "<br>\n";
      print "<hr noshade><pre>", $self->message(), "</pre>\n";
   }
}

sub display_wml {
   my( $self, $format ) = @_;
   my( $cnt ) = 0;

   if ( $format eq "full" ) {
      print "<br/><b>",$self->name(),"</b> ",$self->color(),"\n";
      print "<br/>Updated: ",
             POSIX::strftime($main::DATETIMEFMT,localtime($self->rtime())),"\n";
      print "<br/>",wml_esacpe($self->summary()),"\n";
      print "<br/><br/>\n";
      foreach $line ( split /\r?\n/s,$self->message() ) {
         print "<br/>",wml_escape($line),"\n";
         if ( ++$cnt >= 15 ) { last; }
      }
   }

}


# Returns a message that can be sent to the person who is on call for this
# machine which indicates the problem with this machine.

sub _problem_message {
   my $self = shift;

   if( $self->has_problem() ) {
      return "Problem: " . $self->name() . ", " . $self->summary();
   } else {
      return "No Problem";
   }
}


sub add_action_bar {
   my( $self ) = shift;
   my $name = $self->host();
   my $service = $self->name();
   my $message = &escape( "Host: $name, " . $self->_problem_message() );

   print "<hr>";
   print "<a href=\"telnet://$name\">Connect to Host</a> || ";
   print "<a href=\"$main::WWWACK/$name/$service\">Acknowledge Problem</a> ";

   if( $main::WWWCONTACT ) {
      print "|| <a href=\"$main::WWWCONTACT?host=$name&message=$message\">";
      print "Contact Help</a>"; 
   }
   
   print "<hr>\n";
}

# Escape non-standard HTML characters
sub escape {
    my($toencode) = @_;
    $toencode=~s/([^ a-zA-Z0-9_\-.])/uc sprintf("%%%02x",ord($1))/eg;
    return $toencode;
}

# Escape non-standard WML characters
sub wml_escape {
    my($toencode) = @_;
    $toencode =~ s/&/&amp;/g;
    $toencode =~ s/</&lt;/g;
    $toencode =~ s/>/&gt;/g;
    $toencode =~ s/\"/&quot;/g;
    $toencode =~ s/\'/&apos;/g;
    $toencode =~ s/\?\?/&shy;/g;
    return $toencode;
}
sub format_duration {
    my($self,$stime,$rtime) = @_;

    my $duration = $rtime - $stime;
    my $units = "seconds";
    if ($duration > 86400.0) { $duration /= 86400.0; $units = "days"; }
    elsif ($duration > 3600.0) { $duration /= 3600.0; $units = "hours"; }
    elsif ($duration > 60.0)   { $duration /= 60.0; $units = "minutes"; }

    return sprintf("%.2f %s", $duration, $units);
}

1;
