/* specter_OPRINT.c
 *
 * specter output target for logging to a file
 *
 * (C) 2000-2001 by Harald Welte <laforge@gnumonks.org>
 *
 *  11 Jun 2004, Michal Kwiatkowski <ruby@joker.linuxstuff.pl>:
 *      Fixed for specter.
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <specter/specter.h>
#include <conffile/conffile.h>
#include "lret.h"


#ifndef OPRINT_DEFAULT
#define OPRINT_DEFAULT	"/var/log/specter.oprint"
#endif

#define HIPQUAD(addr) \
        ((unsigned char *)&addr)[3], \
        ((unsigned char *)&addr)[2], \
        ((unsigned char *)&addr)[1], \
        ((unsigned char *)&addr)[0]


static config_entry_t my_config[] = {
	{ .key = "logfile", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE,
		.u = { .string = OPRINT_DEFAULT } },
};

struct my_data {
	FILE *of;
};

static specter_local_ret_t local_ret[] = {
	{ "*", NULL }
};


static void *init_oprint(config_entry_t *ce)
{
	struct my_data *data;

	if (fill_local_ret(local_ret, 1) == -1)
		return NULL;

	if ((data = malloc(sizeof(struct my_data))) == NULL) {
		specter_log(SPECTER_FATAL, "Couldn't allocate data: %s.\n",
				strerror(errno));
		return NULL;
	}

	/* open logfile */
	if (!strcmp(GET_CE(ce,0)->u.string, "stdout")) {
		data->of = stdout;
	}
	else if (!strcmp(GET_CE(ce,0)->u.string, "stderr")) {
		data->of = stderr;
	}
	else {
		data->of = fopen(GET_CE(ce,0)->u.string, "a");
		if (!data->of) {
			specter_log(SPECTER_FATAL, "Couldn't open \"%s\": %s.\n",
				GET_CE(ce,0)->u.string, strerror(errno));
			return NULL;
		}
	}

	return data;
}

static void fini_oprint(config_entry_t *ce, void *data)
{
	struct my_data *md = data;

	if (md->of != stdout && md->of != stderr) {
		fclose(md->of);
	}

	free(data);
}


static int output_oprint(config_entry_t *ce, void *data)
{
	specter_iret_t *ret;
	struct my_data *md = data;

	fprintf(md->of, "===>PACKET BOUNDARY\n");
	for (ret = local_ret[0].p; ret; ret = ret->next) {
		fprintf(md->of,"%s=", ret->name);
		if (!IS_VALID(ret)) {
			fprintf(md->of, "<invalid>\n");
			continue;
		}
		switch (ret->type) {
			case SPECTER_IRET_STRING:
				fprintf(md->of, "%s\n", (char *) ret->value.ptr);
				break;
			case SPECTER_IRET_BOOL:
			case SPECTER_IRET_INT8:
			case SPECTER_IRET_INT16:
			case SPECTER_IRET_INT32:
				fprintf(md->of, "%d\n", ret->value.i32);
				break;
			case SPECTER_IRET_UINT8:
			case SPECTER_IRET_UINT16:
			case SPECTER_IRET_UINT32:
				fprintf(md->of, "%u\n", ret->value.ui32);
				break;
			case SPECTER_IRET_IPADDR:
				fprintf(md->of, "%u.%u.%u.%u\n",
					HIPQUAD(ret->value.ui32));
				break;
			case SPECTER_IRET_NONE:
				fprintf(md->of, "<none>");
				break;
			default:
				fprintf(md->of, "\n");
				break;
		}
	}

	fflush(md->of);
	return 0;
}

static int sighup_handler(config_entry_t *ce, void *data, int signal)
{
	struct my_data *md = data;

	switch (signal) {
	case SIGHUP:
		specter_log(SPECTER_INFO, "Reopening logfile.\n");

		if (md->of == stderr || md->of == stdout)
			break;

		fclose(md->of);
		md->of = fopen(GET_CE(ce,0)->u.string, "a");
		if (!md->of) {
			specter_log(SPECTER_FATAL, "Couldn't open \"%s\": %s.\n",
				GET_CE(ce,0)->u.string, strerror(errno));
			return -1;
		}
		break;
	default:
		break;
	}

	return 0;
}

static specter_output_t oprint_op = {
	.name = "oprint",
	.ce_base = my_config,
	.ce_num = 1,
	.init = &init_oprint,
	.fini = &fini_oprint,
	.output = &output_oprint,
	.signal = &sighup_handler
};


void _init(void)
{
	if (register_output(&oprint_op, 0) == -1) {
		specter_log(SPECTER_FATAL, "Couldn't register.\n");
		exit(EXIT_FAILURE);
	}
}

