(* Defines the semantic routines associated with reading a policy and
   an MLS file, and the routine used to generate the labeled
   transition system from the results. *)

open Formulas

(* Identifier is the data structure associated with an identifier that
   appears in a policy or an MLS file.  One data structure is used to
   represent every kind of identifier.  An identifier can name a
   class, common, attribute, type, type alias, role, or a user, or any
   combination of these kinds of identifiers with the exception that
   an identifier can only be one of a type, type alias, or an
   attribute. *)

type identifier = {
    name : string;

    (* Fields relevant for a class *)
    (* Permissions of class *)
    mutable class_perms : identifier list option;
    (* Write permissions of class *)
    mutable class_write_perms : identifier list option;
    (* Read permissions of class *)
    mutable class_read_perms : identifier list option;
    (* Inherited common *)
    mutable class_parent : identifier option;

    (* Fields relevant for a common *)
    (* Permissions of common *)
    mutable common_perms : identifier list option;
    (* Write permissions of common *)
    mutable common_write_perms : identifier list option;
    (* Read permissions of common *)
    mutable common_read_perms : identifier list option;
    (* Classes inheriting *)
    mutable common_children : identifier list option;

    (* Fields relevant for an attribute *)
    mutable attr_types : identifier list option;

    (* Fields relevant for a type *)
    mutable type_attrs : identifier list option;

    (* Fields relevant for a type alias *)
    mutable type_alias : identifier option;

    (* Fields relevant for a role *)
    mutable role_types : id_ctxs option;

    (* Fields relevant for a user *)
    mutable user_roles : id_ctxs option;
  }

and id_ctxs = identifier list contexts

and id_trans = identifier list transitions

let get_name id = id.name

(* The identifier table -- all identifiers are interned *)

let id_table : (string, identifier) Hashtbl.t =
  Hashtbl.create 500

let find_identifier name =
  let mk_identifier name =
    {name = name;
     class_perms = None;
     class_write_perms = None;
     class_read_perms = None;
     class_parent = None;
     common_perms = None;
     common_write_perms = None;
     common_read_perms = None;
     common_children = None;
     attr_types = None;
     type_attrs = None;
     type_alias = None;
     role_types = None;
     user_roles = None;
   } in
  try
    Hashtbl.find id_table name
  with Not_found ->
    let id = mk_identifier name in
    Hashtbl.add id_table name id;
    id

(* Sets for internal use *)

let cmp {name = name} {name = name'} =
  compare name name'

(* Identifier sets *)

module IS = Set.Make(
  struct
    type t = identifier
    let compare = cmp
  end)

let is_add_list set list =
  let add s e = IS.add e s in
  List.fold_left add set list

(* String sets *)

module SS = Set.Make(
  struct
    type t = string
    let compare = compare
  end)

let ss_add_list set list =
  let add s e = SS.add e s in
  List.fold_left add set list

(* Statements *)

type statements =
    TypeAllow of id_trans
  | NeverAllow of id_trans
  | RoleAllow of id_trans
  | Constraint of id_trans

(* Constants *)

let process_class_name = "process"
let transition_permission_name = "transition"
let object_role_name = "object_r"
let domain_attribute_name = "domain"
let self_name = "self"

(* Predicates *)

let is_common id =
  match id.common_children with
      None -> false
    | Some _ -> true

let is_class id =
  match id.class_perms with
      None -> false
    | Some _ -> true

let is_type id =
  match id.type_attrs with
      None -> false
    | Some _ -> true

let is_type_alias id =
  match id.type_alias with
      None -> false
    | Some _ -> true

let is_attribute id =
  match id.attr_types with
      None -> false
    | Some _ -> true

let is_role id =
  match id.role_types with
      None -> false
    | Some _ -> true

let is_user id =
  match id.user_roles with
      None -> false
    | Some _ -> true

(* Helpers *)

let opt_ids opt =
  match opt with
    None -> []
  | Some list -> list

(* Global state variables *)

let all_commons = ref IS.empty
let all_classes = ref IS.empty
let all_perms = ref IS.empty
let all_attrs = ref IS.empty
let all_types = ref IS.empty
let all_roles = ref IS.empty
let all_users = ref IS.empty
let all_statements = ref ([] : statements list)

(* Call init to reset module during interactive use *)

let init() =
  Hashtbl.clear(id_table);
  all_commons := IS.empty;
  all_classes := IS.empty;
  all_perms := IS.empty;
  all_attrs := IS.empty;
  all_types := IS.empty;
  all_roles := IS.empty;
  all_users := IS.empty;
  all_statements := []

(* Semantic routines *)

(* Classes and commons *)

let class_def ~id =
  all_classes := IS.add id !all_classes

let common_perms_def ~id ~perms =
  if is_common id then
    failwith ("common " ^ id.name ^ " redefined");
  all_commons := IS.add id !all_commons;
  all_perms := is_add_list !all_perms perms;
  id.common_children <- Some [];
  id.common_perms <- Some perms

let class_perms_def ~id ~parent ~perms =
  if is_class id then
    failwith ("class " ^ id.name ^ " redefined");
  all_perms := is_add_list !all_perms perms;
  id.class_perms <- Some perms;
  id.class_parent <- parent;
  match parent with
    None -> ()
  | Some parent ->			(* Add class to common's children *)
      match parent.common_children with
	None -> failwith ("class " ^ id.name ^ " cannot find " ^ parent.name)
      | Some kids ->  parent.common_children <- Some (id :: kids)

(* MLS classes and commons *)

(* Is the named identifier in the list? *)
let rec has_name name list =
  match list with
    [] -> false
  | id :: list ->
      if name = id.name then
	true
      else
	has_name name list

(* Collect perms with dir in mls flow spec. *)
let find_name_in_flow dir flow =
  let rec loop perms flow =
    match flow with
      [] -> perms
    | (perm, dirs) :: flow ->
	if has_name dir dirs then
	  loop (perm :: perms) flow
	else
	  loop perms flow in
  loop [] flow

(* Not currently used. *)
let opt_parent_perms id =
  match id.class_parent with
    None -> []
  | Some common -> opt_ids common.common_perms

let mls_class_perms_def ~id ~flow =
  if not (is_class id) then
    failwith ("mls declaration for non-class " ^ id.name);
  if id.class_write_perms = None then begin
    let perms = is_add_list IS.empty (opt_ids id.class_perms) in
    let writes = is_add_list IS.empty (find_name_in_flow "write" flow) in
    let reads = is_add_list IS.empty (find_name_in_flow "read" flow) in
    id.class_write_perms <- Some (IS.elements (IS.inter perms writes));
    id.class_read_perms <- Some (IS.elements (IS.inter perms reads))
  end
  else
    failwith ("mls class " ^ id.name ^ " redefined")

let mls_common_perms_def ~id ~flow =
  if not (is_common id) then
    failwith ("mls declaration for non-common " ^ id.name);
  if id.common_write_perms = None then begin
    let perms = is_add_list IS.empty (opt_ids id.common_perms) in
    let writes = is_add_list IS.empty (find_name_in_flow "write" flow) in
    let reads = is_add_list IS.empty (find_name_in_flow "read" flow) in
    id.common_write_perms <- Some (IS.elements (IS.inter perms writes));
    id.common_read_perms <- Some (IS.elements (IS.inter perms reads))
  end
  else
    failwith ("mls common " ^ id.name ^ " redefined")

(* Attributes *)

let attribute_def ~id =
  if id.name = self_name then
    failwith "self cannot be an attribute";
  if is_type id then			(* make sure attribute *)
    failwith ("identifier " ^ id.name	(* is not a type *)
	      ^ " cannot be both an attribute and a type");
  all_attrs := IS.add id !all_attrs;
  id.attr_types <- Some []

(* Types *)

let type_attributes_def ~id ~attrs =
  if id.name = self_name then
    failwith "self cannot be a type";
  if is_attribute id then		(* make sure type *)
    failwith ("identifier " ^ id.name	(* is not an attribute *)
	      ^ " cannot be both a type and an attribute");
  if is_type_alias id then		(* make sure type *)
    failwith ("identifier " ^ id.name	(* is not a type alias *)
	      ^ " cannot be both a type and a type alias");
  all_types := IS.add id !all_types;
  let attr_def attr =
    match attr.attr_types with
      None -> failwith ("type " ^ id.name ^ " cannot find " ^ attr.name)
    | Some types -> attr.attr_types <- Some (id :: types) in
  List.iter attr_def attrs;
  match id.type_attrs with
    None -> id.type_attrs <- Some attrs;
  | Some prev_attrs -> id.type_attrs <- Some (attrs @ prev_attrs)

(* Type Aliases *)

let extract_aliases id aliases =
  match aliases with
    Ctx_types l -> l
  | _ -> failwith ("alias names too complex for type " ^ id.name)

let a_type_alias_def id alias =
  if is_attribute alias then		(* make sure alias *)
    failwith ("identifier " ^ alias.name (* is not an attribute *)
	      ^ " cannot be both a type alias and an attribute");
  if is_type alias then			(* make sure alias *)
    failwith ("identifier " ^ alias.name (* is not a type *)
	      ^ " cannot be both a type and a type alias");
  match alias.type_alias with
    None -> alias.type_alias <- Some id;
  | Some thing ->
      if thing != id then
	failwith ("cannot use " ^ alias.name ^ " as alias for both "
		  ^ id.name ^ " and " ^ thing.name)

let type_alias_def ~id ~aliases =
  if not (is_type id) then
    failwith ("cannot alias identifier " ^ id.name ^ ": not a type");
  List.iter (a_type_alias_def id) (extract_aliases id aliases)

(* Type allow statements *)

(* Is there only a self reference? *)
let just_self trans =
  match trans with
    Next_types [id] -> self_name = id.name
  | _ -> false

(* Is self explicitly mentioned in a next type position? *)
let rec has_self trans =
  match trans with
    Tran_true -> false
  | Tran_false -> false
  | Tran_classes _ -> false
  | Tran_permissions _  -> false
  | Tran_types _ -> false
  | Next_types list -> has_name self_name list
  | Same_types -> false
  | Tran_roles _ -> false
  | Next_roles _ -> false
  | Same_roles -> false
  | Tran_users _ -> false
  | Next_users _ -> false
  | Same_users -> false
  | Tran_not trans -> has_self trans
  | Tran_and (arg1, arg2) ->
      has_self arg1 || has_self arg2
  | Tran_or (arg1, arg2) ->
      has_self arg1 || has_self arg2
  | Tran_imply (arg1, arg2) ->
      has_self arg1 || has_self arg2
  | Tran_iff (arg1, arg2) ->
      has_self arg1 || has_self arg2

let rec expunge_self trans =
  match trans with
    Tran_true -> trans
  | Tran_false -> trans
  | Tran_classes _ -> trans
  | Tran_permissions _  -> trans
  | Tran_types _ -> trans
  | Next_types list ->
      if has_name self_name list then
	Next_types (List.filter (fun id -> self_name <> id.name) list)
      else
	trans
  | Same_types -> trans
  | Tran_roles _ -> trans
  | Next_roles _ -> trans
  | Same_roles -> trans
  | Tran_users _ -> trans
  | Next_users _ -> trans
  | Same_users -> trans
  | Tran_not trans -> Tran_not (expunge_self trans)
  | Tran_and (arg1, arg2) ->
      Tran_and (expunge_self arg1, expunge_self arg2)
  | Tran_or (arg1, arg2) ->
      Tran_or (expunge_self arg1, expunge_self arg2)
  | Tran_imply (arg1, arg2) ->
      Tran_imply (expunge_self arg1, expunge_self arg2)
  | Tran_iff (arg1, arg2) ->
      Tran_iff (expunge_self arg1, expunge_self arg2)

let handle_self sources targets classes perms =
  let cp = Tran_and (classes, perms) in
  if just_self targets then
    Tran_and (cp, Tran_and (sources, Same_types))
  else if has_self targets then
    Tran_and (cp, Tran_or (Tran_and (sources, expunge_self targets),
			   Tran_and (sources, Same_types)))
  else
    Tran_and (cp, Tran_and (sources, targets))

let allow_def ~sources ~targets ~classes ~perms =
  let trans = handle_self sources targets classes perms in
  all_statements := TypeAllow trans :: !all_statements

let never_allow_def ~sources ~targets ~classes ~perms =
  let trans = handle_self sources targets classes perms in
  all_statements := NeverAllow trans :: !all_statements

(* Role types *)

let role_types_def ~id ~types =
  all_roles := IS.add id !all_roles;
  match id.role_types with
    None -> id.role_types <- Some types
  | Some prev_types -> id.role_types <- Some (Ctx_or (types, prev_types))

(* Role allow statements *)

let role_allow_def ~sources ~targets =
  all_statements := RoleAllow(Tran_and(sources, targets)) :: !all_statements

(* Constraints *)

let constrant_def ~classes ~perms ~cexpr =
  let trans = Tran_imply (Tran_and (classes, perms), cexpr) in
  all_statements := Constraint trans :: !all_statements

(* User roles *)

let user_roles_def ~id ~roles =
  all_users := IS.add id !all_users;
  match id.user_roles with
    None -> id.user_roles <- Some roles
  | Some prev_roles -> id.user_roles <- Some (Ctx_or (roles, prev_roles))

(* Post parsing rountines *)

(* Declare object_role to be associated with non-process types. *)
let object_role_def () =
  let object_role = find_identifier object_role_name in
  let domain = Ctx_types [find_identifier domain_attribute_name] in
  role_types_def object_role (Ctx_not domain)

(* Converts identifier lists to string lists while eliminating duplicates. *)
let expand_ids list =
  let add set id =
    SS.add id.name set in
  SS.elements (List.fold_left add SS.empty list)

let alias_deref alias =
  match alias.type_alias with
    Some id -> id;
  | None -> failwith ("deref of non-alias identifier " ^ alias.name)

(* Replaces attributes and aliases with their types *)
let expand_types list =
  let rec loop set list =
    let expand_type set id =
      if is_type id then
	SS.add id.name set
      else if is_type_alias id then
	SS.add (alias_deref id).name set
      else if is_attribute id then
	match id.attr_types with
	  Some types -> loop set types
	| None -> failwith ("expand_types: bad attribute " ^ id.name)
      else
	failwith ("expand_types: bad type " ^ id.name) in
    List.fold_left expand_type set list in
  SS.elements (loop SS.empty list)

(* Expand attributes with their types *)
let rec expand_ctx ctx =
  match ctx with
    Ctx_true -> Ctx_true
  | Ctx_false -> Ctx_false
  | Ctx_types set -> Ctx_types (expand_types set)
  | Ctx_roles set -> Ctx_roles (expand_ids set)
  | Ctx_users set -> Ctx_users (expand_ids set)
  | Ctx_not arg -> Ctx_not (expand_ctx arg)
  | Ctx_and (arg1, arg2) ->
      Ctx_and (expand_ctx arg1, expand_ctx arg2)
  | Ctx_or (arg1, arg2) ->
      Ctx_or (expand_ctx arg1, expand_ctx arg2)
  | Ctx_imply (arg1, arg2) ->
      Ctx_imply (expand_ctx arg1, expand_ctx arg2)
  | Ctx_iff (arg1, arg2) ->
      Ctx_imply (expand_ctx arg1, expand_ctx arg2)

(* Initial state *)

let emit_role_types () =
  let emit_role id ctx =
    match id.role_types with
      None -> failwith ("bad role: " ^ id.name)
    | Some types -> Ctx_or (Ctx_and (Ctx_roles [id.name],
				     expand_ctx types),
			    ctx) in
  IS.fold emit_role !all_roles Ctx_false

let emit_user_roles () =
  let emit_user id ctx =
    match id.user_roles with
      None -> failwith ("bad user: " ^ id.name)
    | Some roles -> Ctx_or (Ctx_and (Ctx_users [id.name],
				     expand_ctx roles),
			    ctx) in
  IS.fold emit_user !all_users (Ctx_roles [object_role_name])

let emit_init () =
  Ctx_and (emit_role_types (), emit_user_roles ())

(* Convert a context into a current state transition. *)
let rec ctx2tran ctx =
  match ctx with
    Ctx_true -> Tran_true
  | Ctx_false -> Tran_false
  | Ctx_types set -> Tran_types set
  | Ctx_roles set -> Tran_roles set
  | Ctx_users set -> Tran_users set
  | Ctx_not arg -> Tran_not (ctx2tran arg)
  | Ctx_and (arg1, arg2) ->
      Tran_and (ctx2tran arg1, ctx2tran arg2)
  | Ctx_or (arg1, arg2) ->
      Tran_or (ctx2tran arg1, ctx2tran arg2)
  | Ctx_imply (arg1, arg2) ->
      Tran_imply (ctx2tran arg1, ctx2tran arg2)
  | Ctx_iff (arg1, arg2) ->
      Tran_iff (ctx2tran arg1, ctx2tran arg2)

(* Convert a context into a next state transition. *)
let rec ctx2tran' ctx =
  match ctx with
    Ctx_true -> Tran_true
  | Ctx_false -> Tran_false
  | Ctx_types set -> Next_types set
  | Ctx_roles set -> Next_roles set
  | Ctx_users set -> Next_users set
  | Ctx_not arg -> Tran_not (ctx2tran' arg)
  | Ctx_and (arg1, arg2) ->
      Tran_and (ctx2tran' arg1, ctx2tran' arg2)
  | Ctx_or (arg1, arg2) ->
      Tran_or (ctx2tran' arg1, ctx2tran' arg2)
  | Ctx_imply (arg1, arg2) ->
      Tran_imply (ctx2tran' arg1, ctx2tran' arg2)
  | Ctx_iff (arg1, arg2) ->
      Tran_iff (ctx2tran' arg1, ctx2tran' arg2)

(* Transition relation *)

(* Expand attributes with their types *)
let rec expand_tran tran =
  match tran with
    Tran_true -> Tran_true
  | Tran_false -> Tran_false
  | Tran_classes set -> Tran_classes (expand_ids set)
  | Tran_permissions set -> Tran_permissions (expand_ids set)
  | Tran_types set -> Tran_types (expand_types set)
  | Next_types set -> Next_types (expand_types set)
  | Same_types -> Same_types
  | Tran_roles set -> Tran_roles (expand_ids set)
  | Next_roles set -> Next_roles (expand_ids set)
  | Same_roles -> Same_roles
  | Tran_users set -> Tran_users (expand_ids set)
  | Next_users set -> Next_users (expand_ids set)
  | Same_users -> Same_users
  | Tran_not arg -> Tran_not (expand_tran arg)
  | Tran_and (arg1, arg2) ->
      Tran_and (expand_tran arg1, expand_tran arg2)
  | Tran_or (arg1, arg2) ->
      Tran_or (expand_tran arg1, expand_tran arg2)
  | Tran_imply (arg1, arg2) ->
      Tran_imply (expand_tran arg1, expand_tran arg2)
  | Tran_iff (arg1, arg2) ->
      Tran_imply (expand_tran arg1, expand_tran arg2)

let emit_type_allows () =
  let emit_type_allow trans statement =
    match statement with
      TypeAllow allow ->
	Tran_or (expand_tran allow, trans)
    | _ -> trans in
  List.fold_left emit_type_allow Tran_false !all_statements

let emit_role_allows () =
  let init =
    Tran_not (Tran_and (Tran_classes [process_class_name],
			Tran_permissions [transition_permission_name])) in
  let emit_role_allow trans statement =
    match statement with
      RoleAllow allow ->
	Tran_or (expand_tran allow, trans)
    | _ -> trans in
  List.fold_left emit_role_allow init !all_statements

let emit_constraints () =
  let emit_constraint trans statement =
    match statement with
      Constraint allow ->
	Tran_and (expand_tran allow, trans)
    | _ -> trans in
  List.fold_left emit_constraint Tran_true !all_statements

let emit_allows_and_constraints () =
  Tran_and (emit_type_allows(),
	    Tran_and (emit_role_allows (),
		      emit_constraints ()))

(* Reverse the direction of a transition. *)
let rec rev_tran tran =
  match tran with
    Tran_true -> tran
  | Tran_false -> tran
  | Tran_classes set -> tran
  | Tran_permissions set -> tran
  | Tran_types set -> Next_types set
  | Next_types set -> Tran_types set
  | Same_types -> tran
  | Tran_roles set -> Next_roles set
  | Next_roles set -> Tran_roles set
  | Same_roles -> tran
  | Tran_users set -> Next_users set
  | Next_users set -> Tran_users set
  | Same_users -> tran
  | Tran_not arg -> Tran_not (rev_tran arg)
  | Tran_and (arg1, arg2) ->
      Tran_and (rev_tran arg1, rev_tran arg2)
  | Tran_or (arg1, arg2) ->
      Tran_or (rev_tran arg1, rev_tran arg2)
  | Tran_imply (arg1, arg2) ->
      Tran_imply (rev_tran arg1, rev_tran arg2)
  | Tran_iff (arg1, arg2) ->
      Tran_imply (rev_tran arg1, rev_tran arg2)

(* Class permission associations *)

type perm_spec =
    All_perms
  | Write_perms
  | Read_perms

let common_perms_from_spec perm_spec id =
  match perm_spec with
    All_perms -> opt_ids id.common_perms
  | Write_perms -> opt_ids id.common_write_perms
  | Read_perms -> opt_ids id.common_read_perms

let class_perms_from_spec perm_spec id =
  match perm_spec with
    All_perms -> opt_ids id.class_perms
  | Write_perms -> opt_ids id.class_write_perms
  | Read_perms -> opt_ids id.class_read_perms

let emit_class_permissions perm_spec =
  let emit_common id tran =
    let perms = common_perms_from_spec perm_spec id in
    let classes = opt_ids id.common_children in
    if perms = [] || classes = [] then
      tran
    else
      Tran_or (Tran_and (Tran_classes (expand_ids classes),
			 Tran_permissions (expand_ids perms)),
	       tran) in
  let emit_class id tran =
    let perms = class_perms_from_spec perm_spec id in
    if perms = [] then
      tran
    else
      Tran_or (Tran_and (Tran_classes [id.name],
			 Tran_permissions (expand_ids perms)),
	       tran) in
  let commons = IS.fold emit_common !all_commons Tran_false in
  IS.fold emit_class !all_classes commons

(* When auth is true, generate an authorization transition relation,
   otherwise generate an information flow transition relation. *)

let emit_transition auth initial =
  let user_role = Tran_and (ctx2tran initial, ctx2tran' initial) in
  let permits = emit_allows_and_constraints () in
  if auth then
    let all_perms = emit_class_permissions All_perms in
    Tran_and (user_role, Tran_and(all_perms, permits))
  else
    let write_like = emit_class_permissions Write_perms in
    let read_like = emit_class_permissions Read_perms in
    Tran_and (user_role,
	      Tran_or (Tran_and (write_like,
				 permits),
		       Tran_and (read_like,
				 rev_tran permits)))

(* Specifications *)

(* Compute the set given by a source type. *)
let eval_src_types all_types tran =
  let rec eval tran =
    match tran with
      Tran_true -> all_types
    | Tran_types set -> ss_add_list SS.empty set
    | Tran_not arg -> SS.diff all_types (eval arg)
    | Tran_and (arg1, arg2) ->
	SS.inter (eval arg1) (eval arg2)
    | Tran_or (arg1, arg2) ->
	SS.union (eval arg1) (eval arg2)
    | Tran_imply (arg1, arg2) ->
	SS.union (SS.diff all_types (eval arg1)) (eval arg2)
    | Tran_iff (arg1, arg2) ->
	let arg1 = eval arg1 in
	let arg2 = eval arg2 in
	let both = SS.inter arg1 arg2 in
	let neither = SS.diff all_types (SS.union arg1 arg2) in
	SS.union both neither
    | _ -> SS.empty in
  eval tran

let emit_specifications types =
  let types = ss_add_list SS.empty types in

  let emit_self_specs specs cp src =
    let emit_self_spec specs name =
      Tran_imply (Tran_and (cp, Tran_types [name]),
		  Tran_not (Next_types [name])) :: specs in
    let src_types = SS.elements (eval_src_types types src) in
    List.fold_left emit_self_spec specs src_types in

  let emit_specs specs cp src tgt =
    let cp = expand_tran cp in
    let src = expand_tran src in
    let tgt = expand_tran tgt in
    match tgt with
      Same_types -> emit_self_specs specs cp src
    | _ -> Tran_imply (Tran_and (cp, src), Tran_not tgt) :: specs in

  let emit_never_allow specs statement =
    match statement with
      NeverAllow (Tran_and (cp, Tran_or (Tran_and (src, tgt),
					 Tran_and (src', tgt')))) ->
	let specs = emit_specs specs cp src tgt in
	emit_specs specs cp src' tgt'
    | NeverAllow (Tran_and (cp, Tran_and (src, tgt))) ->
	emit_specs specs cp src tgt
    | _ -> specs in

  List.fold_left emit_never_allow [] !all_statements

(* Main routine *)

let slat ~auth =
  object_role_def ();
  let initial = emit_init () in
  let transition = emit_transition auth initial in
  let types = List.map get_name (IS.elements !all_types) in
  let specs = 				(* Generate specifications *)
    if auth then			(* from neverallows only *)
      emit_specifications types		(* when generating an *)
    else				(* authorization transition *)
      [] in				(* relation *)
  {types = types;
   roles = List.map get_name (IS.elements !all_roles);
   users = List.map get_name (IS.elements !all_users);
   classes = List.map get_name (IS.elements !all_classes);
   permissions = List.map get_name (IS.elements !all_perms);
   initial = initial;
   transition = transition;
   specifications = specs;
 }
