/* # skkinput (Simple Kana-Kanji Input)
 * parseStr.c
 * This file is part of skkinput.
 * Copyright (C) 1997
 * Takashi SAKAMOTO (sakamoto@yajima.kuis.kyoto-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/X.h>
#include <X11/Xlib.h>

#include "commondef.h"
#include "skkkey.h"
#include "kanji.h"

/*
 * parseStr.c ǤζɽŪ
 */
struct keyToMask {
  unsigned char *string ;
  long mask ;
} ;

/*
 * ޥ
 */
#define forceLowerChr(chara)	\
(( (chara) >= 'A' && (chara) <= 'Z' )? ((chara) - 'A' + 'a') : (chara))

/*
 * ץȥ
 */
#if 0
static void dump_buffer( unsigned short *buffer, int size ) ;
#endif

/*
 * ʸɤФؿ
 */
struct myChar *skip_empty_symbols( struct myChar *ptr )
{
  while( !IS_END_OF_STRING( *ptr ) ){
    if( !IS_ASCII_EQUAL( *ptr, 0x20 ) && 
	!IS_ASCII_EQUAL( *ptr, '\t' ) &&
	!IS_ASCII_EQUAL( *ptr, '\n' ) &&
	!IS_ASCII_EQUAL( *ptr, '\r' ) )
      break ;
    ptr ++ ;
  }
  return ptr ;
}

/*
 * Ϳ줿ʸƬʬ˰פʸǤ뤫ɤȽꤹ
 * ؿʸʸζ̤Ϥʤ
 */
static int isPrefixString
( unsigned char *masterstring, unsigned char *prefixstring )
{
  while( *prefixstring != '\0' ){
    /* ʸʸζ̤ʤӤưۤʤ硣*/
    if( forceLowerChr( *masterstring ) != *prefixstring )
      return False ;
    /* ʸǧ*/
    masterstring ++ ;
    prefixstring ++ ;
  }
  return True ;
}

/*
 * X Resource Υ᤹ؿ
 *---
 * δؿ᤹ΤϰʸǤ롣ʸʤ󤫤
 * Ƥ̣Ϥʤ
 */
int parseXrLikeKeyString
( unsigned char *string, KeySym *r_keysym, 
  long *r_modifiers, long *r_checkModifiers )
{
  static struct keyToMask key_masks[] = {
    { "shift",   ShiftMask   }, { "lock", LockMask },
    { "cntrl", ControlMask }, { "mod1", Mod1Mask },
    { NULL,      0L          },
  } ;
  struct keyToMask *kptr ;

  /* ʤΥǤʤ餫 mask ¸ߤ硣*/
  if( *string != '<' ){
    for( kptr = key_masks ; kptr->string != NULL ; kptr ++ )
      if( isPrefixString( string, kptr->string ) ){
	*r_modifiers = *r_checkModifiers = kptr->mask ;
	string += strlen( kptr->string ) ;
	break ;
      }
    /* ⤷mask ְäƤ顣*/
    if( kptr->string == NULL )
      return False ;
    /*  "<" ޤɤФ*/
    while( *string != '<' ){
      if( *string == '\0' || ( *string != ' ' && *string != '\t' ) )
	return False ;
      string ++ ;
    }
  }
  /*  <key> Ƚ񤫤뤳ȤˤʤäƤ뤱ɡɤ褦ʡ
     ȸ٤ʡ*/
  if( !isPrefixString( string, "<key>" ) )
    return False ;
  /* Ĥʸ keysym Ȼפ*/
  *r_keysym = XStringToKeysym( string + 5 ) ;
  return True ;
}

int parseXrLikeKeyStrings
( unsigned char *string, struct XrLikeKey *keytbl )
{
  unsigned char *buffer, *ptr, *tmppoint ;
  int length = strlen( string ), num ;

  buffer = malloc( sizeof( unsigned char ) * ( length + 1 ) ) ;
  if( buffer == NULL )
    return 0 ;
  strcpy( buffer, string ) ;

  tmppoint = buffer ;
  num = 0 ;
  for( ptr = buffer ; ; ptr ++ ){
    switch( *ptr ){
    case '\\' :
      ptr ++ ;
      if( *ptr == '\0' )
	goto exit_loop ;
      break ;
    case ',' :
      /* Ƚλ֤Ʊä顢ʸʤΤ̵뤹롣*/
      if( tmppoint == ptr ){
	tmppoint ++ ;
	break ;
      }
      /* Ͽ褬Τʤ顢ºݤ˲᤹롣*/
      if( keytbl != NULL ){
	/* ǰöѹꤹΤǡbuffer  copy ȤäƤ
	   櫓Ǥ롣⤷ȡwritable Ǥʤʸ뤫
	   ⤷ʤ顣ˤ˸ƤФ뤳ȤϤʤΤǡmalloc 
	   free 򷫤֤ƤפȦǤ롣*/
	*ptr = '\0' ;
	if( parseXrLikeKeyString
	    ( tmppoint, &keytbl->keysym, &keytbl->modifiers,
	      &keytbl->checkModifiers ) ){
	  /* ̵˲褿顢ϿƼʸؤȸ*/
	  keytbl ++ ;
	  num ++ ;
	}
	/* separator ϥޤʤ櫓Ǥ롣*/
	*ptr = ',' ;
      } else {
	num ++ ;
      }
      tmppoint = ptr + 1 ;
      break ;
    case '\0' :
      /* Ƚλ֤Ʊä顢ʸʤΤ̵뤹롣*/
      if( tmppoint != ptr ){
	if( keytbl != NULL ){
	  if( parseXrLikeKeyString
	      ( tmppoint, &keytbl->keysym, &keytbl->modifiers,
		&keytbl->checkModifiers ) ){
	    /* ̵˲褿顢ϿƼʸؤȸ*/
	    keytbl ++ ;
	    num ++ ;
	  }
	} else {
	  num ++ ;
	}
      }
      goto exit_loop ;
    default :
      break ;
    }
  }
exit_loop:
  free( buffer ) ;
  return num ;
}

/*
 * ݥ󥿤ǻؤ줿ɥ쥹³ʸ 8 ʿοͤȤߤƲ
 * ᤹ؿ 
 *----
 * 127 Ķʤ褦˸롣
 */
int parseOctNumber
( struct myChar *string, struct myChar *dest )
{
  int counter = 0, num = 0 ;
  while( IS_ASCII_CHARA( *string ) && 
	 string->chara >= '0' && string->chara < '8' ){
    num = ( num << 3 ) | ( string->chara - '0' ) ;
    /* ʿͤʤ褦ˤʤäƤ롣ǤΤϡchar 
        */
    if( num > 127 ){
      num = num >> 3 ;
      break ;
    }
    counter ++ ;
    string ++ ;
  }
  if( dest != NULL ){
    dest->charset = CHARSET_ASCII ;
    dest->chara   = num ;
  }
  return counter ;
}

/*
 * ݥ󥿤ǻؤ줿ɥ쥹³ʸ 16 ʿοͤȤߤƲ
 * ᤹ؿ 
 *----
 * 127 Ķʤ褦˸롣
 */
int parseHexNumber
( struct myChar *string, struct myChar *dest )
{
  int counter = 0, num = 0 ;
  while( 1 ){
    if( !IS_ASCII_CHARA( *string ) )
      break ;
    /* 16ʤλȤƲǽʤΤϡ0  9  a  f ޤǡ*/
    if( string->chara >= '0' && string->chara <= '9' ){
      num = ( num << 4 ) | ( string->chara - '0' ) ;
    } else if( string->chara >= 'a' && string->chara <= 'f' ){
      num = ( num << 4 ) | ( string->chara - 'a' + 10 ) ;
    } else if( string->chara >= 'A' && string->chara <= 'F' ){
      num = ( num << 4 ) | ( string->chara - 'A' + 10 ) ;
    } else {
      break ;
    }
    /* ʿͤʤ褦ˤʤäƤ롣ǤΤϡchar 
        */
    if( num > 127 ){
      num = num >> 4 ;
      break ;
    }
    counter ++ ;
    string ++ ;
  }
  if( dest != NULL ){
    dest->charset = CHARSET_ASCII ;
    dest->chara   = num ;
  }
  return counter ;
}

/*
 * backslash θ³ü쵭ڤΤѤؿ
 */
int parseBackslashControl
( struct myChar *string, struct myChar *dest )
{
  static unsigned char *cntrlKeys = "@abcdefghijklmnopqrstuvwxyz[\\]^_" ;
  unsigned char *ptr ;
  static unsigned char *miscKeys[] = {
    "Home",	"Left",		"Up",		"Right",
    "Down",	"PageUp",	"PageDown",	"End",
    "Begin",	"Insert",	"Clear",	"Help",
    "Break",	"Henkan_Mode",	"Muhenkan",	"F1",
    "F2",	"F3",		"F4",		"F5",
    "F6",	"F7",		"F8",		"F9",
    "F10",	"F11",		"F12",		NULL,
  } ;
  struct myChar lchara ;
  int i, len ;

  /* ³ʸ ASCII Ǥʤܡ*/
  if( !IS_ASCII_CHARA( *string ) || IS_END_OF_STRING( *string ) )
    return ERR ;

  if( string->chara >= '0' && string->chara < '8' ){
    /* ³ʸ 8ʿˤľܥɻꡣ*/
    return parseOctNumber( string, dest ) ;
  } else if( string->chara == 'x' ){
    /* ³ʸ 16ʿˤľܥɻꡣ*/
    string ++ ;
    return parseHexNumber( string, dest ) ;
  }
  /* Ǥʤˤϡġ*/
  if( IS_ASCII_EQUAL( string[ 1 ], '-' ) ){
    if( IS_END_OF_STRING( string[ 2 ] ) ){
      return ERR ;
    }
    switch( string->chara ){
      /* ȥ륭λǤä硣*/
    case 'C' :
      if( IS_ASCII_EQUAL( string[ 2 ], '\\' ) ){
	if( IS_ASCII_EQUAL( string[ 3 ], '\\' ) ){
	  /* backslash ξ³ʤܤ͡*/
	  if( dest != NULL ){
	    dest->charset = CHARSET_ASCII ;
	    dest->chara   = 28 ;
	  }
	  return 4 ;
	} else {
	  /* ⤦١Ϥ롣*/
	  len = parseBackslashControl( string + 3, &lchara ) ;
	  if( len < 0 )
	    return ERR ;
	  /* Υξˤ̤˥ȥդ᤹롣
	   * ƤΥ򸫤ɤ͡ġǤ⤽ޤǤȽŤ
	   * ʤΤ褦ʵ뤫餷ޤ󡣤⤽⤳
	   * 򸫤Τ⡢Insert ȤäαĹʤǤ
	   */ 
	  if( lchara.chara >= CHARA_HOME &&
	      lchara.chara <	NUMBER_OF_KEYS ){
	    lchara.chara = lchara.chara | 1 ;
	  }
	  if( dest != NULL )
	    *dest = lchara ;
	  return ( len + 3 ) ;
	}
      } else {
	/* Control λȤʸƤ뤫ɤĴ٤롣*/
	for( ptr = cntrlKeys ; *ptr != '\0' ; ptr ++ ){
	  if( IS_ASCII_EQUAL( string[ 2 ], *ptr ) ){
	    /* ʸ褿 */
	    if( dest != NULL ){
	      dest->charset = CHARSET_ASCII ;
	      dest->chara   = ptr - cntrlKeys ;
	    }
	    return 3 ;
	  }
	}
      }
      /* Control+Space  C-@ ƱΤȤư*/
      if( IS_ASCII_EQUAL( string[ 2 ], ' ' ) ){
	if( dest != NULL ){
	  dest->charset = CHARSET_ASCII ;
	  dest->chara   = '\0' ;
	}
	return 3 ;
      }
      return ERR ;
      
      /* ESC λǤä硣*/
    case 'M' :
      if( dest != NULL ){
	dest->charset = CHARSET_ASCII ;
	dest->chara   = 0x1b ;
      }
      return 2 ;
    }
  }
  /* ʳʸäɤȽǤ롣*/
  for( i = 0 ; miscKeys[ i ] != NULL ; i ++ ){
    int len = strlen( miscKeys[ i ] ) ;
    if( !myCharCharStrncmp( string, miscKeys[ i ], len ) ){
      if( dest != NULL ){
	dest->charset = CHARSET_ASCII ;
	dest->chara   = CHARA_HOME + ( i << 1 ) ;
      }
      return len ;
    }
  }
  return ERR ;
}

/*
 * emacs-like ʥ᤹ΤѤؿ
 * buffer  NULL 줿ˤɬפȤʤ short 礭
 * 褦ˤʤäƤ롣
 */
int parseKeyString
( struct myChar *string, struct myChar *buffer, int bufsize )
{
  int i, ret, uselen ;

  uselen = 0 ;
  for( i = 0 ; !IS_END_OF_STRING( *string ) ; i ++ ){
    /* ХåեȤڤäƤʤɤå롣*/
    if( buffer != NULL && uselen >= bufsize )
      break ;
    /* Хåå夬褿硩 */
    if( IS_ASCII_EQUAL( *string, '\\' ) ){
      string ++ ;
      /* Ǥʤʸڤ줿ꤹȤѤǤ롣*/
      if( IS_END_OF_STRING( *string ) )
	return 0 ;
      if( IS_ASCII_EQUAL( *string, '\\' ) ){
	if( buffer != NULL )
	  *buffer ++ = *string ;
	uselen ++ ;
	string ++ ;
      } else {
	/* Хå³ʸβ򤹤롣*/
	if( ( ret = parseBackslashControl( string, buffer ) ) <= 0 ){
	  if( buffer != NULL )
	    *buffer ++ = *string ;
	  uselen ++ ;
	  /* äȼԤ鲿̵äΤΤ褦˼ؤȰư
	   * Ƥġ*/
	  string ++ ;
	} else {
	  /* buffer + 0 ˤ parseBackslashControl η̤äƤ
	   * Τǡؤ뤳Ȥˤ롣*/
	  buffer ++ ;
	  uselen ++ ;
	  /* ᤷʬ롣*/
	  string += ret ;
	}
      }
    } else {
      /* ʳʸ褿ν*/
      if( buffer != NULL )
	*buffer ++ = *string ;
      uselen ++ ;
      string ++ ;
    }
  }
  return uselen ;
}

/*
 * ³ double quote ǰϤޤ줿ʸ emacs-lisp ʸ
 * Ȳᤷơη̤ short Ȥ֤ؿresult ˤϡ
 * Υ롣ä饨顼ˤʤ롣
 */
struct myChar *parseString
( struct myChar *string, struct myChar **dest, int *result )
{
  struct myChar *start, *ptr ;
  int len ;

  /* ޤ̤ϲФƤʤꤷƤ*/
  *dest   = NULL ;
  *result = -1 ;
  /* ǽ˶¸ߤƤפʤ褦ɤФ*/
  string = skip_empty_symbols( string ) ;
  /* ʸγϵǤ double quote ̵С顼Ǥ롣*/
  if( !IS_ASCII_EQUAL( *string, 0x22 ) )
    return string ;
  string ++ ;
  start = string ;
  /* ʸȤĴ٤롣*/
  while( !IS_END_OF_STRING( *string ) &&
	 !IS_ASCII_EQUAL( *string, 0x22 ) ){
    /* ⤷ double quote 줿Ȥ׵᤬äˡġ*/
    if( IS_ASCII_EQUAL( *string, '\\' ) ){
      string ++ ;
      /* ʸ󤬽äƤޤä顢ѡ*/
      if( IS_END_OF_STRING( *string ) )
	return string ;
    }
    string ++ ;
  }
  /* ơɤޤǹԤäʡ */
  if( !IS_ASCII_EQUAL( *string, 0x22 ) ){
    return string ;
  }
  /* ʸġ*/
  if( string == start ){
    /* ʸäƤʤȥ֥륯Ȥǽü *
     * Ƥʤ*/
    *dest   = NULL ;
    *result = 0 ;
    string ++ ;
  } else {
    MYCHAR_SET_END_OF_STRING( *string ) ;
    /* 餫ɬפʸĹ׻Ƥ*/
    if( ( len = parseKeyString( start, NULL, 0 ) ) <= 0 ){
      /* Ȥ᤹*/
      MYCHAR_SET_CHARA( *string, 0x22 ) ;
      string ++ ;
      return string ;
    }
    /* ʸĹmallocޤ*/
    ptr = malloc( ( len + 1 ) * sizeof( struct myChar ) ) ;
    if( ptr != NULL ){
      /* ᤷ̤ dest 롣*/
      parseKeyString( start, ptr, len ) ;
      MYCHAR_SET_END_OF_STRING( ptr[ len ] ) ;
      *dest   = ptr ;
      *result = len ;
    }
    /* ѤƤޤäʸȤ᤹*/
    MYCHAR_SET_CHARA( *string, 0x22 ) ;
    string ++ ;
  }
  return string ;
}

/*
 * emacs-lisp ˤ expand-file-name ؿ
 *-----
 * ֺǽ "~" ä home directory Ÿ뵡ǽ
 * Ƥ롣wild-card ʤȤˤбƤʤ
 * δؿƤ̤ skk-local-jisyo-path 뤳Ȥˤʤ롣
 * skk-local-jisyo-path  NULL Ǥʤä顢δؿƤФʤ
 * ȻפƤ¦Ǥ¿ʬåǤʤΤǡǥå
 * 롣
 */
unsigned char *expand_file_name( unsigned char *path )
{
  unsigned char *ptr ;
  unsigned char *home ;
  int len = strlen( path ) ;

  /* "~" Ȥ home directory ɽ뵭椬äƤν*/
  if( *path == '~' ){
    /* HOME Directory 褫롣*/
    home = getenv( "HOME" ) ;
    /* Home directory + ĤΥѥĹ malloc 롣*/
    ptr = malloc( strlen( home ) + len + 1 ) ;
    if( ptr == NULL ){
      /* malloc ԡĤ̿Ūʥ顼ʤΤǷ³ưϤʤ*/
      fprintf( stderr, "Memory fault.\n" ) ;
      exit( 1 ) ;
    }
    /* "~" Ÿơľ*/
    strcpy( ptr, home ) ;
    strcat( ptr, path + 1 ) ;
  } else {
    /* ɬפ̵硣*/
    ptr = ( unsigned char *)malloc( len + 1 ) ;
    strcpy( ptr, path ) ;
  }
  return ptr ;
}

#if 0
int main( int argc, char *argv[] )
{
  unsigned short buffer[ 256 ] ;
  int len ;
  len = parseKeyString( "\\C-u\\M-xabc", buffer, 256 ) ;
  dump_buffer( buffer, len ) ;
  return 0 ;
}

#endif

void dump_sstring( unsigned short *buffer, int size )
{
  int i ;
  printf( "(dump_sstring)" ) ;
  if( size == 0 ){
    printf( "empty string" ) ;
  }
  for( i = 0 ; i < size; i ++ ){
    printf( "%04x, ", buffer[ i ] ) ;
  }
  printf( "\n" ) ;
  return ;
}

static void skip_charSpace( unsigned char **string )
{
  unsigned char *ptr = *string ;

  while( *ptr != '\0'  ){
    if( *ptr != 0x20 && *ptr != '\t' && *ptr != '\n' &&
	*ptr != '\r' )
      break ;
    ptr ++ ;
  }
  *string = ptr ;
  return ;
}

unsigned char *getOneFontsetFromFontsetList( unsigned char **string )
{
  unsigned char *ptr, *retstring = NULL ;
  int length, next_chara_ignore ;

  /* ƬζʸФ*/
  skip_charSpace( string ) ;

  next_chara_ignore = False ;
  for( ptr = *string, length = 0  ; *ptr != '\0' ; ptr ++, length ++ ){
    if( next_chara_ignore ){
      next_chara_ignore = False ;
      continue ;
    }
    /* "," դȤޤǤĤΥեȥå̾ˤʤ롣*/
    if( *ptr == ',' ){
      /* "," Фߤ"," ޤʸ󥳥ԡΤϷ
       * length 䤵ʤ*/ 
      ptr ++ ;
      break ;
    }
    /* Τɤʬʤɡϼʸ̵뤹ΤȤ
     * ͽ󤷤Ƥ*/
    if( *ptr == '\\' ){
      next_chara_ignore = True ;
    }
  }
  /* դޤ */
  if( length > 0 ){
    retstring = malloc( sizeof( unsigned char ) * ( length + 1 ) ) ;
    if( retstring == NULL )
      return NULL ;
    strncpy( retstring, *string, length ) ;
    retstring[ length ] = '\0' ;
  }
  *string = ptr ;
  return retstring ;
}
