/*  Screem:  screem-link-view-html.c
 *
 *  The link view widget
 *
 *  Copyright (C) 2001  Matt Colyer, David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <libgnomecanvas/gnome-canvas-text.h>

#include <libgnomeui/gnome-popup-menu.h>

#include <glib/gi18n.h>

#include <libgnomevfs/gnome-vfs-utils.h>

#include <gtk/gtk.h>

#include <glade/glade.h>

#include <string.h>

#include "fileops.h"

#include "screem-window.h"

#include "screem-site.h"
#include "screem-page.h"

#include "screem-linkview.h"
#include "screem-link-view-html.h"

#include "support.h"

static void screem_link_view_html_class_init(ScreemLinkViewHtmlClass *klass);
static void screem_link_view_html_init( ScreemLinkViewHtml *link_view_html);
static void screem_link_view_html_finalize( GObject *link_view_html );


static gint item_event( GnomeCanvasItem *group, GdkEvent *event, 
			ScreemLinkViewHtml *html );


ScreemLinkViewHtml *screem_link_view_html_new( ScreemLinkView *view,
					       ScreemPage *page,
					       const gchar *pathname )
{
	ScreemLinkViewHtml *html;
	GType type;

	ScreemSite *site;

	GHashTable *upload_table;

	GnomeVFSURI *uri;
	gboolean islocal;

	type = screem_link_view_html_get_type();

	html = SCREEM_LINK_VIEW_HTML( g_object_new( type, NULL ) );

	g_object_set_data( G_OBJECT( html ), "view", view );

	g_object_get( G_OBJECT( view ),
		      "site", &site,
		      "uploadtable", &upload_table,
		      NULL );

	g_assert( SCREEM_IS_SITE( site ) );

	islocal = TRUE;
	if( pathname && strncmp( "file:", pathname, strlen( "file:" ) ) ) {
		islocal = FALSE;
		uri = gnome_vfs_uri_new( pathname );
		if( uri ) {
			islocal = gnome_vfs_uri_is_local( uri );
		}
	}

	if( page || islocal ) {
		const gchar *sitepath;
		gboolean blank;

		html->page = page;

		sitepath = screem_site_get_pathname( site );
		blank = FALSE;

		if( ! pathname ) {
			blank = TRUE;
			pathname = _( "Untitled" );
		} else if( sitepath ) {
			html->name = relative_path( pathname, sitepath );
		} else {
			html->name = g_strdup( pathname );
		}

		html->filename = g_strdup( pathname );
		html->remoteAddr = NULL;

		/* exists ? */
		if( blank ) {
			html->uploadStatus = file_new;
		} else {
			/* ignored? */
			html->excluded = 
				( screem_site_is_excluded( site,
							   html->filename ) ||
				  screem_site_is_ignored( site,
							  html->filename ) );
			if( uri_exists( pathname, NULL ) ) {
				html->exists = TRUE;
				/*ptrHFD->filesize = s->st_blocks/2;*/
				if( upload_table ) {
					html->uploadStatus = GPOINTER_TO_INT( g_hash_table_lookup( upload_table, pathname ) );
				} else {
					html->uploadStatus = file_new;
				}
			}
		}
	} else {
		/* remote file */
		html->name = NULL;
		html->filename = NULL;	
		html->remoteAddr = g_strdup( pathname );
	}
	html->images = NULL;

	g_object_unref( site );
	
	return html;
}


void screem_link_view_html_build_icon( ScreemLinkViewHtml *html,
				       gdouble x, gdouble y,
				       gdouble angle,
				       gdouble *width,
				       GnomeCanvasGroup *root )
{
	ScreemLinkView *view;
	GnomeCanvasGroup *group;
	GtkType type;
	gint type_of_icon;
	gchar *path;
	
	gchar *font;
	GConfClient *client;
	PangoFontDescription *pfont;
	gchar *temp;
	
	client = gconf_client_get_default();

	view = SCREEM_LINK_VIEW( g_object_get_data( G_OBJECT(html), "view" ) );

	/* Set the label and choose the correct icon*/
	if( html->name ) {
		/* Local File */
		path = (gchar*) html->name;
			
		/* Is it a bad link? */
		if ( html->exists ){
			if( html->excluded ) {
				type_of_icon = NONODE;
			} else if( html->uploadStatus == file_changed ||
				   html->uploadStatus == file_moved ||
				   html->uploadStatus == file_deleted ) {
				type_of_icon = UPDATENODE;
			} else {
				type_of_icon = GOODNODE;
			}
		} else {
			type_of_icon = BADNODE;
		}
		if( html->images ) {
			ScreemLinkViewImage *image;
			for(image= html->images; image; image = image->next) {
				/*g_print("IMAGES: %s\n", image->filename);*/
			}
			/*g_print("\n");*/
		}
	} else if( ! html->remoteAddr ) {
		path = _( "Untitled" );
		type_of_icon = GOODNODE;
	} else {
		/* Remote File */
		path = html->remoteAddr;
		type_of_icon = EXTERNALNODE;
	}
	
	/* create the group */
	type = gnome_canvas_group_get_type();
	group = GNOME_CANVAS_GROUP( gnome_canvas_item_new( root, type,
							   "x", x,
							   "y", y,
							   NULL ) );

	/* update the node data */
	html->drawnX = x;
	html->drawnY = y;

	/* create the image */
       	html->icon = node_draw( group, -5.0, -5.0, type_of_icon, FALSE);
	gnome_canvas_item_raise_to_top( html->icon );
	
	/*Create the text*/
	font = gconf_client_get_string( client,
					"/app/screem/editor/font", NULL );
	type = gnome_canvas_text_get_type();
	pfont = NULL;
	if( font ) {
		pfont = pango_font_description_from_string( font );
	}
	temp = gnome_vfs_unescape_string_for_display( path );
	html->label = gnome_canvas_item_new( group, type, 
					     "x", 0.0,
					     "y", 15.0,
					     "text", temp,
					     "fill_color", "black",
					     "anchor", GTK_ANCHOR_CENTER,
					     NULL );
	g_free( temp );
	if( pfont ) {
		g_object_set( G_OBJECT( pfont ), "font_desc", pfont,
				NULL );
	}
	
	if( font ) {
		g_free( font );
		pango_font_description_free( pfont );
	}
	/* set the label as a data item of item1 */
	g_object_set_data( G_OBJECT( html->icon ), "label", html->label );

	g_object_set_data( G_OBJECT( html->icon ), "ScreemLinkViewHtml",
			   html );
	g_object_set_data( G_OBJECT( html->label ), "ScreemLinkViewHtml",
			   html );

	/* Use the group x,y as they are correct, using the world
	   coords of the item doesn't work, try switching depth and
	   the view shifts, and may hide some of it it so that it
	   can't be scrolled to */
	{
		gdouble true_x;
		gdouble true_y;
		gdouble tx;
		gdouble ty;
		gdouble tx2;
		gdouble ty2;

		true_x = x; true_y = y;

		g_object_get( G_OBJECT( view ), 
			      "minx", &tx,
			      "miny", &ty,
			      "maxx", &tx2,
			      "maxy", &ty2,
			      NULL );
		
		if( tx > true_x ) {
			tx = true_x;
		}
		if( tx2 < true_x ) {
			tx2 = true_x;
		}
		if( ty > true_y ) {
			ty = true_y;
		}
		if( ty2 < true_y ) {
			ty2 = true_y; 
		}

		g_object_set( G_OBJECT( view ), 
			      "minx", tx,
			      "miny", ty,
			      "maxx", tx2,
			      "maxy", ty2,
			      NULL );
	}
	
	g_object_set_data( G_OBJECT( group ), "linkview", view );


	g_signal_connect( G_OBJECT( group ), "event", 
			  G_CALLBACK( item_event ), html );

	g_object_unref( client );
}

/* static stuff */

static gint item_event( GnomeCanvasItem *group, GdkEvent *event, 
			ScreemLinkViewHtml *html )
{
	GnomeCanvasItem *label;
	gint handled = FALSE;

	ScreemLinkView *link_view;
	ScreemSite *site;
	ScreemWindow *window;
	ScreemPage *page;

	GnomeCanvasItem *item;
	const gchar *path;

	gboolean popup;
	GtkAction *action;
	
	item = html->icon;
	if( html->remoteAddr ) {
		path = html->remoteAddr;
	} else if( html->name ) {
		path = html->filename;
	} else {
		path = NULL;
	}

	link_view = g_object_get_data( G_OBJECT( group ), "linkview" );

	g_object_get( G_OBJECT( link_view ),
		      "site", &site,
		      "window", &window,
		      NULL );

	g_assert( SCREEM_IS_SITE( site ) );
	g_assert( SCREEM_IS_WINDOW( window ) );

	if( path ) {
		page = screem_site_locate_page( site, path );
	} else {
		page = NULL;
	}
	
	popup = ( event->type == GDK_BUTTON_PRESS &&
		  event->button.button == 3 );

	if( popup && page ) {
		/* set upload status toggle buttons */
		gboolean status;
		
		status = screem_site_is_excluded( site, path );
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
							"ExcludeFlag" );
		gtk_toggle_action_set_active( GTK_TOGGLE_ACTION( action ),
						status );
		status = screem_site_is_ignored( site, path );
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
							"IgnoreFlag" );
		gtk_toggle_action_set_active( GTK_TOGGLE_ACTION( action ),
						status );
		status = screem_site_is_ascii( site, path );
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
							"ASCIIFlag" );
		gtk_toggle_action_set_active( GTK_TOGGLE_ACTION( action ),
						status );
	} else if( popup ) {
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
							"ExcludeFlag" );
		g_object_set( G_OBJECT( action ), "visible", FALSE, NULL );
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
							"IgnoreFlag" );
		g_object_set( G_OBJECT( action ), "visible", FALSE, NULL );
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
							"ASCIIFlag" );
		g_object_set( G_OBJECT( action ), "visible", FALSE, NULL );
	}

	switch( event->type ) {
	case GDK_ENTER_NOTIFY:
		gnome_canvas_item_set(item, "outline_color", "gray",NULL);
		gnome_canvas_item_request_update(item);
		label = g_object_get_data( G_OBJECT( item ), "label" );
		if( ! ( label->object.flags & GNOME_CANVAS_ITEM_VISIBLE ) ) {
			g_object_set_data( G_OBJECT( label ), "hidden",
					     GINT_TO_POINTER( TRUE ) );
			gnome_canvas_item_show( label );
		}
		handled = TRUE;
		break;
	case GDK_LEAVE_NOTIFY:
		gnome_canvas_item_set(item, "outline_color", "black",NULL);
		gnome_canvas_item_request_update(item);
		label = g_object_get_data( G_OBJECT( item ), "label" );
		if( g_object_get_data( G_OBJECT( label ), "hidden" ) ) {
			gnome_canvas_item_hide( label );
			g_object_set_data( G_OBJECT( label ), "hidden",
					   NULL );
		}
		handled = TRUE;
		break;
	case GDK_BUTTON_PRESS:
		if( event->button.button == 3 ) {
			GtkWidget *menu;

			popup = TRUE;

			/* popup menu for the icons, path holds the 
			   pathname of the file clicked on */
			menu = gtk_ui_manager_get_widget( GTK_UI_MANAGER( window->merge ),
							 "/linkviewmenu" );
			screem_popup_menu_do_popup_modal( menu, 0, 0, 
							 &event->button,
							 0, 0 );
			handled = TRUE;
		}
		break;
	case GDK_2BUTTON_PRESS:
		if( event->button.button == 1 ) {
			/* the item was double clicked on, switch to the 
			   page */
			if( page ) {
				/* switch to it! */
				screem_window_set_document( window, page );
			}
			handled = TRUE;
		}
		break;
	default:
		break;
	}
	
	if( page && popup ) {
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
							"ExcludeFlag" );
		if( gtk_toggle_action_get_active( GTK_TOGGLE_ACTION( action ) ) ) {
			screem_site_add_exclude( site, path );
		} else {
			screem_site_remove_exclude( site, path );
		}
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
							"IgnoreFlag" );
		if( gtk_toggle_action_get_active( GTK_TOGGLE_ACTION( action ) ) ) {
			screem_site_add_ignore( site, path );
		} else {
			screem_site_remove_ignore( site, path );
		}
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
							"ASCIIFlag" );
		if( gtk_toggle_action_get_active( GTK_TOGGLE_ACTION( action ) ) ) {
			screem_site_add_ascii( site, path );
		} else {
			screem_site_remove_ascii( site, path );
		}
	} else if( popup ) {
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
							"ExcludeFlag" );
		g_object_set( G_OBJECT( action ), "visible", TRUE, NULL );
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
							"IgnoreFlag" );
		g_object_set( G_OBJECT( action ), "visible", TRUE, NULL );
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
							"ASCIIFlag" );
		g_object_set( G_OBJECT( action ), "visible", TRUE, NULL );
	}

	g_object_unref( site );
	
	return handled;
}


/* G Object stuff */
#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void screem_link_view_html_class_init( ScreemLinkViewHtmlClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );
	parent_class = g_type_class_peek_parent( klass );


	object_class->finalize = screem_link_view_html_finalize;
}

static void screem_link_view_html_init( ScreemLinkViewHtml *link_view_html )
{

}

static void screem_link_view_html_finalize( GObject *link_view_html )
{
	ScreemLinkViewHtml *html;

	html = SCREEM_LINK_VIEW_HTML( link_view_html );

	if( html->name ) {
		g_free( html->name );
	}
	if( html->filename ) {
		g_free( html->filename );
	}
	if( html->remoteAddr ) {
		g_free( html->remoteAddr );
	}
	if( html->images ) {
		g_object_unref( html->images );
	}

	gtk_object_destroy( GTK_OBJECT( html->label ) );
	gtk_object_destroy( GTK_OBJECT( html->icon ) );

	G_OBJECT_CLASS( parent_class )->finalize( G_OBJECT( html ) );
}

GType screem_link_view_html_get_type()
{
	static guint type = 0;

	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemLinkViewHtmlClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_link_view_html_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemLinkViewHtml ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_link_view_html_init
		};
		
		type = g_type_register_static( PARENT_TYPE,
					       "ScreemLinkViewHtml",
					       &info, 0 );
	}

	return type;
}


