// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "circle.h"

#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Circle::Circle(const Circle& a) : BaseCircle(a) {}

Circle::Circle(FrameBase* p, const Vector& ctr, double r, 
	       const char* clr, int w, const char* f, const char* t, 
	       unsigned short prop, const char* c,
	       const List<Tag>& tag)
  : BaseCircle(p, ctr, 0, clr, w, f, t, prop, c, tag)
{
  annuli = 1;
  radii = new double[1];
  radii[0] = r;

  strcpy(type, "circle");
  handle = new Vector[4];
  numHandle = 4;

  updateBBox();
}

void Circle::updateBBox()
{
  // bound marker
  Vector c = center * parent->refToCanvas;
  Vector r = Vector(radii[0],radii[0]);
  if (!(properties & FIXED))
    r *= parent->getZoom();
  bbox = BBox(c-r, c+r);

  // generate handles
  handle[0] = bbox.ll;
  handle[1] = bbox.lr();
  handle[2] = bbox.ur;
  handle[3] = bbox.ul();

  // bound handles
  for (int i=0; i<numHandle; i++)
    bbox.bound(handle[i]);

  // make room for handles
  bbox.expand(3);

  // calculate overall bbox
  calcAllBBox();
}

void Circle::edit(const Vector& v, int h)
{
  float d = Vector(radii[0],radii[0]).length()-radii[0];
  radii[0] = (v-center).length() - d;

  updateBBox();
  doCallBack(&editCB);
}

void Circle::setRadius(double r)
{
  radii[0] = r;

  updateBBox();
  doCallBack(&editCB);
}

// list

void Circle::list(ostream& str, CoordSystem sys, SkyFrame sky,
		  SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listPre(str,sys,sky,ptr);

      Vector v = ptr->mapFromRef(center,sys);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	  << ptr->mapLenFromRef(radii[0],sys) << ')';

      listPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      listPre(str,sys,sky,ptr);

      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
		<< ptr->mapLenFromRef(radii[0],sys,ARCSEC) << '"' << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
	    wcs >> ra >> dec;
	    str << type << '(' << ra << ',' << dec << ',' 
		<< ptr->mapLenFromRef(radii[0],sys,ARCSEC) 
		<< '"' << ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	    << ptr->mapLenFromRef(radii[0],sys) << ')';
      }

      listPost(str,delim);
    }
    else
      str << "";
    break;
  }
}

void Circle::listCiao(ostream& str, CoordSystem sys, SkyFrame sky,
		      SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listCiaoPre(str);

      Vector v = ptr->mapFromRef(center,PHYSICAL);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	  << ptr->mapLenFromRef(radii[0],PHYSICAL) << ')'
	  << delim;
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      listCiaoPre(str);

      char buf[64];
      ptr->mapFromRef(center,sys,FK5,SEXAGESIMAL,buf,64);
      char ra[16];
      char dec[16];
#if __GNUC__ >= 3
      string x(buf);
      istringstream wcs(x);
#else
      istrstream wcs(buf,64);
#endif
      wcs >> ra >> dec;
      str << type << '(' << ra << ',' << dec << ',' 
	  << ptr->mapLenFromRef(radii[0],sys,ARCMIN) << '\'' << ')'
	  << delim;
    }
    else
      str << "";
    break;
  }
}

void Circle::listPros(ostream& str, CoordSystem sys, SkyFrame sky,
		      SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case DETECTOR:
  case AMPLIFIER:
    sys = IMAGE;
  case PHYSICAL:
    {
      listProsCoordSystem(str,sys,sky);
      str << "; ";
      Vector v = ptr->mapFromRef(center,sys);
      str << type << ' ' << setprecision(8) << v 
	  << ptr->mapLenFromRef(radii[0],IMAGE)
	  << delim;
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      listProsCoordSystem(str,sys,sky);
      str << "; ";

      // display wcs radius in ARCSEC

      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  str << type << ' ' << setprecision(8) << v[0] << "d " << v[1] <<"d " 
	      << ptr->mapLenFromRef(radii[0],sys,ARCSEC) << '"';
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char dec[16];
#if __GNUC__ >= 3
	  string x(buf);
	  istringstream wcs(x);
#else
	  istrstream wcs(buf,64);
#endif
	  wcs >> ra >> dec;
	  str << type << ' ' << ra << ' ';
	  if (dec[0]=='+')
	    str << dec+1 << ' ';
	  else
	    str << dec << ' ';

	  str << ptr->mapLenFromRef(radii[0],sys,ARCSEC) << '"';
	}
	break;
      }

      str << delim;
    }
    else
      str << "";
    break;
  }
}

void Circle::listSAOtng(ostream& str, CoordSystem sys, SkyFrame sky,
		     SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listSAOtngPre(str,delim);

      Vector v = ptr->mapFromRef(center,IMAGE);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	  << ptr->mapLenFromRef(radii[0],IMAGE) << ')';

      listSAOtngPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      listSAOtngPre(str,delim);

      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	      << ptr->mapLenFromRef(radii[0],IMAGE) << ')';
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char dec[16];
#if __GNUC__ >= 3
	  string x(buf);
	  istringstream wcs(x);
#else
	  istrstream wcs(buf,64);
#endif
	  wcs >> ra >> dec;
	  str << type << '(' << ra << ',' << dec << ',' 
	      << ptr->mapLenFromRef(radii[0],IMAGE) << ')';
	}
	break;
      }

      listSAOtngPost(str,delim);
    }
    else
      str << "";
    break;
  }
}

void Circle::listSAOimage(ostream& str, CoordSystem sys, SkyFrame sky,
			   SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  // all coords are in image coords

  if (!(properties&INCLUDE))
    str << '-';

  Vector v = ptr->mapFromRef(center,IMAGE);
  str << "circle(" << setprecision(8) << v[0] << ',' << v[1] 
      << ',' << radii[0] << ')';

  str << delim;
}
