// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "baseellipse.h"
#include "framebase.h"
#include "util.h"

BaseEllipse::BaseEllipse(const BaseEllipse& a) : BaseMarker(a)
{
  flip = a.flip;

  radii = new Vector[a.annuli];
  for (int i=0; i<a.annuli; i++)
    radii[i] = a.radii[i];
}

BaseEllipse::BaseEllipse(FrameBase* p, const Vector& ctr, double a,
			 const char* clr,  int w, const char* f, 
			 const char* t, unsigned short prop, const char* c,
			 const List<Tag>& tag)
  : BaseMarker(p, ctr, 48, clr, w, f, t, prop, c, tag) 
{
  radii = NULL;
  angle = a;

  if (!parent->isIIS())
    flip = FlipY();
}

BaseEllipse::~BaseEllipse()
{
  if (radii)
    delete [] radii;
}

void BaseEllipse::updateCoords(const Matrix& m)
{
  for (int i=0; i<annuli; i++)
    radii[i] *= Scale(m);

  Marker::updateCoords(m);
}

int BaseEllipse::isIn(const Vector& v)
{
  // v is in canvas coords

  Vector r = radii[annuli-1];

  // zero radius

  if (!r[0] || !r[1])
    return 0;

  if (properties & FIXED)
    r /= parent->getZoom();

  Vector p = v * 
    parent->canvasToRef * Translate(-center) * flip * Rotate(-angle);

  if ((p[0]*p[0])/(r[0]*r[0]) + (p[1]*p[1])/(r[1]*r[1]) <= 1)
    return 1;
  else
    return 0;
}

// private

void BaseEllipse::calcPoints(double zoom)
{
  Matrix m;
  if (properties & FIXED)
    m = Scale(1/zoom) * Rotate(angle) * flip * Translate(center);
  else
    m = Rotate(angle) * flip * Translate(center);

  for (int i=0; i<annuli; i++) {
    for (int j=0; j<segments; j++) {
      double theta = 2*M_PI / segments * j;
      vertices[i][j] = 
	Vector(radii[i][0]*cos(theta), radii[i][1]*sin(theta)) * m;
    }
  }

  double theta = degToRad(45);
  vertices[annuli][0] = 
    Vector(radii[annuli-1][0]*cos(theta), radii[annuli-1][1]*sin(theta)) * m;
  vertices[annuli][1] = 
    Vector(-radii[annuli-1][0]*cos(theta), -radii[annuli-1][1]*sin(theta)) * m;
}
