// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <string.h>

#include "colorbar.h"
#include "util.h"

#include "lut.h"
#include "lasc.h"
#include "sao.h"
#include "default.h"

Colorbar::Colorbar(Tcl_Interp* i, Tk_Canvas c, Tk_Item* item) 
  : ColorbarBase(i, c, item)
{
  cmaps = NULL;
  colorIndex = NULL;
  currentcmap = NULL;
  itts = NULL;
  currentitt = NULL;

  bias = .5;
  contrast = 1.0;
  invert = 0;
}

Colorbar::~Colorbar()
{
  ColorMapInfo* ptr = cmaps;
  while (ptr) {
    ColorMapInfo* tmp = ptr->getNext();
    delete ptr;
    ptr = tmp;
  }

  if (colorIndex)
    delete [] colorIndex;
}

void Colorbar::adjustCmd(float c, float b)
{
  contrast = c;
  bias = b;
  updateColors();
}

#if __GNUC__ >= 3
void Colorbar::getBiasCmd()
{
  if (currentcmap) {
    ostringstream str;
    str << bias << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }
  else
    result = TCL_ERROR;
}
#else
void Colorbar::getBiasCmd()
{
  if (currentcmap) {
    char buf[64];
    ostrstream str(buf,64);
    str << bias << ends;
    Tcl_AppendResult(interp, buf, NULL);
  }
  else
    result = TCL_ERROR;
}
#endif

#if __GNUC__ >= 3
void Colorbar::getColormapCmd()
{
  if (currentcmap) {
    ostringstream str;
    str << currentcmap->getID() << ' '
	<< bias << ' ' 
	<< contrast << ' ' 
	<< invert << ' '
	<< colorIndex << ' '
	<< (unsigned short*)colorCells << ' '
	<< colorCount << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }
  else
    result = TCL_ERROR;
}
#else
void Colorbar::getColormapCmd()
{
  if (currentcmap) {
    char buf[64];
    ostrstream str(buf,64);
    str << currentcmap->getID() << ' '
	<< bias << ' ' 
	<< contrast << ' ' 
	<< invert << ' '
	<< colorIndex << ' '
	<< (unsigned short*)colorCells << ' '
	<< colorCount << ends;
    Tcl_AppendResult(interp, buf, NULL);
  }
  else
    result = TCL_ERROR;
}
#endif

void Colorbar::getColormapNameCmd(int id)
{
  ColorMapInfo* ptr = cmaps;
  while (ptr) {
    if (ptr->getID() == id) {
      Tcl_AppendResult(interp, (char*)ptr->getName(), NULL);
      return;
    }
    ptr = ptr->getNext();
  }

  // if we got this far, we did not find it, bail out

  Tcl_AppendResult(interp, " colormap not found.", NULL);
  result = TCL_ERROR;
}

#if __GNUC__ >= 3
void Colorbar::getContrastCmd()
{
  if (currentcmap) {
    ostringstream str;
    str << contrast << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }
  else
    result = TCL_ERROR;
}
#else
void Colorbar::getContrastCmd()
{
  if (currentcmap) {
    char buf[64];
    ostrstream str(buf,64);
    str << contrast << ends;
    Tcl_AppendResult(interp, buf, NULL);
  }
  else
    result = TCL_ERROR;
}
#endif

void Colorbar::getCurrentNameCmd()
{
  if (currentcmap)
    Tcl_AppendElement(interp, (char*)currentcmap->getName());
  else {
    Tcl_AppendResult(interp, " no colormap selected.", NULL);
    result = TCL_ERROR;
  }
}

#if __GNUC__ >= 3
void Colorbar::getCurrentIDCmd()
{
  if (currentcmap) {
    ostringstream str;
    str << currentcmap->getID() << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }
  else {
    Tcl_AppendResult(interp, " no colormap selected.", NULL);
    result = TCL_ERROR;
  }
}
#else
void Colorbar::getCurrentIDCmd()
{
  if (currentcmap) {
    char buf[8];
    ostrstream str(buf,8);
    str << currentcmap->getID() << ends;
    Tcl_AppendResult(interp, buf, NULL);
  }
  else {
    Tcl_AppendResult(interp, " no colormap selected.", NULL);
    result = TCL_ERROR;
  }
}
#endif

void Colorbar::getCurrentFileNameCmd()
{
  if (currentcmap)
    Tcl_AppendElement(interp, (char*)currentcmap->getFileName());
  else {
    Tcl_AppendResult(interp, " no colormap selected.", NULL);
    result = TCL_ERROR;
  }
}

void Colorbar::getInvertCmd()
{
  if (invert)
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void Colorbar::invertCmd(int i)
{
  invert = i ? 1 : 0;
  updateColors();
}

void Colorbar::ittCmd(const char* which)
{
  // special case (usually the default)

  if (!strcmp(which, "none")) {
    currentitt = NULL;
    if (currentcmap) {
      currentcmap->setITT(currentitt);
      updateColors();
    }
    return;
  }

  // else

  ITT* ptr = itts;
  while (ptr) {
    if (!strcmp(ptr->getName(), which)) {
      currentitt = ptr;
      if (currentcmap) {
	currentcmap->setITT(currentitt);
	updateColors();
      }
      return;
    }
    ptr = ptr->getNext();
  }

  // if we got this far, we did not find it, bail out

  Tcl_AppendResult(interp, " intensity table ", which, " not found.", NULL);
  result = TCL_ERROR;
}

void Colorbar::ittCmd(int id)
{
  // special case (usually the default)

  if (id == 0) {
    currentitt = NULL;
    if (currentcmap) {
      currentcmap->setITT(currentitt);
      updateColors();
    }
    return;
  }

  // else

  ITT* ptr = itts;
  while (ptr) {
    if (ptr->getID() == id) {
      currentitt = ptr;
      if (currentcmap) {
	currentcmap->setITT(currentitt);
	updateColors();
      }
      return;
    }
    ptr = ptr->getNext();
  }

  // if we got this far, we did not find it, bail out

  Tcl_AppendResult(interp, " intensity table not found.", NULL);
  result = TCL_ERROR;
}

#if __GNUC__ >= 3
void Colorbar::listIDCmd()
{

  ColorMapInfo* ptr = cmaps;
  while (ptr) {
    ostringstream str;
    str << ptr->getID() << ends;
    Tcl_AppendElement(interp, str.str().c_str());

    ptr = ptr->getNext();
  }
}
#else
void Colorbar::listIDCmd()
{
  char buf[8];
  ColorMapInfo* ptr = cmaps;
  while (ptr) {
    ostrstream str(buf,8);
    str << ptr->getID() << ends;
    Tcl_AppendElement(interp, buf);

    ptr = ptr->getNext();
  }
}
#endif

void Colorbar::listNameCmd()
{
  ColorMapInfo* ptr = cmaps;
  while (ptr) {
    Tcl_AppendElement(interp, (char*)ptr->getName());
    ptr = ptr->getNext();
  }
}

#if __GNUC__ >= 3
void Colorbar::listITTIDCmd()
{
  ITT* ptr = itts;
  while (ptr) {
    ostringstream str;
    str << ptr->getID() << ends;
    Tcl_AppendElement(interp, str.str().c_str());

    ptr = ptr->getNext();
  }
}
#else
void Colorbar::listITTIDCmd()
{
  char buf[8];
  ITT* ptr = itts;
  while (ptr) {
    ostrstream str(buf,8);
    str << ptr->getID() << ends;
    Tcl_AppendElement(interp, buf);

    ptr = ptr->getNext();
  }
}
#endif

void Colorbar::listITTNameCmd()
{
  ITT* ptr = itts;
  while (ptr) {
    Tcl_AppendElement(interp, (char*)ptr->getName());
    ptr = ptr->getNext();
  }
}

void Colorbar::loadCmd(const char* fn, const char* type)
{
  ColorMapInfo* map = newColorMap(fn, type);

  if (map && map->load()) {

    // add new colormap to end of the list

    if (cmaps) {
      ColorMapInfo* ptr = cmaps;
      while (ptr && ptr->getNext())
	ptr = ptr->getNext();

      ptr->setNext(map);
    }
    else
      cmaps = map;

    currentcmap = map;
    reset();
  }
  else {

    // something has gone wrong, clean up, and bail out
    
    delete map;
    Tcl_AppendResult(interp, " error has occured loading colormap: ", fn,NULL);
    result = TCL_ERROR;
  }
}

void Colorbar::loadITTCmd(const char* fn)
{
  ITT* table = newITT(fn);
  if (table && table->load()) {

    // add new colormap to end of the list

    if (itts) {
      ITT* ptr = itts;
      while (ptr && ptr->getNext())
	ptr = ptr->getNext();

      ptr->setNext(table);
    }
    else {
      itts = table;
      currentitt = table;
      if (currentcmap) {
	currentcmap->setITT(itts);
	updateColors();
      }
    }
  }
  else {

    // something has gone wrong, clean up, and bail out
    
    delete table;
    Tcl_AppendResult(interp, " error has occured loading intesity table: ",
		     fn, NULL);
    result = TCL_ERROR;
  }
}

void Colorbar::mapCmd(char* which)
{
  ColorMapInfo* ptr = cmaps;
  char* a = toLower(which);
  while (ptr) {
    char* b = toLower(ptr->getName());
    if (!strcmp(a,b)) {
      currentcmap = ptr;
      reset();
      delete [] a;
      delete [] b;
      return;
    }
    delete [] b;
    ptr = ptr->getNext();
  }

  // if we got this far, we did not find it, bail out

  delete a;
  Tcl_AppendResult(interp, " colormap not found.", NULL);
  result = TCL_ERROR;
}

void Colorbar::mapCmd(int id)
{
  ColorMapInfo* ptr = cmaps;
  while (ptr) {
    if (ptr->getID() == id) {
      currentcmap = ptr;
      reset();
      return;
    }
    ptr = ptr->getNext();
  }

  // if we got this far, we did not find it, bail out

  Tcl_AppendResult(interp, " colormap not found.", NULL);
  result = TCL_ERROR;
}

void Colorbar::saveCmd(const char* fn)
{
  if (currentcmap) {
    if (!currentcmap->save(fn)) {
      Tcl_AppendResult(interp, " unable to save current colormap.", NULL);
      result = TCL_ERROR;
    }
  }
  else {
    Tcl_AppendResult(interp, " no colormap selected.", NULL);
    result = TCL_ERROR;
  }
}

void Colorbar::setColormapCmd(char* which, float b, float c, int i)
{
  ColorMapInfo* ptr = cmaps;
  while (ptr) {
    if (!strcmp(ptr->getName(),which)) {
      currentcmap = ptr;
      bias = b;
      contrast = c;
      invert = i;
      updateColors();
      return;
    }
    ptr = ptr->getNext();
  }

  // if we got this far, we did not find it, bail out

  Tcl_AppendResult(interp, " colormap not found.", NULL);
  result = TCL_ERROR;
}

void Colorbar::setColormapCmd(int id, float b, float c, int i)
{
  ColorMapInfo* ptr = cmaps;
  while (ptr) {
    if (ptr->getID() == id) {
      currentcmap = ptr;
      bias = b;
      contrast = c;
      invert = i;
      updateColors();
      return;
    }
    ptr = ptr->getNext();
  }

  // if we got this far, we did not find it, bail out

  Tcl_AppendResult(interp, " colormap not found.", NULL);
  result = TCL_ERROR;
}

// private

int Colorbar::calcContrastBias(int i)
{
  // if default (contrast = 1.0 && bias = .5) return

  if (fabs(bias - 0.5) < 0.0001 && fabs(contrast - 1.0) < 0.0001)
    return i;
  
  // map i to range of 0 to 1.0
  // shift by bias (if invert, bias = 1-bias)
  // multiply by contrast
  // shift to center of region
  // expand back to number of dynamic colors

  float b = invert ? 1-bias : bias;
  int r = (int)(((((float)i / colorCount) - b) * contrast + .5 ) * colorCount);

  // clip to bounds if out of range

  if (r < 0)
    return 0;
  else if (r >= colorCount)
    return colorCount-1;
  else
    return r;
}

void Colorbar::reset()
{
  bias = 0.5;
  contrast = 1.0;
  invert = 0;
  updateColors();
}

void Colorbar::loadDefaultCMaps()
{
  ColorMapInfo* grey = new GreyColorMap();
  ColorMapInfo* red = new RedColorMap();
  ColorMapInfo* green = new GreenColorMap();
  ColorMapInfo* blue = new BlueColorMap();
  ColorMapInfo* a = new AColorMap();
  ColorMapInfo* b = new BColorMap();
  ColorMapInfo* bb = new BBColorMap();
  ColorMapInfo* he = new HEColorMap();
  ColorMapInfo* i8 = new I8ColorMap();
  ColorMapInfo* aips = new AIPSColorMap();
  ColorMapInfo* sls = new SLSColorMap();
  ColorMapInfo* heat = new HeatColorMap();
  ColorMapInfo* cool = new CoolColorMap();
  ColorMapInfo* rainbow = new RainbowColorMap();
  ColorMapInfo* standard = new StandardColorMap();
  ColorMapInfo* staircase = new StaircaseColorMap();
  ColorMapInfo* color = new ColorColorMap();
  ColorMapInfo* hsv = new HSVColorMap();

  if (grey && red && green && blue && a && b && bb && he && i8 && 
      aips && sls && heat && cool && rainbow && standard && 
      staircase && color && hsv) {

    cmaps = grey;
    currentcmap = grey;
    grey->setNext(red);
    red->setNext(green);
    green->setNext(blue);
    blue->setNext(a);
    a->setNext(b);
    b->setNext(bb);
    bb->setNext(he);
    he->setNext(i8);
    i8->setNext(aips);
    aips->setNext(sls);
    sls->setNext(hsv);
    hsv->setNext(heat);
    heat->setNext(cool);
    cool->setNext(rainbow);
    rainbow->setNext(standard);
    standard->setNext(staircase);
    staircase->setNext(color);
  }
  else {
    cerr << "Colorbar Internal Error: Unable to Create Default Colormaps." 
	 << endl;
    exit(1);
  }
}

ColorMapInfo* Colorbar::newColorMap(const char* fn, const char* type)
{
  // determine colormap type

  char* tmp = dupstr(fn);      // tmp memory, we will mangle it
  char* ptr = tmp;
  while (*ptr++);              // walk forward till end of string
  ptr--;                       // backup one
  while (ptr != tmp && *ptr != '.')  // march backward looking for '.'
    ptr--;
  if (ptr != tmp) {            // are we at '.' or start of string
    *ptr = '\0';               // mark end of string at '.'
    ptr++;
  }

  // Create ColorMap

  ColorMapInfo* map = NULL;

  // if type specified, use it, otherwise, use file extension

  if (type)
    ptr = (char*)type;

  if (strcmp(ptr, "lut") == 0)
    map = new LUTColorMap();
  else if (strcmp(ptr, "lasc") == 0)
    map = new LASCColorMap();
  else
    map = new SAOColorMap();

  // Bail out if we don't have a new ColorMap

  if (!map)
    return NULL;

  // Extract a name from the file name. Any extension has already been removed.

  ptr = tmp;
  while (*ptr++);           // walk forward till end of string
  while (ptr != tmp && *ptr != '/')  // march backward looking for '/'
    ptr--;
  if (ptr != tmp)           // see if we found '/' or at begining of string
    ptr++;

  map->setName(ptr);
  map->setFileName(fn);

  delete tmp;               // clean up
  return map;
}

ITT* Colorbar::newITT(const char* fn)
{
  ITT* table = new ITT();

  // determine colormap type

  char* tmp = dupstr(fn);      // tmp memory, we will mangle it
  char* ptr = tmp;
  while (*ptr++);              // walk forward till end of string
  while (ptr != tmp && *ptr != '.')  // march backward looking for '.'
    ptr--;
  if (ptr != tmp)              // are we at '.' or start of string
    *ptr = '\0';               // mark end of string at '.'

  // Extract a name from the file name. Any extension has already been removed.

  ptr = tmp;
  while (*ptr++);           // walk forward till end of string
  while (ptr != tmp && *ptr != '/')  // march backward looking for '/'
    ptr--;
  if (ptr != tmp)           // see if we found '/' or at begining of string
    ptr++;

  table->setName(ptr);
  table->setFileName(fn);

  delete tmp;               // clean up
  return table;
}
