/*
*  Rubrica
*  file: ui.c
*  
*  Copyright (C) 2000-2001 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include <config.h>

#include <gnome.h>
#include <gconf/gconf-client.h>

#include "ui.h"
#include "callbacks.h"
#include "utils.h"


void rubrica_ui_set_default_icon(GtkWidget *app)
{  
  GdkPixbuf *icon;

  icon = rubrica_ui_pixbuf_new("rub.png");
  if (!icon)
    g_print("\nRubrica: setting default icon failed");

  gtk_window_set_icon(GTK_WINDOW(app), icon);
}



/**
 * rubrica_ui_frame_new
 * @str: the label to show
 *
 * make a new frame with the label "str"
 *
 * return a pointer to a new frame with "str" label
 */
GtkWidget *
rubrica_ui_frame_new(gchar *str)
{
  GtkWidget *frame;
  GtkWidget *label;

  g_return_val_if_fail(str != NULL, NULL);

  frame = gtk_frame_new(NULL);
  label = gtk_label_new(_(str));

  gtk_container_set_border_width(GTK_CONTAINER(frame), 2);
  gtk_frame_set_label_widget(GTK_FRAME(frame), label);
  gtk_misc_set_padding (GTK_MISC (label), 5, 0);

  gtk_widget_show(frame);
  gtk_widget_show(label);

  return (GtkWidget *) frame;  
}


/**
 * rubrica_ui_box_new
 * @type: the type of the new box
 *        HBOX an horizontal box
 *        VBOX a vertical box
 * @homogeneous:
 * @spacing:
 * @border_widtg:
 *
 * create a new vbox/hbox 
 *
 * return: the pointer to new box. The new widget is shown automatically 
 *         (don't call gtk_widget_show())
 */
GtkWidget *
rubrica_ui_box_new(RubricaBoxType type, gboolean homogeneous, 
		   gint spacing, guint border_width)
{
  GtkWidget *box;

  if (type == VBOX)
    box = gtk_vbox_new (homogeneous, spacing);     
  else
    box = gtk_hbox_new (homogeneous, spacing);

  gtk_container_set_border_width (GTK_CONTAINER (box), border_width);
  
  gtk_widget_show (box); 
  
  return (GtkWidget *) box;
}


GtkWidget *
rubrica_ui_default_box_new(RubricaBoxType type)
{
  return (GtkWidget *) rubrica_ui_box_new(type, FALSE, 0, 2);
}


GtkWidget *
rubrica_ui_popup_image_new(gchar *pixfile)
{
  GtkWidget *image = NULL;
  gchar *imagefile = NULL;

  if (!pixfile || !pixfile[0])
    return gtk_image_new ();

  imagefile = rubrica_utils_find_pixmap_file(pixfile);
  
  if (!imagefile)
    {
      g_warning (_("Couldn't find pixmap file: %s"), pixfile);
      
      return gtk_image_new ();
    }
  
  image = gtk_image_new_from_file(imagefile);
  g_free(imagefile);
  
  return (GtkWidget *) image;
}

GtkWidget *
rubrica_ui_image_new(gchar *pixfile)
{
  GtkWidget *image = NULL;
  gchar *imagefile = NULL;

  if (!pixfile || !pixfile[0])
    return gtk_image_new ();

  imagefile = rubrica_utils_find_pixmap_file(pixfile);
  
  if (!imagefile)
    {
      g_warning (_("Couldn't find pixmap file: %s"), pixfile);
      
      return gtk_image_new ();
    }
  
  image = gtk_image_new_from_file(imagefile);
  gtk_widget_show(image);
  g_free(imagefile);
  
  return (GtkWidget *) image;
}


GdkPixbuf*
rubrica_ui_pixbuf_new(gchar *filename)
{
  GdkPixbuf *pix = NULL;
  gchar *imagefile = NULL;
  GError *error = NULL;

  g_return_val_if_fail(filename != NULL, NULL);

  imagefile = rubrica_utils_find_pixmap_file(filename);
      
  if (!imagefile)
    {
      g_warning (_("Couldn't find pixmap file: %s"), filename);
      return NULL;
    }
  
  pix = gdk_pixbuf_new_from_file(imagefile, NULL);
  if (!pix)
    {
      fprintf (stderr, "Failed to load pixbuf file: %s - error: %s\n", 
	       filename, error->message);
      g_error_free (error);
    }  

  g_free(imagefile);
  
  return  pix;
}


GtkWidget *
rubrica_ui_label_new(gchar *str) 
{ 
  GtkWidget *label; 

  g_return_val_if_fail(str != NULL, NULL);
  
  label = gtk_label_new_with_mnemonic(str); 

  gtk_widget_show(label); 
  
  return (GtkWidget *) label; 
} 


GtkWidget *
rubrica_ui_entry_new() 
{ 
  GtkWidget *entry; 
  
  entry = gtk_entry_new(); 

  gtk_widget_show(entry); 
  
  return (GtkWidget *) entry; 
} 


GtkWidget *
rubrica_ui_check_button_new(gchar *str)
{
  GtkWidget *check;
  
  g_return_val_if_fail(str != NULL, NULL);

  check = gtk_check_button_new_with_mnemonic (str);
  gtk_container_set_border_width (GTK_CONTAINER (check), 2);

  gtk_widget_show(check);
  
  return (GtkWidget *) check;
}


GtkWidget *
rubrica_ui_buttonbox_new(RubricaBoxType type)
{
  GtkWidget *box = NULL;

  switch (type)
    {
    case HBOX:
      box = gtk_hbutton_box_new ();
      gtk_button_box_set_layout(GTK_BUTTON_BOX(box), GTK_BUTTONBOX_SPREAD);
      break;

    case VBOX:
      box = gtk_vbutton_box_new ();
      gtk_button_box_set_layout(GTK_BUTTON_BOX(box), GTK_BUTTONBOX_SPREAD);
      break;

    default:
      g_warning("\nCan't build new box");
      break;
    }

    gtk_widget_show(box); 

  return box;
}


GtkWidget*
rubrica_ui_scrolled_window_new(void)
{  
  GtkWidget *scrollwin;
  
  scrollwin = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrollwin), 
				      GTK_SHADOW_ETCHED_IN);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin),
				 GTK_POLICY_AUTOMATIC, 
				 GTK_POLICY_AUTOMATIC);

  gtk_widget_show(scrollwin);

  return (GtkWidget *) scrollwin;
}


GtkWidget *rubrica_ui_option_menu_new(RubricaOptionType *option_list)
{
  RubricaOptionType *list = option_list;
  GtkWidget *optmenu;
  GtkWidget *optitem;
  GtkWidget *menu;

  menu    = gtk_menu_new();
  optmenu = gtk_option_menu_new();
  gtk_option_menu_set_menu(GTK_OPTION_MENU(optmenu), menu);

  while (list->pixmap)
    {
      GtkWidget *image;

      if ((strcmp(list->label, "-") == 0))
	optitem = gtk_separator_menu_item_new();
      else
	optitem = gtk_image_menu_item_new_with_mnemonic(_(list->label));

      if ((strcmp(list->pixmap, "") != 0))
	{
	  image = rubrica_ui_image_new(list->pixmap);
	  gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(optitem), image);
	}      

      gtk_container_add(GTK_CONTAINER(menu), optitem);
      gtk_widget_show(optitem);

      list++;
    }  

  gtk_widget_show(menu);
  gtk_widget_show(optmenu);

  return (GtkWidget *) optmenu;
}



GtkWidget *rubrica_ui_popup_menu_new(RubricaMenu *rubrica_menu)
{
  RubricaMenu *list = rubrica_menu;
  GConfClient *client;
  GtkWidget *menu;
  GtkWidget *submenu;
  GtkWidget *image;
  GtkWidget *menuitem = NULL;

  menu    = gtk_menu_new();
  client  = gconf_client_get_default();

  while (list->type != RM_END)
    {
      RubricaMenuType type = list->type;

      switch (type)
	{  
	case RM_SEPARATOR:
	  menuitem = gtk_separator_menu_item_new();
	  break;
	  
	case RM_ITEM:
	  if (list->pixtype == RMP_STOCK)
	    menuitem = gtk_image_menu_item_new_from_stock(list->pixmap, NULL);
	  else
	    {
	      menuitem = gtk_image_menu_item_new_with_mnemonic(_(list->label));
	      
	      if (list->pixtype == RMP_FILE)
		{
		  image = rubrica_ui_popup_image_new(list->pixmap);
		  gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), 
						image);
		}
	    }
	  if (list->callback)
	    g_signal_connect(G_OBJECT(menuitem), "activate", 
			     G_CALLBACK(list->callback), list->data);
	  break;
	  
	case RM_CHECK_ITEM:
	  menuitem = gtk_check_menu_item_new_with_mnemonic(_(list->label));

	  gconf_client_notify_add(client, list->data, 
				  rubrica_ui_update_popup_item, menuitem,
				  NULL, NULL);
	  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem), 
					 gconf_client_get_bool(client, 
							       list->data, 
							       NULL));

	  if (list->callback)
	    g_signal_connect(G_OBJECT(menuitem), "toggled", 
			     G_CALLBACK(list->callback), list->data);
	  break;
	  
	case RM_SUBTREE:
	  menuitem = gtk_menu_item_new_with_mnemonic(_(list->label));
	  submenu  = rubrica_ui_popup_menu_new(list->more);
	  gtk_menu_item_set_submenu (GTK_MENU_ITEM(menuitem), submenu);
	  break;

	case RM_END:
	default:
	  break; 
	}

      gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);      
      list++;
    }  
  
  return (GtkWidget *) menu;
}


/**
 * rubrica_ui_pack_in_frame_new_box
 * @frame: the container where the new box will be packed, or NULL 
 * @type: the type of the new box
 *        HBOX an horizontal box
 *        VBOX a vertical box
 * @homogeneous:
 * @spacing:
 * @border_widtg:
 *
 * create a new vbox/hbox, and pack it into a frame
 *
 * return: the pointer to new box
*/
GtkWidget*
rubrica_ui_pack_in_frame_new_box(GtkWidget *frame, RubricaBoxType type, 
				 gboolean homogeneous, gint spacing,
				 guint border) 
{
  GtkWidget *box; 

  box = rubrica_ui_box_new(type, homogeneous, spacing, border); 
  gtk_container_add(GTK_CONTAINER(frame), box);
  
  return (GtkWidget *) box; 
} 


/**
 * rubrica_ui_pack_in_box_new_frame
 * @box:  the container where the new frame will be packed, or NULL 
 * @str:
 * 
 * create a new frame with label "str" and pack it into box
 *
 * return a pointer to a new frame 
*/
GtkWidget*
rubrica_ui_pack_in_box_new_frame(GtkWidget *box, gchar *str, 
				 gboolean packstart) 
{
  GtkWidget *frame;
  GtkWidget *label;
  
  frame = gtk_frame_new(NULL);
  label = gtk_label_new(_(str));

  gtk_container_set_border_width(GTK_CONTAINER(frame), 2);
  gtk_frame_set_label_widget(GTK_FRAME(frame), label);
  gtk_misc_set_padding (GTK_MISC (label), 5, 0);

  if (packstart)  
    gtk_box_pack_start(GTK_BOX (box), frame, FALSE, FALSE, 0); 
  else
    gtk_box_pack_end(GTK_BOX (box), frame, FALSE, FALSE, 0); 

  gtk_widget_show(frame);
  gtk_widget_show(label);

  return (GtkWidget *) frame;  
} 


/**
 * rubrica_ui_packed_check_button_new
 * @container: the container where check button will be packed or NULL
 * @str: the string to show whith check button
 *
 * create a check button with label str, packed into a container.
 *
 * return: the pointer to new check button
*/

GtkWidget*
rubrica_ui_pack_in_box_check_button_new(GtkWidget *box, gchar *str) 
{
  GtkWidget *check;  
  
  g_return_val_if_fail(box != NULL, NULL);

  check =  gtk_check_button_new_with_mnemonic(str);  
  gtk_container_set_border_width (GTK_CONTAINER (check), 2);    

  gtk_box_pack_start(GTK_BOX(box), check, FALSE, FALSE, 0);

  gtk_widget_show(check);  
  
  return (GtkWidget *) check;  
}


GtkWidget* 
rubrica_ui_pack_in_box_image_new(GtkWidget *box, gchar *pixfile,
				 gboolean packstart)
{
  GtkWidget *image;

  g_return_val_if_fail(box != NULL, NULL);
  g_return_val_if_fail(pixfile != NULL, NULL);

  image = rubrica_ui_image_new(pixfile);
  
  if (packstart)  
    gtk_box_pack_start(GTK_BOX (box), image, FALSE, FALSE, 0); 
  else
    gtk_box_pack_end(GTK_BOX (box), image, FALSE, FALSE, 0); 

  gtk_misc_set_alignment(GTK_MISC (image), 0, 0.5);
  gtk_misc_set_padding (GTK_MISC (image), 10, 0);
  
  return (GtkWidget *) image;
}


/**
 * rubrica_ui_packed_labelpix_entry_new
 * @container:
 * @str:
 * @pixfile:
 * @type
 * @pack_left:
 *
 * create a new box (vertical or horizontal) packed into container. 
 * A label and an entry and an icon are packed into the box. Icon 
 * can be put on the left or on the rigth 
 *
 * return: a pointer to entry
*/
GtkWidget *
rubrica_ui_pack_in_box_boxed_label_entry_pix_new(GtkWidget *box, 
						 RubricaBoxType type, 
						 gboolean homogeneous,
						 gchar *str, gchar *pixfile, 
						 gboolean first,
						 gboolean packstart)
{
  GtkWidget *entry;
  
  GtkWidget *new_box;
  GtkWidget *label;
  GtkWidget *image;

  g_return_val_if_fail(box != NULL, NULL);
  g_return_val_if_fail(str != NULL, NULL);
  g_return_val_if_fail(pixfile != NULL, NULL);

  new_box = rubrica_ui_box_new(type, homogeneous, 0, 2);
  if (packstart)  
    gtk_box_pack_start(GTK_BOX (box), new_box, FALSE, FALSE, 0); 
  else
    gtk_box_pack_end(GTK_BOX (box), new_box, FALSE, FALSE, 0); 
  
  label = rubrica_ui_label_new(str);
  gtk_box_pack_start (GTK_BOX (new_box), label, FALSE, FALSE, 2);

  entry = gtk_entry_new();
  gtk_box_pack_start (GTK_BOX (new_box), entry, TRUE, TRUE, 2);

  image = rubrica_ui_image_new(pixfile);      
  gtk_box_pack_start (GTK_BOX (new_box), image, FALSE, FALSE, 2);

  gtk_widget_show(entry);
  
  return (GtkWidget *) entry;
}




/**
 * rubrica_ui_packed_label_option_menu_new
 * @container:
 * @str: the string to show into the label
 * @option_list: an array of option to display in option menu
 * @type
 * 
 * create a new box (vertical or horizontal) packed into container. 
 * A label and an option menu are packed into the box.  
 *
 * return: a pointer to option menu
 */
GtkWidget *
rubrica_ui_pack_in_box_boxed_label_option_menu_new(GtkWidget *box, gchar *str, 
						   RubricaOptionType *option_list,
						   RubricaBoxType type, 
						   gboolean packstart)
{
  GtkWidget *optmenu;
  GtkWidget *label;
  GtkWidget *new_box;

  new_box = rubrica_ui_default_box_new(type);
  if (packstart)  
    gtk_box_pack_start(GTK_BOX (box), new_box, FALSE, FALSE, 0); 
  else
    gtk_box_pack_end(GTK_BOX (box), new_box, FALSE, FALSE, 0); 

  label = gtk_label_new(str);
  gtk_box_pack_start(GTK_BOX(new_box), label, FALSE, FALSE, 2);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label), 10, 0);
  
  optmenu = rubrica_ui_option_menu_new(option_list);
  gtk_box_pack_start(GTK_BOX(new_box), optmenu, FALSE, FALSE, 2);

  /* mostro la prima voce del menu (Prima) */
  gtk_option_menu_set_history(GTK_OPTION_MENU(optmenu), 0);   

  gtk_widget_show(label);

  return (GtkWidget *) optmenu;
}

GtkWidget *
rubrica_ui_table_new(gint row, 
		     gint column, 
		     gboolean homogeneous, 
		     guint border, 
		     guint row_spacing, 
		     guint col_spacing)
{
  GtkWidget *table;
  
  table = gtk_table_new (row, column, homogeneous);  
  gtk_container_set_border_width (GTK_CONTAINER (table), border);
  gtk_table_set_row_spacings (GTK_TABLE (table), row_spacing);
  gtk_table_set_col_spacings (GTK_TABLE (table), col_spacing);

  gtk_widget_show(table);

  return (GtkWidget *) table;
}

GtkWidget *
rubrica_ui_pack_in_box_table_new(GtkWidget *box, 
				 gint row, 
				 gint column, 
				 gboolean homogeneous, 
				 guint border, 
				 guint row_spacing, 
				 guint col_spacing)
{
  GtkWidget *table;
  
  table = rubrica_ui_table_new(row, column, homogeneous, border, 
			       row_spacing, col_spacing);

  gtk_box_pack_start (GTK_BOX (box), table, TRUE, TRUE, 0);

  return (GtkWidget *) table;
}

GtkWidget *
rubrica_ui_pack_in_frame_table_new(GtkWidget *frame, 
				   gint row, 
				   gint column, 
				   gboolean homogeneous, 
				   guint border, 
				   guint row_spacing,
				   guint col_spacing)
{
  GtkWidget *table;

  table = rubrica_ui_table_new(row, column, homogeneous, border, 
			       row_spacing, col_spacing);

  gtk_container_add (GTK_CONTAINER(frame), table);

  return (GtkWidget *) table;
}


GtkWidget *
rubrica_ui_pixbutton_new(gchar *str, gchar *pixfile)
{
  GtkWidget *button;
  GtkWidget *box;
  GtkWidget *label;
  GtkWidget *image;
  
  button = gtk_button_new();

  box = gtk_hbox_new (FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (box), 2);
  
  image = rubrica_ui_image_new(pixfile);
  label = gtk_label_new_with_mnemonic(str);
  
  gtk_box_pack_start (GTK_BOX (box), image, FALSE, FALSE, 3);
  gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 3);

  gtk_container_add(GTK_CONTAINER(button), box);
  
  gtk_widget_show(label);
  gtk_widget_show(box);
  gtk_widget_show(button);

  return button; 
}


void 
rubrica_ui_tabled_widget(GtkWidget *table, GtkWidget *child, 
			 guint left_attach, guint right_attach, 
			 guint top_attach,  guint bottom_attach)
{
  gtk_table_attach_defaults (GTK_TABLE (table), child, 
			     left_attach, right_attach, 
			     top_attach, bottom_attach);
}


GtkWidget *
rubrica_ui_tabled_label_new(GtkWidget *table, gchar *str, 
			    guint left_attach, guint right_attach, 
			    guint top_attach,  guint bottom_attach)
{
  GtkWidget *label;

  label = gtk_label_new(str);

  gtk_table_attach_defaults (GTK_TABLE (table), label, 
			     left_attach, right_attach, 
			     top_attach, bottom_attach);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  
  gtk_widget_show(label);

  return (GtkWidget *) label;
}


GtkWidget *
rubrica_ui_tabled_entry_new(GtkWidget *table, 
			    guint left_attach, guint right_attach, 
			    guint top_attach,  guint bottom_attach)
{
  GtkWidget *entry;
  
  entry = gtk_entry_new ();

  gtk_table_attach_defaults (GTK_TABLE (table), entry, 
			     left_attach, right_attach, 
			     top_attach, bottom_attach);

  gtk_widget_show(entry);

  return (GtkWidget *) entry;  
}


GtkWidget *
rubrica_ui_tabled_check_new(GtkWidget *table, gchar *str,
			    guint left_attach, guint right_attach, 
			    guint top_attach,  guint bottom_attach)
{
  GtkWidget *check;

  check = gtk_check_button_new_with_mnemonic(str);
  
  gtk_table_attach_defaults (GTK_TABLE (table), check, 
			     left_attach, right_attach, 
			     top_attach, bottom_attach); 
  
  gtk_widget_show(check);

  return (GtkWidget *) check; 
}


GtkWidget *rubrica_ui_tabled_date_new(GtkWidget *table, guint left_attach, 
				      guint right_attach, guint top_attach, 
				      guint bottom_attach)
{
  GtkWidget *date;
  time_t tm;

  tm = time(NULL);
  date = gnome_date_edit_new(tm, FALSE, FALSE);
  
  gtk_table_attach_defaults (GTK_TABLE (table), date, 
			     left_attach, right_attach, 
			     top_attach, bottom_attach); 
  
  gtk_widget_show(date);

  return (GtkWidget *) date;   
}

GtkWidget *rubrica_ui_tabled_radio_new(GtkWidget *table, GSList *group, 
				       gchar *str,
				       guint left_attach, guint right_attach, 
				       guint top_attach, guint bottom_attach)
{
  GtkWidget *radio;
  
  radio =  gtk_radio_button_new_with_label (group, str);
  gtk_widget_show(radio);

  gtk_table_attach_defaults (GTK_TABLE (table), radio, 
			     left_attach, right_attach, 
			     top_attach, bottom_attach); 

  return (GtkWidget *) radio;
}


GtkWidget *rubrica_ui_tabled_pixmap_new(GtkWidget *table, gchar *pixfile,
					guint left_attach, guint right_attach, 
					guint top_attach, guint bottom_attach)
{
  GtkWidget *image = NULL;
  
  g_return_val_if_fail(pixfile != NULL, NULL);
  
  image = rubrica_ui_image_new(pixfile);
  
  gtk_table_attach_defaults (GTK_TABLE (table), image, 
			     left_attach, right_attach, 
			     top_attach, bottom_attach); 

  return (GtkWidget *) image;
}

GtkWidget *rubrica_ui_tabled_text_view(GtkWidget *table, GtkWidget *text_view,
				       guint left_attach, guint right_attach, 
				       guint top_attach, guint bottom_attach)
{
  GtkWidget *scrollwin;

  scrollwin = rubrica_ui_scrolled_window_new();
  gtk_container_add(GTK_CONTAINER(scrollwin), text_view);

  gtk_table_attach_defaults (GTK_TABLE (table), scrollwin,
			     left_attach, right_attach, 
			     top_attach, bottom_attach); 

  return (GtkWidget *) scrollwin;
}


GtkWidget *rubrica_ui_pack_in_box_widgets(RubricaBoxType type, 
					  GtkWidget* widget1, 
					  GtkWidget* widget2,
					  gboolean pack_start)  
{
  GtkWidget *box;  
 
  box = rubrica_ui_box_new(type, FALSE, 0, 2);  

  if (pack_start) 
    {
      gtk_box_pack_start (GTK_BOX (box), widget1, FALSE, FALSE, 0); 
      gtk_box_pack_start (GTK_BOX (box), widget2, FALSE, FALSE, 0); 
    }
  else 
    {
      gtk_box_pack_end (GTK_BOX (box), widget1, FALSE, FALSE, 0);       
      gtk_box_pack_end (GTK_BOX (box), widget2, FALSE, FALSE, 0);       
    } 

  gtk_widget_show (box);    
 
  return (GtkWidget *) box;    
} 


/*  Text 
*/
GtkWidget*
rubrica_ui_text_view_new(void)
{
  GtkWidget *text_view;
  
  text_view = gtk_text_view_new ();
  gtk_text_view_set_editable (GTK_TEXT_VIEW (text_view), TRUE);
  gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (text_view), GTK_WRAP_WORD);
  gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW (text_view), TRUE);

  gtk_widget_show(text_view);
  
  return (GtkWidget *) text_view;
}


void 
rubrica_ui_clean_text_view(GtkWidget *text_view)
{
  GtkTextIter start, end;
  GtkTextBuffer *text_buffer;  
  
  text_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text_view));
  
  gtk_text_buffer_get_bounds(text_buffer, &start, &end);
  gtk_text_buffer_delete(text_buffer, &start, &end);
}


void 
rubrica_ui_set_text_in_view(GtkWidget *text_view, gchar *buffer)
{
  GtkTextIter start, end;
  GtkTextBuffer *text_buffer;  

  g_return_if_fail(buffer != NULL);
  g_return_if_fail(g_utf8_validate(buffer, -1, NULL));
  
  text_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text_view));  
  gtk_text_buffer_get_bounds(text_buffer, &start, &end);

  gtk_text_buffer_insert(text_buffer, &start, buffer, -1);  
}


/*  Tree/List
*/
GtkWidget*
rubrica_ui_tree_view_new_with_model(GtkTreeModel* model)
{
  GtkWidget *tree_view;
  
  tree_view = gtk_tree_view_new_with_model(model);

  gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(tree_view), TRUE);
  gtk_tree_view_set_reorderable (GTK_TREE_VIEW (tree_view), TRUE);

  g_object_unref(G_OBJECT(model));

  gtk_widget_show(tree_view);

  return (GtkWidget *) tree_view; 
}


void 
rubrica_ui_add_columns(GtkTreeView *view, RubricaColumnsType type)
{
  GtkCellRenderer *renderer;
  GtkTreeModel *model;

  model = gtk_tree_view_get_model(view);

  switch(type)
    {
    case WEB_COLUMNS:
      renderer = gtk_cell_renderer_text_new();
      gtk_tree_view_insert_column_with_attributes (view, /* tree view */
						   -1,  _("Web URL"),
						   renderer, "text",      
						   URI_COLUMN,  
						   NULL);       
      break;

    case EMAIL_COLUMNS:
      renderer = gtk_cell_renderer_text_new();
      gtk_tree_view_insert_column_with_attributes (view, -1, _("Email URL"),
						   renderer, "text", 
						   URI_COLUMN, NULL);
      break;

    case TELEPHONE_COLUMNS:
      renderer = gtk_cell_renderer_text_new();
      gtk_tree_view_insert_column_with_attributes (view, -1, _("Number"), 
						   renderer, "text", 
						   NUMBER_COLUMN, 
						   NULL);   
      
      renderer = gtk_cell_renderer_text_new();
      gtk_tree_view_insert_column_with_attributes (view, -1, _("Type"), 
						   renderer, "text", 
						   TYPE_COLUMN, 
						   NULL);     
      break;

    default:
      break;
    }
}


GtkWidget *rubrica_ui_pack_in_frame_tree_view_new_with_model(GtkWidget *frame, 
							     GtkTreeModel* model)
{
  GtkWidget *scrollwin;
  GtkWidget *tree_view;

  scrollwin = rubrica_ui_scrolled_window_new();
  tree_view = rubrica_ui_tree_view_new_with_model(model);
  gtk_container_add(GTK_CONTAINER(scrollwin), tree_view);

  gtk_container_add(GTK_CONTAINER(frame), scrollwin);
  
  return (GtkWidget *) tree_view;   
}


GtkWidget *rubrica_ui_pack_in_box_tree_view_new_with_model(GtkWidget *box, 
							   GtkTreeModel* model)
{
  GtkWidget *scrollwin;
  GtkWidget *tree_view; 

  scrollwin = rubrica_ui_scrolled_window_new(); 
  tree_view = rubrica_ui_tree_view_new_with_model(model); 
  gtk_container_add(GTK_CONTAINER(scrollwin), tree_view);

  gtk_box_pack_start(GTK_BOX(box), scrollwin, TRUE, TRUE, 2);
  
  return (GtkWidget *) tree_view; 
}

void rubrica_ui_update_popup_item(GConfClient *client, guint cnxn_id, 
				  GConfEntry *entry, gpointer data)
{
  GtkWidget *menuitem = data;
  gboolean bool;

  bool = gconf_value_get_bool(entry->value);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem), bool); 
}
