/* rsrce -- a Macintosh resource fork editor
 * Copyright (C) 2004  Jeremie Koenig
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <netinet/in.h>
#include <endian.h>

#include "resource.h"


struct resource {
	struct res_fork *f;
	struct resource *next;

	restype_t *type;
	int16_t id;
	uint8_t attr;
	char *name, *data;
	int namelen, datalen;
};

struct res_fork {
	struct resource *rlist;
	
	char *buf;
	int len;
	uint16_t attr;
	int rlen;			/* Reserved space at beginning */
};


struct resource *res_lookup(struct res_fork *f, restype_t type, int16_t id)
{
	struct resource *r;
	for(r=f->rlist ; r ; r=r->next) {
		if(memcmp(*r->type, type, sizeof(type)) == 0 &&
		   (id == 0 || r->id == id)) return r;
	}
	return NULL;
}

static void *res_grab(struct res_fork *f, void *data, int len)
{
	if((char *) data >= f->buf  &&  (char *) data < f->buf + f->len) {
		return data;
	} else {
		f->len += len;
		f->buf = realloc(f->buf, f->len);
		memcpy(f->buf + f->len - len, data, len);
		return f->buf + f->len - len;
	}
}

struct resource *res_new(struct res_fork *f, restype_t type, int16_t id)
{
	struct resource *r, **rp;

	r = malloc(sizeof(*r));
	memset(r, 0, sizeof(*r));
	r->f = f;
	r->id = id;

	for(rp = &f->rlist ; *rp && memcmp(*(*rp)->type, type, sizeof(type)) ;
	    rp = &(*rp)->next);
	if(*rp) {
		r->type = (*rp)->type;
		while(*rp && ((*rp)->type == r->type))
			rp = &(*rp)->next;
	} else {
		r->type = res_grab(f, type, sizeof(type));
	}

	r->next = *rp;
	*rp = r;

	return r;
}

void res_getdata(struct resource *r, void **dp, int *len)
{
	*dp  = r->data;
	*len = r->datalen;
}

void res_setdata(struct resource *r, void *p, int len)
{
	if(len > r->datalen) {
		r->data = res_grab(r->f, p, len);
		r->datalen = len;
	} else {
		memcpy(r->data, p, len);
		r->datalen = len;
	}
}

void res_gettype(struct resource *r, restype_t type)
{
	memcpy(type, r->type, sizeof(type));
}

/* FIXME: implementer. */
void res_settype(struct resource *r, restype_t type)
{
}

void res_rename(struct resource *r, char *name, int nlen)
{
	if(nlen == -1) nlen = strlen(name);
	if(!r->name || nlen > r->namelen) {
		r->name = res_grab(r->f, name, nlen);
		r->namelen = nlen;
	} else {
		memcpy(r->name, name, nlen);
		r->namelen = nlen;
	}
}

const char * const res_attrch = "7spLPlw0";
void res_chattr(struct resource *r, const char *spec)
{
	/* FIXME: please a bit more usable */
	r->attr = atoi(spec);
}

void res_delete(struct resource *r)
{
	struct resource **rp;
	
	for(rp = &r->f->rlist ; *rp && *rp != r ; rp = &(*rp)->next);
	assert(*rp);
	*rp = r->next;
	
	free(r);
}

void res_printinfo(struct resource *r, FILE *s)
{	
	int i;
	
	fwrite(r->type, 4, 1, s);
	fprintf(s, " %11d ", r->id);
	for(i=0 ; i<8 ; i++)
		fprintf(s, "%c", ((r->attr << i) & 0x80) ? res_attrch[i] : '-');
	fprintf(s, " %10d", r->datalen);
	if(r->name) {
		fprintf(s, " ");
		fwrite(r->name, 1, r->namelen, s);
	}
	fprintf(s, "\n");
}

void res_ls(FILE *s, struct res_fork *f)
{
	struct resource *r;
	for(r = f->rlist ; r ; r = r->next)
		res_printinfo(r, s);
}


/* Resource header */
struct reshdr {
	uint32_t dofs, mofs, dlen, mlen;
};

struct res_fork *res_newfork()
{
	struct res_fork *f;

	f = malloc(sizeof(*f));
	f->buf = NULL;
	f->len = 0;
	f->rlen = sizeof(struct reshdr);
	f->attr = 0;
	f->rlist = NULL;

	return f;
}

void res_delfork(struct res_fork *f)
{
	struct resource *r;
	
	if(f->buf) free(f->buf);
	while(f->rlist) {
		r = f->rlist->next;
		free(f->rlist);
		f->rlist = r;
	}
	free(f);
}


/* Resource map header */
struct resmaphdr {
	uint8_t reserved[22];
	uint16_t attr;
	uint16_t tlistofs;
	uint16_t nlistofs;
};

/* Resource type list entry */
struct restype {
	restype_t type;
	uint16_t rnum;
	uint16_t refofs;
};

struct restypelist {
	uint16_t tnum;
	struct restype type[0];
};

/* Resource reference list entry */
struct resref {
	uint16_t id;
	uint16_t nameofs;
	unsigned int attr:8, dataofs:24;
	uint32_t reserved;
};

/* Used to manipulate the 3-bytes offset */
#if __BYTE_ORDER == __LITTLE_ENDIAN
#define ntoh3(a) (ntohl(a) >> 8)
#define hton3(a) (htonl(a) >> 8)
#else
#define ntoh3(a) (a)
#define hton3(a) (a)
#endif

/* Resource name list entry */
struct resname {
	uint8_t len;
	char name[0];
};

/* Resource data */
struct resdata {
	uint32_t len;
	uint8_t data[0];
};


#define BUFSZ 4096
#define MAXSZ (100*1024*1024)

static int res_readbuf(struct res_fork *f, FILE *stream)
{
	int n;

	do {
		f->buf = realloc(f->buf, f->len + BUFSZ);
		n = fread(f->buf + f->len, 1, BUFSZ, stream);
		f->len += n;
	} while(n == BUFSZ && f->len < MAXSZ);
	
	if(f->len >= MAXSZ) {
		fprintf(stderr, "This resource fork is rather huge !\n");
		return -1;
	} else if(!feof(stream)) {
		perror(NULL);
		return -1;
	}

	return 0;
}

#define CHECKPTR(p, m) \
if((char *)(p) < f->buf || (char *)(p) + sizeof(*(p)) > f->buf + f->len) { \
	fprintf(stderr, "Wild " m " offset!\n"); \
	return -1; \
}

static int res_parse_reflist(struct res_fork *f, char *dbase, char *tbase,
		char *nbase, struct restype *t)
{
	struct resource *r;
	struct resref *ref;
	struct resname *name;
	struct resdata *data;
	int i;

	ref = (struct resref *) (tbase + ntohs(t->refofs));
	for(i=0 ; i < ntohs(t->rnum) + 1 ; i++) {
		CHECKPTR(ref+i, "reference");
		r = res_new(f, t->type, ntohs(ref[i].id));
		r->attr = ref[i].attr;
		
		if(ref[i].nameofs != htons(-1)) {
			name = (struct resname *)(nbase+ntohs(ref[i].nameofs));
			CHECKPTR(name, "name");
			res_rename(r, name->name, name->len);
		}

		data = (struct resdata *)(dbase + ntoh3(ref[i].dataofs));
		CHECKPTR(data, "data");
		res_setdata(r, data->data, ntohl(data->len));
	}

	return 0;
}

static int res_parse_typelist(struct res_fork *f, char *dbase, char *tbase,
		char *nbase)
{
	struct restypelist *l = (struct restypelist *) tbase;
	int i;

	for(i=0 ; i < ntohs(l->tnum) + 1 ; i++) {
		if(res_parse_reflist(f, dbase, tbase, nbase, l->type + i) < 0)
			return -1;
	}

	return 0;
}

static int res_parsebuf(struct res_fork *f)
{
	struct reshdr *rh;
	struct resmaphdr *mh;

	rh = (struct reshdr *) f->buf;
	mh = (struct resmaphdr *) (f->buf + ntohl(rh->mofs));
	CHECKPTR(mh, "resource map");
	f->rlen = ntohl(rh->dofs) < ntohl(rh->mofs)
		? ntohl(rh->dofs) : ntohl(rh->mofs);
	return res_parse_typelist(f, f->buf + ntohl(rh->dofs),
			(char *) mh + ntohs(mh->tlistofs),
			(char *) mh + ntohs(mh->nlistofs));
}

struct res_fork *res_read(FILE *stream)
{
	struct res_fork *f;

	f = res_newfork();

	if(res_readbuf(f, stream) < 0  ||  res_parsebuf(f) < 0) {
		res_delfork(f);
		return NULL;
	}
	
	return f;
}

void res_recreate(struct res_fork *f)
{
	struct resource *r;
	restype_t *t;
	char *buf;
	int tnum, rnum;
	int mlen, dlen, nlen, rlen, len;

	struct reshdr *rh;
	struct resmaphdr *mh;
	struct restypelist *tl;
	struct restype *type;
	struct resref *ref;
	struct resname *name, *name0;
	struct resdata *data, *data0;
	
	t = NULL;
	dlen = 0;
	mlen = sizeof(struct resmaphdr) + sizeof(struct restypelist);
	rlen = 0;
	nlen = 0;
	tnum = 0;
	rnum = 0;
	for(r = f->rlist ; r ; r = r->next) {
		dlen += sizeof(struct resdata) + r->datalen;
		if(r->type != t) {
			mlen += sizeof(struct restype);
			tnum++;
			t = r->type;
		}
		rlen += sizeof(struct resref);
		if(r->name)
			nlen += sizeof(struct resname) + r->namelen;
		rnum++;
	}

	len = f->rlen + dlen + mlen + rlen + nlen;
	buf = malloc(len);

	rh = (struct reshdr *) buf;
	data = data0 = (struct resdata *) (buf + f->rlen);
	mh = (struct resmaphdr*) (buf + f->rlen + dlen);
	tl = (struct restypelist *) (mh + 1);
	type = tl->type;
	ref = (struct resref  *) (buf + f->rlen + dlen + mlen);
	name = name0 = (struct resname *) (buf + f->rlen + dlen + mlen + rlen);
	
	rh->dofs = htonl(f->rlen);
	rh->dlen = htonl(dlen);
	rh->mofs = htonl(f->rlen + dlen);
	rh->mlen = htonl(mlen + rlen + nlen);
	
	memset(mh->reserved, 0, sizeof(mh->reserved));
	mh->attr = htons(f->attr);
	mh->tlistofs = htons(sizeof(*mh));
	mh->nlistofs = htons(mlen + rlen);
	tl->tnum = htons(tnum - 1);

	t = NULL;
	for(r = f->rlist ; r ; r = r->next) {
		if(r->type != t) {
			if(t) type++;
			type->refofs = htons((char*) ref - (char*) tl);
			memcpy(type->type, r->type, sizeof(restype_t));
			type->rnum = htons(0);
			t = r->type;
		} else {
			type->rnum = htons(ntohs(type->rnum) + 1);
		}
		r->type = &type->type;
		
		ref->id = htons(r->id);
		ref->nameofs = htons(r->name ? (char*)name - (char*)name0 : -1);
		ref->dataofs = hton3((char *) data - (char *) data0);
		ref->attr = r->attr;
		ref->reserved = htonl(0);
		ref++;

		if(r->name) {
			name->len = r->namelen;
			memcpy(name->name, r->name, r->namelen);
			r->name = name->name;
			name = (struct resname *) (name->name + name->len);
		}

		data->len = htonl(r->datalen);
		memcpy(data->data, r->data, r->datalen);
		r->data = data->data;
		data = (struct resdata *) (data->data + r->datalen);
	}

	free(f->buf);
	f->buf = buf;
	f->len = len;
}

int res_write(struct res_fork *f, FILE *stream)
{
	res_recreate(f);
	return fwrite(f->buf, 1, f->len, stream) == f->len ? 0 : -1;
}

