/****h* ROBODoc/HTML_Generator
 * FUNCTION
 *   The generator for HTML output.
 *
 *   The generator supports sections upto 7 levels deep.  It supports
 *   a Table of Contents based on all headers.  A masterindex for
 *   all headertypes and seperate masterindexes for each headertype.
 *
 * MODIFICATION HISTORY
 *   2003-02-03   Frans Slothouber  Refactoring
 *   ????-??-??   Frans Slothouber  V1.0
 *******
 * $Id: html_generator.c,v 1.43 2003/12/30 17:39:36 gumpu Exp $
 */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <assert.h>
#include <stdlib.h>

#include "html_generator.h"
#include "util.h"
#include "globals.h"
#include "robodoc.h"
#include "links.h"
#include "headers.h"
#include "headertypes.h"
#include "generator.h"
#include "items.h"
#include "folds.h"
#include "string.h"
#include "document.h"
#include "directory.h"
#include "path.h"

#ifdef DMALLOC
#include <dmalloc.h>
#endif

char               *css_name = NULL;


void RB_HTML_Generate_String( FILE * dest_doc, char *a_string );




static void RB_HTML_Generate_Source_Tree_Entry(
        FILE *dest_doc, 
        char *dest_name,
        struct RB_Path* parent_path,
        struct RB_Directory* srctree )
{
    struct RB_Path* cur_path;
    struct RB_Filename* cur_filename;

    fprintf( dest_doc, "<ul>\n" );

    for ( cur_filename = srctree->first;
          cur_filename;
          cur_filename = cur_filename->next )
    {
        if ( cur_filename->path == parent_path )
        {
            char               *r = 0;

            r = RB_HTML_RelativeAddress( dest_name, cur_filename->link->file_name );
            fprintf( dest_doc, "<li>\n" );
            fprintf( dest_doc, "<a href=\"%s#%s\"><tt>\n", r,
                    cur_filename->link->label_name );
            RB_HTML_Generate_String( dest_doc, cur_filename->name );
            fprintf( dest_doc, "</tt></a></li>\n" );
        }
    }
    for( cur_path = srctree->first_path;
         cur_path;
         cur_path = cur_path->next )
    {
        if ( cur_path->parent == parent_path )
        {
            fprintf( dest_doc, "<li>\n" );
            RB_HTML_Generate_String( dest_doc, cur_path->name );
            RB_HTML_Generate_Source_Tree_Entry( dest_doc, dest_name, cur_path, srctree );
            fprintf( dest_doc, "</li>\n" );
        }
    }
    fprintf( dest_doc, "</ul>\n" );
}


void RB_HTML_Generate_Source_Tree( 
        FILE *dest_doc, 
        char *dest_name,
        struct RB_Document *document )
{
    struct RB_Directory* srctree;
    srctree = document->srctree;
    RB_HTML_Generate_Source_Tree_Entry(
                dest_doc,
                dest_name,
                NULL,
                srctree );
}



/****if* HTML_Generator/RB_HTML_Generate_String
 * FUNCTION
 *   Write a string to the destination document, escaping
 *   characters where necessary.
 * SYNOPSIS
 *   void RB_HTML_Generate_String( FILE* dest_doc, char* a_string )
 * INPUTS
 *   o dest_doc -- the file the characters are written too
 *   o a_string -- a nul terminated string.
 * SEE ALSO
 *   RB_HTML_Generate_Char()
 * SOURCE
 */

void
RB_HTML_Generate_String( FILE * dest_doc, char *a_string )
{
    int                 i;
    int                 l = strlen( a_string );
    unsigned char       c;

    for ( i = 0; i < l; ++i )
    {
        c = a_string[i];
        RB_HTML_Generate_Char( dest_doc, c );
    }
}

/*******/


/****if* HTML_Generator/RB_HTML_Generate_False_Link
 * FUNCTION
 *   Create a representation for a link that links an word in
 *   a header to the header itself.
 * SYNOPSIS
 *   void RB_HTML_Generate_False_Link( FILE * dest_doc, char *name )
 * INPUTS
 *   dest_doc -- the file the representation is written to.
 *   name     -- the word.
 * SOURCE
 */

void
RB_HTML_Generate_False_Link( FILE * dest_doc, char *name )
{
    fprintf( dest_doc, "<strong>" );
    RB_HTML_Generate_String( dest_doc, name );
    fprintf( dest_doc, "</strong>" );
}

/*******/


/****f* HTML_Generator/RB_HTML_Generate_Extra
 * FUNCTION
 *   Do some additional processing to detect HTML extra's like
 *   file references and other kind of links for the documentation
 *   body of an item.
 * INPUTS
 *   o dest_doc  -- the file to write to.
 *   o item_type -- the kind of item the body belongs to.
 *   o cur_char  -- pointer to a substring of the item's body
 * RESULTS
 *   Number of character produced.
 * SOURCE
 */

int
RB_HTML_Generate_Extra( FILE * dest_doc, int item_type, char *cur_char )
{
    char                link[1024];
    int                 res = 0, prev;
    static char         incomment = 0;  /* are we in comment? */
    static char         quote = 0;      /* double quote */
    static char         squote = 0;     /* single quote */

    if ( ( course_of_action & DO_CMODE ) && ( item_type == SOURCE_ITEM ) )
    {
        if ( !incomment )
        {
            /* hope it is ok to check prev char like this */
            prev = *( ( char * ) ( cur_char - 1 ) );
            /* check & set flags */
            switch ( *cur_char )
            {
            case '\"':
                if ( prev != '\\' )
                {
                    quote = !quote;
                }
                else if ( quote && *( ( char * ) ( cur_char - 2 ) ) == '\\' )
                {
                    quote = !quote;     /* case "... \\" */
                }
                break;
            case '\'':         /* open double quote with ' embedded? */
                if ( prev != '\\' )
                {
                    if ( quote == 0 )
                    {
                        squote = !squote;
                    }
                }
                else if ( squote && *( ( char * ) ( cur_char - 2 ) ) == '\\' )
                {
                    squote = !squote;   /* case '\\' */
                }
                break;
            default:
                break;
            }                   /* end switch(*cur_char) */
        }                       /* end if(!incomment) */
        /* additional C-mode formatting, currently comments only */
        if ( !( quote || squote ) )
        {
            if ( strncmp( "/*", cur_char, 2 ) == 0 )
            {
                /* start of C comment */
                fprintf( dest_doc, "<font color=\"#FF0000\">/*" );
                res = 1;
                incomment = 1;
            }
            else if ( strncmp( "*/", cur_char, 2 ) == 0 )
            {
                /* end of C comment */
                fprintf( dest_doc, "*/</font>" );
                res = 1;
                incomment = 0;
            }
        }                       /* end if (!(quote || squote)) */
    }
    /* return now if something was processed above */
    if ( res > 0 )
    {
        return res;
    }
    if ( incomment || item_type != SOURCE_ITEM )
    {
        if ( strncmp( "http://", cur_char, 7 ) == 0 )
        {
            sscanf( cur_char, "%s", link );
            RB_Say( "found link %s\n", link );
            res = ( strlen( link ) - 1 );
            /* [ 697247 ] http://body. does not skip the '.' */
            if ( link[( strlen( link ) - 1 )] == '.' )
            {
                link[( strlen( link ) - 1 )] = '\0';
                fprintf( dest_doc, "<a href=\"%s\">%s</a>.", link, link );
            }
            else
            {
                fprintf( dest_doc, "<a href=\"%s\">%s</a>", link, link );
            }
        }
        else if ( strncmp( "href:", cur_char, 5 ) == 0 )
        {
            /*
             * handy in relative hyperlink paths, e.g.
             * href:../../modulex/
             */
            sscanf( ( cur_char + 5 ), "%s", link );
            RB_Say( "found link %s\n", link );
            res = ( strlen( link ) + 4 );
            fprintf( dest_doc, "<a href=\"%s\">%s</a>", link, link );
        }
        else if ( strncmp( "file:/", cur_char, strlen( "file:/" ) ) == 0 )
        {
            sscanf( cur_char, "%s", link );
            RB_Say( "found link %s\n", link );
            res = ( strlen( link ) - 1 );
            fprintf( dest_doc, "<a href=\"%s\">%s</a>", link, link );
        }
        else if ( strncmp( "mailto:", cur_char, 7 ) == 0 )
        {
            sscanf( ( cur_char + 7 ), "%s", link );
            RB_Say( "found mail to %s\n", link );
            res = ( strlen( link ) + 6 );
            fprintf( dest_doc, "<a href=\"mailto:%s\">%s</a>", link, link );
        }
        else if ( strncmp( "image:", cur_char, 6 ) == 0 )
        {
            sscanf( ( cur_char + 6 ), "%s", link );
            RB_Say( "found image %s\n", link );
            res = ( strlen( link ) + 5 );
            fprintf( dest_doc, "<img src=\"%s\">", link );
        }

    }
    return res;
}

/******/


void
RB_HTML_Generate_Item_Begin( FILE * dest_doc )
{
    fprintf( dest_doc, "<pre>" );
}

void
RB_HTML_Generate_Item_End( FILE * dest_doc )
{
    fprintf( dest_doc, "</pre>" );
}

void
RB_HTML_Generate_TOC_1( FILE * dest_doc, struct RB_header *header )
{
    assert( 0 );
}

int                 sectiontoc_counters[MAX_SECTION_DEPTH];


/****f* HTML_Generator/RB_HTML_Generate_TOC_Section
 * FUNCTION
 *   Create a table of contents based on the hierarchy of
 *   the headers starting for a particular point in this
 *   hierarchy (the parent).
 * SYNOPSIS
 *   void
 *   RB_HTML_Generate_TOC_Section( FILE * dest_doc, char *dest_name,
 *                                 struct RB_header *parent,
 *                                 struct RB_header **headers, int count,
 *                                 int depth )
 * INPUTS
 *   o dest_doc  -- the file to write to.
 *   o dest_name -- the name of this file.
 *   o parent    -- the parent of the headers for which the the
 *                  current level(depth) of TOC is created.
 *   o headers   -- an array of headers for which the TOC is created
 *   o count     -- the number of headers in this array
 *   o depth     -- the current depth of the TOC
 * NOTES
 *   This is a recursive function and tricky stuff.
 * SOURCE
 */

void
RB_HTML_Generate_TOC_Section( FILE * dest_doc, char *dest_name,
                              struct RB_header *parent,
                              struct RB_header **headers, int count,
                              int depth )
{
    struct RB_header   *header;
    int                 i;

    ++sectiontoc_counters[depth];

    for ( i = depth + 1; i < MAX_SECTION_DEPTH; ++i )
    {
        sectiontoc_counters[i] = 0;
    }

    fprintf( dest_doc, "<li>" );
    for ( i = 1; i <= depth; ++i )
    {
        fprintf( dest_doc, "%d.", sectiontoc_counters[i] );
    }
    fprintf( dest_doc, "   " );
    RB_HTML_Generate_Link( dest_doc, dest_name, parent->file_name,
                           parent->unique_name, parent->name, 0 );
    fprintf( dest_doc, "</li>\n" );


    for ( i = 0; i < count; ++i )
    {
        header = headers[i];
        if ( header->parent == parent )
        {
            RB_HTML_Generate_TOC_Section( dest_doc, dest_name, header,
                                          headers, count, depth + 1 );
        }
        else
        {
            /* Empty */
        }
    }
}

/*******/


void
RB_HTML_Generate_TOC_2( FILE * dest_doc, struct RB_header **headers,
                        int count, struct RB_Part *owner, char *dest_name )
{
    struct RB_header   *header;
    int                 i;
    int                 depth = 1;

    for ( i = 0; i < MAX_SECTION_DEPTH; ++i )
    {
        sectiontoc_counters[i] = 0;
    }
    fprintf( dest_doc, "<h3 align=\"center\">TABLE OF CONTENTS</h3>\n" );
    if ( course_of_action & DO_SECTIONS )
    {
        /* --sections was specified, create a TOC based on the
         * hierarchy of the headers.
         */
        fprintf( dest_doc, "<ul>\n" );
        for ( i = 0; i < count; ++i )
        {
            header = headers[i];
            if ( owner == NULL )
            {
                if ( header->parent )
                {
                    /* Will be done in the subfunction */
                }
                else
                {
                    RB_HTML_Generate_TOC_Section( dest_doc, dest_name, header,
                                                  headers, count, depth );
                }
            }
            else
            {
                /* This is the TOC for a specific RB_Part (MultiDoc
                 * documentation). We only include the headers that
                 * are part of the subtree. That is, headers that are
                 * parth the RB_Part, or that are childern of the
                 * headers in the RB_Part.
                 */
                if ( header->owner == owner )
                {
                    /* BUG 721690 */
                    /* Any of the parents of this header should not
                     * have the same owner as this header, otherwise
                     * this header will be part of the TOC multiple times.
                     */
                    int                 no_bad_parent = TRUE;
                    struct RB_header   *parent = header->parent;

                    for ( ; parent; parent = parent->parent )
                    {
                        if ( parent->owner == owner )
                        {
                            no_bad_parent = FALSE;
                            break;
                        }
                    }
                    if ( no_bad_parent )
                    {
                        RB_HTML_Generate_TOC_Section( dest_doc, dest_name,
                                                      header, headers, count,
                                                      depth );
                    }
                }
                else
                {

                }
            }
        }
        fprintf( dest_doc, "</ul>\n" );
    }
    else
    {
        /* No --section option, generate a plain, one-level
         * TOC
         */
        fprintf( dest_doc, "<ul>\n" );

        for ( i = 0; i < count; ++i )
        {
            header = headers[i];
            if ( header->name && header->function_name )
            {
                if ( ( owner == NULL ) || ( header->owner == owner ) )
                {
                    fprintf( dest_doc, "<li>" );
                    RB_HTML_Generate_Link( dest_doc, dest_name,
                                           header->file_name,
                                           header->unique_name, header->name,
                                           0 );
                    fprintf( dest_doc, "</li>\n" );
                }
            }
        }
        fprintf( dest_doc, "</ul>\n" );
    }
}



void
RB_HTML_Generate_Item_Name( FILE * dest_doc, char *name )
{
    fprintf( dest_doc, "<p><strong>" );
    RB_HTML_Generate_String( dest_doc, name );
    fprintf( dest_doc, "</strong></p>\n" );
}


/****f* HTML_Generator/RB_HTML_Generate_Label
 * FUNCTION
 *   Generate a label (name) that can be refered too.
 *   A label should consist of only alphanumeric characters so
 *   all 'odd' characters are replaced with their ASCII code in
 *   hex format.
 * SYNOPSIS
 *   void RB_HTML_Generate_Label( FILE * dest_doc, char *name )
 * INPUTS
 *   o dest_doc -- the file to write it to.
 *   o name     -- the name of the label.
 * SOURCE
 */

void
RB_HTML_Generate_Label( FILE * dest_doc, char *name )
{
    int                 i;
    int                 l = strlen( name );
    unsigned char       c;

    fprintf( dest_doc, "<a name=\"" );
    for ( i = 0; i < l; ++i )
    {
        c = name[i];
        if ( isalnum( c) ) 
        {
            RB_HTML_Generate_Char( dest_doc, c );
        }
        else
        {
            char buf[4];
            sprintf(buf, "%02x", c );
            RB_HTML_Generate_Char( dest_doc, buf[0] );
            RB_HTML_Generate_Char( dest_doc, buf[1] );
        }
    }
    fprintf( dest_doc, "\">\n" );
}

/********/



int                 section_counters[MAX_SECTION_DEPTH];


void
RB_HTML_Generate_BeginSection( FILE * dest_doc, int depth, char *name )
{
    int                 i;

    ++section_counters[depth];
    for ( i = depth + 1; i < MAX_SECTION_DEPTH; ++i )
    {
        section_counters[i] = 0;
    }
    switch ( depth )
    {
    case 1:
    case 2:
    case 3:
    case 4:
    case 5:
    case 6:
    case 7:
        fprintf( dest_doc, "<h%d>", depth );
        for ( i = 1; i <= depth; ++i )
        {
            fprintf( dest_doc, "%d.", section_counters[i] );
        }
        fprintf( dest_doc, "  " );
        RB_HTML_Generate_String( dest_doc, name );
        fprintf( dest_doc, "  </h%d>\n", depth );
        break;
    default:
        /* too deep, don't do anything. */
        assert( 0 );
    }
}

void
RB_HTML_Generate_EndSection( FILE * dest_doc, int depth, char *name )
{
    /* Empty */
}


char               *
RB_HTML_Get_Default_Extension( void )
{
    return ( ".html" );
}

/****f* HTML_Generator/RB_HTML_Generate_Doc_Start
 * NAME
 *   RB_HTML_Generate_Doc_Start --
 * FUNCTION
 *   Generate the first part of a HTML document.
 *   As far as ROBODoc is concerned a HTML document
 *   consists of three parts.
 *   (1) The start of a document
 *   (2) The body of a document
 *   (3) The end of a document
 * INPUTS
 *   o dest_doc  --  the output file.
 *   o src_name  --  The file or directoryname from which 
 *                   this document is generated.
 *   o name      --  The title for this document
 *   o toc       --  unused!
 *   o dest_name --  the name of the output file.
 *   o charset   --  the charset to be used for the file.
 * HISTORY
 *   12/21/2002 Fold code modifications (PetteriK)
 * SOURCE
 */

void
RB_HTML_Generate_Doc_Start( FILE * dest_doc, char *src_name, char *name,
                            char toc, char *dest_name, char *charset )
{
    /* Append document type and title */
    fprintf( dest_doc,
             "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2//EN\">\n" );
    fprintf( dest_doc, "<html>\n<head>\n" );
    RB_InsertCSS( dest_doc, dest_name );

    if ( charset )
    {
        fprintf( dest_doc,
                 "<meta http-equiv=\"Content-type\" content=\"text/html; charset=%s\">\n",
                 charset );
    }

    fprintf( dest_doc, "<title>%s</title>\n", name );

    /* append SGML-comment with document- and copyright-info. This code
     * ensures that every line has an own comment to avoid problems with
     * buggy browsers */
    fprintf( dest_doc, "<!-- Source: %s -->\n", src_name );
    {
        static const char   copyright_text[]
            = COMMENT_ROBODOC COMMENT_COPYRIGHT;
        size_t              i = 0;
        char                previous_char = '\n';
        char                current_char = copyright_text[i];

        while ( current_char )
        {
            if ( previous_char == '\n' )
            {
                fprintf( dest_doc, "<!-- " );
            }
            if ( current_char == '\n' )
            {
                fprintf( dest_doc, " -->" );
            }
            else if ( ( current_char == '-' ) && ( previous_char == '-' ) )
            {
                /* avoid "--" inside SGML-comment, and use "-_" instead; this
                 * looks a bit strange, but one should still be able to figure
                 * out what is meant when reading the output */
                current_char = '_';
            }
            fputc( current_char, dest_doc );
            i += 1;
            previous_char = current_char;
            current_char = copyright_text[i];
        }
    }

    /* append heading and start list of links to functions */
    fprintf( dest_doc, "</head>\n<body bgcolor=\"#FFFFFF\">\n" );

    /* handle folds bit differently */
    if ( ( extra_flags & FOLD ) && RB_Inside_Fold(  ) )
    {
        fprintf( dest_doc,
                 "Generated with ROBODoc v%s from <i>%s</i> on ",
                 VERSION, name );
        RB_TimeStamp( dest_doc );
        fprintf( dest_doc, "<br /><hr />\n" );
        /* fold name, use ... notation */
        fprintf( dest_doc, "<strong>...</a> %s</strong><br />", src_name );
    }
    else
    {
        /* normal case, no folding business */
        fprintf( dest_doc,
                 "<a name=\"%s\">Generated from %s</a> with <a href=\"http://sourceforge.net/projects/robodoc/\">ROBODoc</a> v%s on ",
                 src_name, src_name, VERSION );
        RB_TimeStamp( dest_doc );
        fprintf( dest_doc, "<br />\n" );
    }
}

/******/


/* TODO */
/*x**if* HTML_Generator/RB_HTML_Generate_Doc_End
 * NAME
 *   RB_HTML_Generate_Doc_End --
 * FUNCTION
 *   Close of the document with the proper end tags.
 ******
 */

void
RB_HTML_Generate_Doc_End( FILE * dest_doc, char *name )
{
    fprintf( dest_doc, "</body>\n</html>\n" );
}


/* TODO */
/*x**if* HTML_Generator/RB_HTML_Generate_Header_Start
 * NAME
 *   RB_HTML_Generate_Header_Start --
 ******
 * OBSOLETE
 */

void
RB_HTML_Generate_Header_Start( FILE * dest_doc, struct RB_header *cur_header )
{
    if ( cur_header->name && cur_header->function_name )
    {
        fprintf( dest_doc, "<hr />\n" );
        fprintf( dest_doc, "\n<h2><a name=\"%s\">", cur_header->unique_name );
        RB_HTML_Generate_Label( dest_doc, cur_header->name );
        RB_HTML_Generate_String( dest_doc, cur_header->name );
        fprintf( dest_doc, "</a></h2>\n\n" );
    }
}

/* TODO */
/*x**f* HTML_Generator/RB_HTML_Generate_Header_End
 * NAME
 *   RB_HTML_Generate_Header_End --
 ******
 */

void
RB_HTML_Generate_Header_End( FILE * dest_doc, struct RB_header *cur_header )
{
    fprintf( dest_doc, "\n" );
}


/* TODO */
/*x**f* HTML_Generator/RB_HTML_Generate_IndexMenu
 * FUNCTION
 *   Generates a menu to jump to the various master index files for
 *   the various header types.  The menu is generated for each of the
 *   master index files.  The current header type is highlighted.
 * INPUTS
 *   dest_doc       -- the output file.
 *   document       -- the gathered documention.
 *   cur_headertype -- the header type that is to be highlighted.
 ******
 */

void
RB_HTML_Generate_IndexMenu( 
        FILE* dest_doc, 
        char* filename,
        struct RB_Document* document, 
        struct RB_HeaderType* cur_type )
{
    char type_char;

    assert( dest_doc );
    assert( document );
    assert( cur_type );


    fprintf( dest_doc, "<p>\n" );

    for ( type_char = MIN_HEADER_TYPE;
          type_char < MAX_HEADER_TYPE;
          ++type_char )
    {
        struct RB_HeaderType* header_type;
        int n;

        header_type = RB_FindHeaderType( type_char );
        if ( header_type )
        {
            n = RB_Number_Of_Links( header_type, NULL, FALSE ) +
                RB_Number_Of_Links( header_type, NULL, TRUE );

            if ( n )
            {
                char* targetfilename = 0;
                targetfilename = RB_Get_SubIndex_FileName( 
                        document->docroot->name, 
                        document->extension, 
                        header_type );
                assert( targetfilename );

                fprintf( dest_doc, "[" );
                RB_HTML_Generate_Link(
                        dest_doc,
                        filename,
                        targetfilename,
                        "top",
                        header_type->indexName,
                        0 );
                fprintf( dest_doc, "]\n" );

                free( targetfilename );
            }
        }
    }

    fprintf( dest_doc, "</p>\n" );
}

/* TODO */
/*x**f* HTML_Generator/RB_HTML_Generate_Index_Page
 * FUNCTION
 *   Generate a single file with a index table for headers
 *   of one specific type of headers
 * INPUTS
 *   o document    -- the document
 *   o header_type -- the type for which the table is to
 *                    be generated.
 ******
 */

void RB_HTML_Generate_Index_Page( struct RB_Document* document, 
        struct RB_HeaderType* header_type )
{
    char* filename = 0;
    FILE* file;

    assert( document );
    assert( header_type );

    filename = RB_Get_SubIndex_FileName( document->docroot->name, 
            document->extension, header_type );
    assert( filename );

    file = fopen( filename, "w" );
    if ( !file )
    {
        RB_Panic( "can't open (%s)!\n", filename );
    }
    else
    {
        /* File opened, now we generate an index
         * for the specified header type 
         */
        RB_HTML_Generate_Doc_Start( file, 
                document->srcroot->name, 
                header_type->indexName, 
                0, 
                filename, 
                document->charset );
        RB_HTML_Generate_IndexMenu(
                file,
                filename,
                document,
                header_type );
        if ( header_type )
        {
            if ( RB_CompareHeaderTypes( header_type, RB_FindHeaderType( HT_SOURCEHEADERTYPE ) ) &&
                 ( header_type->typeCharacter != HT_MASTERINDEXTYPE ) )
            {
                RB_HTML_Generate_Source_Tree( file, filename, document );
            }
            else
            {
                RB_HTML_Generate_Index_Table(
                        file,
                        filename,
                        header_type,
                        header_type->indexName );
            }
        }
        RB_HTML_Generate_Doc_End( file, filename );
        fclose( file );
    }

    free( filename );
}


/* TODO */
/*x**if* HTML_Generator/RB_HTML_Generate_Index
 * NAME
 *   RB_HTML_Generate_Index --
 ******
 */


/* Should be called indexes */
void RB_HTML_Generate_Index( struct RB_Document *document )
{
    unsigned char  type_char = 0;

    assert( document );

    for ( type_char = MIN_HEADER_TYPE;
          type_char < MAX_HEADER_TYPE;
          ++type_char )
    {
        struct RB_HeaderType* header_type;
        int n;

        header_type = RB_FindHeaderType( type_char );
        if ( header_type )
        {
            n = RB_Number_Of_Links( header_type, NULL, FALSE ) +
                RB_Number_Of_Links( header_type, NULL, TRUE );
            if ( n )
            {
                /* There are headers of this type, so create an index page
                 * for them
                 */
                RB_HTML_Generate_Index_Page( document, header_type );
            }
        }
    }

    RB_HTML_Generate_Index_Page( document, RB_FindHeaderType( HT_MASTERINDEXTYPE ) );
}



void
RB_EvenUneven_Class( FILE * f, int n )
{
    if ( n & 1 )
    {
        fprintf( f, " class=\"uneven\"" );
    }
    else
    {
        fprintf( f, " class=\"even\"" );
    }
}



unsigned int
RB_NumberOfColums( struct RB_HeaderType* type, unsigned int max_width,
       int internal )
{
    unsigned int        number_of_col = 0;
    unsigned int        cur_max_width = 0;
    unsigned int        number_of_items = 0;
    unsigned int       *col_maximums =
        calloc( max_width, sizeof( unsigned int ) );

    assert( col_maximums );

    do
    {
        unsigned int        i;
        unsigned int        cur_col = 0;
        unsigned int        width = 0;

        number_of_items = 0;
        cur_max_width = 0;
        ++number_of_col;

        /* Reset the maximums */
        for ( i = 0; i < number_of_col; ++i )
        {
            col_maximums[i] = 0;
        }

        /* Compute maxium width of each column in the table */
        for ( i = 0; i < link_index_size; ++i )
        {
            struct RB_link     *cur_link;

            cur_link = link_index[i];
            if ( cur_link->htype && 
                 RB_CompareHeaderTypes( cur_link->htype, type ) &&
                 ( ( cur_link->is_internal && internal ) ||
                   ( !cur_link->is_internal && !internal ) ) )
            {
                ++number_of_items;
                width = strlen( cur_link->object_name );
                if ( width > col_maximums[cur_col] )
                {
                    col_maximums[cur_col] = width;
                }
                ++cur_col;
                assert( cur_col < max_width );
                if ( cur_col > number_of_col )
                {
                    cur_col = 0;
                }
            }
        }
        /* Compute the maximum table width by adding the maximum witdh
         * of each column.
         */
        cur_max_width = 0;
        for ( i = 0; i < number_of_col; ++i )
        {
            cur_max_width += col_maximums[i];
        }
    }
    while ( ( cur_max_width <= max_width )
            && ( number_of_items > number_of_col ) );
    /* The previous number as the current number will create a
     * table wider than the max_width */
    --number_of_col;
    free( col_maximums );
    return ( number_of_col );
}


void RB_HTML_Generate_Table_Body( FILE* dest, char *dest_name, struct RB_HeaderType* type, int internal )
{
    struct RB_link     *cur_link;
    int                 cur_column;
    int                 number_of_columns;
    int                 row = 0;
    unsigned int        i;
    unsigned int        n = 0;

    /* Compute the number of columns we need for
     * this type of header.
     */
    number_of_columns = RB_NumberOfColums( type, 80, internal );
    cur_column = 0;
    for ( i = 0; i < link_index_size; ++i )
    {
        cur_link = link_index[i];
        if ( cur_link->htype && 
             RB_CompareHeaderTypes( cur_link->htype, type ) &&
             ( ( cur_link->is_internal && internal ) ||
               ( !cur_link->is_internal && !internal ) ) )
        {
            char               *r = 0;

            ++n;
            if ( cur_column == 0 )
            {
                fprintf( dest, "<tr " );
                RB_EvenUneven_Class( dest, row );
                ++row;
                fprintf( dest, ">\n" );
            }

            r = RB_HTML_RelativeAddress( dest_name, cur_link->file_name );
            fprintf( dest, "<td " );
            RB_EvenUneven_Class( dest, n );
            fprintf( dest, "><a href=\"%s#%s\"><tt>\n", r,
                    cur_link->label_name );
            RB_HTML_Generate_String( dest, cur_link->object_name );
            fprintf( dest, "</tt></a></td>\n" );

            ++cur_column;
            if ( cur_column > number_of_columns )
            {
                fprintf( dest, "</tr>\n" );
                cur_column = 0;
                if ( number_of_columns & 1 )
                {
                    ++n;
                }
                else
                {

                }
            }
        }
    }
    for ( ; cur_column <= number_of_columns; )
    {
        if ( cur_column == 0 )
        {
            fprintf( dest, "<tr " );
            RB_EvenUneven_Class( dest, row );
            ++row;
            fprintf( dest, ">\n" );
        }
        fprintf( dest, "<td></td>\n" );
        cur_column++;
    }
    fprintf( dest, "</tr>\n" );
}


/****if* HTML_Generator/RB_HTML_Generate_Index_Table
 * NAME
 *   RB_HTML_Generate_Index_Table --
 * FUNCTION
 *   Create a HTML TABLE containing links to headers of a particular
 *   type.  This creates two tables, a table for normal headers as
 *   well as one for internal headers.
 * INPUTS
 *   o dest        -- the file in which to write the table
 *   o dest_name   -- the name of this file
 *   o type        -- the type of header for which to generate
 *                    the table
 *   o title       -- the title of the table.
 * SOURCE
 */

void
RB_HTML_Generate_Index_Table( 
        FILE * dest, char *dest_name,
        struct RB_HeaderType* type,
        char *title )
{
    /* Compute the number of columns we need for
     * this type of header.
     */

    fprintf( dest, "<h1>" );
    RB_HTML_Generate_String( dest, title );
    fprintf( dest, "</h1>\n" );

    if( RB_Number_Of_Links( type, NULL, FALSE ) ) 
    {
        if (  RB_Number_Of_Links( type, NULL, TRUE ) ) 
        {
            /* only print a title if there are two tables. */
            fprintf( dest, "<h2>Normal</h2>" );
        }
        fprintf( dest, "<table cellspacing=\"3\">\n" );
        RB_HTML_Generate_Table_Body( dest, dest_name, type, FALSE );
        fprintf( dest, "</table>\n" );
    }

    if( RB_Number_Of_Links( type, NULL, TRUE ) ) {
        /* Always print the  Internal title, since
         * these headers are special and the user should know
         * he is looking at something special.
         */
        fprintf( dest, "<h2>Internal</h2>" );
        fprintf( dest, "<table cellspacing=\"3\">\n" );
        RB_HTML_Generate_Table_Body( dest, dest_name, type, TRUE );
        fprintf( dest, "</table>\n" );
    }
}

/********/



/* TODO */
/*x**if* HTML_Generator/RB_HTML_Generate_Empty_Item
 * NAME
 *   RB_HTML_Generate_Empty_Item --
 ******
 */

void
RB_HTML_Generate_Empty_Item( FILE * dest_doc )
{
    fprintf( dest_doc, "<br>\n" );
}




/****f* HTML_Generator/RB_HTML_Generate_Link
 * NAME
 *   RB_HTML_Generate_Link --
 * SYNOPSIS
 *   void RB_HTML_Generate_Link( FILE* dest_doc, char* dest_name, char* filename,
 *          char* labelname, char* linkname, int in_fold )
 * INPUTS
 *   dest_doc  --  the file to which the text is written
 *   dest_name --  the name of the destination file
 *                 (the file from which we link)
 *   filename  --  the name of the file that contains the link
 *                 (the file we link to)
 *   labelname --  the name of the unique label of the link.
 *   linkname  --  the name of the link as shown to the user.
 *   in_fold   --  are we inside a fold.
 * SOURCE
 */

void
RB_HTML_Generate_Link( FILE * dest_doc, char *dest_name, char *filename,
                       char *labelname, char *linkname, int in_fold )
{
    /* Include the file name in the link if we are in fold
     * PetteriK 08.04.2000
     */
    if ( in_fold )
    {
        /* We are in fold, always use the file name in the link,
         * in filename == NULL (i.e. the label is in the current file
         * that we are processing), refer to value in dest_name.
         * This also makes sure that we link correctly if linkname
         * is the same as labelname.
         */
        fprintf( dest_doc, "<a href=\"%s#%s\">%s</a>",
                 ( filename ? filename : dest_name ), labelname, linkname );
    }
    else if ( filename && strcmp( filename, dest_name ) )
    {
        char               *r =
            RB_HTML_RelativeAddress( dest_name, filename );
        fprintf( dest_doc, "<a href=\"%s#%s\">", r, labelname );
        RB_HTML_Generate_String( dest_doc, linkname );
        fprintf( dest_doc, "</a>" );

    }
    else
    {
        fprintf( dest_doc, "<a href=\"#%s\">", labelname );
        RB_HTML_Generate_String( dest_doc, linkname );
        fprintf( dest_doc, "</a>" );
    }
}

/******/


/****f* HTML_Generator/RB_HTML_RelativeAddress
 * FUNCTION
 *   Link to 'that' from 'this' computing the relative path.  Here
 *   'this' and 'that' are both paths.
 *   This function is used to create links from one document to
 *   another document that might be in a completely different 
 *   directory.
 * SYNOPSIS
 *   char* RB_HTML_RelativeAddress( char *thisname, char *thatname )
 * EXAMPLE
 *   this /sub1/sub2/sub3/f.html
 *   that /sub1/sub2/g.html
 *   ==
 *   ../g.html
 *
 *   this /sub1/f.html
 *   that /sub1/sub2/g.html
 *   ==
 *   ./sub2/g.html
 *
 *   this /sub1/f.html
 *   that /sub1/g.html
 *   ==
 *   ./g.html
 *
 *   this /sub1/doc3/doc1/tt.html
 *   that /sub1/doc5/doc2/qq.html
 *   ==
 *   ../../doc5/doc2/qq.html
 * NOTES
 *   notice the execelent docmentation...
 * SOURCE
 */

#define MAX_RELATIVE_SIZE 1024
char               *
RB_HTML_RelativeAddress( char *thisname, char *thatname )
{
    static char         relative[MAX_RELATIVE_SIZE + 1];
    char               *i_this;
    char               *i_that;
    char               *i_this_slash = NULL;
    char               *i_that_slash = NULL;

    relative[0] = '\0';

    assert( thisname );
    assert( thatname );

    for ( i_this = thisname, i_that = thatname;
          ( *i_this && *i_that ) && ( *i_this == *i_that );
          ++i_this, ++i_that )
    {
        if ( *i_this == '/' )
        {
            i_this_slash = i_this;
        }
        if ( *i_that == '/' )
        {
            i_that_slash = i_that;
        }
    }

    if ( i_this_slash && i_that_slash )
    {
        int                 this_slashes_left = 0;
        int                 that_slashes_left = 0;
        char               *i_c;

        for ( i_c = i_this_slash + 1; *i_c; ++i_c )
        {
            if ( *i_c == '/' )
            {
                ++this_slashes_left;
            }
        }
        for ( i_c = i_that_slash + 1; *i_c; ++i_c )
        {
            if ( *i_c == '/' )
            {
                ++that_slashes_left;
            }
        }
        if ( this_slashes_left )
        {
            int                 i;

            for ( i = 0; i < this_slashes_left; ++i )
            {
                strcat( relative, "../" );
            }
            strcat( relative, i_that_slash + 1 );
        }
        else if ( !this_slashes_left && that_slashes_left )
        {
            strcat( relative, "./" );
            strcat( relative, i_that_slash + 1 );
        }
        else if ( !this_slashes_left && !that_slashes_left )
        {
            strcat( relative, "./" );
            strcat( relative, i_that_slash + 1 );
        }
        else
        {
            /* cannot get here... so it must be a BUG */
            assert( 0 );
        }
    }
    return relative;
}

/******/



/****f* HTML_Generator/RB_HTML_Generate_Char
 * NAME
 *   RB_HTML_Generate_Char -- generate a single character for an item.
 * SYNOPSIS
 *   void RB_HTML_Generate_Char (FILE* dest_doc, int c)
 * FUNCTION
 *   This function is called for every character that goes
 *   into an item's body.  This escapes all the reserved
 *   HTML characters such as &, < and >.
 * SOURCE
 */

void
RB_HTML_Generate_Char( FILE * dest_doc, int c )
{
    switch ( c )
    {
    case '\n':
        assert( 0 );
        break;
    case '\t':
        assert( 0 );
        break;
    case '<':
        fprintf( dest_doc, "&lt;" );
        break;
    case '>':
        fprintf( dest_doc, "&gt;" );
        break;
    case '&':
        fprintf( dest_doc, "&amp;" );
        break;
    default:
        fputc( c, dest_doc );
    }
}

/*******/



/****f* HTML_Generator/RB_HTML_Generate_Char
 * FUNCTION
 *   Create the .css file.  Unless the user specified it's own css
 *   file robodoc creates a default one.
 *
 *   For multidoc mode the name of the .css file is
 *      robodoc.css
 *   For singledoc mode the name of the .css file is equal
 *   to the name of the documentation file.
 * INPUTS
 *   o document -- the document for which to create the file.
 * SOURCE
 */

void
RB_Create_CSS( struct RB_Document *document )
{
    size_t              l = 0;
    FILE               *css_file;

    /* compute the complete path to the css file */
    if ( ( document->actions & DO_SINGLEDOC ) ||
         ( document->actions & DO_SINGLEFILE ) ) {
        char* extension = ".css";

        l += strlen( document->singledoc_name );
        l += strlen( extension );
        ++l;
        css_name = malloc( l );
        strcpy( css_name, document->singledoc_name );
        strcat( css_name, extension );
    } else {
        struct RB_Path     *docroot = document->docroot;
        char               *docrootname = docroot->name;
        char               *filename = "robodoc.css";

        l = strlen( filename );
        l += strlen( docrootname );
        ++l;
        css_name = malloc( l );
        strcpy( css_name, docrootname );
        strcat( css_name, filename );
    }

    if ( document->css )
    {
        /* The user specified its own css file,
         * so we use the content of that.
         */
        RB_CopyFile( document->css, css_name );
    }
    else
    {
        css_file = fopen( css_name, "w" );
        if ( css_file )
        {
            fprintf( css_file, "%s",
                    "body\n"
                    "{\n"
                    "    background-color: #ffffff;\n"
                    "    color: #000000;\n"
                    "    font-family: 'Lucida Grande', Verdana, \n"
                    "                     Geneva, Lucida, Arial, \n"
                    "                     Helvetica, sans-serif;\n"
                    "    font-size: 10pt;\n"
                    "    margin: 2% 5%;\n"
                    "}\n"
                    "h1, h2, h3, h4, h5, h6, h7\n"
                    "{\n"
                    "    background-color: #dddddd;\n"
                    "    color: #000000;\n"
                    "    text-align: right;\n"
                    "    font-size: 11pt;\n"
                    "}\n"
                    "td.even\n"
                    "{\n"
                    "    background-color: #eeeeee;\n"
                    "    color: #000000;\n"
                    "    font-size: 10pt;\n"
                    "}\n"
                    "pre\n"
                    "{\n"
                    "    background-color: #ffffff;\n"
                    "    color: #000000;\n" "    font-size: 10pt;\n" "}\n" );
            fclose( css_file );
        }
        else
        {
            RB_Panic( "Can't open %s for writing", css_file );
        }
    }
}

/*******/


void
RB_InsertCSS( FILE * dest_doc, char *filename )
{
    if ( css_name )
    {
        char               *r = RB_HTML_RelativeAddress( filename, css_name );
        if ( strlen(r) == 0 ) {
            printf("%s %s\n", filename, css_name);
        }
        assert( r );
        assert( strlen( r ) );
        fprintf( dest_doc,
                 "<link rel=\"stylesheet\" href=\"%s\" type=\"text/css\">\n",
                 r );
    }
}









