/* Copyright (C) 2003 Nikos Chantziaras.
 *
 * This file is part of the QTads program.  This program is free
 * software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#ifndef QTADSTIMEDISPLAY_H
#define QTADSTIMEDISPLAY_H

#include "config.h"

#include <qlabel.h>
#include <qdatetime.h>


/* A timer-widget that displays elapsed time.
 *
 * TODO: Find a way to reduce flicker (QTextEdit would help, but it's
 * overkill).
 */
class QTadsTimeDisplay: public QLabel {
	Q_OBJECT

  private:
	int fTimerId;
	QTime fTime;
	bool fDisplaySeconds;

	void
	updateTime();

  protected:
	virtual void
	timerEvent( QTimerEvent* e );

  public:
	QTadsTimeDisplay( QWidget* parent = 0, const char* name = 0, WFlags f = 0 );

	// Start the timer.
	void
	start();

	// Stop the timer.
	void
	stop();

	// Reset the timer back to zero.  If the timer isn't running,
	// nothing happens.
	void
	reset();

	// Enable/disable display of seconds.
	void
	displaySeconds( bool yes );
};


inline
QTadsTimeDisplay::QTadsTimeDisplay( QWidget* parent, const char* name, WFlags f )
: QLabel(parent, name, f), fTimerId(0), fDisplaySeconds(false)
{
	QLabel::setIndent(0);
	QLabel::setTextFormat(Qt::PlainText);
	QLabel::setAlignment(Qt::AlignCenter | Qt::SingleLine);
	this->fTime.start();
	this->updateTime();
}


inline void
QTadsTimeDisplay::updateTime()
{
	// Calculate how many hours, minutes and seconds have passed
	// since our last invocation (given in milliseconds).
	int elapsed = this->fTimerId != 0 ? this->fTime.elapsed() : 0;
	int h = elapsed / 3600000;
	elapsed -= h * 3600000;
	int m = elapsed / 60000;
	elapsed -= m * 60000;
	int s = elapsed / 1000;

	// Convert the calculated values to two-digit strings.
	QString hStr(h < 10 ? "0" + QString::number(h) : QString::number(h));
	QString mStr(m < 10 ? "0" + QString::number(m) : QString::number(m));
	QString sStr(s < 10 ? "0" + QString::number(s) : QString::number(s));

	this->setText(this->fDisplaySeconds ? hStr + ":" + mStr + ":" + sStr
		                            : hStr + ":" + mStr);
}


inline void
QTadsTimeDisplay::timerEvent( QTimerEvent* e )
{
	// If the event was not our update-timer, it must be a timer
	// event of our base class.  Process it and return.
	if (e->timerId() != this->fTimerId) {
		QLabel::timerEvent(e);
		return;
	}

	Q_ASSERT(this->fTimerId != 0);

	this->updateTime();
}


inline void
QTadsTimeDisplay::start()
{
	this->fTime.restart();
	if (this->fTimerId == 0) {
		this->fTimerId = this->startTimer(1000);
	}

	Q_ASSERT(this->fTimerId != 0);

	this->updateTime();
}


inline void
QTadsTimeDisplay::stop()
{
	if (this->fTimerId != 0) {
		this->killTimer(this->fTimerId);
		this->fTimerId = 0;
	}
}


inline void
QTadsTimeDisplay::reset()
{
	this->fTime.restart();
	this->updateTime();
}


inline void
QTadsTimeDisplay::displaySeconds( bool yes )
{
	this->fDisplaySeconds = yes;
	this->updateTime();
}

#endif // QTADSTIMEDISPLAY_H
