/***************************************************************************
  miscpage.cpp
  -------------------
  A dialog page for misc ingredients
  -------------------
  Copyright 1999-2004, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qcheckbox.h>
#include <qcombobox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qstringlist.h>
#include <qwhatsthis.h>

#include "fixedspinbox.h"
#include "model.h"
#include "view.h"

#include "miscpage.h"

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// MiscPage()
// ----------
// Constructor

MiscPage::MiscPage(Model *model, View *view, Controller *controller,
                   const char *name)
    : IngredientPage(model, view, controller, name), noteslabel_(0),
      notesedit_(0), currentitem_(0)
{
    // setup listview
    itemslistview_->addColumn("Notes");
    itemslistview_->setColumnAlignment(2, AlignLeft);
    itemslistview_->setSorting(0);
    itemslistview_->setSorting(1);
    itemslistview_->setSorting(2);

    // setup widgets
    noteslabel_ = new QLabel("Notes", this, "noteslabel_");
    notesedit_ = new QLineEdit(this, "notesedit_");
    QWhatsThis::add(notesedit_,
                    "Use this field to list any notes on the item, such as "
                    "form, attenuation, or other factors");

    // setup layout
    rowtwolayout_->addWidget(quantitylabel_, 0);
    rowtwolayout_->addWidget(quantityspin_, 0, AlignRight);
    rowtwolayout_->addWidget(noteslabel_, 0);
    rowtwolayout_->addWidget(notesedit_, 1);

    delete spacer_; // don't need it
    rowtwolayout_->addWidget(addbutton_, 0);
    rowtwolayout_->addWidget(applybutton_, 0);
    rowtwolayout_->addWidget(removebutton_, 0);

    // set the leftmost labels to the same minimum size
    quantitylabel_->adjustSize();
    quantitylabel_->setMinimumWidth(quantitylabel_->width());
    itemlabel_->setMinimumWidth(quantitylabel_->width());

    // fixup tab orders
    QWidget::setTabOrder(itemcombo_, quantityspin_);
    QWidget::setTabOrder(quantityspin_, notesedit_);

    // setup connections
    connect(itemcombo_, SIGNAL(activated(const QString &)),
            this, SLOT(updateEntryItems(const QString &)));
    connect(addbutton_, SIGNAL(clicked()), SLOT(addMiscIngredient()));
    connect(applybutton_, SIGNAL(clicked()), SLOT(applyMiscIngredient()));
    connect(removebutton_, SIGNAL(clicked()), SLOT(removeMiscIngredient()));
    connect(itemslistview_, SIGNAL(selectionChanged(QListViewItem *)),
            SLOT(itemSelected(QListViewItem *)));

    // initialize widgets with values
    documentChanged();
}

MiscPage::~MiscPage() { ; }

//////////////////////////////////////////////////////////////////////////////
// Slots                                                                    //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// updateEntryItems()
// ------------------
// update the misc ingredient entry widgets

void MiscPage::updateEntryItems(const QString &name)
{
    MiscIngredient *thismisc = model_->misc(name);
    if (thismisc) {
        quantityspin_->setValue(thismisc->quantity().amount());
        quantityspin_->setSuffix(" " + thismisc->quantity().unit().symbol());
        notesedit_->setText(thismisc->notes());
    }
}

//////////////////////////////////////////////////////////////////////////////
// documentChanged()
// -----------------
// Received when document has changed

void MiscPage::documentChanged()
{
    if (itemcombo_->count() == 0) {
        itemcombo_->clear();
        itemcombo_->insertStringList(model_->miscList());
    }
    updateEntryItems(itemcombo_->currentText());
    updateListView();
}

//////////////////////////////////////////////////////////////////////////////
// addMiscIngredient()
// -----------------------
// adds a misc ingredient to recipe

void MiscPage::addMiscIngredient()
{
    MiscIngredientIterator it;
    MiscIngredientListItem *item;

    MiscIngredient misc(itemcombo_->currentText(),
                         Quantity(quantityspin_->value(), model_->defaultMiscUnit()),
                         notesedit_->text());
    model_->checkMisc(misc);

    it = model_->recipe()->addMisc(misc);
    item = new MiscIngredientListItem(itemslistview_, (*it).name(),
                                      QString::number((*it).quantity().amount(), 'f', 2)
                                      + " " + (*it).quantity().unit().symbol(),
                                      (*it).notes() );
    item->setMiscIngredientIterator(it);

    view_->flushView();
}

//////////////////////////////////////////////////////////////////////////////
// applyMiscIngredient()
// -------------------------
// applies a misc ingredient to recipe

void MiscPage::applyMiscIngredient()
{
     if (currentitem_ != 0) {
         MiscIngredient misc(itemcombo_->currentText(),
                         Quantity(quantityspin_->value(), model_->defaultMiscUnit()),
                         notesedit_->text());
         model_->checkMisc(misc);

         model_->recipe()->applyMisc(currentitem_->miscIngredientIterator(), misc);
         currentitem_->setToIterator();
         view_->flushView();
     }
}

//////////////////////////////////////////////////////////////////////////////
// removeMiscIngredient()
// --------------------------
// rremoves a misc ingredient from recipe

void MiscPage::removeMiscIngredient()
{
    if (currentitem_ != 0) {
        model_->recipe()->removeMisc(currentitem_->miscIngredientIterator());
        delete currentitem_; currentitem_ = 0;
        view_->flushView();
    }
}

//////////////////////////////////////////////////////////////////////////////
// itemSelected()
// ------------------
// Received when user selects an item in the list view

void MiscPage::itemSelected(QListViewItem *item)
{
    if (item) {
        currentitem_ = dynamic_cast<MiscIngredientListItem*>(item);
        if (currentitem_ == 0) {
            qWarning("qbrew: bad dynamic cast, line" + __LINE__);
            return;
        }
        itemcombo_->setEditText(item->text(0));
        quantityspin_->setValue(item->text(1));
        // TODO: using default unit, but item might not be default?
        quantityspin_->setSuffix(" " + model_->defaultMiscUnit().symbol());
        notesedit_->setText(item->text(2));
    }
}

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// updateListView()
// ----------------
// Update the listview

void MiscPage::updateListView()
{
    MiscIngredientList *list = model_->recipe()->miscs();
    MiscIngredientIterator it;
    MiscIngredientListItem *item;
    itemslistview_->clear();
    for (it=list->begin(); it != list->end(); ++it) {
        item = new MiscIngredientListItem(itemslistview_, (*it).name(),
                   QString::number((*it).quantity().amount(), 'f', 2)
                   + " " + (*it).quantity().unit().symbol(),
                   (*it).notes() );
        item->setMiscIngredientIterator(it);
    }
}
