#!/usr/bin/python
import unittest
from cStringIO import StringIO
import os
import tempfile

# Add build directory to search path
if os.path.exists("build"):
	from distutils.util import get_platform
	import sys
	s = "build/lib.%s-%.3s" % (get_platform(), sys.version)
	s = os.path.join(os.getcwd(), s)
	sys.path.insert(0,s)

from bz2 import *

class BaseTest(unittest.TestCase):
	"Base for other testcases."
	TEXT = 'root:x:0:0:root:/root:/bin/bash\nbin:x:1:1:bin:/bin:\ndaemon:x:2:2:daemon:/sbin:\nadm:x:3:4:adm:/var/adm:\nlp:x:4:7:lp:/var/spool/lpd:\nsync:x:5:0:sync:/sbin:/bin/sync\nshutdown:x:6:0:shutdown:/sbin:/sbin/shutdown\nhalt:x:7:0:halt:/sbin:/sbin/halt\nmail:x:8:12:mail:/var/spool/mail:\nnews:x:9:13:news:/var/spool/news:\nuucp:x:10:14:uucp:/var/spool/uucp:\noperator:x:11:0:operator:/root:\ngames:x:12:100:games:/usr/games:\ngopher:x:13:30:gopher:/usr/lib/gopher-data:\nftp:x:14:50:FTP User:/var/ftp:/bin/bash\nnobody:x:65534:65534:Nobody:/home:\npostfix:x:100:101:postfix:/var/spool/postfix:\nniemeyer:x:500:500::/home/niemeyer:/bin/bash\npostgres:x:101:102:PostgreSQL Server:/var/lib/pgsql:/bin/bash\nmysql:x:102:103:MySQL server:/var/lib/mysql:/bin/bash\nwww:x:103:104::/var/www:/bin/false\n'
	DATA = 'BZh91AY&SY.\xc8N\x18\x00\x01>_\x80\x00\x10@\x02\xff\xf0\x01\x07n\x00?\xe7\xff\xe00\x01\x99\xaa\x00\xc0\x03F\x86\x8c#&\x83F\x9a\x03\x06\xa6\xd0\xa6\x93M\x0fQ\xa7\xa8\x06\x804hh\x12$\x11\xa4i4\xf14S\xd2<Q\xb5\x0fH\xd3\xd4\xdd\xd5\x87\xbb\xf8\x94\r\x8f\xafI\x12\xe1\xc9\xf8/E\x00pu\x89\x12]\xc9\xbbDL\nQ\x0e\t1\x12\xdf\xa0\xc0\x97\xac2O9\x89\x13\x94\x0e\x1c7\x0ed\x95I\x0c\xaaJ\xa4\x18L\x10\x05#\x9c\xaf\xba\xbc/\x97\x8a#C\xc8\xe1\x8cW\xf9\xe2\xd0\xd6M\xa7\x8bXa<e\x84t\xcbL\xb3\xa7\xd9\xcd\xd1\xcb\x84.\xaf\xb3\xab\xab\xad`n}\xa0lh\tE,\x8eZ\x15\x17VH>\x88\xe5\xcd9gd6\x0b\n\xe9\x9b\xd5\x8a\x99\xf7\x08.K\x8ev\xfb\xf7xw\xbb\xdf\xa1\x92\xf1\xdd|/";\xa2\xba\x9f\xd5\xb1#A\xb6\xf6\xb3o\xc9\xc5y\\\xebO\xe7\x85\x9a\xbc\xb6f8\x952\xd5\xd7"%\x89>V,\xf7\xa6z\xe2\x9f\xa3\xdf\x11\x11"\xd6E)I\xa9\x13^\xca\xf3r\xd0\x03U\x922\xf26\xec\xb6\xed\x8b\xc3U\x13\x9d\xc5\x170\xa4\xfa^\x92\xacDF\x8a\x97\xd6\x19\xfe\xdd\xb8\xbd\x1a\x9a\x19\xa3\x80ankR\x8b\xe5\xd83]\xa9\xc6\x08\x82f\xf6\xb9"6l$\xb8j@\xc0\x8a\xb0l1..\xbak\x83ls\x15\xbc\xf4\xc1\x13\xbe\xf8E\xb8\x9d\r\xa8\x9dk\x84\xd3n\xfa\xacQ\x07\xb1%y\xaav\xb4\x08\xe0z\x1b\x16\xf5\x04\xe9\xcc\xb9\x08z\x1en7.G\xfc]\xc9\x14\xe1B@\xbb!8`'

	def decompress(self, data):
		w, r = os.popen2("bunzip2")
		w.write(data)
		w.close()
		data = r.read()
		r.close()
		return data

class BZ2FileTest(BaseTest):
	"Test MCRYPT type miscelaneous methods."

	def setUp(self):
		self.filename = tempfile.mktemp("bz2")
		
	def tearDown(self):
		if os.path.isfile(self.filename):
			os.unlink(self.filename)
	
	def createTempFile(self):
		f = open(self.filename, "w")
		f.write(self.DATA)
		f.close()
	
	def testRead(self):
		"Test BZ2File.read()"
		self.createTempFile()
		bz2f = BZ2File(self.filename)
		self.assertEqual(bz2f.read(), self.TEXT)
		bz2f.close()
	
	def testReadChunk10(self):
		"Test BZ2File.read() in chunks of 10 bytes"
		self.createTempFile()
		bz2f = BZ2File(self.filename)
		text = ''
		while 1:
			str = bz2f.read(10)
			if not str:
				break
			text += str
		self.assertEqual(text, text)
		bz2f.close()
	
	def testRead100(self):
		"Test BZ2File.read(100)"
		self.createTempFile()
		bz2f = BZ2File(self.filename)
		self.assertEqual(bz2f.read(100), self.TEXT[:100])
		bz2f.close()
	
	def testReadLine(self):
		"Test BZ2File.readline()"
		self.createTempFile()
		bz2f = BZ2File(self.filename)
		sio = StringIO(self.TEXT)
		for line in sio.readlines():
			self.assertEqual(bz2f.readline(), line)
		bz2f.close()
	
	def testReadLines(self):
		"Test BZ2File.readlines()"
		self.createTempFile()
		bz2f = BZ2File(self.filename)
		sio = StringIO(self.TEXT)
		self.assertEqual(bz2f.readlines(), sio.readlines())
		bz2f.close()
	
	def testXReadLines(self):
		"Test BZ2File.xreadlines()"
		self.createTempFile()
		bz2f = BZ2File(self.filename)
		sio = StringIO(self.TEXT)
		self.assertEqual(list(bz2f.xreadlines()), sio.readlines())
		bz2f.close()
	
	def testWrite(self):
		"Test BZ2File.write()"
		bz2f = BZ2File(self.filename, "w")
		bz2f.write(self.TEXT)
		bz2f.close()
		f = open(self.filename)
		self.assertEqual(self.decompress(f.read()), self.TEXT)
		f.close()

	def testWriteChunks10(self):
		"Test BZ2File.write() with chunks of 10 bytes"
		bz2f = BZ2File(self.filename, "w")
		n = 0
		while 1:
			str = self.TEXT[n*10:(n+1)*10]
			if not str:
				break
			bz2f.write(str)
			n += 1
		bz2f.close()
		f = open(self.filename)
		self.assertEqual(self.decompress(f.read()), self.TEXT)
		f.close()

	def testWriteLines(self):
		"Test BZ2File.writelines()"
		bz2f = BZ2File(self.filename, "w")
		sio = StringIO(self.TEXT)
		bz2f.writelines(sio.readlines())
		bz2f.close()
		f = open(self.filename)
		self.assertEqual(self.decompress(f.read()), self.TEXT)
		f.close()

class BZ2CompressorTest(BaseTest):
	def testCompress(self):
		"Test BZ2Compressor.compress()/flush()"
		bz2c = BZ2Compressor()
		data = bz2c.compress(self.TEXT)
		data += bz2c.flush()
		self.assertEqual(self.decompress(data), self.TEXT)

	def testCompressChunks10(self):
		"Test BZ2Compressor.compress()/flush() with chunks of 10 bytes"
		bz2c = BZ2Compressor()
		n = 0
		data = ''
		while 1:
			str = self.TEXT[n*10:(n+1)*10]
			if not str:
				break
			data += bz2c.compress(str)
			n += 1
		data += bz2c.flush()
		self.assertEqual(self.decompress(data), self.TEXT)

class BZ2DecompressorTest(BaseTest):
	def testDecompress(self):
		"Test BZ2Decompressor.decompress()"
		bz2d = BZ2Decompressor()
		text = bz2d.decompress(self.DATA)
		self.assertEqual(text, self.TEXT)

	def testDecompressChunks10(self):
		"Test BZ2Decompressor.decompress() with chunks of 10 bytes"
		bz2d = BZ2Decompressor()
		text = ''
		n = 0
		while 1:
			str = self.DATA[n*10:(n+1)*10]
			if not str:
				break
			text += bz2d.decompress(str)
			n += 1
		self.assertEqual(text, self.TEXT)

	def testDecompressUnusedData(self):
		"Test BZ2Decompressor.decompress() with unused data"
		bz2d = BZ2Decompressor()
		unused_data = "this is unused data"
		text = bz2d.decompress(self.DATA+unused_data)
		self.assertEqual(text, self.TEXT)
		self.assertEqual(bz2d.unused_data, unused_data)

	def testEOFError(self):
		"Calling BZ2Decompressor.decompress() after EOS must raise EOFError"
		bz2d = BZ2Decompressor()
		text = bz2d.decompress(self.DATA)
		self.assertRaises(EOFError, bz2d.decompress, "anything")


class FuncTest(BaseTest):
	"Test module functions"
	
	def testCompress(self):
		"Test compress() function"
		data = compress(self.TEXT)
		self.assertEqual(self.decompress(data), self.TEXT)

	def testDecompress(self):
		"Test decompress() function"
		text = decompress(self.DATA)
		self.assertEqual(text, self.TEXT)

if __name__ == "__main__":
	unittest.main()

# vim:ts=4:sw=4
