package PSP::Log;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: Log.pm,v 1.2 2001/02/12 21:30:08 hartmans Exp $

use strict;

=head1 NAME

 PSP::Log - logging facilities for PSP Driver

=head1 SYNOPSIS

 # more later

=head1 DESCRIPTION

more later.

=cut

use Exporter;
use Symbol;
use PSP::Conf qw($psp_driver_log $psp_stuck_seconds);
use PSP::Utils qw(dump_object);

use vars qw(@ISA @EXPORT_OK %EXPORT_TAGS);
BEGIN {
  @ISA = qw(Exporter);
  @EXPORT_OK = qw(&psp_warn
		  &psp_log_event
		  &psp_log
		  &psp_die);
  $EXPORT_TAGS{all} = \@EXPORT_OK;
}

=head1 GLOBALS

=head2 $log_failure_already_warned

This is a boolean quantity that indicates if a log_failure warning has
already been delivered to STDERR.  This variable is to prevent flooding
to STDERR when there is a problem with the logging facility.

=cut

use vars qw($LOG_FH $log_failure_already_warned);

=head1 SUBROUTINES

=head2 psp_log_event

=cut

sub psp_log_event {
  my ($location,$text,$t0,$obj_to_dump) = @_;

  # return unless we have a filename.
  $psp_driver_log or return;

  # get the current time.
  my $t1 = time();

  # convert that time to a string.
  my $datestr = date_string($t1);

  # remove end-space from text.
  $text =~ s/\s+$//;

  # log the output text.
  psp_log("$datestr ($$/$PSP::Driver::n_requests) $location: $text".
	  ($t0 ? " after ".($t1-$t0)." second(s)" : ""));

  # if an initial time and an exception time are specified,
  #   and if the elapsed time exceeds the exception time
  # then dump the $obj_to_dump with some explanation.

  if (($t0 and $psp_stuck_seconds) and ($t1-$t0) > $psp_stuck_seconds) {
    psp_log(fmt_time_exception($obj_to_dump,$t1,$t0));
  }

  # return our current time.
  return $t1;
}

=head2 psp_log

=cut

sub psp_log {
  my ($msg) = @_;

  # return unless we have a filename.
  $psp_driver_log or return;

  # normalize end-space of the input message.
  $msg =~ s/\s*$/\n/s;

  if (!$LOG_FH) {
    $LOG_FH = gensym();
    # warn for this problem only once.
    if (! open $LOG_FH, ">>".$psp_driver_log) {
      unless ($log_failure_already_warned++) {
	psp_warn("open: $psp_driver_log: $!");
      }
      undef $LOG_FH;
      return;
    } else {
      # reset the warning if somehow we were able to open it after all..
      undef $log_failure_already_warned;
    }
  }

  undef $LOG_FH unless print $LOG_FH $msg;
}

=head2 psp_warn

=cut

sub psp_warn {
  my ($msg) = @_;

  # normalize end-space.
  $msg =~ s/\s*$/\n/s;

  # log this message.
  psp_log($msg);

  # warn to stderr
  warn $msg;
}

=head2 psp_die

=cut

sub psp_die {
  my ($msg) = @_;

  # perpend DYING to the message
  $msg = "DYING: ".$msg;

  # warn this message.
  psp_warn($msg);

  # exit.
  exit 1;
}

=head2 fmt_time_exception

=cut

sub fmt_time_exception {
  my ($obj_to_dump,$t1,$t0) = @_;

  # Create some text and a dump to log the exception.
  my $out .= join("\n",
(("-"x78),
 "WARNING!!! Process ran for ".($t1-$t0)." seconds since previous event.",
 "This time exceeds the $psp_stuck_seconds limit.  Dumped data follows.",
 dump_object($obj_to_dump),
 ("-"x78)
))."\n";
  return $out;
}

=head2 date_string

=cut

sub date_string {
  my ($time) = @_;

  # accept an input time, or use the current time.
  $time ||= time();

  # get some conventional time meanings.
  my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($time);
  $mon++; $year+=1900;

  # return the formatted date string.
  return(sprintf('%04d/%02d/%02d %2d:%02d:%02d',
		 $year,$mon,$mday,$hour,$min,$sec));
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<PSP::Loader>, L<PSP::Conf>

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
