/*
 * prismstumbler wireless lan analyzing tool
 *
 * Copyright (C) 2000 and created by Jan Fernquist  <Jan.B.Fernquist@telia.com>
 * Copyright (C) 2003 Florian Boor <florian.boor@kernelconcepts.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * GTK+ frontend 
 *
 */

#define _XOPEN_SOURCE

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <poll.h>
#include <unistd.h>
#include <features.h>
#include "psgui.h"
#include "interface.h"
#include "support.h"
#include "ps_database.h"
#include "helper.h"
#include "analyzer.h"
#include "ps_resolv.h"
#include "preferences.h"
#include "cards.h"
#include "setupdlg.h"

#define COLORSTEP 20
#define UPDATE_INTERVAL 1200
#define BROKEN_SSID "<not displayable>"
#define HELPMESSAGE "Prismstumbler\nVersion " VERSION \
		"\nGTK Frontend\n\nflorian@handhelds.org"

extern psconfig_t cfg;
extern char* cfgfile;
gchar logbuf[255];

extern GtkTreeStore *store;
extern GtkWidget *rbCSpecific;
extern GtkWidget *eDevice;
extern GtkWidget *rbDriverNG;
extern GtkWidget *rbDriverOR;
extern GtkWidget *rbDriverHA;
extern GtkWidget *rbDriverSC;
extern GtkWidget *lpsum;
extern GtkWidget *lpdata;
extern GtkWidget *lpiiv;
extern GtkWidget *llong;
extern GtkWidget *llat;
extern GtkWidget *notebook1;
extern GtkWidget *eSChannel;
extern GtkWidget *txLog;
extern GtkWidget *lPosLong;
extern GtkWidget *lPosLat;
extern GtkWidget *lFix;
extern GtkWidget *fMain;
extern GtkWidget *gpbSignal;
extern GtkWidget *gclDetail;
extern GtkWidget *uiScan;
extern GtkWidget *uiDump;
extern GtkWidget *bScanner;
extern GtkWidget *miScan;
extern GtkWidget *miDump;
extern GtkWidget *cbDevice;
extern GtkWidget *miJumpTo;

static int actual_rows = -1;
static int interval_maxsiglevel = 0;
static int net_request_mode = FALSE;
static guint active_row = 0;
static guint last_row = 0;
static void stop_signal ();
void update_gui_pos(int netpos);
static void update_config (psconfig_t * ncfg);
static gboolean gui_update_busy = FALSE;
static gboolean active_jump_to = FALSE;
gboolean night_mode = FALSE;

extern int sock;
extern int driver_set;
static int config_changed = FALSE;
static int detail_initialized = FALSE;

static GdkPixbuf *node_icons[N_MAX] = { NULL, };

typedef struct
{
	psnetinfo_t *ni;
	GtkTreeIter iter;
	int row;
	gushort color;
	node_t type;
}
t_gnet;

static t_gnet *gnet = NULL;


/*
 * Signal Handler
 */
static void
stop_signal ()
{
	fprintf (stderr, "Received CTRL-C - sniffing aborted\n");
	on_fMain_destroy (NULL, NULL);
}


void 
update_scannerbutton()
{
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(miScan),cfg.scan);
	if (!GTK_IS_IMAGE(uiScan)) return;
	/* update button image */
	if (!GTK_IS_IMAGE(uiScan)) return;
	if (cfg.scan)
		gtk_image_set_from_file(GTK_IMAGE(uiScan),PIC_SCANNER_ON);
	else
		gtk_image_set_from_file(GTK_IMAGE(uiScan),PIC_SCANNER_OFF);
	gtk_widget_set_sensitive(bScanner,strcmp(cfg.device,"none"));
}	


void 
update_dumpbutton()
{
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(miDump),cfg.dumptofile);
	if (!GTK_IS_IMAGE(uiDump)) return;
	/* update button image */
	if (!GTK_IS_IMAGE(uiDump)) return;
	if (cfg.dumptofile)
		gtk_image_set_from_file(GTK_IMAGE(uiDump),PIC_DUMP_ON);
	else
		gtk_image_set_from_stock(GTK_IMAGE(uiDump),
			GTK_STOCK_JUMP_TO,GTK_ICON_SIZE_SMALL_TOOLBAR);
}


/* buttonbar/menu callbacks */

void
on_toggle_jump_clicked(GtkWidget *w)
{
	active_jump_to = !active_jump_to;
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(miJumpTo),active_jump_to);
}

void
on_preferences_clicked(GtkWidget *button, gpointer userdata)
{
	config_changed = ps_do_setup(&cfg);	
}


void
on_preferences_save_as_clicked(GtkWidget *button, gpointer userdata)
{
	preferences_save(&cfg,NULL);
}


void
on_preferences_load_from_clicked(GtkWidget *button, gpointer userdata)
{
	psconfig_t cnew;
	cnew = preferences_load(NULL);
	update_config(&cnew);
	send_config();
	if (cnew.devtype == DT_NONE) send_command (C_DETECT_CARD,0);
}


void
on_preferences_save_clicked(GtkWidget *button, gpointer userdata)
{
	preferences_save(&cfg,cfgfile);
}


void
on_preferences_load_clicked(GtkWidget *button, gpointer userdata)
{
	psconfig_t cnew;
	cnew = preferences_load(cfgfile);
	update_config(&cnew);
	send_config();
	if (cnew.devtype == DT_NONE) send_command (C_DETECT_CARD,0);
}


void
on_dump_file_clicked(GtkWidget *button, gpointer userdata)
{
	/* update only if sender is button to avoid cyclic changes */
	if (gui_update_busy) return;
	gui_update_busy = TRUE;
	cfg.dumptofile = !cfg.dumptofile;
//	config_changed = TRUE;
	send_config();
	update_dumpbutton();
	gui_update_busy = FALSE;
}


void
on_toggle_scanner_clicked(GtkWidget *button, gpointer userdata)
{
	/* update only if sender is button to avoid cyclic changes */
	if (gui_update_busy) return;
	gui_update_busy = TRUE;
	cfg.scan = !cfg.scan;
	config_changed = TRUE;
	update_scannerbutton();
	gui_update_busy = FALSE;
}


/* helper functions */

void
show_message(GtkWidget *parent,GtkMessageType type, char* message)
{
	GtkWidget* dialog;
	
	dialog = gtk_message_dialog_new (GTK_WINDOW(parent),
					 GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
					 type,
					 GTK_BUTTONS_OK,
					 message);
	gtk_dialog_run (GTK_DIALOG(dialog));
	gtk_widget_destroy(dialog);
}


static void
load_icons (void)
{
	GError *err = NULL;

	node_icons[N_NET] =
		gdk_pixbuf_new_from_file (PREFIX "/share/pixmaps/wlan-net.png",&err);
	if (node_icons[N_NET])
		gdk_pixbuf_ref (node_icons[N_NET]);
	node_icons[N_ADHOC] =
		gdk_pixbuf_new_from_file (PREFIX "/share/pixmaps/miniwave-icon.png",&err);
	if (node_icons[N_ADHOC])
		gdk_pixbuf_ref (node_icons[N_ADHOC]);
}


static void
node_get_type (t_gnet * node)
{
	node->type = N_NET;
}


void
printlog (gchar * str)
{
	GtkTextBuffer *log;
	log = gtk_text_view_get_buffer (GTK_TEXT_VIEW (txLog));
	if (log)
		gtk_text_buffer_insert_at_cursor (GTK_TEXT_BUFFER (log), str, -1);
}


void
update_gpsdata (psgps_t * gpsdata)
{
	char *str;

	str = g_strdup_printf ("Longitude: \t%3.8f", gpsdata->Long);
	gtk_label_set_text (GTK_LABEL (lPosLong), str);
	g_free (str);
	str = g_strdup_printf ("Latitude: \t\t%3.8f", gpsdata->Lat);
	gtk_label_set_text (GTK_LABEL (lPosLat), str);
	g_free (str);
	if (gpsdata->quality)
	{
		str = g_strdup_printf ("Quality: \t\t%d", gpsdata->quality);
		gtk_label_set_text (GTK_LABEL (lFix), str);
		g_free (str);
	}
	else
	{
		gtk_label_set_text (GTK_LABEL (lFix),
				    "Quality: \t\t<no fix>");
	}
}


static void
update_config (psconfig_t * ncfg)
{
	memcpy (&cfg, ncfg, sizeof (psconfig_t));
	gtk_entry_set_text (GTK_ENTRY(eDevice), cfg.device);
	switch (cfg.devtype)
	{
	case DT_ORINOCO:
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rbDriverOR), TRUE);
		break;
	case DT_HOSTAP:
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rbDriverHA), TRUE);
	break;
#ifdef WLAN_NG
	case DT_PRISM:
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rbDriverNG), TRUE);
	break;
#endif	
	default:
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rbDriverSC), TRUE);
	}

	gtk_spin_button_set_value(GTK_SPIN_BUTTON(eSChannel),(gdouble)cfg.singlechan);
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(rbCSpecific),
				     cfg.singlechan);
	config_changed = FALSE;
	update_scannerbutton();
	update_dumpbutton();
}


void
ps_do_about (GtkWidget * w)
{
	show_message(fMain,GTK_MESSAGE_INFO,HELPMESSAGE);
}


void
do_track_net (GtkWidget * w)
{
	psconfig_t ucfg = cfg;
	
	ucfg.singlechan = gnet[active_row].ni->channel;
	
	update_config(&ucfg);
	send_config();
}


usernetinfo_t
netinfo_to_usernet(psnetinfo_t *ni)
{
	usernetinfo_t result;
	
	memset(&result,0,sizeof(usernetinfo_t));
	
	sprintf (result.bssid, "%s", ni->bssid);
	sprintf (result.ssid, "%s", ni->ssid);
	result.mode = ni->isadhoc;
	result.wep = ni->wep; /* sense! */
	result.dhcp = /*ni->dhcp;*/ TRUE; /* for now...*/
	result.channel = ni->channel;
/*	
	if (ni->ip_range[0])
		result.netmask[0] = 255;
	if (ni->ip_range[1])
		result.netmask[1] = 255;
	if (ni->ip_range[2])
		result.netmask[2] = 255;
	if (ni->ip_range[3])
		result.netmask[3] = 255;
	memcpy (&result.wep_key, &ni->wep_key, 48);
*/	
	result.inrange = TRUE;
	return result;
}


void
do_try_connect (GtkWidget * w)
{
	usernetinfo_t usernet;
	
	/* sanity check */
	
	if (active_row < 0) return;
		
	/* we need to switch the scanner off */
	on_toggle_scanner_clicked(NULL,NULL);
//	cfg.scan = FALSE;
//	send_config();
	update_scannerbutton();
	
	usernet = netinfo_to_usernet(gnet[active_row].ni);
	
	send_usernet (&usernet);
	send_command (C_ASSOCIATE, SEQ_USERNET);
	net_request_mode = gnet[active_row].ni->isadhoc;
}


static void
do_message_info (psinfo_t * psi)
{
	switch (psi->info)
	{
	case I_SUCCESS:
		show_message(fMain,GTK_MESSAGE_INFO, _("Successfully connected."));
		break;
	case I_FAILED:
		if (net_request_mode)
		{
			show_message(fMain,GTK_MESSAGE_ERROR,_("Ad-Hoc connections setup complete."));
		}
		else
			show_message(fMain,GTK_MESSAGE_ERROR,_("Could not connect to wireless LAN."));
		break;
	case I_ERRCARD:
	case I_NOCARD:
		show_message(fMain,GTK_MESSAGE_ERROR,psi->message);
		break;
	default:
		break;
	}
}


void
update_node_info(nodeinfo_t *node)
{
	char *buf;
	
	buf = g_strdup_printf("Packet from station:\n %s %d.%d.%d.%d Port: %d\n",
			node->mac, node->ip[0], node->ip[1], node->ip[2],node->ip[3],node->port);	
	printlog(buf);
	g_free(buf);
}


gboolean
do_timer_update (void *data)
{
	static psmessage_t msg;
	struct pollfd pfd[1];

	/* send configuration data if necessary */
	if (config_changed)
	{
		config_changed = FALSE;
		send_config ();
	}
	
	
	/* check for incoming messages */

	pfd[0].fd = sock;
	pfd[0].events = (POLLIN | POLLRDNORM | POLLRDBAND | POLLPRI);
	while (poll (pfd, 1, 0) > 0)
	{
		if ((pfd[0].revents & POLLERR) || (pfd[0].revents & POLLHUP))
		{
#ifdef DEBUG
			perror ("Err: connection lost: ");
#endif
			return TRUE;
		}
		if (read (sock, (void *) &msg, sizeof (psmessage_t)) < 0)
		{
			perror ("err receiving data packet");
			close (sock);
			exit (1);
		}
		else
		{
			switch (msg.type)
			{
			case (msg_network):
				update_netlist (&msg.content.net);
				break;
			case (msg_gps):
				update_gpsdata (&msg.content.gps);
				break;
			case (msg_config):
				update_config (&msg.content.cfg);
				break;
			case (msg_info):
				do_message_info(&msg.content.info);
			break;
			case (msg_node):
				update_node_info(&msg.content.node);
			break;
			default:
				break;
			}
		}
	}

	/* update user interface elements */
	gtk_progress_set_value ((GtkProgress *) gpbSignal,
				(gfloat) interval_maxsiglevel);
	interval_maxsiglevel = 0;
	update_colors ();

	if (gtk_notebook_get_current_page (GTK_NOTEBOOK(notebook1)) == 1)
		update_gui_pos(active_row);

	return TRUE;
}


void
update_colors ()
{
	int color;
	int rpos = 0;
	char *colorstr;

	for (rpos = 0; rpos <= actual_rows; rpos++)
	{
		if (gnet[rpos].color < 65535)
		{
			color = gnet[rpos].color;
			if (gnet[rpos].color > (65535 - COLORSTEP))
				color = 65535;
			else
			{
				color = gnet[rpos].color;
				color += ((65535 - gnet[rpos].color) / 12);
			}
			gnet[rpos].color = color;

		}
		colorstr = calc_color (gnet[rpos].color, gnet[rpos].ni->wep);
		gtk_tree_store_set (store, &gnet[rpos].iter,
				    COL_COLOR,
				    colorstr,
				    -1);
		g_free(colorstr);
	}
}


void
update_netlist (psnetinfo_t * anet)
{
	int rpos = 0;
	char *tmpvec[5];
	int found_net = FALSE;
	psnetinfo_t *old_net;
	gchar tmpstr[80];
	GtkTreeIter iter;
	GtkTreePath *path;
	char *colorstr;

	// update signal strength
	if (anet->cursiglevel > interval_maxsiglevel)
		interval_maxsiglevel = anet->cursiglevel;

	for (rpos = 0; rpos <= actual_rows; rpos++)
	{
		// update network data
		if (!strncmp (gnet[rpos].ni->bssid, anet->bssid, 17))
		{
			// lets hack a data-transfer
			found_net = TRUE;
			old_net = gnet[rpos].ni;
			memcpy (old_net, anet, sizeof (psnetinfo_t));

			old_net->isvalid = TRUE;
			if (!check_str_content (anet->ssid))
				sprintf(anet->ssid,"%s",BROKEN_SSID);    
				
			gtk_tree_store_set (store, &gnet[rpos].iter,
				    COL_NAME, anet->ssid,-1);
			sprintf (tmpstr, "%i", anet->channel);
			gtk_tree_store_set (store, &gnet[rpos].iter,
			    COL_CHANNEL, anet->channel,-1);
			sprintf (tmpstr, "%i", anet->cursiglevel);
			gtk_tree_store_set (store, &gnet[rpos].iter,
			    COL_SIGNAL, anet->cursiglevel,-1);
			
			anet->cursiglevel = 0;
			gnet[rpos].type = anet->isadhoc ? N_ADHOC : N_NET;
			gtk_tree_store_set (store, &gnet[rpos].iter,
				COL_TYPE, gnet[rpos].type,-1);
			gtk_tree_store_set (store, &gnet[rpos].iter,
			    COL_ICON, node_icons[gnet[rpos].type],-1);
			gnet[rpos].color = 0;
			colorstr = calc_color(0,gnet[rpos].ni->wep);
			gtk_tree_store_set (store, &gnet[rpos].iter,
			    COL_COLOR, colorstr,-1);
			g_free(colorstr);	
			
			/* gui changes on network update */
			if (active_jump_to)
			{
				path = gtk_tree_model_get_path(GTK_TREE_MODEL(store),&gnet[rpos].iter);
				gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(tree),path,NULL,FALSE,0,0);
				gtk_tree_path_free(path);
			}
			return;
		}
	}

	// we have a new contact
	if (!found_net)
	{
		// new row
		tmpvec[0] = malloc (32 * sizeof (gchar));
		tmpvec[1] = malloc (32 * sizeof (gchar));
		tmpvec[2] = malloc (32 * sizeof (gchar));
		tmpvec[3] = malloc (32 * sizeof (gchar));
		tmpvec[4] = NULL;

		strncpy (tmpvec[0], anet->ssid, 32);
		snprintf (tmpvec[1], 31, "%i", anet->channel);
		strncpy (tmpvec[2], anet->bssid, 31);
		snprintf (tmpvec[3], 31, "%i", anet->cursiglevel);

		actual_rows++;	// s max index!

		free (tmpvec[0]);
		free (tmpvec[1]);
		free (tmpvec[2]);
		free (tmpvec[3]);

		// add data
		gnet = realloc (gnet, (actual_rows + 1) * sizeof (t_gnet));

		gnet[actual_rows].ni = malloc (sizeof (psnetinfo_t));
		memcpy (gnet[actual_rows].ni, anet, sizeof (psnetinfo_t));
		gnet[actual_rows].row = actual_rows;
		gnet[actual_rows].color = 0;
		gnet[actual_rows].type = anet->isadhoc ? N_ADHOC : N_NET;
		if (!check_str_content (gnet[actual_rows].ni->ssid))
			sprintf(gnet[actual_rows].ni->ssid,"%s",BROKEN_SSID);    
		
		sprintf (logbuf, "Found new net: \n %s %s \n",
			 gnet[actual_rows].ni->ssid,
			 gnet[actual_rows].ni->bssid);
		printlog (logbuf);
		
//		node_get_type (&gnet[actual_rows]);

		gtk_tree_store_append (store, &iter, NULL);	/* Acquire an iterator */
		gnet[actual_rows].iter = iter;

		colorstr = calc_color(gnet[actual_rows].color, 
					gnet[actual_rows].ni->wep);
		gtk_tree_store_set (store, &iter,
			    COL_ICON, node_icons[gnet[actual_rows].type],
			    COL_NAME, gnet[actual_rows].ni->ssid,
			    COL_BINID, gnet[actual_rows].ni->bssid,
			    COL_TYPE, gnet[actual_rows].type,
			    COL_SIGNAL, gnet[actual_rows].ni->cursiglevel,
				COL_CHANNEL, gnet[actual_rows].ni->channel,
			    COL_DATA, actual_rows,
				COL_COLOR, colorstr,
			    -1);
		g_free(colorstr);
	
		path = gtk_tree_model_get_path(GTK_TREE_MODEL(store),&gnet[actual_rows].iter);
		gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(tree),path,NULL,FALSE,0,0);
		gtk_tree_path_free(path);
	}
}


int
gridAddNet (void *pArg, int argc, char **argv, char **columnNames)
{
	psnetinfo_t anet;
	GtkTreeIter iter;
	char* colorstr;

	anet.isvalid = 5777;

	// add data
	if (argv && argv[2])
	{
		anet = get_network (argv[2]);
		if (anet.isvalid != 5777)
		{
			actual_rows++;
			anet.isvalid = FALSE;	// we received no data from this net

			gnet = realloc (gnet,
					(actual_rows + 1) * sizeof (t_gnet));
			gnet[actual_rows].ni = malloc (sizeof (psnetinfo_t));
			memcpy (gnet[actual_rows].ni, &anet,
				sizeof (psnetinfo_t));

			gtk_tree_store_append (store, &iter, NULL);	/* Acquire an iterator */
			gnet[actual_rows].iter = iter;
			gnet[actual_rows].row = actual_rows;
			gnet[actual_rows].color = 60000;

			node_get_type (&gnet[actual_rows]);

			colorstr = calc_color (gnet[actual_rows].color, 
						gnet[actual_rows].ni->wep);
			gtk_tree_store_set (store, &iter,
					    COL_ICON,
					    node_icons[gnet[actual_rows].type],
						COL_NAME,
					    gnet[actual_rows].ni->ssid,
					    COL_CHANNEL,
					    gnet[actual_rows].ni->channel,
					    COL_BINID,
					    gnet[actual_rows].ni->bssid,
					    COL_TYPE,
					    gnet[actual_rows].ni->isadhoc,
					    COL_SIGNAL,
					    gnet[actual_rows].ni->cursiglevel,
					    COL_COLOR,
					    colorstr, 
						COL_DATA,actual_rows, -1);
			g_free(colorstr);			
		}
		else
			fprintf (stderr, "urks - got empty net :-/");
	}

	return 0;
}


void
clear_list ()
{
	gtk_tree_store_clear(store);
}


gboolean
do_detect_cards()
{
	GSList *cards, *acard;
	GList *cbDevice_items = NULL;
	
	/* get card list */
	cards = find_wireless_cards ();
	if (cards)
	{
		struct ps_card *card = cards->data;
		snprintf (cfg.device, 6, "%s", card->name);
		for (acard = cards;acard;acard = g_slist_next(acard))
		{
			card = cards->data;
			cbDevice_items = g_list_append (cbDevice_items, card->name);
		}
		g_slist_free(cards);
		gtk_combo_set_popdown_strings (GTK_COMBO (cbDevice), cbDevice_items);
		return TRUE;
	}
	return FALSE;
}


void
on_fMain_destroy (GtkObject * object, gpointer user_data)
{
	int i;
	cfg.scan = FALSE;
	send_config ();
	sleep (1);

	/* save current state */
	for (i = 0; i <= actual_rows; i++)
	{
		save_network (*(gnet[i].ni));
	}

	close_db ();
	gtk_main_quit ();
}


void
on_fMain_show (GtkWidget * widget, gpointer user_data)
{
	GtkTextBuffer *log;

	// we do the rest of initialisation here...
	load_icons ();

	if (init_db() < 0)
		fprintf(stderr,_("Database init failed, unable to save data!\n"));
	
	signal (SIGINT, stop_signal);
	signal (SIGTERM, stop_signal);
	
	log = gtk_text_view_get_buffer (GTK_TEXT_VIEW (txLog));
	do_detect_cards();

	/* init config widgets */
	gtk_entry_set_text (GTK_ENTRY(eDevice), cfg.device);

	switch (cfg.devtype)
	{
	case DT_ORINOCO:
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rbDriverOR), TRUE);
		break;
	case DT_HOSTAP:
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rbDriverHA), TRUE);
		break;
#ifdef WLAN_NG
	case DT_PRISM:
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rbDriverNG), TRUE);
		break;
#endif	
	default:
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rbDriverSC), TRUE);
	}

	if (cfg.singlechan)
	{
		gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(rbCSpecific),TRUE);
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(eSChannel),(gdouble)cfg.singlechan);
	}

	gtk_window_set_title (GTK_WINDOW(fMain), VERSIONSTR);
	gtk_text_buffer_insert_at_cursor (GTK_TEXT_BUFFER (log),
					  "* Eventlog opened *\n", -1);

	getall_networks (gridAddNet);
	config_changed = FALSE;	// update server config....
	
	send_config ();
	if (!driver_set)
		send_command (C_DETECT_CARD,0);

	gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook1),0);
	
	// setup periodically update
	gtk_timeout_add (UPDATE_INTERVAL, (GtkFunction) do_timer_update, NULL);
}


/*
void
on_tree_select_row (GtkTreeView * treeview,
		    GtkTreePath * arg1,
		    GtkTreeViewColumn * arg2, gpointer user_data)
{
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	int nnode;
	
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree));
	gtk_tree_selection_get_selected (selection, NULL, &iter);
	gtk_tree_model_get (GTK_TREE_MODEL (store), &iter, COL_DATA, &nnode, -1);
	selected_net = nnode->ni;
	
}
*/


void
on_notebook1_switch_page (GtkNotebook * notebook,
			  GtkNotebookPage * page,
			  gint page_num, gpointer user_data)
{
	if (actual_rows >= 0)
	{
		if (active_row >= 0)
			update_gui_pos(active_row);
		else
			update_gui_pos(last_row);
	}
}


gboolean
on_gclWlans_button_press_event (GtkWidget * widget,
				GdkEventButton * event, gpointer user_data)
{
	GtkMenu *menu;
	GdkEventButton *event_button;

	menu = GTK_MENU (create_menu1 ());
	if (event->type == GDK_BUTTON_PRESS)
	{
		event_button = (GdkEventButton *) event;
		if (event_button->button == 3)
		{
			gtk_menu_popup (menu, NULL, NULL, NULL, NULL,
					event_button->button,
					event_button->time);
			return TRUE;
		}
	}
	if (event->type == GDK_2BUTTON_PRESS
	    || event->type == GDK_3BUTTON_PRESS)
	{
		gtk_notebook_set_page (GTK_NOTEBOOK(notebook1), 1);
	}
	return FALSE;
}


void
on_delete1_activate (GtkMenuItem * menuitem, gpointer user_data)
{
/*	gchar *dbssid;
	gchar *dchan;

	if (active_row >= 0)
	{
		gtk_clist_get_text (gclWlans, active_row, 2, &dbssid);
		gtk_clist_get_text (gclWlans, active_row, 1, &dchan);
		if (!deletenetwork (dbssid, atoi (dchan)))
		{
			free (gtk_clist_get_row_data (gclWlans, active_row));
			gtk_clist_remove (gclWlans, active_row);
			actual_rows--;
			active_row = -1;
		}
	}
*/	
}


void
on_view_detail1_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	gtk_notebook_set_page (GTK_NOTEBOOK(notebook1), 1);
}


void
on_gclWlans_unselect_row (GtkCList * clist,
			  gint row,
			  gint column, GdkEvent * event, gpointer user_data)
{
#warning update	
	last_row = row;
	active_row = -1;
}


void
on_cbNight_toggled (GtkToggleButton * togglebutton, gpointer user_data)
{
	GtkRcStyle *rc_style;

	static GdkColor blue = { 0, 0, 0x0000, 0x8000 };
	static GdkColor greenb = { 0, 0, 0xc000, 0xe000 };
	static GdkColor white = { 0, 0xffff, 0xffff, 0xffff };
	static GdkColor black = { 0, 0x0000, 0x0000, 0x0000 };

	rc_style = gtk_rc_style_new ();
	
	if (gtk_toggle_button_get_active (togglebutton))
	{
		night_mode = TRUE;
		rc_style->base[GTK_STATE_NORMAL] = blue;
		rc_style->bg[GTK_STATE_NORMAL] = blue;
		rc_style->text[GTK_STATE_NORMAL] = white;
		rc_style->fg[GTK_STATE_NORMAL] = white;
		rc_style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_FG;
		rc_style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_BG;
		rc_style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_TEXT;
		rc_style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_BASE;

		rc_style->base[GTK_STATE_SELECTED] = greenb;
		rc_style->color_flags[GTK_STATE_SELECTED] |= GTK_RC_BASE;
		
		rc_style->bg[GTK_STATE_SELECTED] = greenb;
		rc_style->color_flags[GTK_STATE_SELECTED] |= GTK_RC_BG;
		gtk_widget_modify_style (GTK_WIDGET (tree), rc_style);
		gtk_widget_modify_style (GTK_WIDGET (gclDetail), rc_style);
		gtk_widget_modify_style (GTK_WIDGET (txLog), rc_style);
	}
	else
	{
		night_mode = FALSE;
		rc_style->base[GTK_STATE_NORMAL] = white;
		rc_style->bg[GTK_STATE_NORMAL] = white;
		rc_style->text[GTK_STATE_NORMAL] = black;
		rc_style->fg[GTK_STATE_NORMAL] = black;
		rc_style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_FG;
		rc_style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_BG;
		rc_style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_TEXT;
		rc_style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_BASE;

		rc_style->base[GTK_STATE_SELECTED] = greenb;
		rc_style->color_flags[GTK_STATE_SELECTED] |= GTK_RC_BASE;
		
		rc_style->bg[GTK_STATE_SELECTED] = greenb;
		rc_style->color_flags[GTK_STATE_SELECTED] |= GTK_RC_BG;
		gtk_widget_modify_style (GTK_WIDGET (tree), rc_style);
		gtk_widget_modify_style (GTK_WIDGET (gclDetail), rc_style);
		gtk_widget_modify_style (GTK_WIDGET (txLog), rc_style);
/*		gtk_widget_set_style(txLog,gtk_style_copy(gtk_widget_get_default_style ()));
		gtk_widget_set_style(gclDetail,gtk_style_copy(gtk_widget_get_default_style ()));
		gtk_widget_set_style(tree,gtk_style_copy(gtk_widget_get_default_style ()));
*/	}
	gtk_rc_style_unref(rc_style);
	update_colors();
}


gboolean
on_fMain_key_press_event (GtkWidget * widget,
			  GdkEventKey * event, gpointer user_data)
{
	switch (event->keyval)
	{
	case GDK_d:
		gtk_notebook_set_page (GTK_NOTEBOOK(notebook1), 1);
		return TRUE;
		break;
	case GDK_l:
		gtk_notebook_set_page (GTK_NOTEBOOK(notebook1), 0);
		return TRUE;
		break;
	case GDK_c:
		gtk_notebook_set_page (GTK_NOTEBOOK(notebook1), 2);
		return TRUE;
		break;
	case GDK_s:
		gtk_notebook_set_page (GTK_NOTEBOOK(notebook1), 3);
		return TRUE;
		break;
	case GDK_r:
		on_delete1_activate (NULL, NULL);
		return TRUE;
		break;
	case GDK_p:
	case GDK_Up:
		if (gtk_notebook_current_page(GTK_NOTEBOOK(notebook1)) == 1)
		{
			if (active_row >= 1) update_gui_pos(active_row - 1);
			return TRUE;
		}
		else
			return FALSE;
		break;
	case GDK_n:
	case GDK_Down:
		if (gtk_notebook_current_page(GTK_NOTEBOOK(notebook1)) == 1)
		{
			if ((active_row+1) < actual_rows) update_gui_pos(active_row + 1);
			return TRUE;
		}
		else
			return FALSE;
		break;
	}
	return FALSE;
}


void
on_rbCAll_toggled (GtkToggleButton * togglebutton, gpointer user_data)
{
	cfg.singlechan = 0;
	config_changed = TRUE;
}


void
on_rbCSpecific_toggled (GtkToggleButton * togglebutton, gpointer user_data)
{
	if (gtk_toggle_button_get_active (togglebutton))
	{
		cfg.singlechan = (int)gtk_spin_button_get_value(GTK_SPIN_BUTTON(eSChannel));

		if (cfg.singlechan < 1 || cfg.singlechan > 14)
		{
			cfg.singlechan = 0;
			fprintf (stderr,"Illegal channel!\n");
		}
	}
	config_changed = TRUE;
}


gboolean
on_eSChannel_value_changed (GtkWidget * widget,
			      			gpointer user_data)
{
	on_rbCSpecific_toggled (GTK_TOGGLE_BUTTON(rbCSpecific), NULL);
	return FALSE;
}


void
on_cbFakeAP_toggled (GtkToggleButton * togglebutton, gpointer user_data)
{
	cfg.filter = gtk_toggle_button_get_active (togglebutton);
	config_changed = TRUE;
}


void
on_cbDevice_changed (GtkEditable * editable, gpointer user_data)
{
}


void
on_bStartScan_clicked (GtkButton * button, gpointer user_data)
{
	char *devname;
	char logmessage[40];

	devname = gtk_editable_get_chars (GTK_EDITABLE (eDevice), 0, -1);
	sprintf (cfg.device, "%s", devname);
#ifdef WLAN_NG
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (rbDriverNG)))
		cfg.devtype = DT_PRISM;
#endif	
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (rbDriverOR)))
		cfg.devtype = DT_ORINOCO;
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (rbDriverHA)))
		cfg.devtype = DT_HOSTAP;
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (rbDriverSC)))
		cfg.devtype = DT_SCAN;

	sprintf (logmessage, "Changed capture device to %s.\n", devname);
	printlog (logmessage);
	g_free (devname);
	gtk_widget_set_sensitive(bScanner,strcmp(cfg.device,"none"));
	config_changed = TRUE;
}


void
on_bStopScan_clicked (GtkButton * button, gpointer user_data)
{
	cfg.scan = FALSE;
	config_changed = TRUE;
}


gboolean   
tv_row_clicked(GtkTreeView *treeview, GtkTreePath *arg1, 
	GtkTreeViewColumn *arg2, gpointer user_data)
{
	GtkWidget *details;
	int netpos;
	GtkTreeSelection *selection;
	GtkTreeIter iter;

	selection = gtk_tree_view_get_selection (treeview);
	if (gtk_tree_selection_get_selected (selection, NULL, &iter) == FALSE) 
		return FALSE;
	gtk_tree_model_get (GTK_TREE_MODEL (store), &iter, COL_DATA, &netpos, -1);

	last_row = active_row;
	active_row = netpos;
	update_gui_pos(netpos);
	return TRUE;	
}

void
update_gui_pos(int netpos)
{
	psnetinfo_t *selected_net;
	gchar tmpstr[80];
	gchar tmpstr3[40];
	gchar *tmpstr2;
	gchar *tmpvec[2];
	guint8 binbssid[8];
	unsigned int b0, b1, b2;
	
	if (netpos > actual_rows) return;
	
	selected_net = gnet[netpos].ni;
	
	tmpvec[0] = malloc (80 * sizeof (gchar));
	tmpvec[1] = malloc (80 * sizeof (gchar));
	gtk_clist_freeze (GTK_CLIST(gclDetail));
	if (detail_initialized)
	{
		/* BSSID */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 0, 1, (char**)&tmpstr);
		if (strcmp (tmpstr, selected_net->bssid))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 0, 1,
						selected_net->bssid);
		/* SSID */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 1, 1, (char**)&tmpstr);
		if (strcmp (tmpstr, selected_net->ssid))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 1, 1,
						selected_net->ssid);
		/* channel */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 2, 1, (char**)&tmpstr);
		sprintf (tmpstr3, "%d", selected_net->channel);
		if (strcmp (tmpstr, tmpstr3))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 2, 1, tmpstr3);
		/* speed */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 3, 1, (char**)&tmpstr);
		sprintf (tmpstr3, "%3.1f Mb/s", (float)selected_net->speed/1000.0);
		if (strcmp (tmpstr, tmpstr3))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 3, 1, tmpstr3);
		/* maxsiglevel */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 4, 1, (char**)&tmpstr);
		sprintf (tmpstr3, "%d", selected_net->maxsiglevel);
		if (strcmp (tmpstr, tmpstr3))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 4, 1, tmpstr3);
		/* first seen */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 5, 1, (char**)&tmpstr);
		tmpstr2 = ctime_r (&selected_net->first, tmpstr3);
		tmpstr2[strlen (tmpstr2) - 1] = 0;
		if (strcmp (tmpstr, tmpstr2))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 5, 1, tmpstr2);
		/* last seen */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 6, 1, (char**)&tmpstr);
		tmpstr2 = ctime_r (&selected_net->last, tmpstr3);
		tmpstr2[strlen (tmpstr2) - 1] = 0;
		if (strcmp (tmpstr, tmpstr2))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 6, 1, tmpstr2);
		/* Frametype */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 7, 1, (char**)&tmpstr);
		if (strcmp (tmpstr, selected_net->type))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 7, 1,
						selected_net->type);
		/* AP type */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 8, 1, (char**)&tmpstr);
		switch (atoi (selected_net->ap))
		{
		case AP_NONE:
			sprintf (tmpstr3, "%s", "none");
			break;
		case AP_ONE:
			sprintf (tmpstr3, "%s", "one");
			break;
		case AP_MORE:
			sprintf (tmpstr3, "%s", "> one");
			break;
		case AP_PROBE:
			sprintf (tmpstr3, "%s", "probe for");
			break;
		default:
			sprintf (tmpstr3, "%s", "<unknown>");
			break;
		}
		if (strcmp (tmpstr3, selected_net->type))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 8, 1, tmpstr3);
		/* subnet */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 9, 1, (char**)&tmpstr);
		sprintf (tmpstr3, "%i.%i.%i.%i",
			 selected_net->ip_range[0],
			 selected_net->ip_range[1],
			 selected_net->ip_range[2],
			 selected_net->ip_range[3]);
		if (strcmp (tmpstr, tmpstr3))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 9, 1, tmpstr3);

		/* ad-hoc */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 10, 1, (char**)&tmpstr);
		sprintf (tmpstr3, "%s", selected_net->isadhoc ? "yes" : "no");
		if (strcmp (tmpstr, tmpstr3))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 10, 1, tmpstr3);
		
		/* hidden */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 11, 1, (char**)&tmpstr);
		sprintf (tmpstr3, "%s", selected_net->ishidden ? "yes" : "no");
		if (strcmp (tmpstr, tmpstr3))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 11, 1, tmpstr3);
		/* WEP */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 12, 1, (char**)&tmpstr);
		sprintf (tmpstr3, "%s", selected_net->wep ? "yes" : "no");
		if (strcmp (tmpstr, tmpstr3))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 12, 1, tmpstr3);
		/* dhcp */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 13, 1, (char**)&tmpstr);
		sprintf (tmpstr3, "%s", selected_net->dhcp ? "yes" : "no");
		if (strcmp (tmpstr, tmpstr3))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 13, 1,
						tmpstr3);
		/* ipsec */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 14, 1, (char**)&tmpstr);
		sprintf (tmpstr3, "%s", selected_net->ipsec ? "yes" : "no");
		if (strcmp (tmpstr, tmpstr3))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 14, 1,
						tmpstr3);
		/* manufacturer */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 15, 1, (char**)&tmpstr);
		sscanf (selected_net->bssid, "%x:%x:%x", &b0, &b1,
			&b2);
		binbssid[0] = b0;
		binbssid[1] = b1;
		binbssid[2] = b2;
		sprintf (tmpstr3, "%s", get_manuf_name (binbssid));
		if (strcmp (tmpstr, tmpstr3))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 15, 1,
						tmpstr3);
		/* ip packages types */
		gtk_clist_get_text (GTK_CLIST(gclDetail), 16, 1, (char**)&tmpstr);
		tmpstr3[0]='\0';
		tmpvec[1][0]='\0';
		for (b0 = 0; b0 < selected_net->pcount; b0++)
		{
			sprintf (tmpstr3, "%s%i,", tmpvec[1],
				 selected_net->pvec[b0]);
			strcpy(tmpvec[1],tmpstr3);
		}
		if (strlen (tmpvec[1]))
			tmpstr3[strlen (tmpstr3) - 1] = '\0';
		if (strcmp (tmpstr, tmpstr3))
			gtk_clist_set_text (GTK_CLIST(gclDetail), 16, 1, tmpstr3);
	}
	else
	{
		strcpy (tmpvec[0], "BSSID");
		strcpy (tmpvec[1], selected_net->bssid);
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);
		strcpy (tmpvec[0], "SSID");
		strcpy (tmpvec[1], selected_net->ssid);
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);

		strcpy (tmpvec[0], "Channel");
		sprintf (tmpvec[1], "%i", selected_net->channel);
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);
		strcpy (tmpvec[0], "Speed");
		sprintf (tmpvec[1], "%3.1f Mb/s", (float)selected_net->speed/1000.0);
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);
		
		strcpy (tmpvec[0], "max signal level");
		sprintf (tmpvec[1], "%i", selected_net->maxsiglevel);
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);

		strcpy (tmpvec[0], "first seen");
		memset (tmpstr, 0, 80);
		tmpstr2 = ctime_r (&selected_net->first, (char*)tmpstr);
		strncpy (tmpvec[1], tmpstr2, strlen (tmpstr2));
		tmpvec[1][strlen (tmpstr2) - 1] = 0;
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);

		strcpy (tmpvec[0], "last seen");
		tmpstr2 = ctime_r (&selected_net->last, (char*)tmpstr);
		strncpy (tmpvec[1], tmpstr2, strlen (tmpstr2));
		tmpvec[1][strlen (tmpstr2) - 1] = 0;
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);

		strcpy (tmpvec[0], "Frametype");
		strcpy (tmpvec[1], selected_net->type);
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);

		strcpy (tmpvec[0], "Accesspoint");
		switch (atoi (selected_net->ap))
		{
		case AP_NONE:
			sprintf (tmpvec[1], "%s", "none");
			break;
		case AP_ONE:
			sprintf (tmpvec[1], "%s", "one");
			break;
		case AP_MORE:
			sprintf (tmpvec[1], "%s", "> one");
			break;
		case AP_PROBE:
			sprintf (tmpvec[1], "%s", "probe for");
			break;
		default:
			sprintf (tmpvec[1], "%s", "<unknown>");
			break;
		}
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);

		strcpy (tmpvec[0], "Subnet");
		sprintf (tmpvec[1], "%i.%i.%i.%i",
			 selected_net->ip_range[0],
			 selected_net->ip_range[1],
			 selected_net->ip_range[2],
			 selected_net->ip_range[3]);

		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);

		strcpy (tmpvec[0], "Ad-Hoc");
		sprintf (tmpvec[1], "%s", selected_net->isadhoc ? "yes" : "no");
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);
		strcpy (tmpvec[0], "Hidden");
		sprintf (tmpvec[1], "%s", selected_net->ishidden ? "yes" : "no");
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);
		strcpy (tmpvec[0], "WEP");
		sprintf (tmpvec[1], "%s", selected_net->wep ? "yes" : "no");
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);
		strcpy (tmpvec[0], "DHCP");
		sprintf (tmpvec[1], "%s", selected_net->dhcp ? "yes" : "no");
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);
		strcpy (tmpvec[0], "IPSec");
		sprintf (tmpvec[1], "%s", selected_net->ipsec ? "yes" : "no");
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);

		sscanf (selected_net->bssid, "%x:%x:%x", &b0, &b1,
			&b2);
		strcpy (tmpvec[0], "Manufacturer");
		binbssid[0] = b0;
		binbssid[1] = b1;
		binbssid[2] = b2;
		sprintf (tmpvec[1], "%s", get_manuf_name (binbssid));
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);

		strcpy (tmpvec[0], "Protocols");
		sprintf (tmpvec[1], "");
		for (b0 = 0; b0 < selected_net->pcount; b0++)
			sprintf (tmpvec[1], "%s%i,", tmpvec[1],
				 selected_net->pvec[b0]);
		if (strlen (tmpvec[1]))
			tmpvec[strlen (tmpvec[1]) - 1] = '\0';
		gtk_clist_append (GTK_CLIST(gclDetail), tmpvec);
		detail_initialized = TRUE;
	}
	sprintf (tmpstr, "%ld", selected_net->psum);
	gtk_label_set_text (GTK_LABEL(lpsum), tmpstr);
	sprintf (tmpstr, "%ld", selected_net->pdata);
	gtk_label_set_text (GTK_LABEL(lpdata), tmpstr);
	sprintf (tmpstr, "%ld", selected_net->pint);
	gtk_label_set_text (GTK_LABEL(lpiiv), tmpstr);
	sprintf (tmpstr, "%f", selected_net->longitude);
	gtk_label_set_text (GTK_LABEL(llong), tmpstr);
	sprintf (tmpstr, "%f", selected_net->latitude);
	gtk_label_set_text (GTK_LABEL(llat), tmpstr);

	gtk_clist_thaw (GTK_CLIST(gclDetail));

	free (tmpvec[0]);
	free (tmpvec[1]);
}
