/*****
*
* Copyright (C) 2001, 2002, 2003 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/


#include <stdio.h>
#include <string.h>

#include "config.h"
#include "detect.h"


#define setup_alert(class) setup_arpspoof_alert((class), sizeof((class)))


typedef struct macaddr {
        uint32_t ipaddr;
        uint8_t macaddr[6];
        
        struct macaddr *next;
} macaddr_t;



static int is_enabled = 0;
static macaddr_t *macaddr_list[1024] = { NULL };


static char *watch_list = NULL;
static int check_directed = 0;
static plugin_detect_t plugin;
static subscribtion_t subscribtion[] = {
        { p_arp, NULL },
	{ p_end, NULL },
};



__inline__ static unsigned int hash_addr(uint32_t ipaddr) 
{
        return ipaddr & (1024 - 1);
}



static nids_alert_t *setup_arpspoof_alert(const char *class, size_t clen) 
{
        static nids_alert_t alert;
        static int initialized = 0;
        static idmef_impact_t impact;

        if ( ! initialized ) {
                initialized = 1;
                nids_alert_init(&alert);
                
                alert.impact = &impact;
                impact.severity = impact_medium;
                impact.completion = 0; /* we don't know */
                impact.type = other;
                impact.description.string = NULL;
        }
        
        alert.classification.name.len = clen;
        alert.classification.name.string = class;
        
        return &alert;
}




static int parse_mac_address(const char *string, uint8_t *addr) 
{
        int ret;
        uint32_t tmp[6];
        
        ret = sscanf(string, "%02x:%02x:%02x:%02x:%02x:%02x",
                     &tmp[0], &tmp[1], &tmp[2], &tmp[3], &tmp[4], &tmp[5]);
        
        if ( ret < 6 ) {
                log(LOG_ERR, "Argument is not a valid MAC address.\n");
                return -1;
        }
        
        addr[0] = (uint8_t) tmp[0];
        addr[1] = (uint8_t) tmp[1];
        addr[2] = (uint8_t) tmp[2];
        addr[3] = (uint8_t) tmp[3];
        addr[4] = (uint8_t) tmp[4];
        addr[5] = (uint8_t) tmp[5];
        
        return 0;
}



static int add_hash_entry(const char *ipaddr, const char *macaddr) 
{
        int ret;
        macaddr_t *addr;
        unsigned int key;
        
        addr = malloc(sizeof(macaddr_t));
        if ( ! addr ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }

        addr->next = NULL;
        
        addr->ipaddr = inet_addr(ipaddr);
        if ( addr->ipaddr < 0 ) {
                log(LOG_ERR, "Argument is not a valid IP address.\n");
                goto err;
        }
        
        ret = parse_mac_address(macaddr, addr->macaddr);
        if ( ret < 0 ) 
                goto err;

        key = hash_addr(addr->ipaddr);

        addr->next = macaddr_list[key];
        macaddr_list[key] = addr;
        
        return 0;

 err:
        free(addr);
        return -1;
}



static void check_arp_request(packet_container_t *pc, etherhdr_t *ether, etherarphdr_t *arp) 
{
        int ret;
        static uint8_t bcast[6] = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };

        if ( check_directed && memcmp(ether->ether_dhost, bcast, sizeof(ether->ether_shost)) ) 
                nids_alert((plugin_generic_t *)&plugin, pc, setup_alert("Directed ARP request"),
                           "ARP request directly directed to a host (not broadcasted)");
        
        ret = memcmp(ether->ether_shost, arp->arp_sha, sizeof(ether->ether_shost));
        if ( ret != 0 ) 
                nids_alert((plugin_generic_t *)&plugin, pc, setup_alert("ARP address mismatch"),
                           "Ethernet source address doesn't match ARP sender address");
}




static void check_arp_reply(packet_container_t *pc, etherhdr_t *ether, etherarphdr_t *arp)
{
        int ret;
        
        ret = memcmp(ether->ether_shost, arp->arp_sha, sizeof(ether->ether_shost));
        if ( ret != 0 )                 
                nids_alert((plugin_generic_t *)&plugin, pc, setup_alert("Arp address mismatch"),
                           "Ethernet source address doesn't match ARP sender address");

        ret = memcmp(ether->ether_dhost, arp->arp_tha, sizeof(ether->ether_dhost));
        if ( ret != 0 )
                nids_alert((plugin_generic_t *)&plugin, pc, setup_alert("Arp address mismatch"),
                           "Ethernet destination address doesn't match ARP target address");
}



static void check_cache_overwrite(packet_container_t *pc, etherhdr_t *ether, etherarphdr_t *arp) 
{
        uint32_t spa;
        macaddr_t *tmp;
        unsigned int key;

        /*
         * do not use extract_* function, cause we store
         * the ip address in network byte order.
         */
        spa = align_uint32(&arp->arp_spa);
        key = hash_addr(spa);

        for ( tmp = macaddr_list[key]; tmp != NULL; tmp = tmp->next ) {                
                if ( tmp->ipaddr != spa )
                        continue;
                
                if ( memcmp(ether->ether_shost, tmp->macaddr, 6) != 0 )
                        nids_alert((plugin_generic_t *)&plugin, pc, setup_alert("ARP cache overwrite"),
                                   "Attempted ARP cache overwrite attack : Source Ethernet address "
                                   "is not the same as real hardware address");
                
                if ( memcmp(arp->arp_sha, tmp->macaddr, 6) != 0 )
                        nids_alert((plugin_generic_t *)&plugin, pc, setup_alert("ARP cache overwrite"),
                                   "Attempted ARP cache overwrite attack : Source Hardware Address "
                                   "is not the same as real hardware address");
                
                break;
        }
}




static void arpcheck_run(packet_container_t *packet) 
{
        etherhdr_t *ether;
        uint16_t op, hrd, pro;
        etherarphdr_t *arp = packet->packet[packet->depth].p.arp_hdr;

        if ( packet->packet[packet->depth - 1].proto != p_ether )
                return;

        hrd = extract_uint16(&arp->ea_hdr.ar_hrd);
        if ( hrd != ARPHRD_ETHER )
                return;

        pro = extract_uint16(&arp->ea_hdr.ar_pro);
        if ( pro != ETHERTYPE_IP )
                return;

        ether = packet->packet[packet->depth - 1].p.ether_hdr;
        op = extract_uint16(&arp->ea_hdr.ar_op);
        switch ( op ) {

        case ARPOP_REQUEST:
                check_arp_request(packet, ether, arp);
                break;

        case ARPOP_REPLY:
                check_arp_reply(packet, ether, arp);
                check_cache_overwrite(packet, ether, arp);
                break;
        }
}



static int get_directed(char *buf, size_t size) 
{
        snprintf(buf, size, "%s", (check_directed) ? "enabled" : "disabled");
        return 0;
}



static int set_directed(prelude_option_t *opt, const char *optarg) 
{
        check_directed = 1;
        return prelude_option_success;
}



static int get_watch(char *buf, size_t size) 
{
        snprintf(buf, size, "%s", watch_list);
        return 0;
}



                 
static int set_watch(prelude_option_t *opt, const char *optarg) 
{
        int ret;
        char *ipaddr, *macaddr;
        
        if ( ! optarg ) {
                log(LOG_ERR, "Argument should be: <ip> <macaddr>.\n");
                return prelude_option_error;
        }

        ipaddr = strdup(optarg);
        if ( ! ipaddr ) {
                log(LOG_ERR, "memory exhausted.\n");
                return prelude_option_error;
        }
        
        macaddr = strchr(ipaddr, ' ');
        if ( ! macaddr ) {
                log(LOG_ERR, "Ip address should be followed by a MAC address.\n");
                free(ipaddr);
                return prelude_option_error;
        }

        *macaddr = 0;
        macaddr++;
        
        ret = add_hash_entry(ipaddr, macaddr);
        if ( ret < 0 )
                log(LOG_ERR, "error adding entry to the hash table.\n");

        if ( watch_list )
                free(watch_list);

        watch_list = strdup(optarg);
        
        return prelude_option_success;
}




static int set_arpspoof_state(prelude_option_t *opt, const char *arg) 
{
        int ret;
        
        if ( is_enabled == 1 ) {
                ret = plugin_unsubscribe((plugin_generic_t *) &plugin);
                if ( ret < 0 )
                        return prelude_option_error;
                
                is_enabled = 0;
        }

        else {
                ret = plugin_subscribe((plugin_generic_t *) &plugin);
                if ( ret < 0 )
                        return prelude_option_error;

                is_enabled = 1;
        }
        
        return prelude_option_success;
}



static int get_arpspoof_state(char *buf, size_t size) 
{
        snprintf(buf, size, "%s", (is_enabled == 1) ? "enabled" : "disabled");
        return prelude_option_success;
}



plugin_generic_t *plugin_init(int argc, char **argv)
{
        prelude_option_t *opt;

        opt = prelude_option_add(NULL, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 0, "arpspoof",
                                 "Enable Arpspoof and precede it's option", no_argument,
                                 set_arpspoof_state, get_arpspoof_state);

        prelude_option_add(opt, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 'd', "directed",
                           "Check for directed ARP request", no_argument,
                           set_directed, get_directed);

        prelude_option_add(opt, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 'w', "arpwatch",
                           "Check this address for cache overwrite attack",
                           required_argument, set_watch, get_watch);
        
        plugin_set_name(&plugin, "ArpSpoof");        
        plugin_set_name(&plugin, "ArpSpoof");
        plugin_set_author(&plugin, "Yoann Vandoorselaere");
        plugin_set_contact(&plugin, "yoann@prelude-ids.org");
        plugin_set_desc(&plugin, "Check for several ARP attack.");
        plugin_set_running_func(&plugin, arpcheck_run);
        plugin_set_subscribtion(&plugin, subscribtion);
        
        return (plugin_generic_t *) &plugin;
}






