// ------------------------------------------------------------------------- //
// $Id: nodefactory.cpp,v 1.57 2004/01/02 15:58:25 pandr Exp $
// ------------------------------------------------------------------------- //
 
/*
 * Copyright (c) 2002 
 *				see AUTHORS list
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#if HAVE_CONFIG_H
# include <config.h>
#endif

#include "common.h"
#include "nodefactory.h"
#include "ftdraw.h"
#include "image.h"
#include "primitive.h"

// ------------------------------------------------------------------------- //

NodeFactory NodeFactory::_instance;

NodeFactory* NodeFactory::instance()
{
	return &_instance;
}

Leaf* NodeFactory::image(const char* filename, bool center, bool gray_alpha,
    const rgba& color, float dropshadow)
{
	Image *img = _image_loader.load(filename);
	if (!img) return 0;

	if (gray_alpha)
		img->gray_alpha();

	TileList* pl = pic_primlist(img, false);

	delete img;

	if (!pl) return 0;

	Leaf *l = new Leaf("PicNode");

	for (TileList::iterator i = pl->begin(); i != pl->end(); ++i)
	{
		Primitive *p = new Primitive(i->tile);
		p->offset(i->xoffset, i->yoffset, 0.0f);
		l->add_primitive(p);
	}

	l->set_size(size3(0.0f, pl->get_width(), 0.0f, 
				pl->get_height(), 0.0f, 0.0f));
	l->set_dropshadow(dropshadow);
	l->set_color(color);
	delete pl;
	return l;
}

Leaf* NodeFactory::rectangle(const v2& ll, const v2& ur, const rgba& ll_color,
	const rgba& lr_color, const rgba& ul_color, const rgba& ur_color, bool stretch, 
	const char *texturename)
{
	Leaf *l = new Leaf("Rectangle");
	Primitive *p = 0;

	float width  = ur.x() - ll.x();
	float height = ur.y() - ll.y();

	if (texturename) {
		Image *img = _image_loader.load(texturename);

		if (img) {
			int imagewidth  = img->get_width();
			int imageheight = img->get_height();

			if ( (imageheight > 256) || (imagewidth > 256) ) {
				log_warning("Unable to use a texture image larger than 256x256");
				return 0;
			}

			log_debug("Image (%dx%d)", imagewidth, imageheight);
			Tile::handle tile = TileBank::instance()->get_tile(
					iv2(imagewidth,imageheight));
			if (!tile) {
				log_warning("Unable to get Tile::handle for the texture");
				return 0;
			}
			tile->copy_from_subimage(img,0,0);
			p = new Primitive(tile);
			p->set_prim_type(GL_QUADS); 
			p->set_num_vertices(4);

			v3* vertices = new v3[4];
			vertices[0].set(ll.x(), ll.y(), 0.0f);
			vertices[1].set(ur.x(), ll.y(), 0.0f);
			vertices[2].set(ur.x(), ur.y(), 0.0f);
			vertices[3].set(ll.x(), ur.y(), 0.0f);
		
			p->set_vertices(vertices);

			if (!stretch) {
				// tiled background
				// Calculate proper (s,t) tex-coords for the whole rectangle
				// add test at least 4x4 and height and width must be powers of 2
				float s = width/imagewidth;
				float t = height/imageheight;

				p->set_use_tex_coords(true);
				v2* texcoords = new v2[4];
				texcoords[0].set(0.0f, 0.0f);
				texcoords[1].set(s, 0.0f);
				texcoords[2].set(s, t);
				texcoords[3].set(0.0f, t);
				p->set_tex_coords(texcoords); 
			}

			l->add_primitive(p);
			l->set_size(size3(0.0f, width, 0.0f, height, 0.0f, 0.0f));
		}
		else {
			log_warning("Unable to load the image in NodeFactory::rectangle");
			return 0;
		}
	}
	else {
		p = new Primitive();
		p->set_prim_type(GL_QUADS); 
		p->set_num_vertices(4);
		p->set_use_colors(true);

		rgba *colors = new rgba[4];
		colors[0] = ll_color;
		colors[1] = lr_color;
		colors[2] = ul_color;
		colors[3] = ur_color;
		p->set_colors(colors);

		v3 *vertices = new v3[4];
		vertices[0].set(ll.x(), ll.y(), 0.0f);
		vertices[1].set(ur.x(), ll.y(), 0.0f);
		vertices[2].set(ur.x(), ur.y(), 0.0f);
		vertices[3].set(ll.x(), ur.y(), 0.0f);
		p->set_vertices(vertices);

		l->add_primitive(p);
		l->set_size(size3(0.0f, width, 0.0f, height, 0.0f, 0.0f));
	}

	return l;
}

Letter* NodeFactory::letter(FreeTypeFont::handle font, FT_ULong c, 
		const rgba& color, float dropshadow)
{
	const Glyph* glyph = font->get_tex_glyph(c);
	if (!glyph) {
		log_warning("Unable to load char %d from font", c);
		return 0;
	}
	std::string name("Letter: ");
	name += c < 256 ? char(c) : '?';
	Letter* l = new Letter(name, font, glyph, c);
	l->set_color(color);
	l->set_dropshadow(dropshadow);
	return l;
}

Group* NodeFactory::group(const char* name)
{
	return new Group(name);
}

TileList* NodeFactory::pic_primlist(Image *img, bool center)
{

	int imagewidth  = img->get_width();
	int imageheight = img->get_height();
	int xmeshsize   = min(256, next_pow_two(imagewidth));
	int ymeshsize   = min(256, next_pow_two(imageheight));

	int xtiles = (int)ceil((float)imagewidth  / xmeshsize);
	int ytiles = (int)ceil((float)imageheight / ymeshsize);

	TileList* list = new TileList(imagewidth, imageheight);

	log_debug("Image (%dx%d) tiled %dx%d\n", 
			imagewidth, imageheight, xtiles, ytiles);
	int xoffset = 0;
	int yoffset = 0;
	for (int xtile = 0; xtile < xtiles; xtile++) {
		for (int ytile = 0; ytile < ytiles; ytile++) {
			int tilewidth  = min(xmeshsize, imagewidth  - xoffset);
			int tileheight = min(ymeshsize, imageheight - yoffset);

			Tile::handle tile =
			    TileBank::instance()->get_tile(iv2(tilewidth, tileheight));

			tile->copy_from_subimage(img, xoffset, yoffset);

			list->push_back(TileInfo(tile, xoffset, yoffset));

			yoffset += ymeshsize;
		}
		yoffset = 0;
		xoffset += xmeshsize;
	}
	return list;
}

// ------------------------------------------------------------------------- //

