# --------------------------------------------------------------------------- #
# $Id: merger.py,v 1.5 2003/04/13 13:51:43 weismann Exp $
# --------------------------------------------------------------------------- #

# This file contains modules to facilitate template based merge
# with key word substitution

import os, re
import pllog
import plfiles

class ExportMerger:
	"""
		DESCRIPTION:
		
		Objects of this class is used to store a list containing hashes of
		information and to merge these information with template files. 

		METHODS:

		init
			constructor setting up template file names, necessary export
			information as well as log- and filehandling modules.

		get_exportdir
			return the export directory.

		get_exportbasename
			return the basename of the files exported.

		next_running_no
			return (and increment) the running number in a three digit format.

		add_merge_info
			converts the info recieved as list into hash table and adds this
			hash table to the list of hashes.  Fills necessary info into hash
			previously at the tail of the list, linking its information to the
			info of the new tail element.
		
		show_merge_info
			dump information of list-of-hashes

		dump_raw
			dump minimalistic information extracted from the hashtables in the
			hash list into file

		merge
			merges information from hashes in the list with templatefiles


		VARIABLES:

		mergeinfo
			The list used to store hashes

		running_no
			The running number

		prev_no, prev_thumb_shot_name, prev_shot_width, prev_shot_height
			information from the hash currently at the tail of mergeinfo. Added
			to new tail element, linking new tail hash to last tail hash

	"""

	def __init__(self, slidefile, firstslidefile, lastslidefile, \
	                   indexfile, exportdir, exportbasename, exporttype):
		self.slidefile               = slidefile
		self.firstslidefile          = firstslidefile
		self.lastslidefile           = lastslidefile
		self.indexfile               = indexfile
		self.usingexportdir          = 0
		self.exporttype              = exporttype
		self.exportdir               = ""
		if exportdir:
			self.usingexportdir        = 1
			self.exportdir = exportdir+"/"
		else:
			self.exportdir = "./"+exporttype+"/"
			if not os.path.isdir(self.exportdir):
				os.mkdir(self.exportdir)
		self.exportbasename          = exportbasename
		self.extension               = ".%s" % self.exporttype
		self.mergeinfo               = []
		self.running_no              = 0
		self.prev_no                 = ""
		self.prev_thumb_shot_name    = ""
		self.prev_thumb_shot_width   = 0
		self.prev_thumb_shot_height  = 0

	def get_exportdir(self):
		return self.exportdir

	def get_exportbasename(self):
		return self.exportbasename

	def next_running_no(self):
		self.running_no += 1
		return "%03i"%self.running_no

	def add_merge_info(self, info):

		# Add information as "next info" on the tail element
		if len(self.mergeinfo) > 0:
			self.mergeinfo[-1]['next_thumb_shot_no']     = info[0]
			self.mergeinfo[-1]['next_thumb_shot_name']   = info[4]
			self.mergeinfo[-1]['next_thumb_shot_width']  = info[5]
			self.mergeinfo[-1]['next_thumb_shot_height'] = info[6]

		# Build info
		inf = {}
		inf['shot_no']                = info[0]
		inf['shot_name']              = info[1]
		inf['shot_width']             = info[2]
		inf['shot_height']            = info[3]
		inf['thumb_shot_name']        = info[4]
		inf['thumb_shot_width']       = info[5]
		inf['thumb_shot_height']      = info[6]
		inf['prev_thumb_shot_no']     = self.prev_no
		inf['prev_thumb_shot_name']   = self.prev_thumb_shot_name
		inf['prev_thumb_shot_width']  = self.prev_thumb_shot_width
		inf['prev_thumb_shot_height'] = self.prev_thumb_shot_height
		inf['next_thumb_shot_no']     = ""
		inf['next_thumb_shot_name']   = ""
		inf['next_thumb_shot_width']  = 0
		inf['next_thumb_shot_height'] = 0

		# Save info
		self.mergeinfo.append(inf)

		# Remember some info as future "previous info"
		self.prev_no                = info[0]
		self.prev_thumb_shot_name   = info[4]
		self.prev_thumb_shot_width  = info[5]
		self.prev_thumb_shot_height = info[6]

	def show_merge_info(self):
		for info in self.mergeinfo:
			print "========"
			for key in info.keys():
				print "%s: %s" % (key, info[key])

	def dump_raw(self):
		dumpfilename = "%s%s.export" \
			   %(self.exportdir, \
			     self.exportbasename)
		print "Exporting to '%s'" % dumpfilename
		f = open(dumpfilename, "w")
		for info in self.mergeinfo:
			f.write("%s;%i;%i;%s;%i;%i;\n"%(\
			  info['shot_name'],\
			  info['shot_width'],\
			  info['shot_height'],\
			  info['thumb_shot_name'],\
			  info['thumb_shot_width'],\
			  info['thumb_shot_height']))
		f.close()
			
	def merge(self):
		index_pre                 = []
		index_post                = []
		index_middle              = []
		index_rep_template        = []
		number_of_slides          = len(self.mergeinfo)
		exportindexfilename       = ""
		shortexportindexfilename  = ""
		using_index               = 0
		
		# Prepare index file buffers
		if self.indexfile:
			using_index = 1
			
			# strip extension
			indexfile_striped = os.path.basename(self.indexfile)
			indexfile_striped = "".join(indexfile_striped.split('.')[:-1]) 

			exportindexfilename = "%s%s%s" \
			   %(self.exportdir, \
			     indexfile_striped, \
				 self.extension)
			shortexportindexfilename = "%s%s" \
			   %(indexfile_striped, \
				 self.extension)

			file  = plfiles.open_file(self.indexfile)
			lines = file.readlines()

			# Read index template and split into three buffers
			state = 0
			for line in lines:
				if state == 0 and re.search("%%repeat%%", line):
					state = 1
					continue
				if state == 1 and re.search("%%repeat%%", line):
					state = 2
					continue
				if state == 0 or state == 2:
					line = re.sub("\%\%showname\%\%", self.exportbasename,line)
					line = re.sub("\%\%totalnumber\%\%", \
					              "%i"%number_of_slides, line)
					if state == 0:
						index_pre.append(line)
					elif state == 2:
						index_post.append(line)
				else:
					index_rep_template.append(line)

		# Merge information with export templates
		print "Exporting to '%s'" % self.exportdir
		i = 0
		for info in self.mergeinfo:
			i += 1

			# Setup info
			exportfilename = "%s%s%s%s" \
			   %(self.exportdir, \
			     self.exportbasename, \
				 info['shot_no'], \
				 self.extension)
			shortexportfilename = "%s%s%s" \
			   %(self.exportbasename, \
				 info['shot_no'], \
				 self.extension)
			prevfilename = ""
			nextfilename = ""
			if not info['prev_thumb_shot_name'] == "":
				prevfilename = "%s%s%s" \
				    %(self.exportbasename, info['prev_thumb_shot_no'], 
						self.extension)
			if not info['next_thumb_shot_name'] == "":
				nextfilename = "%s%s%s" \
				    %(self.exportbasename, info['next_thumb_shot_no'], 
						self.extension)

			# Determine which - if any - single slide file to merge with
			singleslidefilename = None
			if i == 1:
				if self.firstslidefile:
					singleslidefilename = self.firstslidefile
			elif i == number_of_slides:
				if self.lastslidefile:
					singleslidefilename = self.lastslidefile
			if (not singleslidefilename):
				if self.slidefile:
					singleslidefilename = self.slidefile
				else:
					pllog.abort("Templatefile unspecified")

			# Merge with single file template
			print "Exporting '%s' (template: '%s')" \
			      % (shortexportfilename, singleslidefilename)
			f = open(exportfilename, "w")
			file    = plfiles.open_file(singleslidefilename)
			lines   = file.readlines()
			for line in lines:
				line = re.sub("\%\%showname\%\%", self.exportbasename, line)
				line = re.sub("\%\%filename\%\%", shortexportfilename, line)
				line = re.sub("\%\%indexfilename\%\%", 
					shortexportindexfilename, line)
				line = re.sub("\%\%totalnumber\%\%", 
					"%i"%number_of_slides, line)
				line = re.sub("\%\%shotnumber\%\%", info['shot_no'], line)
				line = re.sub("\%\%shotname\%\%", info['shot_name'], line)
				line = re.sub("\%\%width\%\%", "%i"%info['shot_width'], line)
				line = re.sub("\%\%height\%\%", "%i"%info['shot_height'], line)
				line = re.sub("\%\%prevfilename\%\%", prevfilename, line)
				line = re.sub("\%\%prevshotnumber\%\%", 
					info['prev_thumb_shot_no'], line)
				line = re.sub("\%\%prevshotname\%\%", 
					info['prev_thumb_shot_name'], line)
				line = re.sub("\%\%prevwidth\%\%", 
					"%i"%info['prev_thumb_shot_width'], line)
				line = re.sub("\%\%prevheight\%\%", 
					"%i"%info['prev_thumb_shot_height'], line)
				line = re.sub("\%\%nextfilename\%\%", nextfilename, line)
				line = re.sub("\%\%nextshotnumber\%\%", 
					info['next_thumb_shot_no'], line)
				line = re.sub("\%\%nextshotname\%\%", 
					info['next_thumb_shot_name'], line)
				line = re.sub("\%\%nextwidth\%\%", 
					"%i"%info['next_thumb_shot_width'], line)
				line = re.sub("\%\%nextheight\%\%", 
					"%i"%info['next_thumb_shot_height'], line)
				f.write(line)
			f.close()
				
			# Merge with indexfile's repeat-template
			for line in index_rep_template:
				line = re.sub("\%\%showname\%\%", self.exportbasename, line)
				line = re.sub("\%\%filename\%\%", shortexportfilename, line)
				line = re.sub("\%\%indexfilename\%\%", 
					shortexportindexfilename, line)
				line = re.sub("\%\%prevfilename\%\%", prevfilename, line)
				line = re.sub("\%\%nextfilename\%\%", nextfilename, line)
				line = re.sub("\%\%totalnumber\%\%", 
					"%i"%number_of_slides, line)
				line = re.sub("\%\%shotnumber\%\%", info['shot_no'], line)
				line = re.sub("\%\%thumbshotname\%\%", 
					info['thumb_shot_name'], line)
				line = re.sub("\%\%thumbwidth\%\%", 
					"%i"%info['thumb_shot_width'], line)
				line = re.sub("\%\%thumbheight\%\%", 
					"%i"%info['thumb_shot_height'], line)
				index_middle.append(line)

		# Write indexfile
		if using_index:
			print "Exporting '%s' (template: '%s')" \
			      % (shortexportindexfilename, self.indexfile)
			f = open(exportindexfilename, "w")
			for line in index_pre:
				f.write(line)
			for line in index_middle:
				f.write(line)
			for line in index_post:
				f.write(line)
			f.close()

# --------------------------------------------------------------------------- #

