/*  Neutrino:  neutrino-preferences.c
 *
 *  Copyright (C) 2003 David A Knight <david@ritter.demon.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <glade/glade.h>
#include <gconf/gconf-client.h>

#include <gtk/gtksizegroup.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>

#include "neutrino-window.h"
#include "neutrino-preferences.h"

static const gchar *toggle_prefs[] = {
	"writeid3", "/apps/neutrino/download/metatoid3",
	"write1.1", "/apps/neutrino/download/id3version1",
	"recursedirs", "/apps/neutrino/upload/recurse",
	"showmeta", "/apps/neutrino/upload/prompt",
	"stripid3", "/apps/neutrino/upload/strip",
	NULL
};
static const gchar *string_prefs[] = {
	"filename", "/apps/neutrino/download/format",
	"downloadto", "/apps/neutrino/download/path",
	"musicpath", "/apps/neutrino/general/musicpath",
	NULL
};

static void neutrino_preferences_toggle_notify( GConfClient *client,
						guint cnxn_id,
						GConfEntry *entry,
						gpointer data );
static void neutrino_preferences_string_notify( GConfClient *client,
						guint cnxn_id,
						GConfEntry *entry,
						gpointer data );

void neutrino_edit_preferences( NeutrinoWindow *window )
{
	GladeXML *xml;
	GtkWidget *widget;
	gint i;
	GConfClient *client;

	GtkSizeGroup *group;
	static const gchar *labels[] = {
		"format", "downloadpath", "music",
		NULL
	};

	client = gconf_client_get_default();

	xml = glade_xml_new( DATADIR"/neutrino/glade/neutrino.glade",
			     "preferences", NULL );

	group = gtk_size_group_new( GTK_SIZE_GROUP_HORIZONTAL );
	for( i = 0; labels[ i ]; ++ i ) {
		widget = glade_xml_get_widget( xml, labels[ i ] );
		gtk_size_group_add_widget( group, widget );
	}

	glade_xml_signal_autoconnect( xml );

	/* handle boolean preferences */
	for( i = 0; toggle_prefs[ i ]; i += 2 ) {
		GtkWidget *toggle;
		gboolean value;
		gint handle;

		toggle = glade_xml_get_widget( xml, toggle_prefs[ i ] );
		value = gconf_client_get_bool( client,
					       toggle_prefs[ i + 1 ],
					       NULL );
		g_object_set_data( G_OBJECT( toggle ), "key",
				   (gpointer)toggle_prefs[ i + 1 ] );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ),
					      value );
		handle = gconf_client_notify_add( client,
						  toggle_prefs[ i + 1 ],
						  neutrino_preferences_toggle_notify,
						  toggle, NULL, NULL );
		g_object_set_data( G_OBJECT( toggle ), "notify",
				   GINT_TO_POINTER( handle ) );
	}
	/* handle string preferences */
	for( i = 0; string_prefs[ i ]; i += 2 ) {
		GtkWidget *entry;
		gchar *value;
		gint handle;

		entry = glade_xml_get_widget( xml, string_prefs[ i ] );
		value = gconf_client_get_string( client,
						 string_prefs[ i + 1 ],
						 NULL );
		if( ! value ) {
			value = g_strdup( "" );
		}

		gtk_entry_set_text( GTK_ENTRY( entry ), value );
		g_free( value );
		
		g_object_set_data( G_OBJECT( entry ), "key",
				   (gpointer)string_prefs[ i + 1 ] );
		handle = gconf_client_notify_add( client,
						  string_prefs[ i + 1 ],
						  neutrino_preferences_string_notify,
						  entry, NULL, NULL );
		g_object_set_data( G_OBJECT( entry ), "notify",
				   GINT_TO_POINTER( handle ) );
	}

	widget = glade_xml_get_widget( xml, "preferences" );
	gtk_window_set_transient_for( GTK_WINDOW( widget ),
				      GTK_WINDOW( window ) );
}

void neutrino_preferences_cleanup( GtkWidget *widget )
{
	GladeXML *xml;
	GConfClient *client;
	gint handle;
	gint i;

	xml = glade_get_widget_tree( widget );

	client = gconf_client_get_default();

	/* handle boolean preferences */
	for( i = 0; toggle_prefs[ i ]; i += 2 ) {
		GtkWidget *toggle;

		toggle = glade_xml_get_widget( xml, toggle_prefs[ i ] );

		handle = GPOINTER_TO_INT( g_object_get_data( G_OBJECT(toggle),
							     "notify" ) );

		gconf_client_notify_remove( client, handle );
	}
	/* handle string preferences */
	for( i = 0; string_prefs[ i ]; i += 2 ) {
		GtkWidget *entry;

		entry = glade_xml_get_widget( xml, string_prefs[ i ] );

		handle = GPOINTER_TO_INT( g_object_get_data( G_OBJECT(entry),
							     "notify" ) );
		gconf_client_notify_remove( client, handle );
	}

	g_object_unref( client );

	gtk_widget_destroy( widget );
}

gboolean neutrino_preferences_toggle_changed( GtkWidget *toggle )
{
	GConfClient *client;
	const gchar *key;
	gboolean state;
	
	client = gconf_client_get_default();

	key = g_object_get_data( G_OBJECT( toggle ), "key" );

	state = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) );

	gconf_client_set_bool( client, key, state, NULL );

	g_object_unref( client );

	return TRUE;
}

gboolean neutrino_preferences_string_changed( GtkWidget *widget )
{
	GConfClient *client;
	const gchar *key;
	const gchar *string;
	
	client = gconf_client_get_default();

	key = g_object_get_data( G_OBJECT( widget ), "key" );

	string = gtk_entry_get_text( GTK_ENTRY( widget ) );

	gconf_client_set_string( client, key, string, NULL );

	g_object_unref( client );

	return FALSE;
}


static void neutrino_preferences_toggle_notify( GConfClient *client,
						guint cnxn_id,
						GConfEntry *entry,
						gpointer data )
{
	GtkWidget *toggle;

	toggle = GTK_WIDGET( data );

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		gboolean wstate = 
                        gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle));
                gboolean state = gconf_value_get_bool( entry->value );
		
                if( wstate != state ) {
                        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(toggle),
                                                      state );
                }
	}
}

static void neutrino_preferences_string_notify( GConfClient *client,
						guint cnxn_id,
						GConfEntry *entry,
						gpointer data )
{
	GtkWidget *widget;

	widget = GTK_WIDGET( data );

	if( entry->value && entry->value->type == GCONF_VALUE_STRING ) {
		const gchar *value;
		const gchar *wvalue;

		value = gconf_value_get_string( entry->value );
		wvalue = gtk_entry_get_text( GTK_ENTRY( widget ) );

		if( ! value ) {
			value = "";
		}
		if( strcmp( value, wvalue ) ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    value );
		}
	}
}

