-- ----------------------------------------------------------------------------------------
-- Copyright (C) 2004 MySQL AB
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-- ----------------------------------------------------------------------------------------


-- ----------------------------------------------------------------------------------------
-- @file _library.lua
-- @brief A collection of common auxiliary functions that is loaded when the Lua loader 
-- gets initialized
-- ----------------------------------------------------------------------------------------


-- ----------------------------------------------------------------------------------------
-- @brief Prepares a return value
--
--   Every function that is exposed to the Grt needs to return its value in a dict that
-- has a "value" entry. Use this function to prepare the result values. The function can
-- also be called without a parameter. In that case 1 will be returned to the Grt.
--
-- Example:
--   function calcDouble(args)
--     return _success(args[1] * 2)
--   end
--
-- @param result  the result value that should be returned to the Grt.
--
-- @return the result value prepared to be returned to the Grt
-- ----------------------------------------------------------------------------------------
function _success(result)
  return result and {value= result} or {value= 1}
end


-- ----------------------------------------------------------------------------------------
-- @brief Returns an error to the Grt
--
--   Use this function to pass an error message back to the Grt.
--
-- Example:
--   function calcDouble(args)
--     return args[1] and 
--       _success(args[1] * 2) or 
--       _error("You need to pass a number as argument")
--   end
--
-- @param errorText  the error message
-- @param errorDetails  detailed information about the error
--
-- @return the error prepared to be returned to the Grt
-- ----------------------------------------------------------------------------------------
function _error(errorText, errorDetails)
  return {error= errorText, detail= errorDetails}
end


-- ----------------------------------------------------------------------------------------
-- @brief Extracts the return value of a Grt function
--
--   A Grt function returns its result as a dict in the "value" entry. This function
-- extracts the value
--
-- @return the 
-- ----------------------------------------------------------------------------------------
function _grtRes(result)
  return getChild(result, "/value")
end


-- ----------------------------------------------------------------------------------------
-- @brief Extracts the return value of a Grt function as a Lua value
--
--   A Grt function returns its result as a dict in the "value" entry. This function
-- extracts the value and converts it to a Lua value
--
-- @return the 
-- ----------------------------------------------------------------------------------------
function _grtRes2Lua(result)
  return grt2Lua(_grtRes(result))
end


-- ----------------------------------------------------------------------------------------
-- @brief Returns a new generated GUID
--
--   Calls the Grt function getGuid() from the module Base to retrieve a GUID. The Grt
-- function result is then converted to a Lua string
--
-- @return a new generated GUID as lua string
-- ----------------------------------------------------------------------------------------
function _getGuid()
  return _grtRes2Lua(Base:getGuid())
end

-- ----------------------------------------------------------------------------------------
-- @brief Checks if a files exists
--
--   Tries to open the files. If the file exists it will close the file again and 
-- return true. If it does not exists it will return false (as Lua will return false 
-- when nothing is returned explicitly
--
-- @return true if the files exists, false if not
-- ----------------------------------------------------------------------------------------
function _fileExists(filename)
  local f = io.open(filename, "r")

  if f then
    f:close()

    return true
  end
end
