/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_model.cpp 
 * @brief Implementation of the model that manages the visual representation in the generic canvas.
 * 
 */

#ifdef _WINDOWS
  #define  WIN32_LEAN_AND_MEAN
  #include <windows.h>
#endif // ifdef _WINDOWS

#include <GL/gl.h>
#include <GL/glu.h>

#include "myx_gc_figure.h"
#include "myx_gc_model.h"
#include "myx_gc_canvas.h"

//----------------------------------------------------------------------------------------------------------------------

CGCModel::CGCModel(CGenericCanvas* Controller)
{
  FCanvas = Controller;
}

//----------------------------------------------------------------------------------------------------------------------

CGCModel::~CGCModel(void)
{
  ClearFigures();
  ClearLayouts();
  ClearStyles();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks if there is any pending OpenGL error (e.g. from style creation). Call is forwared to the canvas and handeled there.
 */
void CGCModel::CheckError(void)
{
  FCanvas->CheckError();
}

//----------------------------------------------------------------------------------------------------------------------

CGenericCanvas* CGCModel::GetController(void)
{
  return FCanvas;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a new figure and puts in into the internal list.
 *
 * @param Type The name of the template from which to create the figure.
 */
CFigure* CGCModel::CreateFigure(wstring Type)
{
  CFigureTemplate* Template = Layout(Type);
  CFigure* Figure = new CFigure(this, Template);
  FFigures.insert(Figure);

  return Figure;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Clears the model, that is, the figures defined in this model.
 */
void CGCModel::ClearFigures(void)
{
  FCanvas->BeginUpdate();
  for (CFigureList::iterator Iterator = FFigures.begin(); Iterator != FFigures.end(); ++Iterator)
    delete *Iterator;
  FFigures.clear();
  FCanvas->EndUpdate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Clears the figure template list (layouts).
 *
 * @note: Existing figures are not concerned by deleting the template list.
 */
void CGCModel::ClearLayouts(void)
{
  for (CLayoutList::iterator Iterator = FLayouts.begin(); Iterator != FLayouts.end(); ++Iterator)
    delete Iterator->second;
  FLayouts.clear();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Clears all defined styles.
 *
 * @note: Deleting all styles means to leave all figures without visual representation.
 */
void CGCModel::ClearStyles(void)
{
  for (CStyleList::iterator Iterator = FStyles.begin(); Iterator != FStyles.end(); ++Iterator)
    delete Iterator->second;
  FStyles.clear();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the layout entry for the given identifier. If ther eis no layout with that name yet then one is created.
 *
 * @param ID The identification of the style.
 * @param The corresponding layout entry.
 */
CFigureTemplate* CGCModel::Layout(wstring ID)
{
  CFigureTemplate* Result = NULL;

  CLayoutList::iterator Iterator = FLayouts.find(ID);
  if (Iterator == FLayouts.end())
  {
    Result = new CFigureTemplate(ID);
    FLayouts[ID] = Result;
  }
  else
    Result = Iterator->second;

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the style entry for the given identifier. If there is no style with that name one is created.
 *
 * @param ID The identification of the style.
 * @return The corresponding style entry, can be NULL if ID is empty.
 */
CGCStyle* CGCModel::Style(wstring ID)
{
  CGCStyle* Result = NULL;

  if (ID.size() != 0)
  {
    CStyleList::iterator Iterator = FStyles.find(ID);
    if (Iterator == FStyles.end())
    {
      Result = new CGCStyle;
      FStyles[ID] = Result;
    }
    else
      Result = Iterator->second;
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

