/* GIMP Minolta Raw Format color plugin
 * Copyright (C) 2002 Laurent HOUDARD <lhoudard@netcourrier.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>

#include "../../config.h"
#include <glib/gthread.h>
#include <gtk/gtk.h>
#include <libgimp/gimp.h>

#include <MRI.h>
#include "PersistentVals.h"
#include "UI_RunVals.h"
#include "preview.h"
#include "histogram.h"
#include "../options.h"
#include "mrwplugin-intl.h"

#include "libeog/eog-image.h"
#include "libeog/eog-scroll-view.h"
#include "libeog/eog-pixbuf-util.h"

#define IMAGE_VIEW_ZOOM_MULTIPLIER 1.05

static gint preview_popup_handler(GtkWidget *widget, GdkEventButton *event);

static Preview preview;

static void
rotate_region (MRI_Region *r, int angle, int w, int h)
{
  MRI_Region n;

  switch (angle) {
    case 90:
      n.x = h - r->y - r->height;
      n.y = r->x;
      n.width = r->height;
      n.height = r->width;
      *r = n;
      break;
    case 180:
      n.x = w - r->x - r->width;
      n.y = h - r->y - r->height;
      n.width = r->width;
      n.height = r->height;
      *r = n;
      break;
    case 270:
      n.x = r->y;
      n.y = w - r->x - r->width;
      n.width = r->height;
      n.height = r->width;
      *r = n;
      break;
  }
}

printr (char *str, MRI_Region *r)
{
    fprintf (stderr, "%s: %d %d %d %d\n", str, r->x, r->y, r->width, r->height);
}

clip_to_preview_region (MRI_Region *irect, MRI_Region *crect)
{
    *crect = *irect;
    if (crect->x < preview.previewRegion.x) {
    	crect->width -= preview.previewRegion.x - crect->x;
	if (crect->width < 0) crect->width = 0;
    	crect->x = preview.previewRegion.x;
    }
    if (crect->x+crect->width > preview.previewRegion.x+preview.previewRegion.width) {
    	crect->width -= preview.previewRegion.x + preview.previewRegion.width - crect->x;
	if (crect->width < 0) crect->width = 0;
    }
    if (crect->y < preview.previewRegion.y) {
    	crect->height -= preview.previewRegion.y - crect->y;
	if (crect->height < 0) crect->height = 0;
    	crect->y = preview.previewRegion.y;
    }
    if (crect->y+crect->height > preview.previewRegion.y+preview.previewRegion.height) {
    	crect->height -= preview.previewRegion.y + preview.previewRegion.height - crect->y;
	if (crect->height < 0) crect->height = 0;
    }
}

image_rect_to_preview_rect (MRI_Region *irect, MRI_Region *prect)
{
    MRI_Region rpr = preview.previewRegion;

    *prect = *irect;
    prect->x -= preview.previewRegion.x;
    prect->y -= preview.previewRegion.y;
    rotate_region (prect, plvals.rotate,
                   preview.previewRegion.width,
		   preview.previewRegion.height);
    prect->x /= plvals.previewShrink;
    prect->y /= plvals.previewShrink;
    prect->width /= plvals.previewShrink;
    prect->height /= plvals.previewShrink;
}

preview_rect_to_image_rect (MRI_Region *prect, MRI_Region *irect)
{
    *irect = *prect;
    irect->x *= plvals.previewShrink;
    irect->y *= plvals.previewShrink;
    irect->width *= plvals.previewShrink;
    irect->height *= plvals.previewShrink;
    if (plvals.rotate == 90 || plvals.rotate == 270)
    rotate_region (irect, 360 - plvals.rotate,
    		   preview.previewRegion.height, preview.previewRegion.width);
    else
    rotate_region (irect, 360 - plvals.rotate,
    		   preview.previewRegion.width, preview.previewRegion.height);
    irect->x += preview.previewRegion.x;
    irect->y += preview.previewRegion.y;
}

gint32
get_preview_selected_rectangle (MRI_Region *region)
{
  MRI_Region retval;
  MRI_Region p = preview.previewRegion;

  eog_scroll_view_get_frame_rectangle (EOG_SCROLL_VIEW(preview.image_view),
  				  &region->x, &region->y, &region->width, &region->height);
  if (region->width > 0 && region->height > 0) {
    region->x *= plvals.previewShrink;
    region->y *= plvals.previewShrink;
    region->width *= plvals.previewShrink;
    region->height *= plvals.previewShrink;
    if (plvals.rotate == 90 || plvals.rotate == 270)
        rotate_region (region, 360 - plvals.rotate, p.height, p.width);
    else
        rotate_region (region, 360 - plvals.rotate, p.width, p.height);
    region->x += p.x;
    region->y += p.y;
    return TRUE;
  }
  else {
    region->x = 0;
    region->y = 0;
    region->width = MRI_GetWidth (preview.mri);
    region->height = MRI_GetHeight (preview.mri);
    return FALSE;
  }
}

void
preview_set_gray_world_region ()
{
  MRI_Region grayWorldRegion;

  if (get_preview_selected_rectangle (&grayWorldRegion))
    mrw_set_gray_world_region (preview.mri, &grayWorldRegion);
}

void
preview_clear_gray_world_region ()
{
  MRI_Region grayWorldRegion;

  grayWorldRegion.x = 0;
  grayWorldRegion.y = 0;
  grayWorldRegion.width = MRI_GetWidth (preview.mri);
  grayWorldRegion.height = MRI_GetHeight (preview.mri);
  mrw_set_gray_world_region (preview.mri, &grayWorldRegion);
}

#define MAX_CROPS 16
MRI_Region cropStack[MAX_CROPS];
guint shrinkStack[MAX_CROPS];
int nCrop = 0;

void
preview_set_crop (GtkWidget *mi, gpointer grunvals)
{
    MRI_Region cropRegion;

    if (nCrop < MAX_CROPS && get_preview_selected_rectangle (&cropRegion)) {
        if (cropRegion.width > 2 && cropRegion.height > 2) {
	    UI_RunVals *runVals = (UI_RunVals *)grunvals;
            cropStack[nCrop] = preview.previewRegion;
            shrinkStack[nCrop] = runVals->pVals->previewShrink;
            nCrop++;
            preview.previewRegion = cropRegion;
            runVals->selection = preview.previewRegion;
            runVals->selection.width = 0;
            runVals->selection.height = 0;
            mrw_set_crop_region (preview.mri, &preview.previewRegion);
        }
    }
}

void
preview_pop_crop (GtkWidget *mi, gpointer grunvals)
{
    if (nCrop != 0) {
	UI_RunVals *runVals = (UI_RunVals *)grunvals;
        runVals->selection = preview.previewRegion;
        nCrop--;
        preview.previewRegion = cropStack[nCrop];
        runVals->pVals->previewShrink = shrinkStack[nCrop];
        update_preview_shrink_scale (runVals->pVals->previewShrink);
        mrw_set_crop_region (runVals->mri, &preview.previewRegion);
    }
}

void
rotate_preview (int angle)
{
  int x, y, h, w;
  int nx, ny, nh, nw;
  int tmp;
  
  if (preview.ui_image == NULL) return;

  EogTransform* trans = eog_transform_rotate_new (angle);

  eog_scroll_view_get_frame_rectangle (EOG_SCROLL_VIEW(preview.image_view), &x, &y, &w, &h);
  switch (angle) {
    case 90:
      eog_image_transform (EOG_IMAGE(preview.ui_image), trans);
      preview.pixbuf = eog_image_get_pixbuf (EOG_IMAGE(preview.ui_image));
      nx = preview.height - y - h;
      ny = x;
      nw = h;
      nh = w;
      tmp = preview.width; preview.width = preview.height; preview.height = tmp;
      break;
    case 180:
      eog_image_transform (EOG_IMAGE(preview.ui_image), trans);
      preview.pixbuf = eog_image_get_pixbuf (EOG_IMAGE(preview.ui_image));
      nx = preview.width - x - w;
      ny = preview.height - y - h;
      nw = w;
      nh = h;
      break;
    case 270:
      eog_image_transform (EOG_IMAGE(preview.ui_image), trans);
      preview.pixbuf = eog_image_get_pixbuf (EOG_IMAGE(preview.ui_image));
      nx = y;
      ny = preview.width - x - w;
      nw = h;
      nh = w;
      tmp = preview.width; preview.width = preview.height; preview.height = tmp;
      break;
    default:
      return;
  }
  eog_scroll_view_set_frame_rectangle (EOG_SCROLL_VIEW(preview.image_view), nx, ny, nw, nh);
  eog_scroll_view_zoom_fit (EOG_SCROLL_VIEW(preview.image_view));
}

static gint
preview_zoom_in (GtkWidget *widget, GdkEventButton *event)
{
  double zoom;

  zoom = eog_scroll_view_get_zoom (EOG_SCROLL_VIEW(preview.image_view));
  eog_scroll_view_set_zoom (EOG_SCROLL_VIEW(preview.image_view),
		            zoom * IMAGE_VIEW_ZOOM_MULTIPLIER);
  return TRUE;
}

static gint
preview_zoom_out (GtkWidget *widget, GdkEventButton *event)
{
  double zoom;

  zoom = eog_scroll_view_get_zoom (EOG_SCROLL_VIEW(preview.image_view));
  eog_scroll_view_set_zoom (EOG_SCROLL_VIEW(preview.image_view),
		            zoom / IMAGE_VIEW_ZOOM_MULTIPLIER);
  return TRUE;
}

static gint
preview_zoom_normal (GtkWidget *widget, GdkEventButton *event)
{
  eog_scroll_view_set_zoom (EOG_SCROLL_VIEW(preview.image_view), 1.0);
  return TRUE;
}

static int sel_x, sel_y;
static int sel2_x, sel2_y;

/*
 * Draw a expanding rectangle on the screen to use as a cursor
 */
static void draw_cursor ( GtkWidget *widget,
                          int    x1,
                          int    y1,
                          int    x2,
                          int    y2)
{
  GdkRectangle update_rect;
  int x, y;
  int width, height;
                                                                                
  update_rect.x = 0;
  update_rect.y = 0;
  update_rect.width = widget->allocation.width;
  update_rect.height = widget->allocation.height;
                                                                                
  if (x2 > x1) {
      width = x2 - x1;
      x = x1;
  }
  else {
      width = x1 - x2;
      x = x2;
  }
  if (y2 > y1) {
      height = y2 - y1;
      y = y1;
  }
  else {
      height = y1 - y2;
      y = y2;
  }
                                                                                
#ifdef DRAWRECT
  gdk_draw_rectangle (pixmap,
                      widget->style->black_gc,
                      FALSE,
                      x, y, width, height);
  gtk_widget_draw (widget, &update_rect);
#else
#if 0
  fprintf (stderr, "set_frame_rectangle %d+%d*%d*%d\n", x, y, width, height);
#endif
  eog_scroll_view_set_frame_rectangle (EOG_SCROLL_VIEW(preview.image_view), x, y, width, height);
#endif
  set_current_selection_size (width*1.0, height*1.0);
}

void
Repaint (GtkWidget *widget )
{
   GdkRectangle  update_rect;
                                                                                
   /* Init3d (); */
                                                                                
   /* --- clear pixmap --- */
#ifdef DRAWRECT
   gdk_draw_rectangle (pixmap,
                       widget->style->white_gc,
                       TRUE,
                       0, 0,
                       widget->allocation.width,
                       widget->allocation.height);
#endif
    /* --- The whole screen --- */
    update_rect.x = 0;
    update_rect.y = 0;
    update_rect.width = widget->allocation.width;
    update_rect.height = widget->allocation.height;
                                                                                
    /* --- Call the expose event - which copies the background
     *     into the widget
     */
    gtk_widget_draw (widget, &update_rect);
}

static gint
preview_button_press (GtkWidget *widget, GdkEventButton *event)
{
  GdkEventButton *event_button;
  event_button = (GdkEventButton *) event;

  if (event->button == 2) {
      double ex = eog_scroll_view_get_pixbuf_x (EOG_SCROLL_VIEW(preview.image_view), event->x);
      double ey = eog_scroll_view_get_pixbuf_y (EOG_SCROLL_VIEW(preview.image_view), event->y);
      int x, y, w, h;

      x = (int)(ex+0.5);
      if (plvals.previewShrink == 1 && (x & 1)) x--;
      y = (int)(ey+0.5);
      if (plvals.previewShrink == 1 && (y & 1)) y--;

      if (event->state & GDK_CONTROL_MASK) {
	  eog_scroll_view_get_frame_rectangle (EOG_SCROLL_VIEW(preview.image_view),
  				  &sel_x, &sel_y, &w, &h);
	  if (2*(x-sel_x) < w)
		sel_x += w;
	  sel2_x = x;
	  if (2*(y-sel_y) < h)
		sel_y += h;
	  sel2_y = y;
	  if (preview.image_view != NULL) {
	      Repaint (widget);
	      draw_cursor (widget, sel_x, sel_y, sel2_x, sel2_y);
	  }
      }
      else {
      	  sel_x = x;
      	  sel_y = y;
      }
      return TRUE;
  }
  else if (event->button == 3) {
      preview_popup_handler(preview.menu, event);
  }
#if 0
  fprintf (stderr, "Preview button press # %d\n", event_button->button);
#endif
  return FALSE;
}

static gint
preview_motion_notify (GtkWidget *widget, GdkEventMotion *event)
{
  int x, y;
  GdkModifierType state;
  double ex, ey;
                                                                                
  if (event->is_hint)
    gdk_window_get_pointer (event->window, &x, &y, &state);
  else
  {
    x = event->x;
    y = event->y;
    state = event->state;
  }

      ex = eog_scroll_view_get_pixbuf_x (EOG_SCROLL_VIEW(preview.image_view), x);
      ey = eog_scroll_view_get_pixbuf_y (EOG_SCROLL_VIEW(preview.image_view), y);
      x = (int)(ex+0.5);
      if (plvals.previewShrink == 1 && (x & 1)) x--;
      y = (int)(ey+0.5);
      if (plvals.previewShrink == 1 && (y & 1)) y--;
                                                                                
  if (state & GDK_BUTTON2_MASK && preview.image_view != NULL)
  {
#ifdef DRAWRECT
    Repaint (widget);
#endif
    draw_cursor (widget, sel_x, sel_y, sel2_x, sel2_y);
  }
  sel2_x = x;
  sel2_y = y;
                                                                                
#if 0
  fprintf (stderr, "Motion notify @ %g,%g\n", event->x, event->y);
#endif
  return TRUE;
}

static void
preview_zoom_fit(GtkWidget *widget, gpointer *data)
{
  eog_scroll_view_zoom_fit (EOG_SCROLL_VIEW(preview.image_view));
}

static gint
preview_popup_handler(GtkWidget *widget, GdkEventButton *event)
{
  GtkMenu *menu;
  GdkEventButton *event_button;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_MENU (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  /* The "widget" is the menu that was supplied when 
   * gtk_signal_connect_object was called.
   */
  menu = GTK_MENU (widget);

  if (event->type == GDK_BUTTON_PRESS)
    {
      event_button = (GdkEventButton *) event;
      if (event_button->button == 3)
	{
	  gtk_menu_popup (menu, NULL, NULL, NULL, NULL, 
			  event_button->button, event_button->time);
	  return TRUE;
	}
    }

  return FALSE;
}

void
preview_new (GtkWidget *parent, UI_RunVals *runVals)
{
  GtkWidget *preview_frame;
  GtkWidget *mi;
  GtkWidget *display;
  int w, h;
  MRI *mri = runVals->mri;
  PersistentVals *loadopt = runVals->pVals;
  int shrink = loadopt->previewShrink;
  w = MRI_GetWidth(mri) / shrink;
  h = MRI_GetHeight(mri) / shrink;

  preview.region.r.x = 0;
  preview.region.r.y = 0;
  preview.region.r.width = MRI_GetWidth (mri);
  preview.region.r.height = MRI_GetHeight (mri);
  preview.previewRegion = preview.region.r;
  preview.mri = mri;
  preview.loadopt = loadopt;

  preview_frame = gtk_frame_new (NULL);
  gtk_widget_show (preview_frame);
  gtk_container_add (GTK_CONTAINER (parent), preview_frame);
  gtk_frame_set_shadow_type (GTK_FRAME(preview_frame), GTK_SHADOW_IN);

  preview.ui_image = NULL;
  preview.image_view = eog_scroll_view_new();

  display = eog_scroll_view_get_display (EOG_SCROLL_VIEW(preview.image_view));
  g_signal_connect (G_OBJECT (display), "button_press_event",
                    G_CALLBACK (preview_button_press), NULL);
  g_signal_connect (G_OBJECT (display), "motion_notify_event",
                    G_CALLBACK (preview_motion_notify), NULL);
  /* eog_scroll_view_set_interp_type (EOG_SCROLL_VIEW(preview.image_view), GDK_INTERP_HYPER); */
  eog_scroll_view_set_antialiasing (EOG_SCROLL_VIEW(preview.image_view), TRUE);
  gtk_widget_set_usize (preview.image_view, previewWidth, previewHeight);
  eog_scroll_view_set_zoom_upscale (EOG_SCROLL_VIEW(preview.image_view), TRUE);

  /* create menu and connect our popup menu handler */
  preview.menu = gtk_menu_new ();
  mi = gtk_menu_item_new_with_label (_("Zoom in"));
  gtk_menu_append (GTK_MENU(preview.menu), mi);
  g_signal_connect_object(G_OBJECT(mi), "activate",
                          G_CALLBACK (preview_zoom_in), G_OBJECT(preview.image_view), 0);
  gtk_widget_show (mi);
  mi = gtk_menu_item_new_with_label (_("Zoom out"));
  gtk_menu_append (GTK_MENU(preview.menu), mi);
  g_signal_connect_object(G_OBJECT(mi), "activate",
      G_CALLBACK (preview_zoom_out), G_OBJECT(preview.image_view), 0);
  gtk_widget_show (mi);
  mi = gtk_menu_item_new_with_label (_("Best fit"));
  gtk_menu_append (GTK_MENU(preview.menu), mi);
  g_signal_connect_object(G_OBJECT(mi), "activate",
      G_CALLBACK (preview_zoom_fit), G_OBJECT(preview.image_view), 0);
  gtk_widget_show (mi);
  mi = gtk_menu_item_new_with_label (_("Normal size"));
  gtk_menu_append (GTK_MENU(preview.menu), mi);
  g_signal_connect_object(G_OBJECT(mi), "activate",
      G_CALLBACK (preview_zoom_normal), G_OBJECT(preview.image_view), 0);
  gtk_widget_show (mi);

  mi = gtk_menu_item_new_with_label (_("Crop"));
  gtk_menu_append (GTK_MENU(preview.menu), mi);
  gtk_signal_connect(GTK_OBJECT(mi), "activate",
      GTK_SIGNAL_FUNC (preview_set_crop), runVals);
  gtk_widget_show (mi);

  mi = gtk_menu_item_new_with_label (_("Pop Crop"));
  gtk_menu_append (GTK_MENU(preview.menu), mi);
  gtk_signal_connect(GTK_OBJECT(mi), "activate",
      GTK_SIGNAL_FUNC (preview_pop_crop), runVals);
  gtk_widget_show (mi);

  mi = gtk_menu_item_new_with_label (_("Set Gray World Region"));
  gtk_menu_append (GTK_MENU(preview.menu), mi);
  g_signal_connect_object(G_OBJECT(mi), "activate",
      G_CALLBACK (preview_set_gray_world_region), G_OBJECT(preview.image_view), 0);
  gtk_widget_show (mi);

  mi = gtk_menu_item_new_with_label (_("Clear Gray World Region"));
  gtk_menu_append (GTK_MENU(preview.menu), mi);
  g_signal_connect_object(G_OBJECT(mi), "activate",
      G_CALLBACK (preview_clear_gray_world_region), G_OBJECT(preview.image_view), 0);
  gtk_widget_show (mi);

  gtk_container_add (GTK_CONTAINER (preview_frame), preview.image_view);
  gtk_widget_show (preview.image_view);
}

struct GimpPreviewerData {
  int	        width, height;
  int		xoff, yoff;
  int	        freedata;
  guchar       *data;
  GdkPixbuf	*gray_pixbuf;
  gint          bpp;
  gint          scan_lines, total_scan_lines;
  gint		y;
};

void
GimpPreviewerStart (void *private, int width, int height, int freedata)
{
  struct GimpPreviewerData *wd = private;
  int i;
  guchar *dest;

  wd->width = width;
  wd->height = height;
  wd->xoff = 0;
  wd->yoff = 0;
  wd->freedata = freedata;
  wd->data = g_malloc (width * wd->bpp);
  wd->y = wd->yoff;
  dest = wd->data;
  for (i = 0; i < width; i++) {
	*dest++ = 0x80;
	*dest++ = 0x80;
	*dest++ = 0x80;
  }
  if (preview.pixbuf == (GdkPixbuf *)0 ||
      preview.width != width || preview.height != height)
  {
      preview.width = width;
      preview.height = height;
      preview.pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, width, height);
      eog_scroll_view_set_frame_rectangle (EOG_SCROLL_VIEW(preview.image_view), 0, 0, 0, 0);
      preview.ui_image = eog_image_new_pixbuf (preview.pixbuf);
      eog_scroll_view_set_image (EOG_SCROLL_VIEW(preview.image_view), EOG_IMAGE(preview.ui_image));
      eog_scroll_view_zoom_fit (EOG_SCROLL_VIEW(preview.image_view));
  }
  wd->gray_pixbuf = gdk_pixbuf_new_from_data (wd->data,
  		GDK_COLORSPACE_RGB, FALSE, 8, width, 1, 0, NULL, NULL);
  gtk_progress_set_value(GTK_PROGRESS(preview.progressBar), 0.0);
}

void
GimpPreviewerRow (void *private, void *data)
{
  struct GimpPreviewerData *wd = private;
  struct MRI_ScanLine *sl = data;
  int i;
  guchar *dest = wd->data + 3 * wd->xoff;

  if (sl->scanLineType != LINETYPE_SHORT) {
	fprintf (stderr, "Error: GimpPreviewerRow: unexpected line type %d\n", sl->scanLineType);
	_exit (1);
  }
  else {
	  unsigned short *R = (unsigned short *)sl->R;
	  unsigned short *G = (unsigned short *)sl->G;
	  unsigned short *B = (unsigned short *)sl->B;

	  for (i = 0; i < wd->width; i++) {
		*dest++ = R[i] >> 8;
		*dest++ = G[i] >> 8;
		*dest++ = B[i] >> 8;
	  }
  }
  gdk_pixbuf_copy_area (wd->gray_pixbuf, 0, 0, wd->width, 1,
        		preview.pixbuf, 0, wd->y++ - wd->yoff);
  gtk_progress_set_value(GTK_PROGRESS (preview.progressBar), (100.0*wd->y)/wd->height);
  gdk_flush ();
}

void
GimpPreviewerClose (void *private)
{
  struct GimpPreviewerData *wd = private;

  gtk_widget_queue_draw (GTK_WIDGET (preview.image_view));
  gtk_progress_set_value(GTK_PROGRESS (preview.progressBar), 100.0);
  gdk_flush ();
  gdk_pixbuf_unref (wd->gray_pixbuf);
  g_free (wd->data);
}

struct link *
GenGimpPreviewer ()
{
  struct GimpPreviewerData *wd = malloc(sizeof (struct GimpPreviewerData));
  struct link *ep = malloc (sizeof (*ep));
  if (wd == (struct GimpPreviewerData *)0 || ep == (struct link*)0) {
	fprintf (stderr, "Error: out of memory\n");
	exit (1);
  }
  ep->start = GimpPreviewerStart;
  ep->row = GimpPreviewerRow;
  ep->close = GimpPreviewerClose;
  ep->private = wd;

  wd->bpp = 3;

  return ep;
}

static GStaticMutex mutex = G_STATIC_MUTEX_INIT;

int
PreviewIdleTask (gpointer grunvals)
{
  int retval;
  UI_RunVals *runVals = (UI_RunVals *)grunvals;

  g_static_mutex_lock (&mutex);
  retval = MRI_DoProcessIterator (preview.pip);
  if (!retval) {
    MRI_FreeProcessIterator (preview.pip);
    preview.previewRunning = 0;
    if (preview.tracefd) { fclose (preview.tracefd); preview.tracefd = NULL; }
    do_histogram(preview.hvals);
    g_free(preview.hvals);
    do_luminance();
    if (runVals->selection.width > 0) {
	GtkWidget *display;
	MRI_Region crect, prect;

	clip_to_preview_region (&runVals->selection, &crect);
	image_rect_to_preview_rect (&crect, &prect);
        sel_x = prect.x;
        sel_y = prect.y;
        sel2_x = sel_x + prect.width;
        sel2_y = sel_y + prect.height;
        display = eog_scroll_view_get_display (EOG_SCROLL_VIEW(preview.image_view));
	draw_cursor (display, sel_x, sel_y, sel2_x, sel2_y);
    }
    gtk_idle_remove (preview.previewIdleTag);
    gdk_flush ();
  }
  g_static_mutex_unlock (&mutex);
  return retval;
}

void
do_preview (const UI_RunVals *runVals)
{
  struct link *previewer;
  int	w, h;
  int	minx, miny;
  PersistentVals *loadopt;

  loadopt = runVals->pVals;
  g_static_mutex_lock (&mutex);
  if (preview.previewRunning) {
  	preview.previewRunning = 0;
  	gtk_idle_remove (preview.previewIdleTag);
        if (preview.tracefd) { fclose (preview.tracefd); preview.tracefd = NULL; }
  }
  preview.region.shrink = loadopt->previewShrink;
  w = MRI_GetWidth(preview.mri) / preview.region.shrink;
  h = MRI_GetHeight(preview.mri) / preview.region.shrink;
  minx = (MRI_GetWidth(preview.mri) - w * preview.region.shrink) / 2;
  miny = (MRI_GetHeight(preview.mri) - h * preview.region.shrink) / 2;
  if (minx & 1) minx--;
  if (miny & 1) miny--;
  preview.region.r.x = minx;
  preview.region.r.y = miny;
  preview.region.r.width = w * preview.region.shrink;
  preview.region.r.height = h * preview.region.shrink;
  /* Copy processing options, overriding finalShrink with previewShrink. */
  preview.vals = *loadopt;
  preview.vals.finalShrink = loadopt->previewShrink;

  previewer = GenGimpPreviewer ();
  preview.hvals = (int *)g_malloc(sizeof(int) * preview.vals.histWidth);
  preview.previewRunning = 1;
#if 0
  { static int tracenum = 0;
    char buffer[256];
    snprintf (buffer, sizeof(buffer), "/tmp/gimp-trace-%d.ppm", ++tracenum);
    preview.tracefd = fopen (buffer, "w");
  }
#else
  preview.tracefd = NULL;
#endif
  preview.Renderer = MakeRenderRaw (preview.mri, previewer, preview.vals.histWidth, preview.hvals, preview.region.shrink, &preview.vals, preview.tracefd);
  preview.pip = MRI_NewProcessIterator (preview.Renderer, preview.mri, preview.vals.rotate, &preview.previewRegion);
  preview.previewIdleTag = gtk_idle_add (PreviewIdleTask, (gpointer)runVals);
  g_static_mutex_unlock (&mutex);
}

extern void rotate_clockwise (GtkWidget *widget, gpointer data);
extern void rotate_anticlockwise (GtkWidget *widget, gpointer data);
extern void rotate_180 (GtkWidget *widget, gpointer data);

GtkWidget *
CreatePreviewPane (UI_RunVals *runVals)
{
  GtkWidget *preview_frame;
  GtkWidget *preview_vbox, *preview_hbox;
  GtkWidget *zoom_in, *zoom_out, *zoom_fit, *zoom_normal;
  GtkWidget *rotate_b90, *rotate_b180, *rotate_b270;
  MRI *mri = runVals->mri;

  preview.pixbuf = (GdkPixbuf *)0;
  preview_frame = gtk_frame_new (_("Preview"));
  gtk_container_set_border_width (GTK_CONTAINER (preview_frame), 6);

  preview_vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (preview_vbox), 6);
  gtk_container_add (GTK_CONTAINER (preview_frame), preview_vbox);
  preview_new (preview_vbox, runVals);
  gtk_widget_show (preview_vbox);

  preview.progressBar = gtk_progress_bar_new();
  gtk_progress_bar_set_orientation (GTK_PROGRESS_BAR(preview.progressBar), GTK_PROGRESS_LEFT_TO_RIGHT);
  gtk_progress_bar_set_bar_style (GTK_PROGRESS_BAR(preview.progressBar), GTK_PROGRESS_CONTINUOUS);
  gtk_progress_set_format_string(GTK_PROGRESS(preview.progressBar), "%p%%");
  gtk_progress_set_show_text(GTK_PROGRESS(preview.progressBar), TRUE);
  gtk_progress_set_value(GTK_PROGRESS(preview.progressBar), 0.0);
  gtk_widget_show (preview.progressBar);
  gtk_box_pack_start (GTK_BOX (preview_vbox), preview.progressBar, FALSE, FALSE, 0);

  preview_hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (preview_vbox), preview_hbox, FALSE, FALSE, 0);

  zoom_in = gtk_button_new_with_label (_("Zoom In"));
  gtk_box_pack_start (GTK_BOX (preview_hbox), zoom_in, FALSE, FALSE, 0);
  gtk_signal_connect_object (GTK_OBJECT (zoom_in), "clicked",
                             GTK_SIGNAL_FUNC (preview_zoom_in), NULL);
  gtk_widget_show (zoom_in);

  zoom_out = gtk_button_new_with_label (_("Zoom Out"));
  gtk_box_pack_start (GTK_BOX (preview_hbox), zoom_out, FALSE, FALSE, 0);
  gtk_signal_connect_object (GTK_OBJECT (zoom_out), "clicked",
                             GTK_SIGNAL_FUNC (preview_zoom_out), NULL);
  gtk_widget_show (zoom_out);

  zoom_fit = gtk_button_new_with_label (_("Best fit"));
  gtk_box_pack_start (GTK_BOX (preview_hbox), zoom_fit, FALSE, FALSE, 0);
  gtk_signal_connect_object (GTK_OBJECT (zoom_fit), "clicked",
                             GTK_SIGNAL_FUNC (preview_zoom_fit), NULL);
  gtk_widget_show (zoom_fit);

  zoom_normal = gtk_button_new_with_label (_("Normal Size"));
  gtk_box_pack_start (GTK_BOX (preview_hbox), zoom_normal, FALSE, FALSE, 0);
  gtk_signal_connect_object (GTK_OBJECT (zoom_normal), "clicked",
                             GTK_SIGNAL_FUNC (preview_zoom_normal), NULL);
  gtk_widget_show (zoom_normal);

  rotate_b270 = gtk_button_new_with_label (_("Rotate Anticlockwise"));
  gtk_box_pack_start (GTK_BOX (preview_hbox), rotate_b270, FALSE, FALSE, 0);
  gtk_signal_connect_object (GTK_OBJECT (rotate_b270), "clicked",
                             GTK_SIGNAL_FUNC (rotate_anticlockwise), NULL);
  gtk_widget_show (rotate_b270);

  rotate_b180 = gtk_button_new_with_label (_("Rotate 180"));
  gtk_box_pack_start (GTK_BOX (preview_hbox), rotate_b180, FALSE, FALSE, 0);
  gtk_signal_connect_object (GTK_OBJECT (rotate_b180), "clicked",
                             GTK_SIGNAL_FUNC (rotate_180), NULL);
  gtk_widget_show (rotate_b180);

  rotate_b90 = gtk_button_new_with_label (_("Rotate Clockwise"));
  gtk_box_pack_start (GTK_BOX (preview_hbox), rotate_b90, FALSE, FALSE, 0);
  gtk_signal_connect_object (GTK_OBJECT (rotate_b90), "clicked",
                             GTK_SIGNAL_FUNC (rotate_clockwise), NULL);
  gtk_widget_show (rotate_b90);

  gtk_widget_show (preview_hbox);
  gtk_widget_show (preview_vbox);
  gtk_widget_show (preview_frame);

  return preview_frame;
}

struct {
  char *baseName;
  char *fullName;
  GdkInterpType type;
} preview_types[] = {
   { N_("Nearest"),  NULL, GDK_INTERP_NEAREST },
   { N_("Tiled"),    NULL, GDK_INTERP_TILES },
   { N_("Bilinear"), NULL, GDK_INTERP_BILINEAR },
   { N_("Hyper"),    NULL, GDK_INTERP_HYPER },
};

static int
set_preview_interpolation_method (GtkWidget *widget, gpointer data)
{
  if (preview.image_view)
    /* eog_scroll_view_set_interp_type (EOG_SCROLL_VIEW(preview.image_view), (GdkInterpType)data); */
    eog_scroll_view_set_antialiasing (EOG_SCROLL_VIEW(preview.image_view), (GdkInterpType)data != GDK_INTERP_NEAREST);
  return TRUE;
}

GtkWidget *
GenPreviewInterpolationMenu ()
{
  GtkWidget *optmenu, *pimethod_menu;
  GtkWidget *mi;
  int	i, active;

  optmenu = gtk_option_menu_new ();
  pimethod_menu = gtk_menu_new ();

  active = 3;
  for (i = 0; i < sizeof(preview_types)/sizeof(preview_types[0]); i++) {
    preview_types[i].fullName = _(preview_types[i].baseName);
    mi = gtk_menu_item_new_with_label (preview_types[i].fullName);
    gtk_widget_show (mi);
    gtk_menu_shell_append (GTK_MENU_SHELL(pimethod_menu), mi);
    gtk_signal_connect(GTK_OBJECT(mi), "activate",
      GTK_SIGNAL_FUNC (set_preview_interpolation_method),
      (gpointer)preview_types[i].type);
  }
  gtk_widget_show (pimethod_menu);
  gtk_option_menu_set_menu (GTK_OPTION_MENU (optmenu), pimethod_menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (optmenu), active);
  gtk_widget_show (optmenu);

  return optmenu;
}
