#ifndef GAMECONTROLBASE_H
#define GAMECONTROLBASE_H

#include <list>

#include "Tools.h"
#include "SDLTools.h"
#include "ObjectFwd.h"
#include "XMLElements.h"
#include "XMLVisitors.h"


//----------------------------------------------------------------------------
#define GAME_STATE_OBJECT(C) \
    STATE_OBJECT(C); \
  protected: \
    C() : GameStateBase(#C) {} \
  public: \
    ~C() {}

//----------------------------------------------------------------------------
class GameControlBase
{
    friend class GameControlTest;

  public:
    //------------------------------------------------------------------------
    enum ShipControl
    {
        ROT_LEFT,
        ROT_RIGHT,
        ROT_OFF,
        THRUST_ON,
        THRUST_OFF,
        ALIGN_ON,
        ALIGN_OFF,
        FIRE
    };

  protected:

    //------------------------------------------------------------------------
    class GameStateBase;
    friend class GameStateBase;
    class GameStateBase
    {
      protected:
        GameStateBase(const char *name) : m_name(name) {}
      public:
        virtual ~GameStateBase() { m_name = NULL; }
        
        //--------------------------------------------------------------------
        inline const char *getName() const { return m_name; }

        //--------------------------------------------------------------------
        virtual void updateShipControl(ShipControl control,
                                       GameControlBase *game);
        virtual void updateBonus(GameControlBase *game);

        virtual void onPreUpdate(GameControlBase *game) {}
        virtual void onPostUpdate(GameControlBase *game) {}
        virtual void onShipLanded(Ship *ship, const Platform *platform,
                                  GameControlBase *game) {}
        virtual void onShipTakeoff(const Ship *ship, GameControlBase *game) {}
        virtual void onShipFiring(const Ship *ship, GameControlBase *game) {}
        virtual void onShipOutOfFuel(const Ship *ship, GameControlBase *game);
        virtual void onObjectDestroyed(const ObjectBase *object,
                                       GameControlBase *game);
        virtual void onPlayerShipDestroyed(Ship *ship, GameControlBase *game);
        virtual bool isFinished(const GameControlBase *game) const
        {
            return false;
        }

      private:
        //--------------------------------------------------------------------
        const char *m_name;
    };

    //------------------------------------------------------------------------
    class Running;
    friend class Running;
    class Running : public GameStateBase
    {
        GAME_STATE_OBJECT(Running)

        void onPreUpdate(GameControlBase *game);
        void onShipLanded(Ship *ship, const Platform *platform,
                          GameControlBase *game);
    };

    //------------------------------------------------------------------------
    class UnloadingCrate;
    friend class UnloadingCrate;
    class UnloadingCrate : public GameStateBase
    {
        GAME_STATE_OBJECT(UnloadingCrate)

        void updateShipControl(ShipControl control, GameControlBase *game) {}
        void onPreUpdate(GameControlBase *game);

        void onPlayerShipDestroyed(Ship *ship, GameControlBase *game);
    };
    
    //------------------------------------------------------------------------
    class UnloadingDelay;
    friend class UnloadingDelay;
    class UnloadingDelay : public GameStateBase
    {
        GAME_STATE_OBJECT(UnloadingDelay)

        void updateShipControl(ShipControl control, GameControlBase *game) {}
        void onPreUpdate(GameControlBase *game);
    };

    //------------------------------------------------------------------------
    class OutOfFuelDelay;
    friend class OutOfFuelDelay;
    class OutOfFuelDelay : public GameStateBase
    {
        GAME_STATE_OBJECT(OutOfFuelDelay)

        void onPreUpdate(GameControlBase *game);
        void onShipOutOfFuel(const Ship *ship, GameControlBase *game) {}
    };

    //------------------------------------------------------------------------
    class ExplosionDelay;
    friend class ExplosionDelay;
    class ExplosionDelay : public GameStateBase
    {
        GAME_STATE_OBJECT(ExplosionDelay)

        void updateShipControl(ShipControl control, GameControlBase *game) {}
        void onPreUpdate(GameControlBase *game);
    };

    //------------------------------------------------------------------------
    class AddFuelToScore;
    friend class AddFuelToScore;
    class AddFuelToScore : public GameStateBase
    {
        GAME_STATE_OBJECT(AddFuelToScore)

        void updateShipControl(ShipControl control, GameControlBase *game) {}
        void updateBonus(GameControlBase *game) {}
        void onPreUpdate(GameControlBase *game);
    };

    //------------------------------------------------------------------------
    class AddBonusToScore;
    friend class AddBonusToScore;
    class AddBonusToScore : public GameStateBase
    {
        GAME_STATE_OBJECT(AddBonusToScore)

        void updateShipControl(ShipControl control, GameControlBase *game) {}
        void updateBonus(GameControlBase *game) {}
        void onPreUpdate(GameControlBase *game);
    };

    //------------------------------------------------------------------------
    class Final;
    friend class Final;
    class Final : public GameStateBase
    {
        GAME_STATE_OBJECT(Final)

        void updateShipControl(ShipControl control, GameControlBase *game) {}
        void updateBonus(GameControlBase *game) {}
        bool isFinished(const GameControlBase *game) const { return true; }
    };


    //------------------------------------------------------------------------
    class ConvertNumberRangesVisitor : public XMLVisitor
    {
      public:
        //--------------------------------------------------------------------
        ConvertNumberRangesVisitor() {}
        ~ConvertNumberRangesVisitor() {}

      private:
        //--------------------------------------------------------------------
        void do_visit(XMLNode *n);

        //--------------------------------------------------------------------
        void convertProperty(XMLNode *n, const char *name);
    };

    //------------------------------------------------------------------------
    class RemoveDestroyObjectiveVisitor : public XMLVisitor
    {
      public:
        //--------------------------------------------------------------------
        RemoveDestroyObjectiveVisitor(const unsigned id) : m_id(id) {}
        ~RemoveDestroyObjectiveVisitor() {}

      private:
        //--------------------------------------------------------------------
        void do_visit(XMLNode *n);

        //--------------------------------------------------------------------
        const unsigned m_id;
    };

    //------------------------------------------------------------------------
    class RemoveReachedObjectivesVisitor : public XMLVisitor
    {
        typedef std::list<XMLNode*> XMLNodeList;

      public:
        //--------------------------------------------------------------------
        RemoveReachedObjectivesVisitor(GameControlBase *gc) : m_gc(gc) {}
        ~RemoveReachedObjectivesVisitor() { m_gc = NULL; }

        //--------------------------------------------------------------------
        void postVisit();

      private:
        //--------------------------------------------------------------------
        void do_visit(XMLNode *n);

        //--------------------------------------------------------------------
        GameControlBase *m_gc;
        XMLNodeList m_list;
    };


  public:

    ~GameControlBase();

    /// (Re)initialize sm_instance from the given XML file.
    static void init(const char *mission, const char *level);

    /// Destroy the sm_instance GameControl.
    static void destroy();

    static inline GameControlBase *getInstance()
    {
        return sm_instance;
    }

    //------------------------------------------------------------------------
    /// Initializes the bonus settings from the given XML node.
    void initBonus(const XMLNode *bonusNode);

    /// Initializes the game control from the given XML node.
    void initStartPosition(const XMLNode *startPositionNode);


    //------------------------------------------------------------------------
    Ship *getPlayerShip()
    {
        return m_player;
    }

    //------------------------------------------------------------------------
    const Ship *getPlayerShip() const
    {
        return m_player;
    }

    //------------------------------------------------------------------------
    inline unsigned getPlayerShipId()
    {
        return m_playerId;
    }

    //------------------------------------------------------------------------
    inline unsigned getPlayerFuel() const
    {
        return m_playerFuel;
    }

    //------------------------------------------------------------------------
    void createPlayerShip();
    void resetPlayerShip();

    //------------------------------------------------------------------------
    /// Initiates the unloading delay of the player's crates.
    void initUnloadingDelay();

    /// Initiates the out-of-fuel handling.
    void initOutOfFuel();

    /// Initiates the explosion of the player's ship.
    void initExplosion();

    /// Initiates the fuel-to-score adding delay.
    void initAddFuelToScore();

    /// Initiates the bonus-to-score adding.
    void initAddBonusToScore();

    inline unsigned getPlayerFuelDecrement() const
    {
        return m_playerFuelDecrement;
    }

    //------------------------------------------------------------------------
    inline void resetFrameCounter()
    {
        m_frameCounter = 0;
    }

    inline void setFrameCounterLimit(unsigned limit)
    {
        m_frameCounterLimit = limit;
    }

    inline bool hasFrameCounterReachedLimit()
    {
        return (m_frameCounter++ == m_frameCounterLimit);
    }

    //------------------------------------------------------------------------
    inline const XMLNode *getDescriptionNode() const
    {
        return m_descriptionNode;
    }

    //------------------------------------------------------------------------
    inline const XMLNode *getGameControlNode() const
    {
        return m_gameControlNode;
    }

    /**
     * Searches for a <destroy> objective with the given object id
     * contained in the id propertie's token string.
     * If found, this object id is removed from the id property.
     */
    void removeDestroyObjective(unsigned id);

    /**
     * If the first objective is a <land> objective with the home platform
     * as first platform entry, the home platform id is removed.
     */
    void removeLandOnHomePlatformObjective();

    /**
     * This method usually is called before hasObjectivesLeft()
     * to remove all reached objective nodes from m_gameControlNode.
     */
    void removeReachedObjectives();

    /**
     * @return true, if there are open objectives, else false.
     */
    inline bool hasObjectivesLeft() const
    {
        return m_gameControlNode->hasNodes();
    }

    /**
     * Checks whether there are open objectives
     * and switches either to the Running or to the AddFuelToScore state.
     */
    void handleGameContinuation();

    //------------------------------------------------------------------------
    void unloadCrate();

    //------------------------------------------------------------------------
    void update();

    //------------------------------------------------------------------------
    inline void setState(GameStateBase *newState)
    {
        m_state = newState;
    }

    inline const char *getNameOfCurrentState() const
    {
        return m_state->getName();
    }

    //------------------------------------------------------------------------
    inline void updateShipControl(ShipControl control)
    {
        m_state->updateShipControl(control, this);
    }

    inline void updateBonus()
    {
        m_state->updateBonus(this);
    }

    //------------------------------------------------------------------------
    inline void onPreUpdate()
    {
        m_state->onPreUpdate(this);
    }

    inline void onPostUpdate()
    {
        m_state->onPostUpdate(this);
    }

    inline void onShipLanded(Ship *ship, const Platform *platform)
    {
        m_state->onShipLanded(ship, platform, this);
    }

    inline void onShipTakeoff(const Ship *ship)
    {
        m_state->onShipTakeoff(ship, this);
    }

    inline void onShipFiring(const Ship *ship)
    {
        m_state->onShipFiring(ship, this);
    }

    inline void onShipOutOfFuel(const Ship *ship)
    {
        m_state->onShipOutOfFuel(ship, this);
    }

    inline void onObjectDestroyed(const ObjectBase *object)
    {
        m_state->onObjectDestroyed(object, this);
    }

    inline void onPlayerShipDestroyed()
    {
        m_state->onPlayerShipDestroyed(m_player, this);
    }

    /**
     * @return true, if the current level is finished (including the
     *         AddFuelToScore handling) and the next level can be started.
     */
    inline bool isFinished() const
    {
        return m_state->isFinished(this);
    }

    //------------------------------------------------------------------------
    /// Called for a Ship-PlayGround collision.
    void onCollision(Ship *ship);

    void onCollision(Ship *ship, Grenade *grenade);
    void onCollision(Ship *ship, Missile *missile);
    void onCollision(Ship *ship, MortarBase *mortar);
    void onCollision(Ship *ship, ProjectileBase *projectile);
    void onCollision(Ship *ship, SAMBatteryBase *sam);
    void onCollision(Ship *ship, Tank *tank);
    void onCollision(Ship *ship, TurretBase *turret);

    /// Called for a Missile-PlayGround collision.
    void onCollision(Missile *missile);

    void onCollision(Missile *missile, Grenade *grenade);
    void onCollision(Missile *missile, MortarBase *mortar);
    void onCollision(Missile *missile, ProjectileBase *projectile);
    void onCollision(Missile *missile, SAMBatteryBase *sam);
    void onCollision(Missile *missile, SwitchBase *s);
    void onCollision(Missile *missile, Tank *tank);
    void onCollision(Missile *missile, TurretBase *turret);

    /// Called for a Grenade-PlayGround collision.
    void onCollision(Grenade *grenade);

    void onCollision(Grenade *grenade, MortarBase *mortar);
    void onCollision(Grenade *grenade, SwitchBase *s);
    void onCollision(Grenade *grenade, Tank *tank);
    void onCollision(Grenade *grenade, TurretBase *turret);

    /// Called for a ProjectileBase-PlayGround collision.
    void onCollision(ProjectileBase *projectile);

    void onCollision(ProjectileBase *projectile, MortarBase *mortar);
    void onCollision(ProjectileBase *projectile, SwitchBase *s);
    void onCollision(ProjectileBase *projectile, Tank *tank);
    void onCollision(ProjectileBase *projectile, TurretBase *turret);


    //------------------------------------------------------------------------
    /**
     * Called by different code parts
     * to handle a particular event of an objective node.
     */
    void handleEvents(const XMLNode *node);

  private:

    //------------------------------------------------------------------------
    GameControlBase();

    //------------------------------------------------------------------------
    void do_updateShipControl(ShipControl control);
    void do_updateBonus();

    //------------------------------------------------------------------------
    void do_explode(Grenade *grenade);
    void do_explode(Missile *missile);
    void do_explode(MortarBase *mortar);
    void do_explode(ProjectileBase *projectile);
    void do_explode(SAMBatteryBase *sam);
    void do_explode(Ship *ship);
    void do_explode(Tank *tank);
    void do_explode(TurretBase *turret);


    //------------------------------------------------------------------------
    /// The player's ship. It will be set to NULL, if it explodes.
    Ship *m_player;

    /// The player's ship id. It remains constant during a level.
    unsigned m_playerId;

    /// The fuel for the player's ship.
    unsigned m_playerFuel;

    /// The fuel decrement value for each Ship::decFuel() call.
    unsigned m_playerFuelDecrement;


    /// The ship's initial x position.
    Sint16 m_initialXPosition;
    
    /// The ship's initial y position.
    Sint16 m_initialYPosition;

    /// The object id of the home platform. If 0, we have a "flying" start.
    unsigned m_homePlatformId;


    /// The content of the level's description node.
    XMLNode *m_descriptionNode;

    /// The content of the level's gamecontrol node.
    XMLNode *m_gameControlNode;

    /// The current crate to unload.
    Crate *m_crateToUnload;

    /// The current platform, where crates are unloaded.
    const Platform *m_platformToUnload;

    unsigned m_bonusFrameCounter;
    unsigned m_bonusFrameCounterLimit;

    /// The bonus decrement per second.
    unsigned m_bonusDecrement;

    /// Used for timeout-specific handling (e.g. ExplosionDelay).
    unsigned m_frameCounter;
    unsigned m_frameCounterLimit;

    GameStateBase *m_state;

    //------------------------------------------------------------------------
    /// The current active GameControl instance.
    static GameControlBase *sm_instance;
};

#endif //GAMECONTROLBASE_H
