/* ======================================================================
 * mesh.cc
 * 
 * This file is part of MeshIO, the general and extensible 3D mesh I/O
 * library.
 * Copyright (c) 1999, 2000 Niklas Elmqvist. All rights reserved.
 *
 * File created 1999-06-05 by Niklas Elmqvist <d97elm@dtek.chalmers.se>.
 * 
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 *
 * ======================================================================
 */

// -- System Includes
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// -- Local Includes
#include "meshio.hh"
#include "vector3d.hh"

using namespace MeshIO;

// -- Code Segment

void MeshIO::mioMeshInit(Mesh *mesh)
{
    // Clear array pointers
    mesh->vertexList  = NULL;
    mesh->normalList  = NULL;
    mesh->mappingList = NULL;
    mesh->faceList    = NULL;
    
    // Clear counters
    mesh->vertexNum  = 0;
    mesh->faceNum    = 0;
    mesh->mappingNum = 0;
}

void MeshIO::mioMeshComputeNormals(Mesh *mesh)
{
    Vertex3D *vList = mesh->vertexList;
    Vector3D vSum;
    int i, vtx;

    // Sanity check
    if (mesh->vertexNum == 0 || mesh->faceNum == 0)
        return;
    
    // Allocate space for temporary vertex normals
    Vector3D *faceNormals = new Vector3D[mesh->faceNum];
    
    // First, traverse all faces and compute normals for them
    for (i = 0; i < mesh->faceNum; i++) {
	
	// Retrieve the vertex indices
	Index ndxVertex = mesh->faceList[i].ndxVertex;
	
	// Compute plane vectors
	Vector3D vVec1 =
	    Vector3D(vList[ ndxVertex.ndxs[2] ].coords) -
	    Vector3D(vList[ ndxVertex.ndxs[0] ].coords);
	Vector3D vVec2 =
	    Vector3D(vList[ ndxVertex.ndxs[1] ].coords) -
	    Vector3D(vList[ ndxVertex.ndxs[0] ].coords);
	
	// Compute plane normal
	faceNormals[i] = vVec2 * vVec1;
	
	// Normalize it
	faceNormals[i].Normalize();
    }
    
    // Now, compute the individual vertex normals
    for (vtx = 0; vtx < mesh->vertexNum; vtx++) {
	
	// Clear summing vector
	vSum.Clear();
	
	// Step through all faces looking for the current vertex
	for (int nF = 0; nF < mesh->faceNum; nF++) {
	    
	    Index ndxVertex = mesh->faceList[nF].ndxVertex;
	    
	    // Is the vertex referenced in this array?
	    if (ndxVertex.ndxs[0] != vtx &&
		ndxVertex.ndxs[1] != vtx &&
		ndxVertex.ndxs[2] != vtx)
		continue;
	    
	    // Yes, it is, add it to the normal
	    vSum += faceNormals[nF];
	}
	
	// Normalize summing vector
	vSum.Normalize();
	
	// Assign the vector to the model
	memcpy(mesh->normalList[vtx].coords, vSum.fCoord, 3 * sizeof(float));
    }
    
    // Delete temporary normal storage
    delete[] faceNormals;
}

void MeshIO::mioMeshClear(Mesh *mesh)
{
    // Delete arrays (if they have been allocated)
    if (mesh->vertexList != NULL) { 
	free(mesh->vertexList); 
	mesh->vertexList = NULL; 
    }
    if (mesh->faceList != NULL) {
	free(mesh->faceList);
	mesh->faceList = NULL;
    }
    if (mesh->mappingList != NULL) {
	free(mesh->mappingList);
	mesh->mappingList = NULL;
    }
    if (mesh->normalList != NULL) {
	free(mesh->normalList);
	mesh->normalList = NULL;
    }
    
    // Update variables
    mesh->faceNum = 0;
    mesh->vertexNum = 0; 
    mesh->mappingNum = 0;
}

Mesh *MeshIO::mioMeshCopy(Mesh *mesh)
{
    // Allocate size for new mesh
    Mesh *dst = (Mesh *) malloc(sizeof(Mesh));
    mioMeshInit(dst);
    
    // Allocate and copy arrays
    if (mesh->vertexList != NULL) {
	dst->vertexList = (Vertex3D *) 
	    malloc(sizeof(Vertex3D) * mesh->vertexNum);
	memcpy(dst->vertexList, mesh->vertexList,
	       sizeof(Vertex3D) * mesh->vertexNum);
	dst->vertexNum = mesh->vertexNum;
    }
    if (mesh->normalList != NULL) {
	dst->normalList = (Vertex3D *) 
	    malloc(sizeof(Vertex3D) * mesh->vertexNum);
	memcpy(dst->normalList, mesh->vertexList,
	       sizeof(Vertex3D) * mesh->vertexNum);
    }
    if (mesh->faceList != NULL) {
	dst->faceList = (Face *) malloc(sizeof(Face) * mesh->faceNum);
	memcpy(dst->faceList, mesh->faceList, sizeof(Face) * mesh->faceNum);
	dst->faceNum = mesh->faceNum;
    }
    if (mesh->mappingList != NULL) {
	dst->mappingList = (Vertex2D *) malloc(sizeof(Vertex2D) *
					       mesh->mappingNum);
	memcpy(dst->mappingList, mesh->mappingList, 
	       sizeof(Vertex2D) * mesh->mappingNum);
	dst->mappingNum = mesh->mappingNum; 
    }

    // Return new mesh
    return dst;
}

void MeshIO::mioMeshDump(Mesh *mesh, bool verbose)
{
    int i;

    // General statistics
    printf("\tMesh vertex number: %d.\n", mesh->vertexNum);
    printf("\tMesh face number: %d.\n", mesh->faceNum);
    printf("\tMesh mapping number: %d.\n", mesh->mappingNum);
        
    // Specific statistics (only in verbose mode)
    if (verbose == true) {
	
	printf("\tMesh vertices:\n");
	
	for (i = 0; i < mesh->vertexNum; i++) {
	    
	    printf("\tx: %0.2f, y: %0.2f, z: %0.2f.\n",
		   mesh->vertexList[i].coord.x, 
		   mesh->vertexList[i].coord.y, 
		   mesh->vertexList[i].coord.z);
	}

	printf("\tMesh faces:\n");
	
	for (i = 0; i < mesh->faceNum; i++) {
	    printf("\ta: %d, b: %d, c: %d, material: %d.\n",
		   mesh->faceList[i].ndxVertex.ndx.a,
		   mesh->faceList[i].ndxVertex.ndx.b,
		   mesh->faceList[i].ndxVertex.ndx.c,
		   mesh->faceList[i].material);
	}
    }
}

void MeshIO::mioMeshScale(Mesh *mesh, float xScale, float yScale, float zScale)
{
    // Step through all vertices and scale them
    for (int i = 0; i < mesh->vertexNum; i++) {

        // Scale on the three axes
        mesh->vertexList[i].coord.x *= xScale;
        mesh->vertexList[i].coord.y *= yScale;
        mesh->vertexList[i].coord.z *= zScale;
    }
}
