/* ======================================================================
 * 3dsfile.cc
 * 
 * This file is part of MeshIO, the general and extensible 3D mesh I/O
 * library.
 * Copyright (c) 1999, 2000 Niklas Elmqvist. All rights reserved.
 *
 * File created 1999-06-05 by Niklas Elmqvist <d97elm@dtek.chalmers.se>.
 *
 * Based on the 3dsrdr.c code by Jare/Iguana available at the 
 * X2FTP site at <ftp://x2ftp.oulu.fi>. 
 * 
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 *
 * ======================================================================
 */

// -- System Includes
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

// -- Local Includes
#include "meshio.hh"
#include "utils.hh"

using namespace MeshIO;

// -- Type Declarations
typedef unsigned char	byte;
typedef unsigned short	word;
typedef unsigned long	dword;

typedef signed char	sbyte;
typedef signed short 	sword;
typedef signed long	sdword;

// -- Enumerated symbols

// Mesh set state.
enum MSetState {
    MSetNoState = 0,            // No specific state
    MSetLight,                  // Defining lights
    MSetMaterialAmbient,        // Ambient material properties
    MSetMaterialDiffuse,        // Diffuse material properties
    MSetMaterialSpecular        // Specular material properties
};

// .3ds chunk types
enum {
    CHUNK_RGBF      = 0x0010,
    CHUNK_RGBB      = 0x0011,
    //    CHUNK_RBGB2     = 0x0012,       // ?? NOT HLS.
    
    CHUNK_MAIN      = 0x4D4D,
    	CHUNK_OBJMESH   = 0x3D3D,
    	    CHUNK_BKGCOLOR  = 0x1200,
    	    CHUNK_AMBCOLOR  = 0x2100,
    	    CHUNK_OBJBLOCK  = 0x4000,
                CHUNK_TRIMESH   = 0x4100,
                    CHUNK_VERTLIST  = 0x4110,
                    CHUNK_FACELIST  = 0x4120,
                    CHUNK_FACEMAT   = 0x4130,
                    CHUNK_MAPLIST   = 0x4140,
                    CHUNK_SMOOLIST  = 0x4150,
                    CHUNK_TRMATRIX  = 0x4160,
                CHUNK_LIGHT     = 0x4600,
                    CHUNK_SPOTLIGHT = 0x4610,
                CHUNK_CAMERA    = 0x4700,
        CHUNK_MATERIAL  = 0xAFFF,
            CHUNK_MATNAME   = 0xA000,
            CHUNK_AMBIENT   = 0xA010,
            CHUNK_DIFFUSE   = 0xA020,
            CHUNK_SPECULAR  = 0xA030,
            CHUNK_TEXTURE   = 0xA200,
            CHUNK_BUMPMAP   = 0xA230,
            CHUNK_MAPFILE   = 0xA300,
        CHUNK_KEYFRAMER = 0xB000,
            CHUNK_KEYKKK    = 0xB001,       // No idea.
            CHUNK_TRACKINFO = 0xB002,
                CHUNK_TRACKOBJNAME  = 0xB010,
                CHUNK_TRACKPOS      = 0xB020,
                CHUNK_TRACKROTATE   = 0xB021,
                CHUNK_TRACKSCALE    = 0xB022,
                CHUNK_OBJNUMBER     = 0xB030,
            CHUNK_FRAMES    = 0xB008,
};

// -- Structure Definitions

typedef struct {
    MSetState state;		// Current mesh state
    MeshSet *set;		// Current mesh set
    Mesh *mesh;			// Current mesh
    Material *material;		// Current material
    ListNode *materialList;	// Temporary material list
} FileInfo;

typedef struct {
    word id;
    const char *name;
    bool (*func)(FILE *f, int ind, long p, FileInfo *info);
} ChunkInfo; 

// -- Local Function Prototypes
static bool ChunkReader		(FILE *f, int ind, long p, FileInfo *info);
static bool RGBFReader		(FILE *f, int ind, long p, FileInfo *info);
static bool RGBBReader		(FILE *f, int ind, long p, FileInfo *info);
static bool ASCIIZReader	(FILE *f, int ind, long p, FileInfo *info,
				 char *buf = NULL, int len = 0);
static bool ObjBlockReader	(FILE *f, int ind, long p, FileInfo *info);
static bool VertListReader	(FILE *f, int ind, long p, FileInfo *info);
static bool FaceListReader	(FILE *f, int ind, long p, FileInfo *info);
static bool FaceMatReader	(FILE *f, int ind, long p, FileInfo *info);
static bool MapListReader	(FILE *f, int ind, long p, FileInfo *info);
static bool SmooListReader	(FILE *f, int ind, long p, FileInfo *info);
static bool TrMatrixReader	(FILE *f, int ind, long p, FileInfo *info);
static bool LightReader		(FILE *f, int ind, long p, FileInfo *info);
static bool SpotLightReader	(FILE *f, int ind, long p, FileInfo *info);
static bool CameraReader	(FILE *f, int ind, long p, FileInfo *info);
static bool MaterialReader	(FILE *f, int ind, long p, FileInfo *info);
static bool MatNameReader	(FILE *f, int ind, long p, FileInfo *info);
static bool MatAmbient		(FILE *f, int ind, long p, FileInfo *info);
static bool MatDiffuse		(FILE *f, int ind, long p, FileInfo *info);
static bool MatSpecular		(FILE *f, int ind, long p, FileInfo *info);
static bool MapFileReader	(FILE *f, int ind, long p, FileInfo *info);
static bool FramesReader	(FILE *f, int ind, long p, FileInfo *info);
static bool TrackObjNameReader	(FILE *f, int ind, long p, FileInfo *info);
static bool TrackPosReader	(FILE *f, int ind, long p, FileInfo *info);
static bool TrackRotReader	(FILE *f, int ind, long p, FileInfo *info);
static bool TrackScaleReader	(FILE *f, int ind, long p, FileInfo *info);
static bool ObjNumberReader	(FILE *f, int ind, long p, FileInfo *info);

static void BuildMaterialArray  (FileInfo *info);

// -- Constants

const ChunkInfo ChunkNames[] = {
    { CHUNK_RGBF,        "RGB float",        RGBFReader		},
    { CHUNK_RGBB,        "RGB byte",         RGBBReader		},
    
    { CHUNK_MAIN,        "Main",             NULL		},
    { CHUNK_OBJMESH,     "Object Mesh",      NULL		},
    { CHUNK_BKGCOLOR,    "Background color", NULL		},
    { CHUNK_AMBCOLOR,    "Ambient color",    NULL		},
    { CHUNK_OBJBLOCK,    "Object Block",     ObjBlockReader	},
    { CHUNK_TRIMESH,     "Tri-Mesh",         NULL		},
    { CHUNK_VERTLIST,    "Vertex list",      VertListReader	},
    { CHUNK_FACELIST,    "Face list",        FaceListReader	},
    { CHUNK_FACEMAT,     "Face material",    FaceMatReader	},
    { CHUNK_MAPLIST,     "Mappings list",    MapListReader	},
    { CHUNK_SMOOLIST,    "Smoothings",       SmooListReader	},
    { CHUNK_TRMATRIX,    "Matrix",           TrMatrixReader	},
    { CHUNK_LIGHT,       "Light",            LightReader       	},
    { CHUNK_SPOTLIGHT,   "Spotlight",        SpotLightReader	},
    { CHUNK_CAMERA,      "Camera",           CameraReader	},
    
    { CHUNK_MATERIAL,    "Material",         MaterialReader   	},
    { CHUNK_MATNAME,     "Material name",    MatNameReader	},
    { CHUNK_AMBIENT,     "Ambient color",    MatAmbient		},
    { CHUNK_DIFFUSE,     "Diffuse color",    MatDiffuse		},
    { CHUNK_SPECULAR,    "Specular color",   MatSpecular       	},
    { CHUNK_TEXTURE,     "Texture map",      NULL		},
    { CHUNK_BUMPMAP,     "Bump map",         NULL		},
    { CHUNK_MAPFILE,     "Map filename",     MapFileReader	},

    { CHUNK_KEYFRAMER,   "Keyframer data",   NULL		},
    { CHUNK_KEYKKK,      "No idea",          NULL		},
    { CHUNK_TRACKINFO,   "Track info",       NULL		},
    { CHUNK_FRAMES,      "Frames",           FramesReader	},
    { CHUNK_TRACKOBJNAME,"Track Obj. Name",  TrackObjNameReader	},
    { CHUNK_TRACKPOS,    "Position keys",    TrackPosReader	},
    { CHUNK_TRACKROTATE, "Rotation keys",    TrackRotReader	},
    { CHUNK_TRACKSCALE,  "Scale keys",       TrackScaleReader	},
    { CHUNK_OBJNUMBER,   "Object number",    ObjNumberReader	},    
};

// -- Code Segment

mioResult MeshIO::mio3DSLoad(const char *filename, MeshSet *mset)
{
    FileInfo info;
    FILE *f;
    long p;
    
    // Attempt to open the desired file
    if ((f = fopen(filename, "rb")) == NULL) {
        printf("Can't open %s!\n", filename);
	return MIO_FILE_NOT_FOUND;
    }
    
    // Initialize mesh mset
    info.set = mset;
    mioMSetInit(info.set);
    
    // Find file size.
    fseek(f, 0, SEEK_END);
    p = ftell(f);
    fseek(f, 0, SEEK_SET);
    
    // Go! (Start reading the file)
    if (ChunkReader(f, 0, p, &info) == false) {
	
	// An error occurred, close up shop
	fclose(f);
	return MIO_READ_ERROR;
    }
    
    // Close the file
    fclose(f);

    // Build the material array
    BuildMaterialArray(&info);
    
    // Compute the normals
    mioMSetComputeNormals(info.set);
	    
    // Return with success
    return MIO_NO_ERROR;
}

int FindMaterial(ListNode *list, char *name)
{
    int ndx = 0;
    Material *node;
    
    // Traverse the list looking for a match
    while (list != NULL) {
	node = (Material *) list->data;
	if (strcmp(name, node->name) == 0)
	    break;
	list = list->next;
	ndx++;
    }
    
    // Return with index (-1 for not found)
    if (list == NULL)
	return -1;
    else return ndx;
}

void BuildMaterialArray(FileInfo *info)
{
    // Allocate space for material array
    info->set->materialList = (Material *)
	malloc(sizeof(Material) * info->set->materialNum);
    
    // Copy the materials to the array
    for (int i = 0; i < info->set->materialNum; i++) {

	// Bitwise copy
	memcpy(&info->set->materialList[i], info->materialList->data,
	       sizeof(Material));
	
	// Delete the material and node
	ListNode *temp = info->materialList;
	info->materialList = info->materialList->next;
	free(temp->data);
	free(temp);
    }
}

int FindChunk(word id) 
{
    for (int i = 0; i < int(sizeof(ChunkNames)/sizeof(ChunkNames[0])); i++)
        if (id == ChunkNames[i].id)
            return i;
    return -1;
}

bool ChunkReader(FILE *f, int ind, long p, FileInfo *info)
{
    bool bResult;
    int n;
    long pc;
    word id;
    dword len;
    
    // Repeat until we have read the entire allotted space
    while (ftell(f) < p) {
	
	// Grab the current file position
        pc = ftell(f);
	
	// Read the chunk header (contains the ID and the length of the chunk)
	if (ReadData(f, &id, sizeof(id)) == false) return false;
	if (ReadData(f, &len, sizeof(len)) == false) return false;
	
	// Identify the chunk (using the chunk name table)
	n = FindChunk(id);
	
	// Is this a valid chunk?
        if (n < 0) {
	    
	    // No, skip it
            fseek(f, pc + len, SEEK_SET);
	    
        } else {
	    
#ifdef DEBUG
            printf("%*sChunk type \"%s\", offset 0x%lX, size %ld bytes\n",
                   ind, "", ChunkNames[n].name, pc, len);
#endif
	    
	    // Yes, continue
            pc = pc + len;
	    
	    // Call relevant reader function
            if (ChunkNames[n].func != NULL)
                bResult = ChunkNames[n].func(f, ind + 2, pc, info);
            else
		// No relevant reader function, call recursively
                bResult = ChunkReader(f, ind + 2, pc, info);
	    
	    // Did we succeed?
	    if (bResult == false) return false;
	    
	    // Find the end of the chunk
            fseek(f, pc, SEEK_SET);
        }
	
	// If there is an error, exit from this loop
	// @@@ FIX THIS TO GENERATE ERROR
        if (ferror(f)) break;	
    }
    
    // Return with success
    return true;
}

// --[ Reader Functions ]------

// Read RGB color value (float format)
bool RGBFReader(FILE *f, int ind, long p, FileInfo *info) 
{
    float c[3];
    Material *mat = info->material;
    
    if (ReadData(f, &c, sizeof(float), 3) == false) return false;
    
    // Check state
    switch (info->state) {
    case MSetMaterialAmbient:
	memcpy(mat->ambient, c, 3 * sizeof(float));
	break;
    case MSetMaterialDiffuse:
	memcpy(mat->diffuse, c, 3 * sizeof(float));
	break;
    case MSetMaterialSpecular:
	memcpy(mat->specular, c, 3 * sizeof(float));
	break;
    default:
	break;
    }
#ifdef DEBUG
    printf("%*s    Red: %f, Green: %f, Blue: %f\n", ind, "", c[0], c[1], c[2]);
#endif
    
    // Return with success
    return true;
}

// Read RGB color value (byte format)
bool RGBBReader (FILE *f, int ind, long p, FileInfo *info) 
{
    Material *mat = info->material;
    byte b[3];
    float c[3];
    
    if (ReadData(f, &b, sizeof(byte), 3) == false) return false;
    
    for (int i = 0; i < 3; i++)
	c[i] = float(b[i]) / 255.0f;
    
    // Check state
    switch (info->state) {
    case MSetMaterialAmbient:
	memcpy(mat->ambient, c, 3 * sizeof(float));
	break;
    case MSetMaterialDiffuse:
	memcpy(mat->diffuse, c, 3 * sizeof(float));
	break;
    case MSetMaterialSpecular:
	memcpy(mat->specular, c, 3 * sizeof(float));
	break;
    default:
	break;
    }
    
#ifdef DEBUG
    printf("%*s    Red: %d, Green: %d, Blue: %d\n", ind, "", b[0], b[1], b[2]);
#endif 
    
    // Return with success
    return true;
}

// Read ASCII zero-terminated string
bool ASCIIZReader(FILE *f, int ind, long p, FileInfo *info, char *buf, int len)
{
    int c, ndx = 0;
    
    // Read ASCIIZ name
    while ((c = fgetc(f)) != EOF && c != '\0') {
#ifdef DEBUG
        putchar(c);
#endif
	if (buf != NULL && ndx < (len - 1))
	    buf[ndx++] = c;
    }
#ifdef DEBUG
    printf("\"\n");
#endif
    
    // Put a terminator in the string if needed
    if (buf != NULL)
	buf[ndx] = '\0';
    
    // Return with success
    return true;
}

// Read object chunk (recursive chunk)
bool ObjBlockReader (FILE *f, int ind, long p, FileInfo *info) 
{
    Mesh *newMesh = (Mesh *) malloc(sizeof(Mesh));
    
    // Read ASCIIZ object name (recursive chunk)
#ifdef DEBUG
    printf("%*sObject name \"", ind, "");
#endif
    ASCIIZReader(f, ind, p, info);
    
    // Initialize mesh and add it to the set
    mioMeshInit(newMesh);
    mioMSetAddMesh(info->set, newMesh);
    info->mesh = newMesh;
    
    // Read rest of chunks inside this one.
    bool bResult = ChunkReader(f, ind, p, info);
    
    // Return with result
    return bResult;
}

// Read a list of [x y z] vertices
bool VertListReader (FILE *f, int ind, long p, FileInfo *info) 
{
    word nv;
    float c[3];
    Mesh *mesh = info->mesh;
    
    // First, read the number of vertices to be added
    if (ReadData(f, &nv, sizeof(nv)) == false) return false;
    mesh->vertexNum = nv;
    
#ifdef DEBUG
    printf("%*sVertices: %d\n", ind, "", mesh->vertexNum);
#endif 
    
    // Allocate space in mesh for vertices (and their corresponding normals)
    mesh->vertexList = (Vertex3D *) malloc(sizeof(Vertex3D) * mesh->vertexNum);
    mesh->normalList = (Vertex3D *) malloc(sizeof(Vertex3D) * mesh->vertexNum);
    
    // Read the actual vertex data
    for (int i = 0; i < mesh->vertexNum; i++) {
	
	// Do a binary read
	if (ReadData(f, &c, sizeof(float), 3) == false) return false;
	
#ifdef DEBUG
        printf("%*s    X: %f, Y: %f, Z: %f\n", ind, "", c[0], c[1], c[2]);
#endif
	
	// Assign data into mesh struct
	for (int j = 0; j < 3; j++)
	    mesh->vertexList[i].coords[j] = c[j];
    }

    // Return with success
    return true;
}

// Read a list of faces (triangles)
bool FaceListReader (FILE *f, int ind, long p, FileInfo *info) 
{
    word nv;
    word c[4];
    Mesh *mesh = info->mesh;
    
    // Read the number of faces to add
    if (ReadData(f, &nv, sizeof(nv)) == false) return false;
    mesh->faceNum = nv;

#ifdef DEBUG
    printf("%*sFaces: %d\n", ind, "", nv);
#endif
    
    // Allocate memory for the face list
    mesh->faceList = (Face *) malloc(sizeof(Face) * mesh->faceNum);
    
    // Load the entire face list
    for (int i = 0; i < mesh->faceNum; i++) {
	
	// Read data into temporary array
	if (ReadData(f, &c, sizeof(word), 4) == false) return false;
	
#ifdef DEBUG
        printf("%*s    (# %d) A: %d, B: %d, C: %d, Flags 0x%X\n",
               ind, "", i, c[0], c[1], c[2], c[3]);
#endif
	
	// Set data into mesh struct
	for (int j = 0; j < 3; j++) {
	    mesh->faceList[i].ndxVertex.ndxs[j] = c[j];
	    mesh->faceList[i].ndxMapping.ndxs[j] = c[j];
	}
    }
    
    // Read rest of chunks inside this one.
    bool bResult = ChunkReader(f, ind, p, info);
    
    // Return with result
    return bResult;
}

// Read material name for faces
bool FaceMatReader(FILE *f, int ind, long p, FileInfo *info) 
{
    word n, nf;
    int ndx;
    char buf[MAX_STRING_LENGTH];
    Mesh *mesh = info->mesh;
    
    // Read ASCIIZ material name
#ifdef DEBUG
    printf("%*sMaterial name for faces: \"", ind, "");
#endif
    ASCIIZReader(f, ind, p, info, buf, MAX_STRING_LENGTH);

    // Search for the material index
    ndx = FindMaterial(info->materialList, buf);
    
#ifdef DEBUG
    // Emit output if error occurred
    if (ndx == -1)
	printf("Error! Material \"%s\" not found.", buf);
#endif
    
    if (ReadData(f, &n, sizeof(n)) == false) return false;
#ifdef DEBUG
    printf("%*sFaces with this material: %d\n", ind, "", n);
#endif
    
    // Read indices
    while (n-- > 0) {
	if (ReadData(f, &nf, sizeof(nf)) == false) return false;
#ifdef DEBUG	
        printf("%*s    Face %d\n", ind, "", nf);
#endif
	
	// Update current mesh
	mesh->faceList[nf].material = ndx;
    }
    
    // Return with success
    return true;
}

// Read texture map coordinates [U, V] for all vertices
bool MapListReader (FILE *f, int ind, long p, FileInfo *info) 
{
    word nv;
    float c[2];
    Mesh *mesh = info->mesh;
    
    // Read the number of map coordinates to get
    if (ReadData(f, &nv, sizeof(nv)) == false) return false;
    
    // Store away this value
    mesh->mappingNum = nv;
    
#ifdef DEBUG
    printf("%*sVertices (TexCoord): %d\n", ind, "", nv);
#endif
    
    // Allocate memory for mapping coordinates
    mesh->mappingList = (Vertex2D *) malloc(sizeof(Vertex2D) * nv);
    
    // Start reading
    for (int i = 0; i < nv; i++) {
	
	// First, read the [u,v] floating values
	if (ReadData(f, &c, sizeof(float), 2) == false) return false;
	
#ifdef DEBUG
        printf("%*s    U: %f, V: %f\n", ind, "", c[0], c[1]);
#endif 
	
	// Set the data into the mesh struct
	for (int j = 0; j < 2; j++)
	    mesh->mappingList[i].coords[j] = c[j];
    }
    
    // Return with success
    return true;
}

// Read smoothing groups
bool SmooListReader (FILE *f, int ind, long p, FileInfo *info) 
{
    dword s;
    
    // Start reading words as long as we can
    while (ftell(f) < p) {
	
	// Read one word
	if (ReadData(f, &s, sizeof(s)) == false) return false;
	
#ifdef DEBUG // No use for this chunk at all
	int i;
	
        printf("%*sSmoothing groups: ", ind, "");
        for (i = 0; i < 32; i++)
            if (s & (1 << i))
                printf("%d, ", i + 1);
        printf("\n");
#endif
    }

    // Return with success
    return true;
}

// Read transformation matrices
bool TrMatrixReader(FILE *f, int ind, long p, FileInfo *info)
{
    float rot[9];
    float trans[3];
    
    // First, read the rotation matrix
    if (ReadData(f, &rot, sizeof(float), 9) == false) return false;
    
    // Then, read the translation vector 
    if (ReadData(f, &trans, sizeof(float), 3) == false) return false;
    
    // @@@ NEED to save this later in Mesh!
    
#ifdef DEBUG
    printf("%*sRotation matrix:\n", ind, "");
    printf("%*s    %f, %f, %f\n", ind, "", rot[0], rot[1], rot[2]);
    printf("%*s    %f, %f, %f\n", ind, "", rot[3], rot[4], rot[5]);
    printf("%*s    %f, %f, %f\n", ind, "", rot[6], rot[7], rot[8]);

    printf("%*sTranslation matrix: %f, %f, %f\n",
           ind, "", trans[0], trans[1], trans[2]);
#endif 
    
    // Return with success
    return true;
}

// Read lights (recursive chunk)
bool LightReader(FILE *f, int ind, long p, FileInfo *info) 
{
    float c[3];
    
    if (ReadData(f, &c, sizeof(float), 3) == false) return false;

#ifdef DEBUG
    printf("%*s    X: %f, Y: %f, Z: %f\n", ind, "", c[0], c[1], c[2]);
#endif
    
    // Set light reading state
    info->state = MSetLight;
    
    // Read rest of chunks inside this one.
    bool bResult = ChunkReader(f, ind, p, info);

    // Return with result
    return bResult;
}

// Read spotlight
bool SpotLightReader(FILE *f, int ind, long p, FileInfo *info)
{
    float c[5];
    
    if (ReadData(f, &c, sizeof(float), 5) == false) return false;
    
#ifdef DEBUG
    printf("%*s    Target X: %f, Y: %f, Z: %f; Hotspot %f, Falloff %f\n",
           ind, "", c[0], c[1], c[2], c[3], c[4]);
#endif 

    // Return with success
    return true;
}

// Read camera information
bool CameraReader(FILE *f, int ind, long p, FileInfo *info) 
{
    float c[8];
    
    if (ReadData(f, &c, sizeof(float), 8) == false) return false;
    
#ifdef DEBUG // No use for this chunk
    printf("%*s    Position: X: %f, Y: %f, Z: %f\n", ind, "", 
	   c[0], c[1], c[2]);
    printf("%*s    Target: X: %f, Y: %f, Z: %f\n", ind, "", c[3], c[4], c[5]);
    printf("%*s    Bank: %f, Lens: %f\n", ind, "", c[6], c[7]);
#endif
    
    // Return with success
    return true;
}

// Read and initialize a material
bool MaterialReader(FILE *f, int ind, long p, FileInfo *info)
{
    Material *material = (Material *) malloc(sizeof(Material));
    
    // Initialize material and add it to the mesh set
    mioMaterialInit(material);
    info->materialList = AddNode(info->materialList, material);
    info->material = material;
    info->set->materialNum++;
    
    // Read the chunk
    return ChunkReader(f, ind, p, info);
}

// Read material name 
bool MatNameReader(FILE *f, int ind, long p, FileInfo *info) 
{
    char buf[MAX_STRING_LENGTH];
    Material *mat = info->material;
    
    // Read ASCIIZ object name
#ifdef DEBUG
    printf("%*sMaterial name \"", ind, "");
#endif
    ASCIIZReader(f, ind, p, info, buf, MAX_STRING_LENGTH);
    mat->name = strdup(buf);
    
    // Return with success
    return true;
}

// Read ambient material color
bool MatAmbient(FILE *f, int ind, long p, FileInfo *info)
{
    // Set new state
    info->state = MSetMaterialAmbient;
    
    // Read the chunk
    return ChunkReader(f, ind, p, info);
}

// Read diffuse material color
bool MatDiffuse(FILE *f, int ind, long p, FileInfo *info)
{
    // Set new state
    info->state = MSetMaterialDiffuse;
    
    // Read the chunk
    return ChunkReader(f, ind, p, info);
}

// Read specular material color
bool MatSpecular(FILE *f, int ind, long p, FileInfo *info)
{
    // Set new state
    info->state = MSetMaterialSpecular;
    
    // Read the chunk
    return ChunkReader(f, ind, p, info);
}

// Read texturemap filename
bool MapFileReader(FILE *f, int ind, long p, FileInfo *info)
{
    char buf[MAX_STRING_LENGTH];
    Material *mat = info->material;

    // Read ASCIIZ filename
#ifdef DEBUG
    printf("%*sMap filename \"", ind, "");
#endif
    ASCIIZReader(f, ind, p, info, buf, MAX_STRING_LENGTH);
    mat->texFile = strdup(buf);
    
    // Return with success
    return true;
}

// Read frame information
bool FramesReader(FILE *f, int ind, long p, FileInfo *info) 
{
    dword c[2];
    
    if (ReadData(f, &c, sizeof(dword), 2) == false) return false;
#ifdef DEBUG
    printf("%*s    Start: %ld, End: %ld\n",
           ind, "", c[0], c[1]);
#endif 

    // Return with success
    return true;
}

// Read object tracking information
bool TrackObjNameReader(FILE *f, int ind, long p, FileInfo *info) 
{
    word w[3];
    
    // Read ASCIIZ name
#ifdef DEBUG
    printf("%*sTrack object name \"", ind, "");
#endif
    ASCIIZReader(f, ind, p, info);
    
    if (ReadData(f, &w, sizeof(word), 3) == false) return false;
#ifdef DEBUG
    printf("%*sObject name data: 0x%X, 0x%X, 0x%X\n", ind, "", w[0], w[1], w[2]);
#endif 

    // Return with succes
    return true;
}

/* Key info flags for position, rotation and scaling:
   0x80000000      == "Ease to" float follows.
   
   Until I know the meaning of each bit I assume all include
   a following float data. Dunno in which order come those, too.
*/

// Read object tracking position information
bool TrackPosReader(FILE *f, int ind, long p, FileInfo *info) 
{
    word n, nf;
    float pos[3];
    dword flags;
    
    fseek(f, 10, SEEK_CUR);
    if (ReadData(f, &n, sizeof(n)) == false) return false;
#ifdef DEBUG
    printf("%*sPosition keys: %d\n", ind, "", n);
#endif
    fseek(f, 2, SEEK_CUR);
    while (n-- > 0) {
        int i;
        float dat;
	if (ReadData(f, &nf, sizeof(nf)) == false) return false;
	if (ReadData(f, &flags, sizeof(flags)) == false) return false;
        for (i = 0; i < 32; i++)
            if (flags & (1 << i))
                if (fread(&dat, sizeof(dat), 1, f) != 1) return false;
	if (ReadData(f, &pos, sizeof(float), 3) == false) return false;
#ifdef DEBUG
        printf("%*s  Frame %d: Flags 0x%lX, X: %f, Y: %f, Z: %f\n",
               ind, "", nf, flags, pos[0], pos[1], pos[2]);
#endif
    }
    
    // Return with success
    return true;
}

// Read object tracking rotation information 
bool TrackRotReader(FILE *f, int ind, long p, FileInfo *info) 
{
    word n, nf;
    float pos[4];
    dword flags;
    
    fseek(f, 10, SEEK_CUR);
    if (ReadData(f, &n, sizeof(n)) == false) return false;
#ifdef DEBUG
    printf("%*sRotation keys: %d\n", ind, "", n);
#endif 
    fseek(f, 2, SEEK_CUR);
    while (n-- > 0) {
        int i;
        float dat;
	if (ReadData(f, &nf, sizeof(nf)) == false) return false;
	if (ReadData(f, &flags, sizeof(flags)) == false) return false;
	for (i = 0; i < 32; i++)
            if (flags & (1 << i))
                if (fread(&dat, sizeof(dat), 1, f) != 1) return false;
	if (ReadData(f, &pos, sizeof(float), 4) == false) return false;
#ifdef DEBUG
        printf("%*s  Frame %d: Flags 0x%lX, Angle: %f, X: %f, Y: %f, Z: %f\n",
               ind, "", nf, flags, pos[0]*180.0/PI, pos[1], pos[2], pos[3]);
#endif
    }

    // Return with success
    return true;
}

// Read object tracking scaling information
bool TrackScaleReader(FILE *f, int ind, long p, FileInfo *info) 
{
    word n, nf;
    float pos[3];
    dword flags;
    
    fseek(f, 10, SEEK_CUR);
    if (ReadData(f, &n, sizeof(n)) == false) return false;
#ifdef DEBUG
    printf("%*sScale keys: %d\n", ind, "", n);
#endif
    fseek(f, 2, SEEK_CUR);
    while (n-- > 0) {
        int i;
        float dat;
	if (ReadData(f, &nf, sizeof(nf)) == false) return false;
	if (ReadData(f, &flags, sizeof(flags)) == false) return false;
        for (i = 0; i < 32; i++)
            if (flags & (1 << i))
                if (fread(&dat, sizeof(dat), 1, f) != 1) return false;
	if (ReadData(f, &pos, sizeof(float), 3) == false) return false;
#ifdef DEBUG
        printf("%*s  Frame %d: Flags 0x%lX, X: %f, Y: %f, Z: %f\n",
               ind, "", nf, flags, pos[0], pos[1], pos[2]);
#endif
    }

    // Return with success
    return true;
}

// Read object number
bool ObjNumberReader(FILE *f, int ind, long p, FileInfo *info) 
{
    sword n;
    
    if (ReadData(f, &n, sizeof(n)) == false) return false;
#ifdef DEBUG
    printf("%*sObject number: %d\n", ind, "", n);
#endif 

    // Return with success
    return true;
}



