/* mg-work-form.c
 *
 * Copyright (C) 2002 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include <libgnomedb/libgnomedb.h>
#include "mg-server.h"
#include "mg-work-form.h"
#include "mg-work-core.h"
#include "mg-work-widget.h"
#include "mg-query.h"
#include "mg-target.h"
#include "mg-entity.h"
#include "mg-renderer.h"
#include "mg-result-set.h"
#include "mg-form.h"
#include "mg-parameter.h"
#include "mg-context.h"
#include "mg-qfield.h"
#include "utility.h"
#include "mg-util.h"

#ifdef debug
#include "mg-graphviz.h"
#endif

static void mg_work_form_class_init (MgWorkFormClass * class);
static void mg_work_form_init (MgWorkForm * wid);
static void mg_work_form_dispose (GObject   * object);

static void mg_work_form_set_property (GObject              *object,
				       guint                 param_id,
				       const GValue         *value,
				       GParamSpec           *pspec);
static void mg_work_form_get_property (GObject              *object,
				       guint                 param_id,
				       GValue               *value,
				       GParamSpec           *pspec);

static void mg_work_form_initialize (MgWorkForm *form, MgQuery *orig_query, GtkWidget *layout, GHashTable *box_widgets);

static void nullified_core_cb (MgWorkCore *core, MgWorkForm *form);

static GtkWidget *modif_buttons_make (MgWorkForm *form);
static void       modif_buttons_update (MgWorkForm *form);

static void       update_simple_form (MgWorkForm *form);
static void       arg_param_changed_cb (MgContext *context, MgWorkForm *form);
static void       work_param_changed_cb (MgForm *simple_form, MgParameter *param, gboolean is_user_modif, MgWorkForm *form);

static void modif_actions_real_do (MgWorkForm *form, gchar action);
static void arrow_actions_real_do (MgWorkForm *form, gint movement);

/* MgWorkWidget interface */
static void            mg_work_form_widget_init         (MgWorkWidgetIface *iface);
static void            mg_work_form_run                 (MgWorkWidget *iface, guint mode);
static void            mg_work_form_set_mode            (MgWorkWidget *iface, guint mode);
static void            mg_work_form_set_entry_editable  (MgWorkWidget *iface, MgQfield *field, gboolean editable);
static void            mg_work_form_show_entry_actions (MgWorkWidget *iface, MgQfield *field, gboolean show_actions);
static void            mg_work_form_show_global_actions (MgWorkWidget *iface, gboolean show_actions);
static MgParameter    *mg_work_form_get_param_for_field (MgWorkWidget *iface, MgQfield *field, const gchar *field_name, 
							 gboolean in_exec_context);
static gboolean        mg_work_form_has_been_changed    (MgWorkWidget *iface);
static MgContext      *mg_work_form_get_exec_context    (MgWorkWidget *iface);
static GtkActionGroup *mg_work_form_get_actions_group   (MgWorkWidget *iface);

enum {
	SIMPLE_FORM_UPDATE,
	SIMPLE_FORM_INSERT
};


struct _MgWorkFormPriv
{
	MgWorkCore        *core;
	gboolean           has_run;
	guint              cursor;

	GtkWidget         *title;
	GtkWidget         *notebook;
	GtkWidget         *basic_form;
	guint              basic_form_mode;
	
	guint              mode;
	GtkTooltips       *tooltips;

	GtkUIManager      *uimanager;
	GtkActionGroup    *actions_group;
	GtkWidget         *modif_all;
	GtkWidget         *nav_scale;
	GtkWidget         *nav_current;
	
	gboolean           intern_form_changes;
};

#define PAGE_NO_DATA 0
#define PAGE_FORM    1

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

/* properties */
enum
{
        PROP_0,
	PROP_TITLE_VISIBLE,
	PROP_TITLE_STRING,
	PROP_ACTIONS_VISIBLE
};

guint
mg_work_form_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MgWorkFormClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) mg_work_form_class_init,
			NULL,
			NULL,
			sizeof (MgWorkForm),
			0,
			(GInstanceInitFunc) mg_work_form_init
		};		
		
		static const GInterfaceInfo work_widget_info = {
                        (GInterfaceInitFunc) mg_work_form_widget_init,
                        NULL,
                        NULL
                };
		
		type = g_type_register_static (GTK_TYPE_VBOX, "MgWorkForm", &info, 0);
		g_type_add_interface_static (type, MG_WORK_WIDGET_TYPE, &work_widget_info);
	}

	return type;
}

static void
mg_work_form_widget_init (MgWorkWidgetIface *iface)
{
	iface->run = mg_work_form_run;
	iface->set_mode = mg_work_form_set_mode;
	iface->set_entry_editable = mg_work_form_set_entry_editable;
	iface->show_entry_actions = mg_work_form_show_entry_actions;
	iface->show_global_actions = mg_work_form_show_global_actions;
	iface->get_param_for_field = mg_work_form_get_param_for_field;
        iface->has_been_changed = mg_work_form_has_been_changed;
	iface->get_exec_context = mg_work_form_get_exec_context;
	iface->get_actions_group = mg_work_form_get_actions_group;
}


static void
mg_work_form_class_init (MgWorkFormClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);
	object_class->dispose = mg_work_form_dispose;

	/* Properties */
        object_class->set_property = mg_work_form_set_property;
        object_class->get_property = mg_work_form_get_property;
	g_object_class_install_property (object_class, PROP_ACTIONS_VISIBLE,
                                         g_param_spec_boolean ("actions_visible", NULL, NULL, FALSE,
                                                               G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_TITLE_VISIBLE,
                                         g_param_spec_boolean ("title_visible", NULL, NULL, FALSE,
                                                               G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_TITLE_STRING,
                                         g_param_spec_string ("title_string", NULL, NULL, NULL,
							      G_PARAM_WRITABLE));
}

static void
mg_work_form_init (MgWorkForm * wid)
{
	wid->priv = g_new0 (MgWorkFormPriv, 1);
	wid->priv->core = NULL;
	wid->priv->has_run = FALSE;

	wid->priv->notebook = NULL;
	wid->priv->basic_form = NULL;
	wid->priv->basic_form_mode = SIMPLE_FORM_UPDATE;

	wid->priv->cursor = 0;

	wid->priv->mode = 0;
	wid->priv->tooltips = NULL;
}

/**
 * mg_work_form_new
 * @query: a #MgQuery object
 * @modified: a #MgTarget object, or %NULL
 *
 * Creates a new #MgWorkForm widget.
 *
 * @query must be a SELECT query (no union, etc selection query)
 *
 * The @modified target must belong to @query and represent
 * modifiable entity (a #MgDbTable for example). If @modified is %NULL then
 * no modification will be allowed.
 *
 * Returns: the new widget
 */
GtkWidget *
mg_work_form_new (MgQuery *query, MgTarget *modified)
{
	return mg_work_form_new_in_layout (query, modified, NULL, NULL);
}

/**
 * mg_work_form_new_in_layout
 * @query: a #MgQuery object
 * @modified: a #MgTarget object, or %NULL
 * @layout: a #GtkWidget object
 * @box_widgets: a #GHashTable of #GtkBox widgets
 *
 * Creates a new #MgWorkForm widget.
 *
 * @query must be a SELECT query (no union, etc selection query)
 *
 * The @modified target must belong to @query and represent
 * modifiable entity (a #MgDbTable for example). If @modified is %NULL then
 * no modification will be allowed.
 *
 * This function is similar to mg_work_form_new() but provides a #GtkWidget to pack
 * entries in. The @box_widgets hash table has keys corresponding to the
 * query fields of @query, and corresponding values pointing to the #GtkBox widgets
 * where the #MGDataEntry widgets will be packed.
 *
 * If any of @layout or @box_widgets is %NULL, then this function is equivalent to mg_work_form_new().
 *
 * Returns: the new widget
 */
GtkWidget *
mg_work_form_new_in_layout (MgQuery *query, MgTarget *modified, GtkWidget *layout, GHashTable *box_widgets)
{
	GObject *obj;
	MgWorkForm *form;

	g_return_val_if_fail (query && IS_MG_QUERY (query), NULL);
	g_return_val_if_fail (mg_query_get_query_type (query) == MG_QUERY_TYPE_SELECT, NULL);
	
	if (modified) {
		g_return_val_if_fail (IS_MG_TARGET (modified), NULL);
		g_return_val_if_fail (mg_target_get_query (modified) == query, NULL);
		g_return_val_if_fail (mg_entity_is_writable (mg_target_get_represented_entity (modified)), NULL);
	}

	obj = g_object_new (MG_WORK_FORM_TYPE, NULL);
	form = MG_WORK_FORM (obj);

	form->priv->core = MG_WORK_CORE (mg_work_core_new (query, modified));
	g_signal_connect (G_OBJECT (form->priv->core), "nullified",
			  G_CALLBACK (nullified_core_cb), form);

	mg_work_form_initialize (form, query, layout, box_widgets);

	return GTK_WIDGET (obj);
}

static void
nullified_core_cb (MgWorkCore *core, MgWorkForm *form)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (core),
					      G_CALLBACK (nullified_core_cb), form);
	
	if (form->priv->core->work_context) 
		g_signal_handlers_disconnect_by_func (G_OBJECT (form->priv->basic_form),
						      G_CALLBACK (work_param_changed_cb), form);

	if (form->priv->has_run && form->priv->core->args_context)
		g_signal_handlers_disconnect_by_func (G_OBJECT (form->priv->core->args_context),
						      G_CALLBACK (arg_param_changed_cb), form);
	
	g_object_unref (G_OBJECT (form->priv->core));
	form->priv->core = NULL;
	gtk_widget_set_sensitive (GTK_WIDGET (form), FALSE);
}

static void
mg_work_form_dispose (GObject *object)
{
	MgWorkForm *form;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_WORK_FORM (object));
	form = MG_WORK_FORM (object);

	if (form->priv) {
		/* core */
		if (form->priv->core)
			nullified_core_cb (form->priv->core, form);

		/* tooltips */
		if (form->priv->tooltips) {
			gtk_object_destroy (GTK_OBJECT (form->priv->tooltips));
			form->priv->tooltips = NULL;
		}

		/* UI */
		if (form->priv->actions_group) 
			g_object_unref (G_OBJECT (form->priv->actions_group));

		if (form->priv->uimanager)
			g_object_unref (G_OBJECT (form->priv->uimanager));
		
		/* the private area itself */
		g_free (form->priv);
		form->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static void
mg_work_form_set_property (GObject              *object,
			   guint                 param_id,
			   const GValue         *value,
			   GParamSpec           *pspec)
{
	MgWorkForm *form;

        form = MG_WORK_FORM (object);
        if (form->priv) {
                switch (param_id) {
                case PROP_TITLE_VISIBLE:
			if (g_value_get_boolean (value))
				gtk_widget_show (form->priv->title);
			else
				gtk_widget_hide (form->priv->title);
                        break;
                case PROP_TITLE_STRING:
			gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (form->priv->title), g_value_get_string (value));
			gtk_widget_show (form->priv->title);
                        break;
		case PROP_ACTIONS_VISIBLE:
			if (g_value_get_boolean (value))
				gtk_widget_show (form->priv->modif_all);
			else
				gtk_widget_hide (form->priv->modif_all);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }
}

static void
mg_work_form_get_property (GObject              *object,
			   guint                 param_id,
			   GValue               *value,
			   GParamSpec           *pspec)
{
	MgWorkForm *form;

        form = MG_WORK_FORM (object);
        if (form->priv) {
                switch (param_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }	
}


/*
 * this function (re)computes the SELECT query and (re)runs it, resulting
 * in a change in the displayed values
 *
 * It does not use the 'param' argument
 */
static void
arg_param_changed_cb (MgContext *context, MgWorkForm *form)
{
	GError *error = NULL;

	if (mg_work_core_run_select_query (form->priv->core, &error)) {
		if (form->priv->cursor >= mg_resultset_get_nbtuples (form->priv->core->data_rs))
			form->priv->cursor = mg_resultset_get_nbtuples (form->priv->core->data_rs) - 1;
		update_simple_form (form);
	}
	else {
		/* we don't want non relevant error messages so we check that the args context is valid before */
		if ((form->priv->mode & MG_ACTION_REPORT_ERROR) && mg_context_is_valid (form->priv->core->args_context)) {
			GtkWidget *dlg;
			gchar *message;
			GtkWidget *parent;
			
			/* find the top level window of the form */
			parent = gtk_widget_get_parent (GTK_WIDGET (form));
			while (parent && !GTK_IS_WINDOW (parent)) 
				parent = gtk_widget_get_parent (parent);
			
			if (error) {
				message = g_strdup (error->message);
				g_error_free (error);
			}
			else
				message = g_strdup_printf (_("An unknown error occurred while executing the query."));
			
			dlg = gtk_message_dialog_new (GTK_WINDOW (parent), 0,
						      GTK_MESSAGE_ERROR,
						      GTK_BUTTONS_CLOSE,
						      message);
			g_free (message);
			gtk_dialog_run (GTK_DIALOG (dlg));
			gtk_widget_destroy (dlg);
		}
		update_simple_form (form);
	}

	/* updating the various possible actions */
	modif_buttons_update (form);
}

static void
work_param_changed_cb (MgForm *simple_form, MgParameter *param, gboolean is_user_modif, MgWorkForm *form)
{
	if (! form->priv->intern_form_changes) {
		modif_buttons_update (form);
		if (is_user_modif && (form->priv->basic_form_mode != SIMPLE_FORM_INSERT)) {
			if (form->priv->mode & MG_ACTION_MODIF_COMMIT_IMMEDIATE) {
				modif_actions_real_do (form, 'c');
			}
		}
	}
}

/*
 * Real initialization
 */
static void 
mg_work_form_initialize (MgWorkForm *form, MgQuery *orig_query, GtkWidget *layout, GHashTable *box_widgets)
{
	MgConf *conf = mg_base_get_conf (MG_BASE (form->priv->core->query_select));
	GtkWidget *realform, *wid, *nb, *group;
	GHashTable *fbw = NULL;

	/* title */
	form->priv->title = gnome_db_gray_bar_new (_("No title"));
        gtk_box_pack_start (GTK_BOX (form), form->priv->title, FALSE, TRUE, 2);
	gtk_widget_show (form->priv->title);


	/*
	 * the "No Data" notice
	 */
	nb = gtk_notebook_new ();
	form->priv->notebook = nb;
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (nb), FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK (nb), FALSE);
	gtk_box_pack_start (GTK_BOX (form), nb, TRUE, TRUE, 0);
	gtk_widget_show (nb);
	wid = gtk_label_new (_("No data to be displayed"));
	gtk_widget_show (wid);
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), wid, NULL);

	/* 
	 * the form's attributes
	 */
	if (mg_base_get_name (MG_BASE (form->priv->core->query_select)))
		gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (form->priv->title),
					    mg_base_get_name (MG_BASE (form->priv->core->query_select)));
	else
		gtk_widget_hide (form->priv->title);

	/*
	 * the actions part
	 */
	group = modif_buttons_make (form);
	gtk_box_pack_start (GTK_BOX (form), group, FALSE, FALSE, 0);
	gtk_widget_show (group);

	/*
	 * handling of the layout
	 */
	if (layout && box_widgets) {
		/* REM: the @box_widgets has keys for query fields, and we need to provide
		   mg_form_new_in_layout() with a hash tbale which has keys for MgContextNode;
		   this is why we need to use another hash table: fbw */
		GSList *list;
		MgParameter *param;
		MgContextNode *node;
		gpointer widget;

		fbw = g_hash_table_new (NULL, NULL);
		g_object_get (G_OBJECT (orig_query), "really_all_fields", &list, NULL);
		while (list) {
			widget = g_hash_table_lookup (box_widgets, list->data);
			if (widget) {
				node = mg_work_core_find_context_node (form->priv->core, MG_QFIELD (list->data));
				if (node)
					g_hash_table_insert (fbw, node, widget);
			}
			list = g_slist_next (list);
		}
	}

	/* 
	 * the form itself 
	 */
	realform = mg_form_new_in_layout (conf, form->priv->core->work_context, layout, fbw);
	if (fbw)
		g_hash_table_destroy (fbw);
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), realform, NULL);
	gtk_widget_show (realform);
	if (!layout && ! form->priv->core->modif_target)
		mg_form_show_entries_actions (MG_FORM (realform), FALSE);
	form->priv->basic_form = realform;

	g_signal_connect (G_OBJECT (form->priv->basic_form), "param_changed",
			  G_CALLBACK (work_param_changed_cb), form);


	/* tooltips */
	form->priv->tooltips = gtk_tooltips_new ();
}

/*
 * updates all the parameters in form->priv->core->work_context to the values of form->priv->core->data_rs
 * pointed by the form->priv->cursor cursor
 */
static void
update_simple_form (MgWorkForm *form)
{
	GSList *list = form->priv->core->nodes;
	const GdaValue *value;

	form->priv->intern_form_changes = TRUE;
	if (form->priv->basic_form_mode == SIMPLE_FORM_INSERT) {
		/* INSERT mode */
		if (list)
			gtk_notebook_set_current_page (GTK_NOTEBOOK (form->priv->notebook), PAGE_FORM);
		else
			gtk_notebook_set_current_page (GTK_NOTEBOOK (form->priv->notebook), PAGE_NO_DATA);

		while (list) {
			MgParameter *param;

			g_object_get (G_OBJECT (MG_WORK_CORE_NODE (list->data)->param), "full_bind", &param, NULL);
			if (! param || ! g_slist_find (form->priv->core->args_context->parameters, param)) {
				value = mg_context_get_param_default_value (form->priv->core->work_context,
									    MG_WORK_CORE_NODE (list->data)->param);
				mg_parameter_set_value (MG_WORK_CORE_NODE (list->data)->param, value);
			}

			list = g_slist_next (list);
		}
		mg_form_set_entries_default (MG_FORM (form->priv->basic_form));
		mg_form_set_entries_auto_default (MG_FORM (form->priv->basic_form), TRUE);
	}
	else {
		/* UPDATE mode */
		mg_form_set_entries_auto_default (MG_FORM (form->priv->basic_form), FALSE);
		if (!form->priv->core->data_rs ||
		    (form->priv->core->data_rs && (mg_resultset_get_nbtuples (form->priv->core->data_rs) == 0))) {
			/* no data to be displayed */
			gtk_notebook_set_current_page (GTK_NOTEBOOK (form->priv->notebook), PAGE_NO_DATA);
			while (list) {
				mg_parameter_declare_invalid (MG_WORK_CORE_NODE (list->data)->param);
				list = g_slist_next (list);
			}
		}
		
		if (list) {
			gtk_notebook_set_current_page (GTK_NOTEBOOK (form->priv->notebook), PAGE_FORM);
			while (list) {
				value = mg_resultset_get_gdavalue (form->priv->core->data_rs, form->priv->cursor, 
								   MG_WORK_CORE_NODE (list->data)->position);
				mg_parameter_set_value (MG_WORK_CORE_NODE (list->data)->param, value);
				list = g_slist_next (list);
			}

			/* Tell the real form that the parameters, as they are now set, are the new default values */
			mg_form_set_current_as_orig (MG_FORM (form->priv->basic_form));
		}
	}
	form->priv->intern_form_changes = FALSE;
}




/*
 *
 * Modification buttons (Commit changes, Reset form, New entry, Delete)
 *
 */
static void action_new_cb (GtkAction *action, MgWorkForm *form);
static void action_delete_cb (GtkAction *action, MgWorkForm *form);
static void action_commit_cb (GtkAction *action, MgWorkForm *form);
static void action_reset_cb (GtkAction *action, MgWorkForm *form);
static void action_first_record_cb (GtkAction *action, MgWorkForm *form);
static void action_prev_record_cb (GtkAction *action, MgWorkForm *form);
static void action_next_record_cb (GtkAction *action, MgWorkForm *form);
static void action_last_record_cb (GtkAction *action, MgWorkForm *form);

static GtkActionEntry ui_actions[] = {
	{ "WorkWidgetNew", GTK_STOCK_NEW, "_New", NULL, "Create a new data entry", G_CALLBACK (action_new_cb)},
	{ "WorkWidgetDelete", GTK_STOCK_DELETE, "_Delete", NULL, "Delete the selected entry", G_CALLBACK (action_delete_cb)},
	{ "WorkWidgetCommit", GTK_STOCK_SAVE, "_Commit", NULL, "Commit the latest changes", G_CALLBACK (action_commit_cb)},
	{ "WorkWidgetReset", GTK_STOCK_REFRESH, "_Reset", NULL, "Reset the data", G_CALLBACK (action_reset_cb)},
	{ "WorkWidgetFirstRecord", GTK_STOCK_GOTO_FIRST, "_First record", NULL, "Go to first record of records", 
	  G_CALLBACK (action_first_record_cb)},
	{ "WorkWidgetLastRecord", GTK_STOCK_GOTO_LAST, "_Last record", NULL, "Go to last record of records", 
	  G_CALLBACK (action_last_record_cb)},
	{ "WorkWidgetPrevRecord", GTK_STOCK_GO_BACK, "_Previous record", NULL, "Go to previous record of records", 
	  G_CALLBACK (action_prev_record_cb)},
	{ "WorkWidgetNextRecord", GTK_STOCK_GO_FORWARD, "Ne_xt record", NULL, "Go to next record of records",
	  G_CALLBACK (action_next_record_cb)}
};

static const gchar *ui_actions_info =
"<ui>"
"  <toolbar  name='ToolBar'>"
"    <toolitem action='WorkWidgetNew'/>"
"    <toolitem action='WorkWidgetDelete'/>"
"    <toolitem action='WorkWidgetCommit'/>"
"    <toolitem action='WorkWidgetReset'/>"
"    <separator/>"
"    <toolitem action='WorkWidgetFirstRecord'/>"
"    <toolitem action='WorkWidgetPrevRecord'/>"
"    <toolitem action='WorkWidgetNextRecord'/>"
"    <toolitem action='WorkWidgetLastRecord'/>"
"  </toolbar>"
"</ui>";

static void actions_arrows_value_changed_cb (GtkRange *range, MgWorkForm *form);
static GtkWidget *
modif_buttons_make (MgWorkForm *form)
{
	GtkActionGroup *actions;
	GtkUIManager *ui;
	GtkWidget *hbox, *wid;

	hbox = gtk_hbox_new (FALSE, 0);
	
	/* action buttons */
	actions = gtk_action_group_new ("Actions");
	form->priv->actions_group = actions;

	gtk_action_group_add_actions (actions, ui_actions, G_N_ELEMENTS (ui_actions), form);

	ui = gtk_ui_manager_new ();
	gtk_ui_manager_insert_action_group (ui, actions, 0);
	gtk_ui_manager_add_ui_from_string (ui, ui_actions_info, -1, NULL);
	form->priv->uimanager = ui;
	form->priv->modif_all = gtk_ui_manager_get_widget (ui, "/ToolBar");
	gtk_widget_show (form->priv->modif_all);
	gtk_box_pack_start (GTK_BOX (hbox), form->priv->modif_all, TRUE, TRUE, 0);

	/* Nav Scale */
	wid = gtk_hscale_new_with_range (0, 1, 1);
	gtk_range_set_update_policy (GTK_RANGE (wid), GTK_UPDATE_DELAYED);
	gtk_scale_set_draw_value (GTK_SCALE (wid), TRUE);
	gtk_scale_set_digits (GTK_SCALE (wid), 0);
	gtk_box_pack_start (GTK_BOX (hbox), wid, TRUE, TRUE, 2);
	gtk_widget_show (wid);
	gtk_widget_set_sensitive (wid, FALSE);
	form->priv->nav_scale = wid;
	g_signal_connect (G_OBJECT (wid), "value_changed",
			  G_CALLBACK (actions_arrows_value_changed_cb), form);

	/* rows counter */
	wid = gtk_label_new ("? / ?");
	gtk_widget_show (wid);
	form->priv->nav_current = wid;
	gtk_box_pack_start (GTK_BOX (hbox), wid, FALSE, FALSE, 2);

	return hbox;
}

static void
action_new_cb (GtkAction *action, MgWorkForm *form)
{
	modif_actions_real_do (form, 'i');
}

static void
action_delete_cb (GtkAction *action, MgWorkForm *form)
{
	modif_actions_real_do (form, 'd');
}

static void
action_commit_cb (GtkAction *action, MgWorkForm *form)
{
	modif_actions_real_do (form, 'c');
}

static void
action_reset_cb (GtkAction *action, MgWorkForm *form)
{
	modif_actions_real_do (form, 'r');
}

static void
action_first_record_cb (GtkAction *action, MgWorkForm *form)
{
	arrow_actions_real_do (form, -2);
}

static void
action_prev_record_cb (GtkAction *action, MgWorkForm *form)
{
	arrow_actions_real_do (form, -1);
}

static void
action_next_record_cb (GtkAction *action, MgWorkForm *form)
{
	arrow_actions_real_do (form, 1);
}

static void
action_last_record_cb (GtkAction *action, MgWorkForm *form)
{
	arrow_actions_real_do (form, 2);
}

static void
actions_arrows_value_changed_cb (GtkRange *range, MgWorkForm *form)
{
	gint value = gtk_range_get_value (GTK_RANGE (form->priv->nav_scale));
	if ((value >= 1) &&
	    (value <= mg_resultset_get_nbtuples (form->priv->core->data_rs)))
		form->priv->cursor = value - 1;

	/* updates */
	update_simple_form (form);
	modif_buttons_update (form);
}

static void
modif_buttons_update (MgWorkForm *form)
{
	gint nrows = 0;
	gboolean changed;
	gboolean valid;
	GtkAction *action;
	gboolean can_modif;

	if (! form->priv->basic_form)
		return;

	/* action buttons */
	changed = mg_form_has_been_changed (MG_FORM (form->priv->basic_form));
	can_modif = form->priv->core->query_insert ? TRUE : FALSE;

	/* assert the validity of the INSERT query (by construction the validity is the same as for the
	 * UPDATE query) */ 
	if (form->priv->core->query_insert)
		valid = mg_renderer_is_valid (MG_RENDERER (form->priv->core->query_insert), 
					      form->priv->core->work_context, NULL);
	else
		valid = FALSE;

	if (form->priv->core->data_rs)
		nrows = mg_resultset_get_nbtuples (form->priv->core->data_rs);

	action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetCommit");
	g_object_set (G_OBJECT (action), "sensitive", changed && valid ? TRUE : FALSE, NULL);

	action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetReset");
	g_object_set (G_OBJECT (action), "sensitive", TRUE, NULL);

	action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetNew");
	g_object_set (G_OBJECT (action), "sensitive", 
		      can_modif && (form->priv->basic_form_mode != SIMPLE_FORM_INSERT) ? TRUE : FALSE, NULL);

	action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetDelete");
	g_object_set (G_OBJECT (action), "sensitive", 
		      can_modif && 
		      form->priv->core->data_rs && (nrows > 0) && 
		      (form->priv->basic_form_mode == SIMPLE_FORM_UPDATE) ? TRUE : FALSE, NULL);

	/* navigation buttons */
	/* global mode */
	if (form->priv->mode & MG_ACTION_NAVIGATION_ARROWS) {
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetFirstRecord");
		g_object_set (G_OBJECT (action), "visible", TRUE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetPrevRecord");
		g_object_set (G_OBJECT (action), "visible", TRUE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetNextRecord");
		g_object_set (G_OBJECT (action), "visible", TRUE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetLastRecord");
		g_object_set (G_OBJECT (action), "visible", TRUE, NULL);

		if (form->priv->mode & MG_ACTION_NAVIGATION_SCROLL) 
			gtk_widget_show (form->priv->nav_scale);
		else 
			gtk_widget_hide (form->priv->nav_scale);
	}
	else {
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetFirstRecord");
		g_object_set (G_OBJECT (action), "visible", FALSE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetPrevRecord");
		g_object_set (G_OBJECT (action), "visible", FALSE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetNextRecord");
		g_object_set (G_OBJECT (action), "visible", FALSE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetLastRecord");
		g_object_set (G_OBJECT (action), "visible", FALSE, NULL);

		gtk_widget_hide (form->priv->nav_scale);
		gtk_widget_hide (form->priv->nav_current);
		return; /* END HERE */
	}

	/* sensitiveness of the widgets */
	if (form->priv->basic_form_mode == SIMPLE_FORM_INSERT) {
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetFirstRecord");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetPrevRecord");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetNextRecord");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetLastRecord");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);

		gtk_widget_set_sensitive (form->priv->nav_scale, FALSE);
	}
	else {
		gint nrows = 0;
		
		if (form->priv->core->data_rs)
			nrows = mg_resultset_get_nbtuples (form->priv->core->data_rs);
		
		if (form->priv->core->data_rs && (nrows > 0)) {
			gchar *str;
			
			action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetFirstRecord");
			g_object_set (G_OBJECT (action), "sensitive", (form->priv->cursor == 0) ? FALSE : TRUE, NULL);
			action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetPrevRecord");
			g_object_set (G_OBJECT (action), "sensitive", (form->priv->cursor == 0) ? FALSE : TRUE, NULL);
			action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetNextRecord");
			g_object_set (G_OBJECT (action), "sensitive", (form->priv->cursor == nrows -1) ? FALSE : TRUE, NULL);
			action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetLastRecord");
			g_object_set (G_OBJECT (action), "sensitive", (form->priv->cursor == nrows -1) ? FALSE : TRUE, NULL);
			
			str = g_strdup_printf ("%d / %d", form->priv->cursor+1, nrows);
			gtk_label_set_text (GTK_LABEL (form->priv->nav_current), str);
			g_free (str);
			
			gtk_widget_set_sensitive (form->priv->nav_scale, nrows == 1 ? FALSE : TRUE);
			if (nrows == 1)
				gtk_range_set_range (GTK_RANGE (form->priv->nav_scale), 1, 2);
			else
				gtk_range_set_range (GTK_RANGE (form->priv->nav_scale), 1, nrows);
			
			if (gtk_range_get_value (GTK_RANGE (form->priv->nav_scale)) != form->priv->cursor+1)
				gtk_range_set_value (GTK_RANGE (form->priv->nav_scale), form->priv->cursor+1);
		}
		else {
			action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetFirstRecord");
			g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
			action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetPrevRecord");
			g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
			action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetNextRecord");
			g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
			action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/WorkWidgetLastRecord");
			g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
			gtk_label_set_text (GTK_LABEL (form->priv->nav_current), "? / ?");
			gtk_widget_set_sensitive (form->priv->nav_scale, FALSE);
			gtk_range_set_range (GTK_RANGE (form->priv->nav_scale), 0, 1);
		}
	}	
}

static void
modif_actions_real_do (MgWorkForm *form, gchar action)
{
	MgQuery *query = NULL;
	guint initial_mode = form->priv->basic_form_mode;

	switch (action) {
	case 'c':
		if (form->priv->basic_form_mode == SIMPLE_FORM_INSERT)
			query = form->priv->core->query_insert;
		else
			query = form->priv->core->query_update;
		form->priv->basic_form_mode = SIMPLE_FORM_UPDATE;
		break;
	case 'r':
		form->priv->intern_form_changes = TRUE;
		form->priv->basic_form_mode = SIMPLE_FORM_UPDATE;
		mg_form_reset (MG_FORM (form->priv->basic_form));
		form->priv->intern_form_changes = FALSE;
		break;
	case 'i':
		form->priv->basic_form_mode = SIMPLE_FORM_INSERT;
		break;
	case 'd':
		query = form->priv->core->query_delete;
		form->priv->basic_form_mode = SIMPLE_FORM_UPDATE;
		break;
	default:
		g_assert_not_reached ();
	}

	if (query) {
		gboolean user_cancelled, query_error;
		GtkWidget *parent;
				
		parent = gtk_widget_get_parent (GTK_WIDGET (form));
		while (parent && !GTK_IS_WINDOW (parent)) 
			parent = gtk_widget_get_parent (parent);

		if (mg_util_query_execute_modif (query, form->priv->core->work_context,
						 form->priv->mode & MG_ACTION_ASK_CONFIRM_INSERT,
						 form->priv->mode & MG_ACTION_ASK_CONFIRM_UPDATE,
						 form->priv->mode & MG_ACTION_ASK_CONFIRM_DELETE,
						 parent, &user_cancelled, &query_error))
			arg_param_changed_cb (NULL, form); /* reload the form's data */ 
		else {
			if ((query_error && !user_cancelled) ||
			    (!query_error && user_cancelled))
				form->priv->basic_form_mode = initial_mode;
		}
	}
	else
		if (form->priv->basic_form_mode != initial_mode)
			update_simple_form (form);

	modif_buttons_update (form);
}


static void
arrow_actions_real_do (MgWorkForm *form, gint movement)
{
	if (form->priv->mode & MG_ACTION_MODIF_AUTO_COMMIT) {
		/* see if some data have been modified and need to be written to the DBMS */
		if (mg_form_has_been_changed (MG_FORM (form->priv->basic_form))) 
			/* simulate a clicked of 'SAVE' button */
			modif_actions_real_do (form, 'c');
	}
	
	switch (movement) {
	case -2:
		form->priv->cursor = 0;
		break;
	case -1:
		if (form->priv->cursor > 0)
			form->priv->cursor--;
		break;
	case 1:
		if (form->priv->core->data_rs &&
		    (form->priv->cursor < mg_resultset_get_nbtuples (form->priv->core->data_rs) - 1 ))
			form->priv->cursor++;
		break;
	case 2:
		if (form->priv->core->data_rs)
			form->priv->cursor = mg_resultset_get_nbtuples (form->priv->core->data_rs) - 1;
		break;
	default:
		g_assert_not_reached ();
	}

	/* updates */
	update_simple_form (form);
	modif_buttons_update (form);
}



/*
 * MgWorkWidget interface implementation
 */
static void
mg_work_form_run (MgWorkWidget *iface, guint mode)
{
	GSList *list;
	MgWorkForm *form;

	g_return_if_fail (iface && IS_MG_WORK_FORM (iface));
	form = MG_WORK_FORM (iface);
	g_return_if_fail (form->priv);
	g_return_if_fail (form->priv->core->query_select);
	
	/* hiding some parameters */
	list = form->priv->core->no_show_params;
	while (list) {
		mg_form_entry_show (MG_FORM (form->priv->basic_form), 
				    MG_PARAMETER (list->data), FALSE);
		list = g_slist_next (list);
	}

	/* REM : we don't check for missing parameters: the user must do it himself
	 * using the mg_work_widget_get_exec_context() function and mg_context_is_valid() */

	/*
	 * Signals connecting
	 */
	if (form->priv->core->args_context)
		g_signal_connect (G_OBJECT (form->priv->core->args_context), "changed",
				  G_CALLBACK (arg_param_changed_cb), form);
	form->priv->has_run = TRUE;

	/*
	 * Actual start
	 */
	if (mode)
		form->priv->mode = mode;
	arg_param_changed_cb (NULL, form);
}


static void
mg_work_form_set_mode (MgWorkWidget *iface, guint mode)
{
	MgWorkForm *form;

	g_return_if_fail (iface && IS_MG_WORK_FORM (iface));
	form = MG_WORK_FORM (iface);
	g_return_if_fail (form->priv);

	form->priv->mode = mode;

	/* updating the various possible actions */
	modif_buttons_update (form);
}

void
mg_work_form_set_entry_editable (MgWorkWidget *iface, MgQfield *field, gboolean editable)
{
	MgWorkForm *form;

	g_return_if_fail (iface && IS_MG_WORK_FORM (iface));
	form = MG_WORK_FORM (iface);
	g_return_if_fail (form->priv);

	TO_IMPLEMENT;
	/* What needs to be done:
	 * - create a mg_form_set_entry_editable() function for MgForm, and call it
	 * - in the MgDataEntry, add a MG_DATA_ENTRY_EDITABLE property which defaults to TRUE.
	 * - imtplement the mg_form_set_entry_editable() in the same way as mg_form_set_entries_default()
	 *   by setting that new property
	 * - implement the new property in MgEntryWrapper and MgEntryCombo.
	 */
}

static void
mg_work_form_show_entry_actions (MgWorkWidget *iface, MgQfield *field, gboolean show_actions)
{
	MgWorkForm *form;
	
	g_return_if_fail (iface && IS_MG_WORK_FORM (iface));
	form = MG_WORK_FORM (iface);
	g_return_if_fail (form->priv);
	
	/* REM: don't take care of the @field argument */
	mg_form_show_entries_actions (MG_FORM (form->priv->basic_form), show_actions);
}

static void
mg_work_form_show_global_actions (MgWorkWidget *iface, gboolean show_actions)
{
	MgWorkForm *form;

	g_return_if_fail (iface && IS_MG_WORK_FORM (iface));
	form = MG_WORK_FORM (iface);
	g_return_if_fail (form->priv);

	if (show_actions)
		gtk_widget_show (form->priv->modif_all);
	else
		gtk_widget_hide (form->priv->modif_all);
}

static MgParameter *
mg_work_form_get_param_for_field (MgWorkWidget *iface, MgQfield *field, const gchar *field_name, gboolean in_exec_context)
{
	MgWorkForm *form;
	MgParameter *param = NULL;

	g_return_val_if_fail (iface && IS_MG_WORK_FORM (iface), NULL);
	form = MG_WORK_FORM (iface);
	g_return_val_if_fail (form->priv, NULL);
	g_return_val_if_fail (field || (field_name && *field_name), NULL);
	
	if (field) {
		/* use the 'field' argument */
		g_return_val_if_fail (field && IS_MG_QFIELD (field), NULL);
		param = mg_work_core_find_param (form->priv->core, field, in_exec_context);
	}
	else {
		/* use the 'field_name' argument */
		MgField *f = mg_entity_get_field_by_name (MG_ENTITY (form->priv->core->query_select), field_name);
		if (f)
			param = mg_work_core_find_param (form->priv->core, MG_QFIELD (f), in_exec_context);	
	}

	return param;
}

static gboolean
mg_work_form_has_been_changed (MgWorkWidget *iface)
{
	MgWorkForm *form;

	g_return_val_if_fail (iface && IS_MG_WORK_FORM (iface), FALSE);
	form = MG_WORK_FORM (iface);
	g_return_val_if_fail (form->priv, FALSE);

	return mg_form_has_been_changed (MG_FORM (form->priv->basic_form));
}

static MgContext *
mg_work_form_get_exec_context (MgWorkWidget *iface)
{
	MgWorkForm *form;

	g_return_val_if_fail (iface && IS_MG_WORK_FORM (iface), NULL);
	form = MG_WORK_FORM (iface);
	g_return_val_if_fail (form->priv, NULL);
	
	return form->priv->core->args_context;
}

static GtkActionGroup *
mg_work_form_get_actions_group (MgWorkWidget *iface)
{
	MgWorkForm *form;
	
	g_return_val_if_fail (iface && IS_MG_WORK_FORM (iface), NULL);
	form = MG_WORK_FORM (iface);
	g_return_val_if_fail (form->priv, NULL);

	return form->priv->actions_group;
}
