////////////////////////////////////////////////////////////////////////
// Boilerplate code for proxying arbitrary types via arbitrary
// proxy functors.
// It is compatible with ALL types which can be proxied via an
// existing functor.
// License: Public Domain
// Author: stephan@s11n.net
////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////
// NOTE: this file does not use a conventional BLAH_H_INCLUDED guard.
// Yes, that's on purpose.
////////////////////////////////////////////////////////////////////////
// Boilerplate code for a
// proxies.
//
//
// The s11n header files are expected to have been included by the
// time this file is ever included.
//
//
// Sample usage:
//
// #define S11N_TYPE std::map<Foo,Bar>
// #define S11N_NAME "std::map<Foo,Bar>"
// #include [this file]
//
// // repeat for each type:
// #define S11N_TYPE std::map<Foo *,Bar *>
// #define S11N_NAME "std::map<Foo*,Bar*>"
// #define S11N_SERIALIZE_FUNCTOR some_functor_name
// #include [this file]
//
// If S11N_TYPE is a subtype of another Serializable,
// and should be available via that base-type's classloader
// (Yes, you normally WANT this) then define this macro:
//
//   S11N_BASE_TYPE
//
// to the type name of S11N_TYPE's BaseType base type.
//
// Not setting S11N_BASE_TYPE is functionaly identical
// to setting it to S11N_TYPE.
//
// Optionally, also define one or both of:
// #define S11N_SERIALIZE_FUNCTOR some_serializer_functor
// #define S11N_DESERIALIZE_FUNCTOR some_deserializer_functor
//
// If S11N_DESERIALIZE_FUNCTOR is defined, that is used
// as the Deserialization proxy, otherwise the first
// functor is used for both cases.
//
// If neither are set then no functor proxy is installed - the type is
// on it's own!
//
// After each include all of these macros are unset so that they may
// be immediately re-used for another registration.
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
// General notes:
//
// By s11n convention S11N_NAME should contain no spaces, even for
// template types. Thus, please use:
//
//    "std::list<std::list<foo*>>"
//
// instead of:
//
//     "std::list<std::<list<foo *> >"
//
// C++ needs a space between "> >", but s11n does not.
// That said, s11n DOESN'T CARE what class names you use, as long
// as you're consistent.
//
////////////////////////////////////////////////////////////////////////


#ifndef S11N_TYPE
#    error "S11N_TYPE is not set. Set it to the type you want to proxy before including this file!"
#endif

#ifndef S11N_NAME
#    error "S11N_NAME must be set to the string form of S11N_TYPE"
#endif

#ifndef S11N_BASE_TYPE
#    define S11N_BASE_TYPE S11N_TYPE
#endif

#ifndef S11N_SERIALIZE_FUNCTOR
#    error "S11N_SERIALIZE_FUNCTOR must be set to the name of a functor which will handle requests on behalf of S11N_TYPE."
#endif


#ifndef S11N_DESERIALIZE_FUNCTOR
#  define S11N_DESERIALIZE_FUNCTOR S11N_SERIALIZE_FUNCTOR
#endif

////////////////////////////////////////////////////////////////////////
// set up the type name registry
#define NAME_TYPE S11N_TYPE
#define TYPE_NAME S11N_NAME
#include <S11N_NS/name_type.h>
////////////////////////////////////////////////////////////////////////

namespace { // anonymous namespace is important for complex linking reasons.


#ifndef S11N_NS_PROXY_REG_CONTEXT_DEFINED
#define S11N_NS_PROXY_REG_CONTEXT_DEFINED 1
        ///////////////////////////////////////////////////////////////
        // we must not include this more than once per compilation unit
        ///////////////////////////////////////////////////////////////
        // A unique (per Context/per compilation unit) space to assign
        // a bogus value for classloader registration purposes (see
        // the classloader docs for a full description of how this
        // works).
        template <typename Context>
        struct proxy_reg_context
        {
                static bool placeholder;
        };
        template <typename Context> bool proxy_reg_context<Context>::placeholder = false;
#endif
// !S11N_NS_PROXY_REG_CONTEXT_DEFINED
////////////////////////////////////////////////////////////////////////////////

        	


        ////////////////////////////////////////////////////////////////////////
        // Register a factory with the classloader (this could be done any
	// number of ways, actually):
	bool proxy_reg_context< S11N_TYPE >::placeholder=
	        (S11N_NS::classloader_register<
                 S11N_BASE_TYPE ,
                 S11N_TYPE
                 >( ::classname< S11N_TYPE >() )
                 ,true);

#ifdef S11N_SERIALIZE_FUNCTOR
        ////////////////////////////////////////////////////////////////////////
        // give s11n de/serialize implementations for S11N_TYPE:
	template <> struct s11n_api_marshaler< S11N_TYPE >
        {
                typedef S11N_TYPE serializable_type;
                typedef S11N_SERIALIZE_FUNCTOR serialize_proxy_type;
                typedef S11N_DESERIALIZE_FUNCTOR deserialize_proxy_type;
                template <typename NodeType>
                static bool serialize( NodeType &dest, const serializable_type & src )
                {
                        dest.impl_class( ::classname< S11N_TYPE >() );
                        //dest.impl_class( ::clname::classname( &src ) );
                        return serialize_proxy_type()( dest, src );
                }

                template <typename NodeType>
                static bool deserialize( const NodeType & src, serializable_type & dest ) 
                {
                        return deserialize_proxy_type()( src, dest );
                }

        }; // struct s11n_api_marshaler< S11N_TYPE >


#endif // S11N_SERIALIZE_FUNCTOR
} // anon namespace 

////////////////////////////////////////////////////////////////////////////////
// end proxy code for [S11N_TYPE]
////////////////////////////////////////////////////////////////////////////////

#undef S11N_TYPE
#undef S11N_NAME

#ifdef S11N_BASE_TYPE
#  undef S11N_BASE_TYPE
#endif

#ifdef S11N_SERIALIZE_FUNCTOR
#    undef S11N_SERIALIZE_FUNCTOR
#endif

#ifdef S11N_DESERIALIZE_FUNCTOR
#    undef S11N_DESERIALIZE_FUNCTOR
#endif

