/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  menuitem.cc - GtkMenuItem C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "menuitem.h"
#include "private/menuitem_p.h"
#include "menu.h"
#include "accellabel.h"
#include "box.h"
#include "image.h"

using namespace Inti;

/*  Gtk::MenuItem
 */

Gtk::MenuItem::MenuItem(GtkMenuItem *menu_item, bool reference)
: Item((GtkItem*)menu_item, reference)
{
}

Gtk::MenuItem::MenuItem()
: Item((GtkItem*)MenuItemClass::create())
{
}

Gtk::MenuItem::MenuItem(const String& label, bool use_underline)
: Item((GtkItem*)MenuItemClass::create())
{
	add_label(label, use_underline);
}

Gtk::MenuItem::MenuItem(const String& label, Menu& submenu, bool use_underline)
: Item((GtkItem*)MenuItemClass::create())
{
	add_label(label, use_underline);
	set_submenu(submenu);
}

Gtk::MenuItem::~MenuItem()
{
}

GtkMenuItemClass*
Gtk::MenuItem::gtk_menu_item_class() const
{
	return get_class<GtkMenuItemClass>();
}

Gtk::MenuItem::operator GtkMenuItem* () const 
{
	return this ? gtk_menu_item() : 0; 
}

void
Gtk::MenuItem::add_label(const String& label, bool use_underline)
{
	AccelLabel *accel_label = new AccelLabel(label, use_underline);
	accel_label->set_alignment(0.0, 0.5);
	add(*accel_label);
	accel_label->set_accel_widget(*this);
	accel_label->show();
}

bool
Gtk::MenuItem::has_submenu() const
{
	return gtk_menu_item()->submenu;
}

bool
Gtk::MenuItem::get_right_justified() const
{
	return gtk_menu_item_get_right_justified(gtk_menu_item());
}

Gtk::Menu*
Gtk::MenuItem::get_submenu() const
{
	return gtk_menu_item()->submenu ? G::Object::wrap<Menu>((GtkMenu*)gtk_menu_item()->submenu) : 0;
}

void 
Gtk::MenuItem::set_label(const String& label, bool use_underline)
{
	std::vector<Widget*> children;
	if (get_children(children))
	{
		int count = children.size();
		for (int i = 0; i < count; i++)
		{
			AccelLabel *accel_label = dynamic_cast<AccelLabel*>(children[i]);
			if (accel_label)
			{
				accel_label->set_text(label);
				accel_label->set_use_underline(use_underline);
				break;
			}
		}
	}
	else
		add_label(label, use_underline);
}

void
Gtk::MenuItem::set_submenu(Menu& submenu)
{
	gtk_menu_item_set_submenu(gtk_menu_item(), submenu.gtk_widget());
}

void
Gtk::MenuItem::remove_submenu()
{
	gtk_menu_item_remove_submenu(gtk_menu_item());
}

void
Gtk::MenuItem::activate()
{
	gtk_menu_item_activate(gtk_menu_item());
}

void
Gtk::MenuItem::toggle_size_request(int *requisition)
{
	gtk_menu_item_toggle_size_request(gtk_menu_item(), requisition);
}

void
Gtk::MenuItem::toggle_size_allocate(int allocation)
{
	gtk_menu_item_toggle_size_allocate(gtk_menu_item(), allocation);
}

void
Gtk::MenuItem::set_right_justified(bool right_justified)
{
	gtk_menu_item_set_right_justified(gtk_menu_item(), right_justified);
}

void
Gtk::MenuItem::set_accel_path(const char *accel_path)
{
	gtk_menu_item_set_accel_path(gtk_menu_item(), accel_path);
}

/*  Gtk::MenuItemClass
 */

void
Gtk::MenuItemClass::init(GtkMenuItemClass *g_class)
{
 	ItemClass::init((GtkItemClass*)g_class);
	g_class->activate = &activate_proxy;
	g_class->activate_item = &activate_item_proxy;
	g_class->toggle_size_request = &toggle_size_request_proxy;
	g_class->toggle_size_allocate = &toggle_size_allocate_proxy;
}

GType
Gtk::MenuItemClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_MENU_ITEM, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::MenuItemClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gtk::MenuItemClass::activate_proxy(GtkMenuItem *menu_item)
{
	MenuItem *tmp_menu_item = G::Object::pointer<MenuItem>(menu_item);
	if (tmp_menu_item)
		tmp_menu_item->on_activate();
	else
	{
		GtkMenuItemClass *g_class = G::TypeInstance::class_peek_parent<GtkMenuItemClass>(GTK_MENU_ITEM_GET_CLASS(menu_item));
		if (g_class->activate)
			g_class->activate(menu_item);
	}
}

void
Gtk::MenuItemClass::activate_item_proxy(GtkMenuItem *menu_item)
{
	MenuItem *tmp_menu_item = G::Object::pointer<MenuItem>(menu_item);
	if (tmp_menu_item)
		tmp_menu_item->on_activate_item();
	else
	{
		GtkMenuItemClass *g_class = G::TypeInstance::class_peek_parent<GtkMenuItemClass>(GTK_MENU_ITEM_GET_CLASS(menu_item));
		if (g_class->activate_item)
			g_class->activate_item(menu_item);
	}
}

void
Gtk::MenuItemClass::toggle_size_request_proxy(GtkMenuItem *menu_item, gint *requisition)
{
	MenuItem *tmp_menu_item = G::Object::pointer<MenuItem>(menu_item);
	if (tmp_menu_item)
		tmp_menu_item->on_toggle_size_request(requisition);
	else
	{
		GtkMenuItemClass *g_class = G::TypeInstance::class_peek_parent<GtkMenuItemClass>(GTK_MENU_ITEM_GET_CLASS(menu_item));
		if (g_class->toggle_size_request)
			g_class->toggle_size_request(menu_item, requisition);
	}
}

void
Gtk::MenuItemClass::toggle_size_allocate_proxy(GtkMenuItem *menu_item, gint allocation)
{
	MenuItem *tmp_menu_item = G::Object::pointer<MenuItem>(menu_item);
	if (tmp_menu_item)
		tmp_menu_item->on_toggle_size_allocate(allocation);
	else
	{
		GtkMenuItemClass *g_class = G::TypeInstance::class_peek_parent<GtkMenuItemClass>(GTK_MENU_ITEM_GET_CLASS(menu_item));
		if (g_class->toggle_size_allocate)
			g_class->toggle_size_allocate(menu_item, allocation);
	}
}

/*  Signal handlers
 */

void
Gtk::MenuItem::on_activate()
{
	GtkMenuItemClass *g_class = class_peek_parent<GtkMenuItemClass>(gtk_menu_item_class());
	if (g_class->activate)
		g_class->activate(gtk_menu_item());
}

void
Gtk::MenuItem::on_activate_item()
{
	GtkMenuItemClass *g_class = class_peek_parent<GtkMenuItemClass>(gtk_menu_item_class());
	if (g_class->activate_item)
		g_class->activate_item(gtk_menu_item());
}

void
Gtk::MenuItem::on_toggle_size_request(int *requisition)
{
	GtkMenuItemClass *g_class = class_peek_parent<GtkMenuItemClass>(gtk_menu_item_class());
	if (g_class->toggle_size_request)
		g_class->toggle_size_request(gtk_menu_item(), requisition);
}

void
Gtk::MenuItem::on_toggle_size_allocate(int allocate)
{
	GtkMenuItemClass *g_class = class_peek_parent<GtkMenuItemClass>(gtk_menu_item_class());
	if (g_class->toggle_size_allocate)
		g_class->toggle_size_allocate(gtk_menu_item(), allocate);
}

/*  Signals
 */

const Gtk::MenuItem::ActivateSignalType Gtk::MenuItem::activate_signal("activate");

const Gtk::MenuItem::ActivateItemSignalType Gtk::MenuItem::activate_item_signal("activate_item");

const Gtk::MenuItem::ToggleSizeRequestSignalType Gtk::MenuItem::toggle_size_request_signal("toggle_size_request");

const Gtk::MenuItem::ToggleSizeAllocateSignalType Gtk::MenuItem::toggle_size_allocate_signal("toggle_size_allocate");

