/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/atk/image.h
//! @brief An AtkImage C++ wrapper interface.
//!
//! Provides Image, an interface implemented by components which expose image or pixmap content on-screen.

#ifndef INTI_ATK_IMAGE_H
#define INTI_ATK_IMAGE_H

#ifndef INTI_G_TYPE_H
#include <inti/glib/type.h>
#endif

#ifndef INTI_ATK_UTIL_H
#include <inti/atk/util.h>
#endif

#ifndef __ATK_IMAGE_H__
#include <atk/atkimage.h>
#endif

namespace Inti {

namespace Atk {

class ImageIface;

//! @class Image image.h inti/atk/image.h
//! @brief An AtkImage C++ wrapper class.
//!
//! Image should be implemented by Atk::Object subtypes on behalf of components which
//! display image or pixmap information on-screen, and which provide information (other
//! than just widget borders, etc.) via that image content. For instance, icons, buttons
//! with icons, toolbar elements, and image viewing panes typically should implement Image.
//!
//! Image primarily provides two types of information: coordinate information (useful for
//! screen review mode of screen readers, and for use by on-screen magnifiers), and
//! descriptive information. The descriptive information is provided for alternative, 
//! text-only presentation of the most significant information present in the image.

class Image : public virtual G::TypeInterface
{
	friend class ImageIface;

protected:
//! @name Constructors
//! @{

	virtual ~Image() = 0;
	//!< Destructor.

//! @}
//  Override these do_ methods when you want to change the default behaviour of the AtkImage.
	
	virtual void do_get_image_position(int *x, int *y, AtkCoordType coord_type);

	virtual const char* do_get_image_description();

	virtual void do_get_image_size(int *width, int *height);

	virtual bool do_set_image_description(const char *description);

public:
//! @name Accessors
//! @{

	AtkImage* atk_image() const { return (AtkImage*)instance; }
	//!< Get a pointer to the AtkImage structure.

	AtkImageIface* atk_image_iface() const;
	//!< Get a pointer to the AtkImageIface structure.

	operator AtkImage* () const;
	//!< Conversion operator; safely converts an Image to an AtkImage pointer.

	String get_image_description() const;
	//!< Get a textual description of the image.
	//!< @return A string representing the image description.

	void get_image_size(int *height, int *width) const;
	//!< Get the width and height in pixels for the specified image. 
	//!< @param width The address of an int filled with the image width, or null.
	//!< @param height The address of an int filled with the image height, or null.
	//!<
	//!< <BR>The values of width and height are returned as -1 if the values
	//!< cannot be obtained.
	
	void get_image_position(int *x, int *y, CoordType coord_type) const;
	//!< Gets the position of the image in the form of a point specifying
	//!< the images top-left corner. 
	//!< @param x The address of an int to put the x coordinate position, or null.
	//!< @param y The address of an int to put the y coordinate position, or null.
	//!< @param coord_type Specifies whether the coordinates are relative to the screen
	//!< or to the components top level window.
	//!<
	//!< <BR>The values of <EM>x</EM> and <EM>y</EM> are returned as -1 if
	//!< the values cannot be obtained.

//! @}
//! @name Methods
//! @{

	bool set_image_description(const String& description);
	//!< Sets the textual description for the image.
	//!< @param description A string description to set for image.
	//!< @return <EM>true</EM> if the description is set, <EM>false</EM> otherwise.

//! @}
};

} // namespace Atk

} // namespace Inti

#endif // INTI_ATK_IMAGE_H

