/*
 * $Id: cap_file.c,v 1.1.1.1.4.3 2001/01/16 07:38:25 agmorgan Exp $
 *
 * Copyright (c) 1997-2000 Andrew G Morgan <morgan@linux.kernel.org>
 *
 * This file deals with setting capabilities on files.
 */

#include "libcap.h"

#if (_LINUX_CAPABILITY_VERSION == 0x19980330)

/* This kernel does not support filesystem capabilities later ones do */

cap_t cap_get_fd(int fildes)
{
    _cap_debug("compilation does not support filecaps");
    errno = ENOSYS;
    return NULL;
}

cap_t cap_get_file(const char *filename)
{
    _cap_debug("compilation does not support filecaps");
    errno = ENOSYS;
    return NULL;
}

int cap_set_fd(int fildes, cap_t cap_d)
{
    _cap_debug("compilation does not support filecaps");
    errno = ENOSYS;
    return -1;
}

int cap_set_file(const char *filename, cap_t cap_d)
{
    _cap_debug("compilation does not support filecaps");
    errno = ENOSYS;
    return -1;
}

#else /* ie. (_LINUX_CAPABILITY_VERSION != 0x19980330) */

/*
 * attempt to have defined behavior when running with an older kernel
 */

static int _libcap_capfile(cap_t cap_d)
{
    cap_d->head.version = _libcap_kernel_version;

    _cap_debug("version: %x, features: %x",
	       cap_d->head.version, cap_d->features);

    if (cap_d->features & CAP_FEATURE_FILE) {
	_cap_debug("ok");
	return 0;
    } else {
	_cap_debug("library compiled with a kernel not supporting filecaps");
	errno = ENOSYS;
	return -1;
    }
}

/*
 * Get the capabilities of an open file, as specified by its file
 * descriptor.
 */

cap_t cap_get_fd(int fildes)
{
    int retval;
    cap_t result;

    /* allocate a new capability set */
    result = cap_init();
    if (result == NULL) {
	_cap_debug("failed to allocate a result");
	return NULL;
    }

    if (_libcap_capfile(result)) {
	_cap_debug("running kernel has no support for filecaps");
	goto fail;
    }

    result->head.type = CAP_USERHEADER_FILEDES;
    result->head.u.fd = fildes;

    retval = capget(&result->head, &result->set);

    result->head.type = 0;
    result->head.u.pid = 0;

    if (!retval) {
	if (_libcap_kernel_features & CAP_FEATURE_TO_32) {
	    if (_libcap_n1_to_n2(result, 1, __CAP_BLKS)) {
		goto fail;
	    }
	}
	return result;
    }

fail:
    cap_free(result);
    result = NULL;

    return NULL;
}

/*
 * Set the capabilities on a named file.
 */

cap_t cap_get_file(const char *filename)
{
    int retval;
    cap_t result;

    /* allocate a new capability set */
    result = cap_init();
    if (result == NULL) {
	_cap_debug("failed to allocate a result");
	return NULL;
    }

    if (_libcap_capfile(result)) {
	_cap_debug("running kernel has no support for filecaps");
	goto fail;
    }

    result->head.type = CAP_USERHEADER_FILE;
    result->head.u.file = filename;

    retval = capget(&result->head, &result->set);

    result->head.type = 0;
    result->head.u.pid = 0;

    if (!retval) {
	if (_libcap_kernel_features & CAP_FEATURE_TO_32) {
	    if (_libcap_n1_to_n2(result, 1, __CAP_BLKS)) {
		goto fail;
	    }
	}
	return result;
    }

fail:
    cap_free(result);
    result = NULL;

    return NULL;
}

/*
 * Set the capabilities of an open file, as specified by its file
 * descriptor.
 */

int cap_set_fd(int fildes, cap_t cap_d)
{
    int retval;

    if (!good_cap_t(cap_d)) {
	errno = EINVAL;
	return -1;
    }

    if (_libcap_capfile(cap_d)) {
	_cap_debug("running kernel has no support for filecaps");
	return -1;
    }

    cap_d->head.type = CAP_USERHEADER_FILEDES;
    cap_d->head.u.fd = fildes;

    if (_libcap_kernel_features & CAP_FEATURE_TO_32) {
	retval = _libcap_n1_to_n2(cap_d, __CAP_BLKS, 1);
    }
    
    retval = capset(&cap_d->head, &cap_d->set);
    _cap_debug("capset returned %d", retval);

    if (_libcap_kernel_features & CAP_FEATURE_TO_32) {
	(void) _libcap_n1_to_n2(cap_d, 1, __CAP_BLKS);
    }

    cap_d->head.type = 0;
    cap_d->head.u.pid = 0;

    return retval;
}

/*
 * Set the capabilities of a named file.
 */

int cap_set_file(const char *filename, cap_t cap_d)
{
    int retval;

    if (!good_cap_t(cap_d)) {
	errno = EINVAL;
	return -1;
    }

    if (_libcap_capfile(cap_d)) {
	_cap_debug("running kernel has no support for filecaps");
	return -1;
    }

    cap_d->head.type = CAP_USERHEADER_FILE;
    cap_d->head.u.file = filename;

    if (_libcap_kernel_features & CAP_FEATURE_TO_32) {
	retval = _libcap_n1_to_n2(cap_d, __CAP_BLKS, 1);
    }
    
    _cap_debug("capset version 0x%x", cap_d->head.version);
    _cap_debug("capset type %d", cap_d->head.type);
    retval = capset(&cap_d->head, &cap_d->set);
    _cap_debug("capset returned %d", retval);

    if (_libcap_kernel_features & CAP_FEATURE_TO_32) {
	(void) _libcap_n1_to_n2(cap_d, 1, __CAP_BLKS);
    }

    cap_d->head.type = 0;
    cap_d->head.u.pid = 0;

    return retval;
}

#endif /* (_LINUX_CAPABILITY_VERSION == 0x19980330) */
