package Lire::GroupField;

use strict;

use Carp;

use Lire::DataTypes qw/ check_xml_name is_quantity_type is_time_type /;
use Lire::I18N qw/ dgettext dgettext_para /;
use Lire::ReportSpec;
use Lire::Utils qw/ check_param check_object_param /;

=pod

=head1 NAME

Lire::GroupField - Class that implements the field element in report specifications.

=head1 SYNOPSIS

    use Lire::GroupField;

=head1 DESCRIPTION

This module is used to represent the field elements that appears in
the group aggregator element in report specifications.

=head1 CONSTRUCTOR

=head2 new( %params )

Creates a new Lire::GroupField object. It takes several parameters
similar to Lire::ReportOperator.

=over

=item name

The DLF field's name that this GroupField represents. This shouldn't
be quantity or time field (other aggregators should be used for
these).

=item report_spec

The report specification in which this the group field is added.

=item label

The column's label that should be used for this field.

=back

=cut

sub new {
    my $proto = shift;
    my $class = ref( $proto) || $proto;

    my %params = @_;
    check_param( $params{'name'}, 'name', \&check_xml_name,
                 "invalid 'name' parameter" );
    check_object_param( $params{'report_spec'}, 'report_spec',
                        "Lire::ReportSpec" );
    croak "'$params{'name'}' field name is not defined in the specification's schemas"
	unless $params{'report_spec'}->has_field( $params{'name'} );

    my $type = $params{'report_spec'}->field( $params{'name'} )->type();
    croak "'$params{'name'}' field is of type '$type'"
      . ". Use Lire::Rangegroup for this type\n"
	if ( is_quantity_type( $type ) );
    croak "'$params{'name'}' field is of type '$type'"
      . ". Use Lire::Timeslot or Lire::Timegroup for this type\n"
	if ( is_time_type( $type ) );

    bless { 'name'	=> $params{'name'},
	    'report_spec' => $params{'report_spec'},
            'i18n_domain' => $params{'i18n_domain'} || 'lire',
	    'label'	=> $params{'label'},
	  }, $class;
}

=pod

=head2 name()

Returns the DLF field name which we represent.

=cut

sub name {
    $_[0]{'name'};
}

=pod

=head2 label( [$new_label] )

Returns the label that should be used for the column we generate in
the report. Default is to use the DLF field's label.

It can be changed by passing a $new_label parameter.

=cut

sub label {
    my ( $self, $label ) = @_;

    $self->{'label'} = $label
      if @_ == 2;

    return dgettext( $self->{'i18n_domain'},
                     $self->{'label'}
                     ? $self->{'label'}
                     : $self->field()->label() );
}

=pod

=head2 field()

Returns the DLF field which we represents as Lire::Field object.

=cut

sub field {
    return $_[0]{'report_spec'}->field( $_[0]{'name'} );
}

# ------------------------------------------------------------------------
# Method print( $fh, $pfx )
#
# The Lire::Group implementation delegates writing the field element
# to us.
sub print {
    my ($self, $fh, $pfx ) = @_;
    $pfx ||= 0;
    $fh  ||= \*STDOUT;

    print $fh ' ' x $pfx;
    print $fh qq{<lire:field name="$self->{'name'}"};
    print $fh qq{ label="$self->{'label'}"}
      if $self->{'label'};
    print $fh "/>\n";
}

# The Lire::ReportOperator create_categorical_info method is delegated
# to us by Lire::Group implementation.
sub create_categorical_info {
    my ( $self, $info ) = @_;

    $info->create_column_info( $self->name(), 'categorical',
			       $self->field()->type(), $self->label() );
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::Group(3pm)

=head1 AUTHORS

  Francis J. Lacoste <flacoste@logreport.org>
  Wolfgang Sourdeau <wolfgang@logreport.org>

=head1 VERSION

$Id: GroupField.pm,v 1.18 2004/05/17 17:05:19 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001-2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut

