// LabPlot : PlotPolar.cc

#include <iostream>
#include <klocale.h>
#include <kdebug.h>
#include "PlotPolar.h"

using namespace std;

//! general polar Plot class
PlotPolar::PlotPolar(Worksheet *p)
	: Plot(p)
{
	title = new Label(i18n("Polar Plot"),QFont("Adobe Times",22),QColor(Qt::black));
	title->setPosition(0.4,0.02);
	
	for (int i=0;i<4;i++)
		gridenabled[i]=TRUE;
	
	axis[0].setTicsLabelFormat(DEGREE);
	axis[0].setTicsLabelPrecision(0);
}

//! Infos for project explorer
QStringList PlotPolar::Info() {
	QStringList s;
	s<<"Polar";

	s<<QString::number(position.X())+QString(" , ")+QString::number(position.Y());
	s<<QString::number(size.X())+QString(" X ")+QString::number(size.Y());
	if (transparent)
		s<<QString("yes");
	else
		s<<QString("no");
	s<<bgcolor.name();
	s<<gbgcolor.name();

	return s;
}

void PlotPolar::draw(QPainter *p, int w, int h) {
	kdDebug()<<"PlotPolar::draw() w/h : "<<w<<' '<<h<<endl;
	int xmin = (int)(w*(size.X()*p1.X()+position.X()));
	int xmax = (int)(w*(size.X()*p2.X()+position.X()));
	int ymin = (int)(h*(size.Y()*p1.Y()+position.Y()));
	int ymax = (int)(h*(size.Y()*p2.Y()+position.Y()));

	kdDebug()<<"XMIN/MXAX/YMIN/YMAX = "<<xmin<<','<<xmax<<','<<ymin<<','<<ymax<<endl;
	kdDebug()<<"p1 = "<<p1.X()<<'/'<<p1.Y()<<" p2 = "<<p2.X()<<'/'<<p2.Y()<<endl;

	if (!transparent) {
		// background color
		p->setBrush(bgcolor);
		p->setPen(Qt::NoPen);
		p->drawRect((int)(w*position.X()),(int)(h*position.Y()),(int)(w*size.X()),(int)(h*size.Y()));

		// graph background color
		p->setBrush(gbgcolor);
		p->setPen(Qt::NoPen);
	}

	kdDebug()<<"PlotPolar : title->draw() pos:"<<position.X()<<' '<<position.Y()<<endl;
	kdDebug()<<" 			size:"<<size.X()<<' '<<size.Y()<<endl;
	title->draw(p,position,size,w,h,0);

	drawCurves(p, w, h);

	if(legend.enabled())
		legend.draw(p,type,graphlist,position,size,w,h);
}

void PlotPolar::drawCurves(QPainter *p, int w, int h) {
	kdDebug()<<"PlotPolar::drawCurves()"<<endl;
	int xmin = (int)(w*(size.X()*p1.X()+position.X()));
	int xmax = (int)(w*(size.X()*p2.X()+position.X()));
	int ymin = (int)(h*(size.Y()*p1.Y()+position.Y()));
	int ymax = (int)(h*(size.Y()*p2.Y()+position.Y()));

	// used as miny=0
	//double miny = actrange[1].rMin();
	double maxy = actrange[1].rMax();

	kdDebug()<<"xmin/xmax ymin/ymax : "<<xmin<<'/'<<xmax<<' '<<ymin<<'/'<<ymax<<endl;
	kdDebug()<<"width/height : "<<w<<'/'<<h<<endl;
	
	// TODO : tic color ???
	p->setPen(Qt::black);

	int major_tics = axis[1].MajorTics();
	int minor_tics = axis[1].MinorTics();
	for (int i=0;i<major_tics;i++) {
		int x = xmin+(int)(i*(xmax-xmin)/2.0/major_tics);
		int xw = (xmax-xmin)-(int)(i*(xmax-xmin)/major_tics);
		int y = ymin+(int)(i*(ymax-ymin)/2.0/major_tics);
		int yh =(ymax-ymin)-(int)(i*(ymax-ymin)/major_tics);

		// major tics
		if (axis[1].MajorTicsEnabled()) {
			p->drawLine(x,y+yh/2-5,x,y+yh/2+5);
			p->drawLine(x+xw,y+yh/2-5,x+xw,y+yh/2+5);
			p->drawLine(x+xw/2-5,y,x+xw/2+5,y);
			p->drawLine(x+xw/2-5,y+yh,x+xw/2+5,y+yh);
		}

		// major grid r
		if (gridenabled[2])
			p->drawEllipse(x,y,xw,yh);

		for (int j=0;j<minor_tics;j++) {
			// TODO : minor tics
			// TODO ; minor grid r
			// if (gridenabled[3])
		}
	}

	// r values
	int xw = xmax-xmin, yh = ymax-ymin;
	int prec = axis[1].TicsLabelPrecision();
	TFormat atlf = axis[1].TicsLabelFormat();
	if (axis[1].enabled())  {
		for (int i=0;i<major_tics+1;i++) {
			double value = i/(double)major_tics*maxy;
		
			QString label = TicLabel(1,atlf,prec,axis[1].DateTimeFormat(),value);	

			// apply prefix & suffix
			label.prepend(axis[1].TicsLabelPrefix());
			label.append(axis[1].TicsLabelSuffix());

			// TODO : use richtext drawing for richtext strings (see Plot2D.h : "draw it")
			
			//  scale font size with plot
			QFont f = axis[1].TicsFont();
			f.setPointSize((int)(f.pointSize()*size.X()));
			p->setFont(f);
			p->drawText(xmin+(int)(xw/2+i/(double)major_tics*xw/16)+5,
				ymin+(int)(yh/2-i/(double)major_tics*yh/2),label);
		}
	}
	
	// phi values
	if (axis[0].enabled()) {
		prec = axis[0].TicsLabelPrecision();
		atlf = axis[0].TicsLabelFormat();
		for (int i=0;i<12;i++) {
			double rx = xw/2+20, ry = yh/2+20;
			double phi =30.0*i*M_PI/180.0;	

			QString label = TicLabel(1,atlf,prec,axis[0].DateTimeFormat(),phi);

			// apply prefix & suffix
			label.prepend(axis[0].TicsLabelPrefix());
			label.append(axis[0].TicsLabelSuffix());

			// TODO : use richtext drawing for richtext strings (see Plot2D.h)		
			
			//  scale font size with plot
			QFont f = axis[0].TicsFont();
			f.setPointSize((int)(f.pointSize()*size.X()));
			p->setFont(f);
			p->drawText((int)(xmin+xw/2+rx*cos(phi)-10),(int)(ymin+yh/2-ry*sin(phi)+5),label);
		}
	}

	// axes lines
	p->drawLine(xmin,(int)((ymax+ymin)/2.0),xmax,(int)((ymax+ymin)/2.0));
	p->drawLine((int)((xmax+xmin)/2.0),ymin,(int)((xmax+xmin)/2.0),ymax);

	if (gridenabled[0]) {	// phi major grid
		double rx = xw/2, ry = yh/2;
		double phi =30.0*M_PI/180.0;
		p->drawLine((int)(xmin+xw/2+rx*cos(phi)),(int)(ymin+yh/2-ry*sin(phi)),
			(int)(xmin+xw/2+rx*cos(phi+M_PI)),(int)(ymin+yh/2-ry*sin(phi+M_PI)));
		phi =60.0*M_PI/180.0;
		p->drawLine((int)(xmin+xw/2+rx*cos(phi)),(int)(ymin+yh/2-ry*sin(phi)),
			(int)(xmin+xw/2+rx*cos(phi+M_PI)),(int)(ymin+yh/2-ry*sin(phi+M_PI)));
		phi =120.0*M_PI/180.0;
		p->drawLine((int)(xmin+xw/2+rx*cos(phi)),(int)(ymin+yh/2-ry*sin(phi)),
			(int)(xmin+xw/2+rx*cos(phi+M_PI)),(int)(ymin+yh/2-ry*sin(phi+M_PI)));
		phi =150.0*M_PI/180.0;
		p->drawLine((int)(xmin+xw/2+rx*cos(phi)),(int)(ymin+yh/2-ry*sin(phi)),
			(int)(xmin+xw/2+rx*cos(phi+M_PI)),(int)(ymin+yh/2-ry*sin(phi+M_PI)));
	}
		
	// clipping rect with some space (clipoffset)
	p->setClipRect(xmin-clipoffset,ymin-clipoffset,xmax-xmin+2*clipoffset,ymax-ymin+2*clipoffset);
	
	// draw curves
	for (unsigned int nr=0; nr < graphlist->getNumber() ; nr++) {
		if(graphlist->getGraph(nr)->isShown() == FALSE)
			continue;
		
		GRAPHType s = graphlist->getStruct(nr);
		if (s == GRAPH2D) {
			Graph2D *g = graphlist->getGraph2D(nr);	

			kdDebug()<<"GRAPH2D / 2D number="<<g->Number()<<endl;

			Point *d = g->Data();
			int N = g->Number();
			Style style =  g->getStyle();
			Symbol symbol = g->getSymbol();	

			double oldx=0.0, oldy=0.0;
			for (int i=0;i<N;i++) {
				double phi = d[i].X(), r =  d[i].Y();

				int x = xmin+xw/2+(int)(r/maxy*xw/2*cos(phi));
				int y = ymin+yh/2-(int)(r/maxy*yh/2*sin(phi));

//				kdDebug()<<"phi/r = "<<phi<<' '<<r<<endl;
//				kdDebug()<<"x/y = "<<x<<' '<<y<<endl;
				symbol.draw(p,x,y);
			
				if(i>0)
					drawStyle(p,style,(int)oldx,(int)oldy,(int)x,(int)y,xmin,xmax,ymin,ymax);
				oldx=x;oldy=y;
			}
		}
	}
	p->setClipping(false);
}

void PlotPolar::saveAxes(QTextStream *t) {
	for (int i = 0; i < 2; i++)
		saveAxis(t,&axis[i],gridenabled[2*i],borderenabled[i],gridenabled[2*i+1]);
}

void PlotPolar::openAxes(QTextStream *t, int version) {
	for(int i = 0;i<2;i++)
		openAxis(t,version, &axis[i],&gridenabled[2*i],&borderenabled[i],&gridenabled[2*i+1]);
}
