/*******************************************************************************
**                        backupprofile.cpp
**                             part of
**                konserve -- A small backup application
**			 -------------------
**  copyright: (C) 2002 - 2004 by Florian Simnacher
**  email    : simnacher AT gmx DOT de
*******************************************************************************/

/*******************************************************************************
**  This program is free software; you can redistribute it and/or             **
**  modify it under the terms of the GNU General Public License               **
**  as published by the Free Software Foundation; either version 2            **
**  of the License, or (at your option) any later version.                    **
**                                                                            **
**  This program is distributed in the hope that it will be useful,           **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of            **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             **
**  GNU General Public License for more details.                              **
**                                                                            **
**  You shougld have received a copy of the GNU General Public License        **
**  along with this program; if not, write to the Free Software               **
**  Foundation, Inc., 59 Temple Place - Suite 330, Boston,                    **
**  MA  02111-1307, USA.                                                      **
*******************************************************************************/

// Qt includes
#include <qfileinfo.h>
#include <qdir.h>

// KDE includes
#include <kmessagebox.h>
#include <ktar.h>
#include <klocale.h>
#include <kdebug.h>

// My includes
#include "konservedebug.h"
#include "backupprofile.h"
#include "helper.h"

/*******************************************************************************
void __debug_print_kurl( KURL url, QString obj )
{
    kdDebug() << obj << "." << "directory(false, false): "
              << url.directory(false, false)
              << endl;
    kdDebug() << obj << "." << "fileName(false): " << url.fileName(false)
              << endl;
    kdDebug() << obj << "." << "path(): " << url.path() << endl;
    kdDebug() << obj << "." << "url(): " << url.url() << endl << endl;
}
*******************************************************************************/

////////////////////////////////////////////////////////////////////////////////
///////                  class BackupProfile                             ///////
////////////////////////////////////////////////////////////////////////////////

BackupProfile::BackupProfile( Backup*      backup,
                              QString      identifier,
                              TimeInterval interv,
                              QDateTime    last,
                              bool         active,
                              bool         readonly )
    : mIdentifier( identifier ),
      mTimeToBackup( interv ),
      mLastBackupTime( last ),
      mbIsActivated( active ),
      mbReadOnly( readonly ),
      mbFilesModified( true )
{
    TRACE();
    mpBackup = backup;
    mpTimer = new QTimer();
    QObject::connect(
        mpBackup, SIGNAL(finishedBackup(bool)),
        this,     SLOT(slotFinishedBackup(bool))
        );
}

BackupProfile::~BackupProfile()
{
    TRACE();
    Q_ASSERT( mpBackup );
    Q_ASSERT( mpTimer );

    delete mpBackup;
    delete mpTimer;
}

void BackupProfile::activate()
{
    TRACE();
    if ( mbReadOnly )
        return;
    if ( mbIsActivated )
        setupTimer();
}

void BackupProfile::restore()
{
    TRACE();
    QFileInfo q( mpBackup->getArchiveUrl().path() );
    KURL      toRestore;

    if ( ! restoreActionIsSave() )
        return;
    toRestore = getLatestBackup();

    KArchive *archive = new KTar( toRestore.path() );
    if ( ! archive->open( IO_ReadOnly ) )
    {
        KMessageBox::error( 0L, i18n( "Sorry, somethings wrong with your archive."
                                      "I cannot restore the backup" ) );
        return;
    }
    KArchiveEntry *root = const_cast<KArchiveDirectory*>( archive->directory() );
    this->extractArchiveEntry( root, QString::null );
    archive->close();
}

BackupProfile * BackupProfile::newBackupProfile(
    KURL      sourceUrl,
    KURL      backupUrl,
    QString   identifier,
    int       time,
    QString   measurement,
    bool      activ,
    QDateTime last
    )
{
    TimeInterval  t( (unsigned int) time, measurement );
    Backup *backup = new Backup(
        sourceUrl, backupUrl
        );
    return new BackupProfile( backup, identifier, t, last, activ );
}

BackupProfile *BackupProfile::newBackupProfileReadOnly(
    KURL      sourceUrl,
    KURL      backupUrl,
    QString   identifier,
    int       time,
    QString   measurement,
    bool      activ,
    QDateTime last
    )
{
    TRACE();
    TimeInterval  t( (unsigned int) time, measurement );

    Backup *backup = new Backup(
        sourceUrl, backupUrl, true
        );
    return new BackupProfile(
        backup, identifier, t, last, activ, true
        );
}

BackupProfile *BackupProfile::newBackupProfileReadOnly(
    const BackupProfile * const b
    )
{
    TRACE();
    TimeInterval  t( b->getTimeBetweenBackups() );
    Backup  *backup     = new Backup(
        b->getSourceUrl(), b->getArchiveUrl(), true
        );
    return new BackupProfile(
        backup, b->getIdentifier(), t, b->getLastTime(), b->isActivated(), true
        );
}

void BackupProfile::slotDoOneTimeBackup()
{
    TRACE();
    if ( mbReadOnly )
        return;

    doBackupIfNeeded();
}

void BackupProfile::slotFinishedBackup( bool successful )
{
    TRACE();
    mLastBackupTime = QDateTime::currentDateTime();
    if ( !successful )
        KMessageBox::error(
            0L, i18n("The backup %1 was not made successfully.\n"
                     "Konserve will try again later").arg(mIdentifier)
            );
}

void BackupProfile::slotSearchForModifiedFiles( KIO::Job* /* aJob */,
                                                const KIO::UDSEntryList& list )
{
    TRACE();
    QDateTime changedTime;
    QDateTime lastBackupTime;

    KURL lastArchiveUrl = getLatestBackup();
    kdDebug(0) << "Looking for files newer than: " << lastArchiveUrl.url() << endl;

    if (lastArchiveUrl.isEmpty())
    {
        mbFilesModified = true;
        return;
    }
    KFileItem lastBackup(
        KFileItem::Unknown, KFileItem::Unknown, lastArchiveUrl
        );
    lastBackupTime.setTime_t( lastBackup.time( KIO::UDS_MODIFICATION_TIME ) );
    kdDebug(0) << "Last backup from: " << lastBackupTime.toString() << endl;

    KIO::UDSEntryListConstIterator it = list.begin();
    for (; it != list.end(); ++it)
    {
        KFileItem f( *it, mpBackup->getSourceUrl() );

        // Only filter out "..". If "." has been modified, a backup is needed
        if ( f.text() == ".." )
            continue;
        else
        {
            changedTime.setTime_t( f.time( KIO::UDS_MODIFICATION_TIME ) );

            // Backup needed, if even one file modified after last backup
            if( changedTime > lastBackupTime )
            {
                kdDebug(0) << "Found newer file" << endl;
                mbFilesModified = true;
            }
        }
    }
}

void BackupProfile::slotFinishedSearchingModifiedFiles( KIO::Job* aJob )
{
    TRACE();
    if ( 0L != aJob )
    {
        if ( aJob->error() )
        {
            aJob->showErrorDialog();

            if ( mbIsActivated )
                setupTimer();
            return;
        }
    }
    if( mbFilesModified )
        mpBackup->doBackup();
    else // No files where modified
    {
        mLastBackupTime = QDateTime::currentDateTime();
        setupTimer();
    }
}

BackupProfile::BackupProfile( const BackupProfile& b )
    : QObject(),
      mIdentifier( b.mIdentifier ),
      mTimeToBackup( b.mTimeToBackup ),
      mLastBackupTime( b.mLastBackupTime ),
      mbIsActivated( false ),
      mbReadOnly( true )
{
    TRACE();
    mpTimer = new QTimer();
}

void BackupProfile::operator=( const BackupProfile& )
{
    TRACE();
    Q_ASSERT( false );
}

void BackupProfile::setupTimer()
{
    TRACE();
    int msecs = 0;

    if ( mpTimer->isActive() )
        mpTimer->stop();

    if ( !mLastBackupTime.isValid() )
    {
        kdDebug(0) << "BackupProfile::setupTimer: mLastBackupTime is not valid"
                   << endl;
        msecs = 1000 * 60;
    }
    else
    {
        QDateTime     now = QDateTime::currentDateTime();
        int    difference = mLastBackupTime.secsTo(now);
        int remainingTime = mTimeToBackup.secs() - difference;

        kdDebug() << "-----* Backup profile: " << mIdentifier      << endl;
        kdDebug() << "Last backup: " << mLastBackupTime.toString() << endl;
        kdDebug() << "Now        : " << now.toString()             << endl;
        kdDebug() << "Difference : " << difference                 << endl;
        kdDebug() << "Remaining  : " << remainingTime              << endl;

        if (remainingTime >= 0)
        {
            msecs = 1000 * remainingTime;
        }
        else
        {
            kdDebug(0) << "Backup is overdue. Starting in one minute" << endl;
            msecs = 1000 * 60; // Start in one minute
        }
    }
    mpTimer->singleShot( msecs, this, SLOT( slotDoOneTimeBackup()));
}

void BackupProfile::extractArchiveEntry( KArchiveEntry* entry, QString path )
{
    TRACE();
    if ( entry->isDirectory() )
    {
        QDir dir;
        if ( dir.exists( path ) || QString::null == path )
            /* NOOP */
            ;
        else
        {
            if ( ! dir.mkdir( path ) )
                KMessageBox::error(
                    0L, i18n( "Shit happend: Could not create dir: %1" ).arg(
                        path )
                    );
        }

        QStringList list = ( dynamic_cast<KArchiveDirectory*>( entry ) )->entries();
        QStringList::iterator it = list.begin();
        while ( list.end() != it )
        {
            extractArchiveEntry(
                ( dynamic_cast<KArchiveDirectory*>( entry ) )->entry( *it ),
                path + "/" + *it
                );
            it++;
        }
    }
    else if ( entry->isFile() )
    {
        QFile file( path );
        if ( !file.open( IO_WriteOnly ) )
            KMessageBox::error(
                0L, i18n( "Shit happend! Could not open file: %1" ).arg( path )
                );
        else
        {
            file.writeBlock( ( dynamic_cast<KArchiveFile*>( entry ) )->data() );
            file.close();
        }
    }
    else
        KMessageBox::error(
            0L, i18n( "Shit happend! Entry not dir nor file!" )
            );
}

KURL BackupProfile::getLatestBackup()
{
    TRACE();
    QFileInfo qfinfo( mpBackup->getArchiveUrl().path() );
    DUMP(mpBackup->getArchiveUrl().path());
    QStringList list;
    KURL result;

    if ( qfinfo.isDir() )
    {
        list = qfinfo.dir().entryList(
            mpBackup->getSourceUrl().fileName() + "-??????????????.tar.gz",
            QDir::Files,
            QDir::Name | QDir::Reversed
            );
        if (list.size() == 0)
        {
            kdDebug(0) << "No previous backup found" << endl;
            return KURL();
        }
        KURL result( mpBackup->getArchiveUrl().url() + *list.begin() );
        kdDebug(0) << "Latest backup is: " << result.url() << endl;
        return result;
    }
    else
    {
        kdDebug(0) << "Latest backup is: " << mpBackup->getArchiveUrl().url()
                   << endl;
        if ( QFileInfo( mpBackup->getArchiveUrl().path() ).exists() )
        {
            DUMP(mpBackup->getArchiveUrl())
                return mpBackup->getArchiveUrl();
        }
        else
        {
            PRINT("No valid URL");
            return KURL();
        }
    }
}

bool BackupProfile::restoreActionIsSave()
{
    TRACE();
    QFileInfo q( mpBackup->getSourceUrl().path() );

    if ( q.exists() )
    {
        KMessageBox::sorry(
            0L, i18n( "The url %1 to be restored still exists.\n"
                      "Please remove or rename ist first.").arg(
                          mpBackup->getSourceUrl().path()),
            i18n( "Url exists" )
            );
        return false;
    }
    return true;
}

void BackupProfile::doBackupIfNeeded()
{
    TRACE();
    mbFilesModified = false;

    if ( QFileInfo( mpBackup->getSourceUrl().path() ).isFile() ||
         QFileInfo( mpBackup->getSourceUrl().path() ).isSymLink() )
    {
        KURL lastBackup = getLatestBackup();
        if (lastBackup.isValid())
        {
            QDateTime lastBackupTime;
            QDateTime originalTime;

            KFileItem backupItem(
                KFileItem::Unknown, KFileItem::Unknown, lastBackup
                );
            KFileItem originalItem(
                KFileItem::Unknown, KFileItem::Unknown, mpBackup->getSourceUrl()
                );

            originalTime.setTime_t(
                originalItem.time( KIO::UDS_MODIFICATION_TIME )
                );
            lastBackupTime.setTime_t(
                backupItem.time( KIO::UDS_MODIFICATION_TIME )
                );

            DUMP(originalTime.toString());
            DUMP(lastBackupTime.toString());

            if( originalTime > lastBackupTime )
            {
                PRINT("Files are modifed");
                mbFilesModified = true;
            }
        }
        else
        {
            PRINT("Doing backup no backup made yet");
            mbFilesModified = true;
        }
        slotFinishedSearchingModifiedFiles(0L);
    }
    else if ( QFileInfo( mpBackup->getSourceUrl().path() ).isDir() )
    {
        KIO::Job* job = KIO::listRecursive( mpBackup->getSourceUrl(), false, true );
        connect(
            job, SIGNAL( entries( KIO::Job*, const KIO::UDSEntryList&)),
            SLOT( slotSearchForModifiedFiles( KIO::Job*, const KIO::UDSEntryList&))
            );
        connect(
            job, SIGNAL( result( KIO::Job* ) ),
            SLOT( slotFinishedSearchingModifiedFiles( KIO::Job* ) )
            );
    }
    else
    {
        Q_ASSERT( false );
    }
}

