/* @(#) loader.c 1.11 @(#) */
/***************************************************************\
*	Copyright (c) 1999-2000 First Step Internet Services, Inc.
*		All Rights Reserved
*	Distributed under the BSD Licenese
*
*	Module: KoalaMod
\***************************************************************/

#define _KOALAMUD_LOADER_C "@(#) nitehawk@localhost.1ststep.net|lib/koalamod/loader.c|20000911023214|18831 @(#)"

#include "autoconf.h"
#include <sys/stat.h>
#include <dlfcn.h>

#include "version.h"
#include "module.h"
#include "llist.h"
#include "log.h"

/* This file contains the code that loads individual modules.  */

listnodeptr dynmodlist = NULL;

/* Load a dynamic module for koala. */
loadererr_t loadmodule(char *modname, char *path, modtype_t modtype,
		pdescriptor desc)
{
	dynmod_t *newmod = NULL;
	dynmod_t *pmod = NULL;
	listnodeptr curmod = dynmodlist;

	/* Error check */
	if (modtype == MODTYPE_NULL || modname == NULL || path == NULL)
	{
		return LOADERERR_BADARGS;
	}

	/* Verify that the module is not already loaded */
	while (curmod)
	{
		pmod = (dynmod_t *)curmod->data.generic;
		if (pmod == NULL)
			break;

		if (strcasecmp(pmod->modname, modname) == 0)
		{
			logmsg(LOGWARN, "loadmodule: Module '%s' already loaded!",
					modname);
			return LOADERERR_LOADFAIL;
		}
		curmod = listnextnode(curmod);
	}


	logmsg(LOGINFO, "loadmodule: Loading module '%s'", modname);

	/* Attempt to allocate new dynamic module */
	if ((newmod = malloc(sizeof(dynmod_t))) == NULL)
	{
		logmsg(LOGCRIT, "loadmodule: Unable to allocate memory");
		return LOADERERR_NOMEM;
	}

	/* Copy module information to dynamic module tracker */
	if ((newmod->path = malloc(strlen(path))) == NULL)
	{
		logmsg(LOGCRIT, "loadmodule: Unable to allocate memory");
		free(newmod);
	}
	if ((newmod->modname = malloc(strlen(modname))) == NULL)
	{
		logmsg(LOGCRIT, "loadmodule: Unable to allocate memory");
		free(newmod->path);
		free(newmod);
	}
	strcpy(newmod->path, path);
	strcpy(newmod->modname, modname);
	newmod->type = modtype;

	/* Create a linked list if it doesen't already exist */
	if (dynmodlist == NULL)
	{
		if ((dynmodlist = listcreate(newmod)) == NULL)
		{
			logmsg(LOGCRIT, "loadmodule: Unable to allocate memory");
			free(newmod->path);
			free(newmod->modname);
			free(newmod);
			return LOADERERR_NOMEM;
		}
	}
	else
	{
		listaddnode(dynmodlist, newmod);
	}

	/* First we need to verify that we have access to the module */
		/* If access does not return 0, then we don't have the right access */
	if (access(newmod->path, R_OK))
	{
		logmsg(LOGERR, "loadmodule: Access denied: %s", newmod->path);
		listremovenode(dynmodlist, newmod);
		free(newmod->modname);
		free(newmod->path);
		free(newmod);
		return LOADERERR_NOACCESS;
	}

	memset((void*)&(newmod->module), 0, sizeof(module_t));

	/* We know we can access the file, now lets try loading it up */
	if ((newmod->modhandle = dlopen(newmod->path, RTLD_LAZY)) == NULL)
	{
		logmsg(LOGERR, "loadmodule: dlopen error: %s", dlerror());
		listremovenode(dynmodlist, newmod);
		free(newmod->modname);
		free(newmod->path);
		free(newmod);
		return LOADERERR_LOADFAIL;
	}

	/* Find a symbol matching the module name */
	if ((newmod->startupfunc = dlsym(newmod->modhandle, newmod->modname)) == NULL)
	{
		logmsg(LOGERR, "loadmodule: Error starting module");
		dlclose(newmod->modhandle);
		listremovenode(dynmodlist, newmod);
		free(newmod->modname);
		free(newmod->path);
		free(newmod);
		return LOADERERR_BADMOD;
	}

	/* Now that we have the startup function, we need to call it to
	 *    initialize everything - 0 == success */
	if ((*newmod->startupfunc)(&modinf))
	{
		/* Startup failed - bail on the load */
		logmsg(LOGERR, "loadmodule: Unable to startup module");
		dlclose(newmod->modhandle);
		listremovenode(dynmodlist, newmod);
		free(newmod->modname);
		free(newmod->path);
		free(newmod);
		return LOADERERR_STARTUPFAIL;
	}

	return LOADERERR_SUCCESS;
}

loadererr_t unloadmodule(char *modname, pdescriptor desc)
{
	listnodeptr cur = dynmodlist;
	dynmod_t *mod;

	/* Error check parameters */
	if (modname == NULL || cur == NULL)
	{
		logmsg(LOGWARN, "Attempting to unload module with no modules loaded");
		return LOADERERR_BADARGS;
	}

	logmsg(LOGINFO, "Unloading module: %s", modname);

	/* Find the module in the list */
	for (;cur; cur = listnextnode(cur))
	{
		mod = (dynmod_t *)(cur->data.generic);
		if (mod == NULL || strcasecmp(mod->modname, modname) == 0)
			break;
	}
	if (mod == NULL)
	{
		logmsg(LOGERR, "unloadmodule: Module not loaded");
		return LOADERERR_UNKNOWNMOD;
	}

	/* Shutdown module */
	if (mod->module.shutdownfn)
	{
		(mod->module.shutdownfn)();
	}

	/* At this point, the module is no longer linked together */
	dlclose(mod->modhandle);

	logmsg(LOGDEBUG, "unloadmodule: module unloaded - Freeing memory");

	/* Release tracking data */
	listremovenode(dynmodlist, mod);
	free(mod);
	
	/* Module has been successfully unloaded */
	return LOADERERR_SUCCESS;
}

/* Generic init function - use this as an example for other loader functions
 */
void initgeneric(modtype_t type)
{
}

/* Get a pointer to the head of the module list */
listnodeptr getmodlisthead(void)
{
	return dynmodlist;
}
