/*
 * decode_s410.c
 *
 * Copyright (C) 2001-2003, Toms Oliveira e Silva
 *
 * e-mail: tos@det.ua.pt
 * www:    http://www.ieeta.pt/~tos
 *
 * GNU/Linux program to decode the data sent by the Polar S410 heart rate monitor.
 * Please send comments, suggestions and corrections to the e-mail given above.
 * Currently, support for the Polar S510 heart rate monitor is very rudimentary
 * (raw output only).
 *
 * The SonicLink waveform (fs=44100Hz, mono, 16bits per sample, lsb first)
 * is read from a file, from the standard input, or acquired from /dev/dsp
 *
 * Examples of utilization:
 *
 *   > modprobe es1371
 *   > decode_s410 /dev/dsp
 *
 *   > modprobe snd-card-sbawe
 *   > amixer set "Input Gain" 75% front
 *   > amixer set "MIC" 100% front unmute capture
 *   > arecord -w -s 44100 -f s16l | decode_s410
 *
 *
 * Released under the GNU general public license (version 2 or any later version); see the
 * gpl.txt file (or the page http://www.gnu.org/licenses/gpl.html) for details.
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define USE_DSP 1
#define VERBOSE 1

#include <stdio.h>
#include <stdlib.h>
#if USE_DSP > 0
# include <string.h>
# include <sys/ioctl.h>
# include <linux/soundcard.h>
#endif

#include "hrmdata.h"
#include "s410.h"

/*
 * Polar heart rate monitor model
 */

#define None 0
#define S410 1
#define S510 2

static int monitor_type = None;


/*
 * rudimentary progress report (text only)
 */

#if VERBOSE > 0

static char  *v_status   = "";
static double v_level    = 0.0;
static double v_progress = 0.0;
static int exit_now      = 0;

static void display_info(void)
{
  fprintf(stderr,"%5.3f  %3.0f%%  %-24s\r", v_level, 100.0 * v_progress, v_status);
}

static void update_status(char *s,int new_line)
{
  v_status = s;
  display_info();
  if(new_line)
    fprintf(stderr,"\n");
}

static void update_level(double l)
{
  static double i_level = 0.0;
  static int c = 0;

  if(l > i_level)
    i_level = l;
  if(++c >= 4410)
  { /* update level every 0.1 seconds (sampling frequency = 44100) */
    v_level = i_level;
    i_level = 0.0;
    c = 0;
    display_info();
  }
}

static void update_progress(double p)
{
  v_progress = p;
  display_info();
}

#else

#define update_status(s,new_line)
#define update_level(l)
#define update_progress(p)

#endif


/*
 * fatal error handling
 */

static void fail(char *message)
{
  fprintf(stderr,"\n%s\n",message);
  exit_now = 1;
}


/*
 * read signal samples
 * 
 * we assume a 16bit mono signal (least significant byte first) with a
 * sampling frequency of 44100Hz
 */

static char *data_file_name = NULL;

#if USE_DSP > 0

static FILE *open_dsp(void)
{
  FILE *dsp_fp,*mixer_fp;
  int fd,val;

  /* open mixer device */
  mixer_fp = fopen("/dev/mixer","r");
  if(mixer_fp == NULL)
  {
    perror("fopen(\"/dev/mixer\",\"r\")");
    return NULL;
  }
  fd = fileno(mixer_fp);
  /* set recording source (mic) */
  val = SOUND_MASK_MIC;
  if(ioctl(fd,SOUND_MIXER_WRITE_RECSRC,&val))
  {
    perror("ioctl(fd,SOUND_MIXER_WRITE_RECSRC,&val)");
    return NULL;
  }
  fprintf(stderr,"Recording source set to %04X\n",val);
  /* set mic recording volume */
  val = 0xFFFF;
  if(ioctl(fd,SOUND_MIXER_WRITE_MIC,&val))
  {
    perror("ioctl(fd,SOUND_MIXER_WRITE_MIC,&val)");
    return NULL;
  }
  fprintf(stderr,"Mic volume set to %04X\n",val);
  /* set master recording volume */
  val = 0xFFFF;
  if(ioctl(fd,SOUND_MIXER_WRITE_VOLUME,&val))
  {
    perror("ioctl(fd,SOUND_MIXER_WRITE_VOLUME,&val)");
    return NULL;
  }
  fprintf(stderr,"Master recording volume set to %04X\n",val);
  fclose(mixer_fp);

  /* open dsp device */
  dsp_fp = fopen("/dev/dsp","r");
  if(dsp_fp == NULL)
  {
    perror("fopen(\"/dev/dsp\",\"r\")");
    return NULL;
  }
  fd = fileno(dsp_fp);
  /* set one channel (mono) */
  val = 1;
  if(ioctl(fd,SNDCTL_DSP_CHANNELS,&val))
  {
    perror("ioctl(fd,SNDCTL_DSP_CHANNELS,&val)");
    return NULL;
  }
  fprintf(stderr,"Number of channels: %d\n",val);
  /* set format (16 bits, lsb first) */
  val = AFMT_S16_LE;
  if(ioctl(fd,SNDCTL_DSP_SETFMT,&val))
  {
    perror("ioctl(fd,SNDCTL_DSP_SETFMT,&val)");
    return NULL;
  }
  fprintf(stderr,"Number of bits: %d\n",val);
  /* set sampling frequency (44100Hz) */
  val = 44100;
  if(ioctl(fd,SNDCTL_DSP_SPEED,&val))
  {
    perror("ioctl(fd,SNDCTL_DSP_SPEED,&val)");
    return NULL;
  }
  fprintf(stderr,"Sampling frequency: %d\n",val);

  return dsp_fp;
}

#endif

static int next_sample(gint reset_fp)
{
  static FILE *fp = NULL;
  int c1,c2;

  if (reset_fp) {
    fp = NULL;
    return 0;
  }
  
  if(fp == NULL)
  {
    printf("DATA FILE: %s\n", data_file_name);
    
#if USE_DSP > 0
    if(strcmp(data_file_name,"/dev/dsp") == 0
    || strcmp(data_file_name,"dsp") == 0) {
      fp = open_dsp();

      if (!fp) {
        exit_now = 1;
        return 0;
      }
    }

#endif
    else
    {
      fp = fopen(data_file_name,"r");
      if(fp == NULL) {
        fail("unable to open data file");
        return 0;
      }
    }
    for(c1 = 0;c1 < 44;c1++) /* skip the first 44 bytes (size of the header of a wav file) */
      if(getc(fp) == EOF) {
        fail("EOF reached in data file");
        return 0;
      }
  }
  c1 = getc(fp);
  c2 = getc(fp);
  if(c1 == EOF || c2 == EOF) {
    fail("EOF reached in data file");
    return 0;
  }
  c1 += c2 << 8;
  if(c1 > 32767)
    c1 -= 65536;
  return c1;
}

/*
 * pass input signal through a (linear phase FIR) high-pass filter
 *
 * y = filter(remez(40,[0 2000 4000 22050]/22050,[0 0 1 1],[2 1]),1,x);
 */

static double filter(double x)
{
  static double h[41] =
  {
     0.00379802504960,-0.01015567605831,-0.00799539667861,-0.00753846964193,
    -0.00613119820747,-0.00283788804837, 0.00237999784896, 0.00894445674876,
     0.01562664364293, 0.02085836025169, 0.02288210087128, 0.02006200096880,
     0.01129559749322,-0.00374264061795,-0.02440163523552,-0.04905850773028,
    -0.07526471722031,-0.10006201874319,-0.12045502068751,-0.13386410688608,
     0.86146086608477,-0.13386410688608,-0.12045502068751,-0.10006201874319,
    -0.07526471722031,-0.04905850773028,-0.02440163523552,-0.00374264061795,
     0.01129559749322, 0.02006200096880, 0.02288210087128, 0.02085836025169,
     0.01562664364293, 0.00894445674876, 0.00237999784896,-0.00283788804837,
    -0.00613119820747,-0.00753846964193,-0.00799539667861,-0.01015567605831,
     0.00379802504960
  };
  static double input_signal[64];
  static int pos = -1;
  double y;
  int i;

  if(pos == -1)
  {
    pos = 0;
    for(i = 0;i < 64;i++)
      input_signal[i] = 0.0;
  }
  input_signal[pos] = x;
  y = 0.0;
  for(i = 0;i <= 40;i++)
    y += h[i] * input_signal[(pos - i) & 63];
  pos = (pos + 1) & 63;
  return y;
}


/*
 * dilate the rectified signal
 *
 * y = delay(3,ordfilt2(abs(x),7,ones(1,7)));
 */

static double dilate(double x)
{
  static double input_signal[8];
  static int pos = -1;
  double y;
  int i;

  if(pos == -1)
  {
    pos = 0;
    for(i = 0;i < 8;i++)
      input_signal[i] = 0.0;
  }
  input_signal[pos] = (x >= 0.0) ? x : -x;
  y = input_signal[pos];
  for(i = 1;i <= 6;i++)
    if(input_signal[(pos - i) & 7] > y)
      y = input_signal[(pos - i) & 7];
  pos = (pos + 1) & 7;
  return y;
}


/*
 * apply a median filter to the signal
 *
 * y = delay(2,ordfilt2(x,3,ones(1,5)));
 */

static double median(double x)
{
  static double input_signal[8];
  static int pos = -1;
  double t,y[5];
  int i,j;

  if(pos == -1)
  {
    pos = 0;
    for(i = 0;i < 8;i++)
      input_signal[i] = 0.0;
  }
  input_signal[pos] = x;
  for(i = 0;i <= 4;i++)
  { /* sort */
    t = input_signal[(pos - i) & 7];
    for(j = 0;j < i;j++)
      if(t < y[j])
      {
        y[i] = y[j];
        y[j] = t;
        t = y[i];
      }
    y[i] = t;
  }
  pos = (pos + 1) & 7;
  return y[2];
}


/*
 * decide if each signal sample corresponds to part of a one or not
 *
 * y = x;
 * for k=2:length(x)
 *   y(k) = max(x(k),0.9999*y(k-1));
 * end
 * amplitude = delay(20,ordfilt2(y,41,ones(1,41)));
 * decisions = [ delay(20,x) >= level*amplitude  amplitude >= 15*delay(30,amplitude) ]);
 */

static double make_decision(double x,int *decisions)
{ /*decisions[0..4] = above/below threshold for five different threshold levels
    decisions[5] = best above/below decision based on a majority rule
    decisions[6] = abrupt change/no abrupt change in the amplitude (start flag)*/
  static double threshold_levels[5] = { 0.25,0.30,0.35,0.40,0.45 };
  static double input_signal[64],amplitude[64];
  static int pos = -1,count_down;
  double t;
  int i;

  if(pos == -1)
  {
    pos = 0;
    count_down = 64;
    for(i = 0;i < 64;i++)
    {
      input_signal[i] = 0.0;
      amplitude[i] = 0.0;
    }
  }
  /* compute the amplitude */
  input_signal[pos] = x;
  amplitude[pos] = 0.9999 * amplitude[(pos - 1) & 63];
  if(x > amplitude[pos])
    amplitude[pos] = x;
  /* dilate the amplitude */
  t = amplitude[pos];
  for(i = 1;i <= 40;i++)
    if(amplitude[(pos - i) & 63] > t)
      t = amplitude[(pos - i) & 63];
  /* compare the (delayed) signal with the (dilated) amplitude scaled by
   * a threshold level */
  decisions[5] = 0;
  for(i = 0;i < 5;i++)
  {
    decisions[i] = (input_signal[(pos - 20) & 63] >= threshold_levels[i] * t) ? 1 : 0;
    decisions[5] += decisions[i];
  }
  decisions[5] = (decisions[5] >= 3) ? 1 : 0; /* majority rule (median) */
  decisions[6] = (amplitude[pos] >= 15.0 * amplitude[(pos - 30) & 63]) ? 1 : 0;
  if(count_down > 0)
  {
    --count_down;
    decisions[6] = 0;
  }
  i = pos;
  pos = (pos + 1) & 63;
  return amplitude[i];
}


/*
 * apply a median filter to the decisions
 *
 * d = delay(2,ordfilt2(d,3,ones(1,5)));
 */

static void b_median(int *d)
{
  static int inputs[8][8];
  static int pos = -1;
  int i,j;

  if(pos == -1)
  {
    pos = 0;
    for(i = 0;i < 8;i++)
      for(j = 0;j < 8;j++)
        inputs[i][j] = 0;
  }
  for(i = 0;i < 7;i++)
  {
    inputs[i][pos] = d[i];
    for(j = 1;j <= 4;j++)
      d[i] += inputs[i][(pos - j) & 7];
    d[i] = (d[i] >= 3) ? 1 : 0;  /* median (majority rule) */
  }
  pos = (pos + 1) & 7;
}


/*
 * byte decode
 *
 * 0..255  new byte
 * -1      continue
 * -2      error
 */

static int byte_decode(int restart)
{
  static int t,t_active,t_byte;
  static int byte,n_bytes,c[6];
  int i,j,k,d[7];
  int tmp;
  double x;

  if(restart)
  {
    t = t_active = 0;
    update_progress(0.0);
    update_status("no signal",0);
  }

  /* process next sample */
  t++;
  tmp = next_sample (0);

  if (exit_now)
    return -2;

  x = (double) tmp / 32768.0;
  x = filter(x);
  x = dilate(x);
  x = median(x);
  x = make_decision(x,d);
  b_median(d);
  update_level(x);

  /* activation test */
  if(d[6] && t_active == 0)
  {
    t_active = t;
    t_byte = 0;
    n_bytes = 0;
    update_status("processing signal ...",0);
  }
  if(t_active == 0)
    return -1;

  /* deactivation test after a long period of inactivity */
  if(t > t_active + 10000)
  {
    update_status("byte start time out",1);
    return -2;
  }

  /* new byte? */
  if(t_byte == 0 && d[5])
  { /* new byte */
    t_active = t_byte = t;
    n_bytes++;
    byte = 0;
    for(i = 0;i < 6;i++)
      c[i] = 0;
  }

  /* processing a byte? */
  if(t_byte > 0)
  {
    j = (t - t_byte) / 88;  /* bit number; the duration of a bit is very close
                               to 2ms=88.2 samples */
    if(j >= 10)
    { /* end of the byte */
      if((byte & 1) == 0)
      { /* first bit (lsb) must be a one */
        update_status("bad byte start",1);
        return -2;
      }
      if(byte >= 512)
      { /* last bit (msb) must be a zero */
        update_status("bad byte finish",1);
        return -2;
      }
      byte >>= 1;
      t_byte = 0;
      if(n_bytes > 5)
        return byte;
      if(byte != 0xAA)
      { /* bad synchronization byte */
        update_status("bad synchronization byte",1);
        return -2;
      }
      return -1; /* discard synchronization byte */
    }
    /* sample number inside the bit */
    k = t - t_byte - 88 * j; 
    
    if(k < 64) /* the bit burst has a duration of close to 60 samples (here
                  we use 64 but latter we use 60...) */
      for(i = 0;i < 6;i++)
        c[i] += d[i];
    else if(k == 64)
    {
      k = 0;
      for(i = 0;i < 6;i++)
      {
        if(c[i] >= 30) /* threshold = 60/2 */
          k += (i < 5) ? 1 : 2;
        c[i] = 0;
      }
      if(k >= 4) /* majority rule */
        byte += 1 << j;
    }
  }
  return -1;
}


/*
 * raw decode (with error detection)
 */

static int extract_bytes(int *data,int max_size)
{
  int i,byte,restart,size,section_start,section_number,crc;

  goto error;
  for(;;)
  {
    byte = byte_decode(restart);
    restart = 0;
    
    /* If EOF is reached or something, return with -1 and don't exit (). */
    if (exit_now)
      return -1;
    
    if(byte == -2)
    {
error:
      restart = 1;
      size = 0;
      section_start = 8;
      section_number = 1;
      crc = 0;
      continue;
    }
    if(byte >= 0)
    {
      if(size >= max_size) 
        return -1;
      
      data[size++] = byte;
      for(i = 7;i >= 0;--i)
      {
        crc <<= 1;
        if(byte & (1 << i))
          crc ^= 1;
        if(crc & 0x10000)
          crc ^= 0x18005; /* CRC-16 polynomial */
      }
      if(size == 1 && data[0] != 85)
      {
        update_status("bad first section header (byte 1)",1);
        goto error;
      }
      if(size == 2)
        switch(data[1])
        {
          case 10:
            monitor_type = S410;
            fprintf(stderr,"Polar S410 SonicLink data detected\n");
            break;
          case 81:
            monitor_type = S510;
            fprintf(stderr,"Polar S510 SonicLink data detected\n");
            break;
          default:
            fprintf(stderr,"Polar S410 or S510 SonicLink data NOT detected (%d)\n",data[1]);
            break;
        }
      if(size == 3 && data[2] != 1)
      {
        update_status("bad first section header (byte 3)",1);
        goto error;
      }
      if(size == 8 && crc)
      {
        update_status("first section crc error",1);
        goto error;
      }
      if(size <= 8)
        continue;
      if(size == section_start + 3)
      { /* new section (header complete) */
        if(data[section_start] != 85 || data[section_start + 1] != section_number || data[section_start + 2] < 1 || data[section_start + 2] > 60)
        {
          update_status("bad section header",1);
          goto error;
        }
      }
      if(size >= section_start + 3 && size == section_start + data[section_start + 2] + 5)
      { /* end of a section */
        if(crc)
        {
          update_status("section crc error",1);
          goto error;
        }
        section_start = size;
        section_number++;
      }
      i = 8 + 5 * data[3] + data[4] + 256 * data[5] + 1; /* total number of bytes (8 for the preamble,
                                                            5 extra for each section, and 1 for the end) */
      update_progress((double)size / (double)i);
      if(size == i)
      {
        if(data[size - 1] != 7)
        {
          update_status("bad trailer byte",1);
          goto error;
        }
        if(section_start != size - 1 || section_number != data[3] + 1)
        {
          update_status("bad section structure",1);
          goto error;
        }
        /* all is well! */
        break;
      }
    }
  }
  update_status("done!",1);
  fprintf(stderr,"%d bytes (%d data bytes)\n",i,data[4] + 256 * data[5]);
  return size;
}


/*
 * extract Polar S410 data
 */

static void exercise_summary(HrmData *hrm_data,char *text,int *data)
{
  static char *months[16] = { "00?","Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec","13?","14?","15?" };
  static char *charset = "0123456789 ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz-%/()*+.:?";
  int i;

 /*
  * exercise summary format
  *
  *  0:   x  D  exercise_set number
  *  1:   x  C         | name (char 1) charset is 0-9 space A-Z a-z - % / ( ) * + . : ?
  *  2:   x  C         | name (char 2) 
  *  3:   x  C         | name (char 3) all chars=0 if exercise_set number=0
  *  4:   x  C         | name (char 4)
  *  5:   x  C         | name (char 5)
  *  6:   x  C         | name (char 6)
  *  7:   x  C         | name (char 7)
  *  8:   x  H         | start seconds
  *  9:   x  H               | minutes
  * 10:   x  H               | hours + 0x80*IsPM
  * 11:   x  H               | day + 0x80*Use12HourClock
  * 12:   x  H               | year
  * 13:   x  D         | start month + 16*duration_tenths
  * 14:   x  H         | duration seconds
  * 15:   x  H                  | minutes
  * 16:   x  H                  | hours
  * 17:   x  D         | FCavg
  * 18:   x  D         | FCmax
  */

  for(i = printf("%s ",text);i < 26;i++)
    printf("-");
  /* printf(" E%d [%02X %02X %02X %02X %02X %02X ?] E%d\n",data[0],
     data[1],data[2],data[3],data[4],data[5],data[6],data[7]); */
  printf(" E%d (",data[0]);
  if(data[0] == 0)
  {
    printf("basic use");
    for(i = 1;i < 8 && data[i] == 0;i++)
      ;
    if(i < 8)
      printf(" ? %02X %02X %02X %02X %02X %02X %02X",data[1],data[2],data[3],
             data[4],data[5],data[6],data[7]);
  
  } else
    for(i = 1;i < 8;i++)
      if(data[i] < 73)
        printf("%c",charset[data[i]]);
      else
        printf("<%02X>",data[i]);
  printf(")\n");
  printf("         starting date --- %s %X, 20%02X\n",months[data[13] & 0x0F],data[11] & 0x7F,data[12]);

 
  i = (data[10] & 0x0F) + 10 * ((data[10] >> 4) & 0x07);
  if (i < 12 && data[10] & 0x80)
    i += 12;
    
  printf("         starting time --- %02d:%02X:%02X\n",i,data[9],data[8]);
  printf("        total duration --- %02X:%02X:%02X.%d\n",data[16],data[15],
        data[14],data[13] >> 4);
  printf("           heart rates --- avg(%3d) max(%3d)\n",data[17],data[18]);

  if (hrm_data) {
    hrm_data->d = data[11] & 0x7F;
    hrm_data->m = data[13] & 0x0F;
    hrm_data->y = 2000 + data[12];

    hrm_data->h = i;
    hrm_data->min = data[9];
    hrm_data->s = data[8];
  
    hrm_data->duration_h = data[16];
    hrm_data->duration_m = data[15];
    hrm_data->duration_s = data[14];
    hrm_data->duration_t = data[13] >> 4;
  
    hrm_data->avg_hr = data[17];
    hrm_data->max_hr = data[18];
  }
}

static void measurement_summary(int n,int *data,int type)
{
  int i;

  /*
  * measurements format
  *
  *  0:   x  D  lap/interval seconds+64*(tenths%4)
  *  1:   x  D             | minutes+64*(tenths/4)
  *  2:   x  D?            | hours
  *  3:   x  D             | FCend
  *  4:   x  D             | FCavg
  *  5:   x  D             | FCmax
  *
  * end of warmup information for long format (interval data):
  *
  *  6:   x  H             | always zero (measurement number inexistent)
  *  7:   x  ?             | (always zero ?)
  *  8:   x  D             | always 0xFE
  *  9:   x  D             | (always 0x20 ?)
  *
  * end of intervals information for long format (interval data):
  *
  *  6:   x  H             | end of warmup measurement number
  *  7:   x  ?             | (always zero ?)
  *  8:   x  D             | always 0xFE
  *  9:   x  D             | 0x40+number of intervals
  *
  * end of cooldown information for long format (interval data):
  *
  *  6:   x  H             | end of intervals measurement number
  *  7:   x  ?             | (always zero ?)
  *  8:   x  D             | always 0xFE
  *  9:   x  D             | 0x60+number of intervals
  *
  * interval information for long format (interval data):
  *
  *  6:   x  H             | recovery time: seconds
  *  7:   x  H             |             |  minutes
  *  8:   x  D             | recovery heart rate drop
  *  9:   x  D             | 0x40+interval number (starts at 0x41)
  *
  * lap information for long format (interval data):
  *
  *  6:   x  H             | previous lap measurement number (starts at 1, 0 means start of exercise)
  *  7:   x  ?             | (always zero ?)
  *  8:   x  D             | always 0xFF
  *  9:   x  D             | lap number (starts at one)
  */
  for(i = printf("measurement #%d ",n);i < 26;i++)
    printf("-");
  printf(" time(%d:%02d:%02d.%d)",data[2],data[1] & 0x3F,data[0] & 0x3F,4 * (data[1] >> 6) + (data[0] >> 6));
  printf(" avg(%3d) end(%3d) max(%3d)",data[4],data[3],data[5]);
  if(type)
  {
    printf(" [%02X %02X %02X %02X]",data[6],data[7],data[8],data[9]);
    switch(data[8])
    {
      case 0xFE:
        if(data[9] == 0x20)
          printf(" warmup end (starts in measurement %X)",data[6]);
        else if(data[9] >= 0x41 && data[9] < 0x60)
          printf(" intervals end (%d interval%s, starts in measurement %X)",data[9] - 0x40,(data[9] > 0x41) ? "s" : "",data[6]);
        else if(data[9] >= 0x60)
          printf(" cooldown end (starts in measurement %X)",data[6]);
        break;
      case 0xFF:
        printf(" lap %d (starts in measurement %X)",data[9],data[6]);
        break;
      default:
        printf(" interval %d",data[9] - 0x40);
        if(data[8] > 0)
          printf(" (recovery time: %02X:%02X, hr drop: %d)",data[7],data[6],data[8]);
        break;
    }
  }
  printf("\n");
}

static int s410_data (HrmData *hrm_data, int *data,int size)
{
  int i,j,p,n_samples,sampling_period,n_measurements,n_laps;

  p = 0;
  /*
  * header section format:
  *
  *  0:  85  D  section header
  *  1:  10  D  always 10
  *  2:   1  D  always 1
  *  3:   x  D  number of sections
  *  4:   x  D  data_size % 256
  *  5:   x  D  data_size / 256
  *  6: crc  D  CRC-16 checksum (msb)
  *  7: crc  D  CRC-16 checksum (lsb)
  */
  printf("number of sections ------- %d\n",data[p + 3]);
  printf("number of data bytes ----- %d\n",data[p + 4] + 256 * data[p + 5]);
  p += 8;
  /*
  * first section format
  *
  *  0:  85  D  section header
  *  1:   1  D  section number (starts at one)
  *  2:  60  D  section size (maximum of 60 bytes)
  *  3:   x  D  number of hr samples
  *  4:   x  D  sample rate(0=15,1=30,2=60,3=120,4=240,5=480) + 16*number of stored exercises ?
  *  5:   x  D  exercise summary (19 bytes)
  * 24:   x  H  NMeasurements (Nlaps+...)
  * 25:   x  H  Nlaps
  * 26:   ?
  * 27:   ?
  * 28:   ?
  * 29:   ?
  * 30:   x  D  lim1 low
  * 31:   x  D     | high
  * 32:   x  D  lim2 low
  * 33:   x  D     | high
  * 34:   x  D  lim3 low
  * 35:   x  D     | high
  * 36:   x  H  recover seconds
  * 37:   x  H  recover minutes
  * 38:   x  D  recover beats (251 = no measurement)
  * 39:   x  H  below Lim1 seconds
  * 40:   x  H           | minutes
  * 41:   x  H           | hours
  * 42:   x  H  inside Lim1 seconds
  * 43:   x  H            | minutes
  * 44:   x  H            | hours
  * 45:   x  H  above Lim1 seconds
  * 46:   x  H           | minutes
  * 47:   x  H           | hours
  * 48:   x  H  below Lim2 seconds
  * 49:   x  H           | minutes
  * 50:   x  H           | hours
  * 51:   x  H  inside Lim2 seconds
  * 52:   x  H            | minutes
  * 53:   x  H            | hours
  * 54:   x  H  above Lim2 seconds
  * 55:   x  H           | minutes
  * 56:   x  H           | hours
  * 57:   x  H  below Lim3 seconds
  * 58:   x  H           | minutes
  * 59:   x  H           | hours
  * 60:   x  H  inside Lim3 seconds
  * 61:   x  H            | minutes
  * 62:   x  H            | hours
  * 63: crc  D  CRC-16 checksum (msb)
  * 64: crc  D  CRC-16 checksum (lsb)
  */
  if(data[p + 2] != 60) {
    printf ("warn %d\n", data[p+2]);
    fail("unexpected size of section 1");
    return 0;
  }
  n_samples = data[p + 3];
  hrm_data->n_samples = data[p + 3];

  
  sampling_period = 15;
  for (i = data[p + 4] & 0x0F;i > 0;--i)
    sampling_period <<= 1;
  if (n_samples == 120 && sampling_period > 15)
    sampling_period >>= 1; /* the sampling period is adjusted immediately, but
                              the samples are decimated only when a new sample
                              is received */

  hrm_data->sampling_period = sampling_period;
  
  printf("section1 data[4]/16 ------ %X [number of archived exercises ?]\n",data[p + 4] >> 4);
  exercise_summary(hrm_data,"last exercise",data + p + 5);
  n_measurements = (data[p + 24] & 0x0F) + 10 * (data[p + 24] >> 4);
  printf("number of measurements --- %d\n",n_measurements);
  n_laps = (data[p + 25] & 0x0F) + 10 * (data[p + 25] >> 4);
  printf("number of laps ----------- %d\n",n_laps);
  printf("section1 data[26..29] ---- [%02X %02X %02X %02X ?]\n",data[p + 26],
         data[p + 27],data[p + 28],data[p + 29]);

  hrm_data->n_measurements = n_measurements;
  hrm_data->n_laps = n_laps;
  
  if(data[p + 38] <= 240)
    printf("recovery data ------------ time(%X:%02X) drop(%3d)\n",data[p + 37], data[p + 36],data[p + 38]);
  else if(data[p + 38] == 251)
    printf("recovery data ------------ none [%02X %02X %02X]\n",data[p + 36],data[p + 37],data[p + 38]);
  else
    printf("recovery data ------------ [%02X %02X %02X ?]\n",data[p + 36],
           data[p + 37],data[p + 38]);
  printf("limit 1 data ------------- min(%3d) max(%3d)",data[p + 30],
         data[p + 31]);
  printf(" below(%02X:%02X:%02X)",data[p + 41],data[p + 40],data[p + 39]);
  printf(" inside(%02X:%02X:%02X)",data[p + 44],data[p + 43],data[p + 42]);
  printf(" above(%02X:%02X:%02X)\n",data[p + 47],data[p + 46],data[p + 45]);
  printf("limit 2 data ------------- min(%3d) max(%3d)",data[p + 32],data[p + 33]);
  printf(" below(%02X:%02X:%02X)",data[p + 50],data[p + 49],data[p + 48]);
  printf(" inside(%02X:%02X:%02X)",data[p + 53],data[p + 52],data[p + 51]);
  printf(" above(%02X:%02X:%02X)\n",data[p + 56],data[p + 55],data[p + 54]);
  printf("limit 3 data ------------- min(%3d) max(%3d)",data[p + 34],data[p + 35]);
  printf(" below(%02X:%02X:%02X)",data[p + 59],data[p + 58],data[p + 57]);
  printf(" inside(%02X:%02X:%02X)",data[p + 62],data[p + 61],data[p + 60]);
  p += 60 + 5;
  /*
  * second section format
  *
  *  0:  85  D  section header
  *  1:   2  D  section number
  *  2:  17  D  section size
  *  3:   x  H  above Lim3 seconds
  *  4:   x  H           | minutes
  *  5:   x  H           | hours
  *  6:   x  H  best lap number  (usually incorrect when intervals=on or when the best lap is the last lap)
  *  7:   x  H         | duration: hundreds of seconds
  *  8:   x  H                  |  seconds
  *  9:   x  H                  |  minutes
  * 10:   x  H                  |  hours
  * 11:   x  H  kcal % 10
  * 12:   x  H  (kcal / 10) % 100
  * 13:   x  H  kcal / 1000
  * 14:   x  H  Total kcal % 100
  * 15:   x  H  Total kcal / 100
  * 16:   x  H  Total kcal / 10000
  * 17:   x  H  Total hours % 100
  * 18:   x  H  Total hours / 100
  * 19:   x  H  Total minutes
  * 20: crc  D  CRC-16 checksum (msb)
  * 21: crc  D  CRC-16 checksum (lsb)
  */
  if(data[p + 2] != 17) {
    fail("unexpected size of section 2");
    return 0;
  }
  printf(" above(%02X:%02X:%02X)\n",data[p + 5],data[p + 4],data[p + 3]);
  printf("best lap number ---------- %X (possibly incorrect)\n",data[p + 6]);
  printf("              duration --- %02X:%02X:%02X.%02X\n",data[p + 10],data[p + 9],data[p + 8],data[p + 7]);
  i = (data[p + 11] & 0x0F) + 10 * (data[p + 11] >> 4)
    + 100 * (data[p + 12] & 0x0F) + 1000 * (data[p + 12] >> 4)
    + 10000 * (data[p + 13] & 0x0F) + 100000 * (data[p + 13] >> 4);
  printf("kcal used in exercise ---- %d.%d\n",i / 10,i % 10);
  i = (data[p + 14] & 0x0F) + 10 * (data[p + 14] >> 4)
    + 100 * (data[p + 15] & 0x0F) + 1000 * (data[p + 15] >> 4)
    + 10000 * (data[p + 16] & 0x0F) + 100000 * (data[p + 16] >> 4);
  printf("total kcal used ---------- %d\n",i);
  i = (data[p + 17] & 0x0F) + 10 * (data[p + 17] >> 4) + 100 * (data[p + 18] & 0x0F) + 1000 * (data[p + 18] >> 4);
  printf("total time used ---------- %d:%02X:00\n",i,data[p + 19]);
  p += 17 + 5;
  /*
  * hr section format
  *
  *  0:  85  D  section header
  *  1:   x  D  section number
  *  2:   x  D  section size
  *  3:   x  D  hr samples array
  *     ...
  *  x: crc  D  CRC-16 checksum (msb)
  *  x: crc  D  CRC-16 checksum (lsb) 
  */
  
  hrm_data->hr_samples = malloc (hrm_data->n_samples * sizeof (guint));
  j = 0;
  while(n_samples > 0)
  {
    if(data[p + 2] > n_samples) {
      fail("unexpected size of the hr data section");
      return 0;
    }
    for(i = 0;i < data[p + 2];i += 10)
    {
      if(j == 0)
      {
        printf("number of samples -------- %d\n",n_samples);
        printf("sampling period ---------- %ds\n",sampling_period);
        printf("heart rate samples -------");
      }
      else
        printf("                       ---");
      for(j = 0;j < 10 && i + j < data[p + 2];j++) {
        printf("%4d",data[p + 3 + i + j]);
        hrm_data->hr_samples[i + j] = data[p + 3 + i + j];
      }
      printf("\n");
    }
    n_samples -= data[p + 2];
    hrm_data->n_samples -= data[p+2];
    p += data[p + 2] + 5;
  }
 /*
  * Measurements section format (without intervals)
  *
  *  0:  85  D  section header
  *  1:   x  D  section number
  *  2:   x  D  section size (multiple of 6)
  *  x:   x  D  measurement summary (7 bytes)
  *     ...
  *  x: crc  D  CRC-16 checksum (msb)
  *  x: crc  D  CRC-16 checksum (lsb)
  *
  * Measurements section format (with intervals)
  *
  *  0:  85  D  section header
  *  1:   x  D  section number
  *  2:   x  D  section size (multiple of 10)
  *  x:   x  D  measurement summary (10 bytes)
  *     ...
  *  x: crc  D  CRC-16 checksum (msb)
  *  x: crc  D  CRC-16 checksum (lsb)
  */
  printf("measurement #0 ----------- time(0:00:00.0) start of the exercise\n");
  if(n_measurements == n_laps)
  {
    j = 0;
    while(n_laps > 0)
    {
      if(data[p + 2] > 6 * n_laps || data[p + 2] % 6) {
        fail("unexpected size of the lap data section");
        return 0;
      }
      for(i = 0;i < data[p + 2];i += 6)
        measurement_summary(++j,data + p + 3 + i,0);
      n_laps -= data[p + 2] / 6;
      p += data[p + 2] + 5;
    }
  }
  else
  {
    j = 0;
    while(n_measurements > 0)
    {
      if(data[p + 2] > 10 * n_measurements || data[p + 2] % 10) {
        fail("unexpected size of the interval data section");
        return 0;
      }
      for(i = 0;i < data[p + 2];i += 10)
        measurement_summary(++j,data + p + 3 + i,1);
      n_measurements -= data[p + 2] / 10;
      p += data[p + 2] + 5;
    }
  }
 /*
  * previous exercises summary section
  *
  *  0:  85  D  section header
  *  1:   x  D  section number
  *  2:   x  D  section size (multiple of 19)
  *  x:   x  D  exercise summary (19 bytes)
  *     ...
  *  x: crc  D  CRC-16 checksum (msb)
  *  x: crc  D  CRC-16 checksum (lsb)
  */
  while (p < size - 1)
  {
    if (data[p + 2] % 19) {
      fail("unexpected size of the exercise summary section");
      return 0;
    }
    for (i = 0;i < data[p + 2];i += 19)
      exercise_summary(NULL, "previous exercise", data + p + 3 + i);
    p += data[p + 2] + 5;
  }
  if (p != size - 1 || data[p] != 7) {
    fail("bad section structure");
    return 0;
  }
  return 1;
}


/*
 * extract raw data
 */

static void raw_data(int *data,int size)
{
  int i,j,k;

  /*
   * output raw data
   */
  printf("# Raw data:\n#\n");
  j = 8;
  k = 11;
  printf("# dec hex comment\n");
  printf("# --- --- --------------------------\n");
  for(i = 0;i < size;i++)
  {
    printf("# %3d  %02X",data[i],data[i]);
    if(i == 0 || k == 3)
      printf((i < size - 1) ? " section start" : " no more sections");
    if(i == 1)
      printf(" data format type");
    if(i == 3)
      printf(" number of sections");
    if(i == 4)
      printf(" number of data bytes (lsb)");
    if(i == 5)
      printf(" number of data bytes (msb)");
    if(k == 2)
      printf(" section number");
    if(k == 1)
      printf(" section size");
    if(j == 1 || j == 2)
      printf(" CRC-16 data");
    printf("\n");
    if(--j == 0)
      printf("#\n");
    if(--k == 0)
    {
      j = data[i] + 2;
      k = j + 3;
    }
  }
  printf("\n");
}


/*
 * main program
 */

#if 0
static void usage(void)
{
  fprintf(stderr,"usage: decode_s410 -h\n");
  fprintf(stderr,"usage: decode_s410 [input_file_name] [output_format]\n");
  fprintf(stderr,"\n");
  fprintf(stderr,"The standard input will be used when the input_file_name argument\n");
  fprintf(stderr,"  is missing or when it is equal to \"-\"\n");
#if USE_DSP > 0
  fprintf(stderr,"The /dev/dsp device will be used when the input_file_name argument\n");
  fprintf(stderr,"  is equal to \"/dev/dsp\" or equal to \"dsp\"\n");
#endif
  fprintf(stderr,"\n");
  fprintf(stderr,"The output_format argument controls the format of the output:\n");
  fprintf(stderr,"  0 --- no output (useless)\n");
  fprintf(stderr,"  1 --- raw data\n");
  fprintf(stderr,"  2 --- decoded data (default)\n");
  fprintf(stderr,"  3 --- raw data + decoded data\n");
  exit(1);
}
#endif 

#if 0
int main(int argc,char **argv)
{
  int i,size,data[8192];

  if(argc == 2 && argv[1][0] == '-' && argv[1][1] == 'h')
    usage();
  data_file_name = NULL;
  if(argc > 1 && (argv[1][0] != '-' || argv[1][1] != 0))
    data_file_name = memcpy (argv[1]);
  i = 2;
  if(argc > 2 && argv[2][0] >= '0' && argv[2][0] <= '3' && argv[2][1] == 0)
    i = argv[2][0] - '0';
  size = extract_bytes(data,sizeof(data));
  if(i & 1 || monitor_type != S410)
    raw_data(data,size);
  if(i & 2 && monitor_type == S410)
    s410_data(data,size);
  return 0;
}
#endif

void
s410_set_file (const char *file)
{
  if (data_file_name != NULL)
    free (data_file_name);
  
  data_file_name = strdup (file);
}

int
s410_extract_bytes (HrmData *hrm_data, int *data, int max_size)
{
  int size;
  int data2[8192];

  printf ("Extracting max %d bytes from %s.\n", max_size, data_file_name);
  
  size = extract_bytes (data2, sizeof (data2));

  if (size < 0)
    g_warning ("Some error happened, couldn't get training data!\n");
  else {
    if (!s410_data (hrm_data, data2, size))
      return -1;
  }

  return size;
}

void 
s410_clean (void)
{
  if (data_file_name)
    free (data_file_name);

  g_print ("Cleaning things\n");
  
  /* Reset static file handle */
  next_sample (1);
  
  exit_now = 0;
}


