/****************************************************************************
 *                               ivi_dlfcn.h
 *
 * Author: Matthew Ballance
 * Desc:   Compat header for dll-loading functions... Borrowed from IVL...
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/

// removed MACOSX-specific libdl implementations

#include "config.h"
#include "ivi_dlfcn.h"

#if defined(__MINGW32__)
# include <windows.h>
# include <stdio.h>
#elif defined(HAVE_DLFCN_H) 
# include  <dlfcn.h>
#elif defined(HAVE_DL_H)
# include  <dl.h>
#endif

#if defined(__MINGW32__)
ivi_dll_t ivi_dlopen(const char *name)
{ 
    return (void *)LoadLibrary(name); 
}

void *ivi_dlsym(ivi_dll_t dll, const char *nm)
{ 
    return (void *)GetProcAddress((HINSTANCE)dll,nm);
}

void ivi_dlclose(ivi_dll_t dll)
{ 
    (void)FreeLibrary((HINSTANCE)dll);
}

const char *ivi_dlerror(void)
{ static char s[30]; 
    sprintf(s,"DLError:%ld", GetLastError()); 
    return s;
}

#elif defined(HAVE_DLFCN_H)
ivi_dll_t ivi_dlopen(const char*name)
{ 
    return dlopen(name,RTLD_LAZY); 
}

void *ivi_dlsym(ivi_dll_t dll, const char*nm)
{ 
	void *p;

	p = dlsym(dll, nm);

	#ifdef MAC_OS_X
		// keep MACOSX-specific handling of leading underscore???
		if (!p)
		{	// Mac OS X only
			//     in case we did not find the symbol
			//     try to prepend an underscore
			//     I am not sure, why this is necessary
			//     for some symbols (but not for others)
			char nm2[1024]; // caution: nm2 must be at least strlen(nm+1) !!!
			nm2[0]='_';
			strncpy(nm2+1, nm, sizeof(nm2)-2);
			nm2[sizeof(nm2)-1]= '\0';
			p= dlsym(dll, nm2);
		}
	#endif

    return p;
}

void ivi_dlclose(ivi_dll_t dll)
{ 
    dlclose(dll); 
}

const char *ivi_dlerror(void)
{
    return dlerror();
}

#elif defined(HAVE_DL_H)
ivi_dll_t ivi_dlopen(const char*name)
{ 
    return shl_load(name, BIND_IMMEDIATE, 0); 
}

void *ivi_dlsym(ivi_dll_t dll, const char*nm)
{
      void *sym;
      shl_t  dllt = (shl_t)dll;
      
      int rc = shl_findsym(&dllt, nm, TYPE_PROCEDURE, &sym);
      return (rc == 0) ? sym : 0;
}

void ivi_dlclose(ivi_dll_t dll)
{ 
    shl_unload((shl_t)dll); 
}

const char *ivi_dlerror(void)
{ 
    return strerror( errno ); 
}

#endif

