/****************************************************************************
 *                            StimSeqInst.cc
 *
 * Author: Matthew Ballance
 * Desc:   Describes an instance of a stimulus sequence. The instance walks
 *         the stimulus sequence and drives the values onto the specified
 *         signal...
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "StimSeqInst.h"
#include "StimulusItem.h"
#include "StimulusSeq.h"
#include "StimulusMgr.h"
#include "IviSim.h"
#include <memory.h>

#define FP stderr
#undef  DEBUG_PUT

#ifdef DEBUG_PUT
#define PUT_MSG(x) fprintf x
#else
#define PUT_MSG(x)
#endif

/********************************************************************
 * StimSeqInst()
 ********************************************************************/
StimSeqInst::StimSeqInst(vpiHandle vpiHndl, StimulusSeq *seqTempl) :
    levStack()
{
    s_vpi_value      value_s;
    s_vpi_time       time_s;
    IviSim          *sim;
    StimulusItem    *item;
    StimSeqLevel    *lev;
    Uint32           msb, lsb;
    vpiHandle        tHndl;
    s_vpi_value      v_val;

    mgr = seqTempl->get_parent();
    sim = mgr->getSim();
    this->vpiHndl  = vpiHndl;

    lev = new StimSeqLevel(seqTempl);
    levStack.push(lev);

    cb_time.type  = vpiSimTime;
    cb_time.low   = 0;
    cb_time.high  = 0;
    repeatCnt     = 0;

    memset(&cb_data, 0, sizeof(cb_data));
    cb_data.reason       =  cbReadWriteSynch;
    cb_data.time         = &cb_time;
    cb_data.cb_rtn       = StimSeqInst::SequenceCB;
    cb_data.user_data    = (char *)this;

    /**** Find the width of the signal ****/
    v_val.format = vpiIntVal;
    tHndl = sim->vpi_handle(vpiRightRange, vpiHndl);
    sim->vpi_get_value(tHndl, &v_val);
    msb = v_val.value.integer;
    tHndl = sim->vpi_handle(vpiLeftRange, vpiHndl);
    sim->vpi_get_value(tHndl, &v_val);
    lsb = v_val.value.integer;
    d_sig_width = (msb>lsb)?(msb-lsb+1):(lsb-msb+1);

    /**** Start off the sequence... ****/
    if (!(item = seqTempl->get_values()->idx(0))) {
        fprintf(stderr, "ERROR :: NULL first item\n");
        return;
    }

    mgr->addSeqInst(this);

    outMode = (seqTempl->getInitDelay())?ModeDelay:ModeOutput;
    StimSeqInst::SequenceCB(&cb_data);
}

/********************************************************************
 * ~StimSeqInst
 ********************************************************************/
StimSeqInst::~StimSeqInst()
{
    mgr->delSeqInst(this);
}

/********************************************************************
 * SequenceCB
 ********************************************************************/
int StimSeqInst::SequenceCB(p_cb_data cb_data_p)
{
    StimSeqInst     *inst     = (StimSeqInst *)cb_data_p->user_data;
    StimSeqLevel    *lev      = inst->levStack.top();
    StimulusSeq     *seq;
    StimulusMgr     *mgr;
    IviSim          *sim;
    StimulusItem    *item;
    s_vpi_value      value_s;
    s_vpi_time       time_s;
    Uint32           delay, i, done = 0;

    if (!(lev = inst->levStack.top())) {
        return 0;
    }
    seq = lev->seq;
    mgr = seq->get_parent();
    sim = mgr->getSim();

    time_s.type = vpiSimTime;
    sim->vpi_get_time(0, &time_s);

    PUT_MSG((FP, "----> SequenceCB() => Time %d\n", time_s.low));


    while (!done) {
        /**** Get the next value... ****/
        item = seq->get_values()->idx(lev->seq_idx);
        PUT_MSG((FP, "\tfetching item at %d\n", lev->seq_idx));
        PUT_MSG((FP, "\t\tval-len = %d\n", seq->get_values()->length()));

        switch (inst->outMode) {

            /********************************************************
             * ModeRepeatDelay
             ********************************************************/
            case ModeRepeatDelay:
                PUT_MSG((FP, "\tModeRepeatDelay\n"));
                
                    inst->cb_time.high = 0;
                    inst->cb_time.low = time_s.low + seq->getRepeatDelay();
                    PUT_MSG((FP, "\tSetting callback for %d\n", 
                                inst->cb_time.low));
                    sim->vpi_register_cb(&inst->cb_data);
                    inst->outMode = ModeOutput;
                    done = 1;
                break;

            /********************************************************
             * ModeDelay
             ********************************************************/
            case ModeDelay:
                PUT_MSG((FP, "\tModeDelay\n"));

                if (!lev->seq_idx) {
                    delay = seq->getInitDelay();
                } else {
                    delay = item->get_delay();
                }

                inst->outMode = ModeOutput;

                if (delay) {
                    inst->cb_time.high = 0;
                    inst->cb_time.low = time_s.low + delay;
                    PUT_MSG((FP, "\tSetting callback for %d\n", 
                                inst->cb_time.low));
                    sim->vpi_register_cb(&inst->cb_data);

                    /**** We're really done... ****/
                    done = 1;
                    break;
                }

                PUT_MSG((FP, "\t\tDrop through to ModeOutput\n"));
                /**** Drop Through... ****/

            /********************************************************
             * ModeOutput
             *
             * We have an item... 
             * - This may be a value, in which case we output the value. 
             * - If the item is a sequence, push our current item on the
             *   stack and recurse...
             * We have an item... 
             ********************************************************/
            case ModeOutput:
                PUT_MSG((FP, "\tModeOutput\n"));
                if (item->is_sequence()) {
                    StimSeqLevel *level = new StimSeqLevel(item->get_seq());
                    inst->levStack.push(level);

                    PUT_MSG((FP, "\t\tMove to new level...\n"));

                    /**** Now, set the new level as the current level and
                     **** branch back to the top so setup delay...
                     ****/
                    lev = level;
                    seq = lev->seq;
                    inst->outMode = ModeDelay;
                    break;
                } else {
                    /**** Otherwise, output the current value and move to the 
                     **** next...
                     ****/
                    value_s.format       = vpiVectorVal;
                    item->get_value()->get_vecval(
                            inst->d_sig_width, &value_s.value.vector);

                    /**** Set the specified sig to that value... ****/
                    PUT_MSG((FP, "\tPut value %s\n", value_s.value.str));
                    sim->vpi_put_value(inst->vpiHndl, &value_s, 
                            NULL, vpiNoDelay);

                    PUT_MSG((FP, "\tMoving to next item (%d)\n",
                                lev->seq_idx+1));

                }
                inst->outMode = ModeSetupNext;
                /**** Drop through to SetupNext... ****/

            case ModeSetupNext:
                lev->seq_idx++;
                /**** See what we need to do... ****/
                if (lev->seq_idx >= seq->get_values()->length()) {
                    PUT_MSG((FP, "\t\tDone with this sequence...\n"));

                    /**** If we need a repeat, recycle... ****/
                    if ((++lev->repeat_cnt >= seq->getNumRepeat()) &&
                            (seq->getNumRepeat() < 0xFFFFFFFF)) {
                        /**** Time to move on... We go to where???
                         **** - See if there is another level to pop
                         ****   back to...
                         ****/
                        delete lev;
                        inst->levStack.pop();
                        PUT_MSG((FP, "\tPop back a level... (Stack %d deep)\n",
                                    inst->levStack.depth()));
                        if (!(lev = inst->levStack.top())) {

                             /**** We're done... Quit... ****/
                            PUT_MSG((FP, "\tSequence complete...\n"));
                            delete inst;
                            return 0;
                        } 

                        /**** Otherwise, we move back to the prev level
                         ****/
                        seq = lev->seq;

                        /**** Loop back to SetupNext... This allows us
                         **** to check for sequence-done
                         ****/
                        break;
                    } else {
                        /**** Else, back to the beginnings... ****/
                        lev->seq_idx = 0;
                        if (seq->getRepeatDelay()) {
                            PUT_MSG((FP, "\tRepeat Delay...\n"));
                            inst->outMode = ModeRepeatDelay;
                        } else {
                            /**** Don't want to get initDelay ****/
                            PUT_MSG((FP, "\tRepeat Delay...\n"));
                            inst->outMode = ModeOutput;
                        }
                    }
                } else {
                    /**** Set mode back to delay... ****/
                    inst->outMode = ModeDelay;
                }
                break;
        }
    }

    PUT_MSG((FP, "<---- SequenceCB()\n"));
    return 0;
}



