/*****************************************************************************
 *                                DesignDB.cc
 * Author: Matthew Ballance
 * Desc:   Design database. This class allows the ui to access the design
 *         being simulated as a hierarchical tree...
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *****************************************************************************/
#include "DesignDB.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "CmdSwitcher.h"
#include "WidgetManager.h"
#include "CallbackMgr.h"

#undef  MODULE_DEBUG
#include "debug.h"

#undef ALL_PATHEL_REGEXP

static int TclCmd_DdbInstCmd(ClientData cdata, Tcl_Interp *interp,
                      int argc, char **argv);

static int DesignDB_ScanDesign(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        Tcl_Obj          *const objv[]);

/*********************************************************
 * DesignDB()
 *********************************************************/
DesignDB::DesignDB(
        Tcl_Interp    *interp,
        int            argc,
        char         **argv) : instName(argv[1]), tmpString(256)
{
    TreeNode    *moduleType, *signalType, *functionType;
    TreeField   *tmpField;

    this->interp = interp;

    Tcl_CreateCommand(interp, argv[1], 
            (Tcl_CmdProc *)TclCmd_DdbInstCmd, this, NULL);
    WidgetMgr_AddInst(interp, WIDGET_TYPE_DDB, argv[1], this);

    Tcl_AppendResult(interp, argv[1], 0);

    simName        = 0;
    sim            = 0;
    dfioName       = 0;
    dfio           = 0;
    LoadDesignCbId = 0;
    sepChar = '.'; 

    /**** Setup callbacks... ****/

    /**** Bail if there are no resources to obtain info from... ****/

    designTree = new TreeNode(sepChar);

    populateTree(designTree);

    tmpGlobVect     = new Vector<TreeNode>;
    matchList       = new Vector<TreeNode>;
    nextMatchList   = new Vector<TreeNode>;

    /*** Create the TaskType type ***/

    /**** Configure attributes ****/
    if (Configure(argc-2, &argv[2], 0) != TCL_OK) {
        fprintf(stderr, "ERROR :: Configure failed\n");
        fprintf(stderr, "\t%s\n", Tcl_GetStringResult(interp));
        return;
    }
}

/********************************************************************
 * getConfigSpec()
 ********************************************************************/
Tk_ConfigSpec *DesignDB::getConfigSpec()
{
    static Tk_ConfigSpec   ddbConfigSpec[] = {
        {TK_CONFIG_STRING, "-sim", "sim", "Sim",
            (char *)NULL, Tk_Offset(DesignDB, simName), 0},
        {TK_CONFIG_STRING, "-dfio", "dfio", "DFIO",
            (char *)NULL, Tk_Offset(DesignDB, dfioName), 0},
        {TK_CONFIG_END, (char *)NULL, (char *)NULL, (char *)NULL,
            (char *)NULL, 0, 0}
    };
    return ddbConfigSpec;
}

/********************************************************************
 * Configure()
 ********************************************************************/
int DesignDB::Configure(Uint32 argc, Char **argv, Uint32 flags)
{
    Int32 ret;

    ret = Tcl_ConfigureWidget(interp, 0, getConfigSpec(), argc, argv,
            (char *)this, flags);

    if (optionSpecified(Cfg_Sim) && simName) {
        sim = (IviSim *)WidgetMgr_GetObjHandle(interp, "IviSim", simName);

        if (!sim) {
            Tcl_AppendResult(interp, "no IviSim object \"", simName, "\"", 0);
            return TCL_ERROR;
        }

        if (LoadDesignCbId) {
            CallbackMgr_DeleteCb(LoadDesignCbId);
        }

        CallbackMgr_AddCb(CBTYPE_SIM_DESIGN_LOAD, simName, 
                DesignDB_ScanDesign, this, &LoadDesignCbId);

//        scanDesign();
    }

    if (optionSpecified(Cfg_Dfio) && dfioName) {
        dfio = (DFIO *)WidgetMgr_GetObjHandle(interp, "DFIO", dfioName);

        if (!dfio) {
            Tcl_AppendResult(interp, "no DFIO object \"", dfioName, "\"", 0);
            return TCL_ERROR;
        }

        scanDFIO();
    }

    return ret;
} 
/*********************************************************
 * populateTree
 *********************************************************/
void DesignDB::populateTree(TreeNode *tree)
{
    TreeNode    *moduleType, *signalType, *functionType;
    TreeField   *tmpField;

    tree->addChild("@modules@");

    /*** Create the ModuleType type ***/
    moduleType = tree->newNodeType("ModuleType");
    moduleType->addChild("@signals@");
    moduleType->addChild("@functions@");
    moduleType->addChild("@tasks@");
    moduleType->addChild("@modules@");
    tmpField = new TreeField("vpiHandle", (void *)0);
    moduleType->addField(tmpField);
    tmpField = new TreeField("fullName",  "");
    moduleType->addField(tmpField);

    /*** Create the SignalType type ***/
    signalType = tree->newNodeType("SignalType");
    tmpField = new TreeField("vpiHandle", (void *)0);
    signalType->addField(tmpField);
    tmpField = new TreeField("sigType", (void *)0);
    signalType->addField(tmpField);
    tmpField = new TreeField("sigMsb", (void *)0);
    signalType->addField(tmpField);
    tmpField = new TreeField("sigLsb", (void *)0);
    signalType->addField(tmpField);
    tmpField = new TreeField("fullName",  "");
    signalType->addField(tmpField);
   
    /*** Create the FunctionType type ***/
    functionType = tree->newNodeType("FunctionType");
    functionType->addChild("parameters");
    tmpField = new TreeField("return", (void *)0);
    functionType->addField(tmpField);
    functionType->addChild("@signals@"); 
}

/*********************************************************
 * scanTrace()
 *
 * Helper for scanDFIO. Returns a tree-node for the
 * specified signal...
 *********************************************************/
void DesignDB::scanTrace(DFIOTrace *trace)
{
    TreeNode    *modules, *node, *tmp_node, *signals;
    Int32        el_start=0, el_end;

    tmpString = trace->get_name_str();

    /**** Start at design root... ****/
    modules = designTree->findNode("@modules@");

    node = designTree;
    /**** Start with the first element of the path... ****/
    while ((el_end = tmpString.strchr(sepChar, el_start)) > 0) {
        tmpString[el_end] = 0;

        /**** Name is now el_start-> string_end ****/
        if (!(tmp_node = node->findNode(&tmpString.value()[el_start]))) {
            modules = node->findNode("@modules@");

            /**** Add a node... ****/
            tmp_node = designTree->newNodeOfType("ModuleType",
                    &tmpString.value()[el_start]);
            tmp_node->findField("fullName")->setStringVal(
                    tmpString.value());

            modules->addChild(tmp_node);
            node->addChild(tmp_node);
        }

        tmpString[el_end] = sepChar;
        node = tmp_node;
        el_start = el_end+1;
    }

    /**** When there is no more sepChar's, we're left with one path
     **** element. This is the name of the signal... This extends from
     **** el_start to the (real) string-end
     ****/

    signals = node->findNode("@signals@");

    tmp_node = node->newNodeOfType("SignalType",
            &tmpString.value()[el_start]);
    tmp_node->findField("vpiHandle")->setVoidVal(trace);
    tmp_node->findField("fullName")->setStringVal(trace->get_name());

    tmp_node->findField("sigMsb")->setVoidVal((void *)trace->get_msb());
    tmp_node->findField("sigLsb")->setVoidVal((void *)trace->get_lsb());

    signals->addChild(tmp_node);
    node->addChild(tmp_node);
}


/*********************************************************
 * scanDFIO()
 *
 * Scans a DFIO for structure information...
 *********************************************************/
void DesignDB::scanDFIO()
{
    Uint32       i, num_traces;
    DFIOTrace   *trace;

    if (!dfio) {
        return;
    }

    designTree = new TreeNode(sepChar);
    populateTree(designTree);

    /**** Okay, scan through the traces in a linear manner... ****/
    num_traces = dfio->traces->length();

    for(i=0; i<num_traces; i++) {
        trace = dfio->traces->idx(i);

        /**** Scan this trace's name to generate structure info... ****/
        scanTrace(trace);
    }
}

/*********************************************************
 * scanDesign()
 * Use the simulator's VPI interface to obtain info on 
 * the design. This builds the design tree...
 *********************************************************/
void DesignDB::scanDesign()
{
    vpiHandle    modIt, modHndl;
    TreeNode    *thisLev;
    TreeNode    *module;
    TreeNode    *modules;

    if (!sim) {
        return;
    }

    if (designTree) {
        delete designTree;
    }

    designTree = new TreeNode(sepChar);
    thisLev = designTree;
    populateTree(designTree);

    modules     = designTree->findNode("@modules@");

    if (modIt = sim->vpi_iterate(vpiModule, 0)) {
        while (modHndl = sim->vpi_scan(modIt)) {
            module = scanModule(thisLev, modHndl);
            designTree->addChild(module);
            modules->addChild(module);
        }
    }

    CallbackMgr_Invoke(CBTYPE_DDB_UPDATE, instName.value(), 0, 0);

    DEBUG_1("    Top-level = %s\n", module->nodeName->value());
}

/*********************************************************
 * scanModule()
 *********************************************************/
TreeNode *DesignDB::scanModule(TreeNode *parent, vpiHandle modHndl)
{
    TreeNode    *module, *tmpNode;
    TreeNode    *signals, *tasks, *functions, *modules;
    vpiHandle    modIt, modIn;

    module = parent->newNodeOfType("ModuleType", 
             sim->vpi_get_str(vpiName, modHndl));
    module->findField("vpiHandle")->setVoidVal(modHndl);
    module->findField("fullName")->setStringVal(
        sim->vpi_get_str(vpiFullName, modHndl));

    signals     = module->findNode("@signals@");
    tasks       = module->findNode("@tasks@");
    functions   = module->findNode("@functions@");

   /**** modules holds the module nodes that are sub-modules of 
    **** this module
    ****/
    modules     = module->findNode("@modules@");

    DEBUG_1("Scanning module \"%s\"\n", sim->vpi_get_str(vpiName, modHndl));

    if (modIt = sim->vpi_iterate(vpiModule, modHndl)) {
        while ((modIn = sim->vpi_scan(modIt))) {

            if (sim->vpi_get(vpiType, modIn) != vpiModule) {
                fprintf(stderr, "ERROR :: type is NOT Module\n");
            }

            DEBUG_0("    vpiModule\n");
            tmpNode = scanModule(parent, modIn);
            modules->addChild(tmpNode);
            module->addChild(tmpNode);
        }
    }

    if (modIt = sim->vpi_iterate(vpiTask, modHndl)) {
        while ((modIn = sim->vpi_scan(modIt))) {
            if (sim->vpi_get(vpiType, modIn) != vpiTask) {
                fprintf(stderr, "ERROR :: type is NOT vpiTask\n");
            }

            tmpNode = scanModule(parent, modIn);
            modules->addChild(tmpNode);
            module->addChild(tmpNode);
        }
    }

    if (modIt = sim->vpi_iterate(vpiReg, modHndl)) {
        while ((modIn = sim->vpi_scan(modIt))) {
            if (sim->vpi_get(vpiType, modIn) != vpiReg &&
                    sim->vpi_get(vpiType, modIn) != vpiIntegerVar) {
                fprintf(stderr, "ERROR :: type is NOT Reg\n");
                fprintf(stderr, "\ttype is \"%s\"\n", 
                        IviSim::GetVpiObjString(sim->vpi_get(vpiType, modIn)));
            }

            DEBUG_0("    vpiReg\n");
            tmpNode = scanSignal(module, modIn, (Uint32)vpiReg); 
            signals->addChild(tmpNode);
            module->addChild(tmpNode);
        }
    }

    if (modIt = sim->vpi_iterate(vpiNet, modHndl)) {
        while ((modIn = sim->vpi_scan(modIt))) {
            if (sim->vpi_get(vpiType, modIn) != vpiNet) {
                fprintf(stderr, "ERROR :: type is NOT Net\n");
            }

            DEBUG_0("    vpiNet\n");
            tmpNode = scanSignal(module, modIn, (Uint32)vpiNet); 
            signals->addChild(tmpNode);
            module->addChild(tmpNode);
        }
    }

    if (modIt = sim->vpi_iterate(vpiInternalScope, modHndl)) {
        while (modIn = sim->vpi_scan(modIt)) {
            if (sim->vpi_get(vpiType, modIn) != vpiModule) {
                /*
                fprintf(stderr, "NOTE :: InternalScope :: %s\n",
                        sim->vpi_get_str(vpiName, modIn));
                 */
            }
        }
    }

    return module;     
}

/*********************************************************
 * globElems
 *
 * Parameters:
 * - Parent
 *   Specifies an alternate root-node to use. This can
 *   be used to allow scope-relative searching
 *   Parent will be used if:
 *   - If the path is not '.'-rooted
 *   - Parent is non-null
 * - PathSpec
 *   A '.'-delimited path specifying the path to a 
 *   resource.
 * - elemType
 *   Specifies the type of element of interest. May be:
 *   - @modules@
 *   - @signals@
 *   - @functions@
 *   - @tasks@
 * - flags
 *   Allows user-specified search options such as -r
 *********************************************************/
Vector<TreeNode> *DesignDB::globElems(
    TreeNode     *Parent, 
    const char   *PathSpec,
    const char   *elemType,
    Uint32        flags)
{
    Char                buf[1024], *tmpPath, mbuf[128];
    TreeNode           *module, *child, *signals, *modules, *searchRoot = 0; 
    TreeNode           *resource;
    Vector<TreeNode>   *children, *modulev;
    Char               *slPtr, *nPtr;
    Char               *elemPtrs[128], tmpbuf[1024], *sptr;
    Uint32              numElems = 0, currElem = 0;
    Uint32              i, x, y;
    Int32               matchRes, sigs;
    Tcl_RegExp          regexp;

    DEBUG_3("globSigs(%s, %s, %d)\n", PathSpec, elemType, flags);
  
   /******************************************************** 
    **** Create indexes into the path specification 
    **** - In addition, decide whether to use designTree
    ****   or the supplied Parent node as the search root
    ********************************************************/
    tmpPath = buf;
    slPtr = 0;
    strcpy(tmpPath, PathSpec);
    if (*tmpPath == sepChar) {
        tmpPath++;
        searchRoot = designTree; 
    } else {
        if (Parent) {
            searchRoot = Parent;
        } else {
            searchRoot = designTree;
        }
    } 

    numElems = 0;
    do {
        if (slPtr = strchr(tmpPath, sepChar)) {
            *slPtr = 0;
            elemPtrs[numElems++] = tmpPath;
        } else if (*tmpPath != 0) {
            elemPtrs[numElems++] = tmpPath;
        }
        if (slPtr) tmpPath += (strlen(tmpPath)+1);
    } while (slPtr);

   /********************************************************
    **** Initialize the lists...
    ********************************************************/
    tmpGlobVect->empty();
    matchList->empty();
    nextMatchList->empty();

   /********************************************************
    **** Locate the root modules
    **** - First search 
    ********************************************************/
    DEBUG_1("    numElems = %d\n", numElems);
  
   /**** If path is of form '/pattern' 
    **** - Only the final loop will be executed.
    **** - FinalLoop expects root modules if (elemType != @modules@)
    ****     - else expects searchRoot
    ****/ 
    if (String::equal(elemType, "@modules@") && (numElems == 0)) {
        DEBUG_0("    numElems == 0 and resource = modules\n");
        modules = searchRoot->findNode("@modules@");
        children = modules->getChildren();
        tmpGlobVect->append(children);
    } else {
        if (numElems > 0) {
            modules = searchRoot->findNode("@modules@");
            children = modules->getChildren();
            matchList->append(children);
        }
    } 


   /**** Now, modify the search root further.
    **** - We have the list of root modules.
    **** - If the first path element does not refer to one of the
    ****   root modules, then check to see if the first path element
    ****   matches one of the children of one of the root modules
    ****/
    if (numElems > 1) {
#ifdef ALL_PATHEL_REGEXP
        regexp = Tcl_RegExpCompile(interp, elemPtrs[0]);
#endif
        for (i=0; i<matchList->length(); i++) {
            DEBUG_2("    Checking %s == %s\n",
                matchList->idx(i)->nodeName->value(),
                elemPtrs[0]); 
#ifdef ALL_PATHEL_REGEXP
            matchRes = Tcl_RegExpExec(interp, regexp, 
                matchList->idx(i)->nodeName->value(),
                matchList->idx(i)->nodeName->value());
#else
            matchRes = String::equal(
                matchList->idx(i)->nodeName->value(),
                elemPtrs[0]);
#endif

            if (matchRes == 1) {
                nextMatchList->append(matchList->idx(i));
                DEBUG_1("    Adding node \"%s\"\n", matchList->idx(i)->nodeName->value());
            } else if (matchRes == -1) {
                goto ddb_exit;
            }
        }

        /**** If none of the roots matched, then try the 
         **** children of the roots
         ****/
        if (nextMatchList->length() == 0) {
            DEBUG_0("    nextMatchList->length() == 0\n");
            for (i=0; i<matchList->length(); i++) {
                module = matchList->idx(i); 
                children = module->findNode("@modules@")->getChildren();
                for (y=0; y<children->length(); y++) { 
                    DEBUG_2("        Checking %s == %s\n",
                        children->idx(y)->nodeName->value(),
                        elemPtrs[0]);
#ifdef ALL_PATHEL_REGEXP
                    matchRes = Tcl_RegExpExec(interp, regexp, 
                        children->idx(y)->nodeName->value(),
                        children->idx(y)->nodeName->value());
#else
                    matchRes = String::equal(
                        children->idx(y)->nodeName->value(),
                        elemPtrs[0]);
#endif

                    if (matchRes == 1) {
                        nextMatchList->append(children->idx(y));
                    } else if (matchRes == -1) {
                        goto ddb_exit;
                    }
                }
            }
        } 

        if (nextMatchList->length() == 0) {
            goto ddb_exit;
        } else {
            matchList->setEqual(nextMatchList);
            nextMatchList->empty();
        }           
    }         


    for (i=0; (i<numElems && i<(numElems-1)); i++) {
#ifdef ALL_PATHEL_REGEXP
        regexp = Tcl_RegExpCompile(interp, elemPtrs[i]);
#endif
        for (x=0; x<matchList->length(); x++) {
            module = matchList->idx(x);
            DEBUG_1("    Pulling off module \"%s\"\n", module->nodeName->value());
            DEBUG_2("        matching against elems[%d] = %s\n", i, elemPtrs[i]);
           
#ifdef ALL_PATHEL_REGEXP
            matchRes = Tcl_RegExpExec(interp, regexp,
                module->nodeName->value(), 
                module->nodeName->value());
#else
            matchRes = String::equal(
                module->nodeName->value(), 
                elemPtrs[i]);
#endif

            DEBUG_1("         matchRes = %d\n", matchRes);
            if (matchRes == 1) {
                if (i < (numElems-2)) {
                    children = module->findNode("@modules@")->getChildren();
                    for (y=0; y<children->length(); y++) {
                        DEBUG_1("    Adding %s\n", 
                                children->idx(y)->nodeName->value());
                        nextMatchList->append(children->idx(y));
                    }
                } else {
                    nextMatchList->append(module);
                }
            } else if (matchRes == -1) {
                fprintf(stderr, "ERROR: bad pattern!!\n");
                goto ddb_exit;
            }
        }

        matchList->setEqual(nextMatchList);
        nextMatchList->empty();
    }

    /**** FinalLoop
     **** - retrive childNode "elemType" from each element of
     ****   matchList.
     **** - Locate the children of these nodes
     **** - Match the childrens' node-names against the
     ****   final path-segment
     ****/
    DEBUG_0("Entering final loop...\n");
#ifdef UNDEFINED
    if (matchList->length()) {
        if ((sptr = strchr(elemPtrs[numElems-1], '*'))) {
            memcpy(tmpbuf, elemPtrs[numElems-1], sptr-elemPtrs[numElems-1]);
            tmpbuf[(sptr-elemPtrs[numElems-1])] = 0;
	    strcat(tmpbuf, ".");
	    strcat(tmpbuf, sptr);
            regexp = Tcl_RegExpCompile(interp, tmpbuf);
	} else {
            regexp = Tcl_RegExpCompile(interp, elemPtrs[numElems-1]);
	}

	if (!regexp) {
                fprintf(stderr, "ERROR: Bad reg-exp \"%s\"\n",
                    elemPtrs[numElems-1]);
                goto ddb_exit;
        }
    }
#endif

    for (i=0; i<matchList->length(); i++) {
        module = matchList->idx(i);

        resource = module->findNode((char *)elemType);
        if (resource == 0) {
            goto ddb_exit;
        }
 
        /**** Now, get the children of this 'resource' node
         ****/
        children = resource->getChildren();

        for (x=0; x<children->length(); x++) {
            matchRes = children->idx(x)->nodeName->match(elemPtrs[numElems-1]);

            /*
            matchRes = Tcl_RegExpExec(interp, regexp,
                children->idx(x)->nodeName->value(),
                children->idx(x)->nodeName->value());
             */
            if (matchRes == 1) {
                tmpGlobVect->append(children->idx(x));
            } else if (matchRes == -1) {
                fprintf(stderr, "ERROR: bad pattern \"%s\"\n",
                    elemPtrs[numElems-1]);
                goto ddb_exit;
            } 
        }  
    }

ddb_exit:
    return tmpGlobVect;
}


/*********************************************************
 * scanSignal()
 *********************************************************/
TreeNode *DesignDB::scanSignal(TreeNode *parent, vpiHandle sigHndl,
                               Uint32    sigType)
{
    TreeNode *newSig;
    vpiHandle    sigBounds;
    s_vpi_value  v_val;
    s_vpi_value  r_val, l_val;

    DEBUG_1("    scanSignal(%s)\n", sim->vpi_get_str(vpiName, sigHndl));

    newSig = parent->newNodeOfType("SignalType", 
                                     sim->vpi_get_str(vpiName, sigHndl));

    newSig->findField("vpiHandle")->setVoidVal(sigHndl);
    newSig->findField("sigType")->setVoidVal((void *)sigType);
    newSig->findField("fullName")->setStringVal(
        sim->vpi_get_str(vpiFullName, sigHndl));

    r_val.format = vpiIntVal;
    r_val.value.integer = 0;
    sigBounds = sim->vpi_handle(vpiRightRange, sigHndl);
    sim->vpi_get_value(sigBounds, &r_val);

    l_val.format = vpiIntVal;
    l_val.value.integer = 0;
    sigBounds = sim->vpi_handle(vpiLeftRange, sigHndl);
    sim->vpi_get_value(sigBounds, &l_val);

    if (l_val.value.integer == r_val.value.integer) {
        l_val.value.integer = r_val.value.integer = 0;
    }

    DEBUG_1("        lsb = 0x%08x\n", r_val.value.integer);
    newSig->findField("sigLsb")->setVoidVal((void *)r_val.value.integer);

    DEBUG_1("        msb = 0x%08x\n", l_val.value.integer);
    newSig->findField("sigMsb")->setVoidVal((void *)l_val.value.integer);

    return newSig;
}

/*********************************************************
 * getSignals
 *
 * Parameters:
 *    - parent
 *        Allows relative pathing. If set to NULL, then
 *    absolute path is used...
 *
 * Returns:
 *    - Number of signals found
 *********************************************************/
int DesignDB::getSignals(
    TreeNode         *parent, 
    Vector<TreeNode> *foundSigs,
    Char            **matchPatterns,
    Uint32            numPatterns
    )
{
    Uint32   i;
    Char    *pat;

    return -1;
}

/*********************************************************
 * dumpDesign()
 * 
 * This is a testing routine, used to verify that the
 * design has been read in correctly...
 *********************************************************/
void DesignDB::dumpDesign(void)
{
    Vector<TreeNode>    *children;

}

/*********************************************************
 * makeTclHandle()
 *
 * Handle is of format:
 *   ddb:TreeNode:vpiHandle
 *********************************************************/
int DesignDB::makeTclHandle(char *desPath, TreeNode *startPt,
                            char *tclHandle)
{
    TreeNode   *target;
    TreeField  *field;

    if (!startPt) startPt = designTree;

    if (desPath[0] == sepChar && desPath[1] == 0) {
        target = startPt->getChildren()->idx(0);
        if (target == 0) return 0;
    } else {
        target = startPt->findNode(desPath);
        if (target == 0) return 0;
    }

    field = target->findField("vpiHandle");
    if (field == 0) {
        fprintf(stderr, "ERROR: can't find field vpiHandle!!\n");
        return 0;
    }

    sprintf(tclHandle, "ddb:0x%08x:0x%08x", 
            target, target->findField("vpiHandle")->getVoidVal()); 
 
    return 1;
}

/*********************************************************
 * tclCmdGlob()
 *********************************************************/
typedef enum {
    GlobOpt_Signals    = 1,
    GlobOpt_Modules    = 2,
    GlobOpt_Size       = 3
 } globOptType;

static OptTypeStruct    globCmdOpts[] = {
    {"-sigs",    GlobOpt_Signals, true},
    {"-signals", GlobOpt_Signals, true},
    {"-mod",     GlobOpt_Modules, true}, 
    {"-mods",    GlobOpt_Modules, true}, 
    {"-modules", GlobOpt_Modules, true}, 
    {"-size",    GlobOpt_Size,    true},
    {"",         0,              false} 
 };

int DesignDB::tclCmdGlob(
    int         argc,
    char      **argv
    )
{
    Char                *argvO[128], *optO[128], buf[128];
    Int32                optTO[128], ret;
    Uint32               argcO, optcO, i, getSigs = 1, len = 0;
    Uint32               msb, lsb;
    Vector<TreeNode>    *result;
    Char                *elemType = "@signals@"; 
    boolean              size     = false;
    Tcl_Obj             *name_list[512], *sigList = 0;
    

    ret = ParseArgs(globCmdOpts, argc, argv, &argcO, argvO,
              &optcO, optO, optTO);

    for (i=0; i<optcO; i++) {
        switch(optTO[i]) {
            case GlobOpt_Signals:
                elemType = "@signals@";
                getSigs = 1;
            break;
 
            case GlobOpt_Modules:
                elemType = "@modules@";
            break;

            case GlobOpt_Size: 
                size = true;
            break;
        }
    }

    if (argcO < 1) {
	Tcl_AppendResult(interp, "No pathSpec given\n", 0);
	goto ddb_error_exit;
    }

    result = globElems(0, argvO[0], elemType, 0); 

   /**** Process result list...
    ****/

    for (i=0; (i<512 && i<result->length()); i++) {
        if (size && getSigs) { 
            msb = (Uint32)result->idx(i)->findField("sigMsb")->getVoidVal();
            lsb = (Uint32)result->idx(i)->findField("sigLsb")->getVoidVal();
            if (lsb || msb) {
                len = sprintf(buf, "%s[%d:%d]", 
                    result->idx(i)->nodeName->value(),
                    msb, lsb);
            } else {
                len = sprintf(buf, "%s", 
                    result->idx(i)->nodeName->value());
            }
        } else {
            len = sprintf(buf, "%s", result->idx(i)->nodeName->value());
        }
        name_list[i] = Tcl_NewStringObj(buf, len);
    }

    sigList = Tcl_NewListObj((result->length()>=512)?512:
                              result->length(), name_list);

    Tcl_SetObjResult(interp, sigList);

ddb_exit:
    return TCL_OK;

ddb_error_exit:
    Tcl_AppendResult(interp, "ddbPath glob [options] pathSpec\n", 
        "options: -signals, -modules, -size", -0);
    return TCL_ERROR;
}


/*********************************************************
 * tclSigNameList()
 *********************************************************/
Tcl_Obj *DesignDB::tclSigNameList(char *modPath, TreeNode *startPt,
                                  char *resource)
{
    Vector<TreeNode>   *list;
    Tcl_Obj             *name_list[512];
    Tcl_Obj             *ret = 0;
    Uint32               i, msb, lsb;
    Char                 buf[128];

    if (modPath[1] == 0) {
        sprintf(buf, "%s.*", modPath);
    } else { 
        sprintf(buf, "%s/.*", modPath);
    }

    list = globElems(0, modPath, 
             (!strcmp(resource, "signals"))?
             (char *)"@signals@":
             (char *)"@modules@", 0);

   
    for (i=0; (i<512 && i<list->length()); i++) {
        if (!strcmp(resource, "signals")) {
            msb = (Uint32)list->idx(i)->findField("sigMsb")->getVoidVal();
            lsb = (Uint32)list->idx(i)->findField("sigLsb")->getVoidVal();
            if (lsb || msb) {
                sprintf(buf, "%s[%d:%d]", list->idx(i)->nodeName->value(),
                             msb, lsb);
            } else {
                sprintf(buf, "%s", list->idx(i)->nodeName->value());
            }
            name_list[i] = Tcl_NewStringObj(buf, strlen(buf));
        } else {
            name_list[i] = Tcl_NewStringObj(
                list->idx(i)->nodeName->value(),
                list->idx(i)->nodeName->length());
        }
    }
         

   ret = Tcl_NewListObj(list->length(), name_list);

   return ret;
}

/*********************************************************
 * ddbCmds
 *********************************************************/
typedef enum {
    DDB_Glob    = 1,
    DDB_Scan    = 2,
    DDB_Config  = 3,
    DDB_Cget    = 4
} DdbReaderCmds;

static CmdSwitchStruct    ddbCmds[] = {
    {"glob",           DDB_Glob},
    {"scan_design",    DDB_Scan},
    {"config",         DDB_Config},
    {"configure",      DDB_Config},
    {"cget",           DDB_Cget},
    {"",               0}
    };

/********************************************************************
 * InstCmd()
 ********************************************************************/
int DesignDB::InstCmd(int argc, char **argv)
{

    if (argc < 1) {
        Tcl_AppendResult(interp, 
         "ERROR: too few arguments\n    ",
         "ddb [get] [path_to_entity]", 0);
        return TCL_ERROR;
    }

    switch(CmdSwitch(ddbCmds, argv[0])) {
        case DDB_Config:
            if (argc < 3) {
                Tcl_AppendResult(interp, "too few args", 0);
                return TCL_ERROR;
            }
            return Configure(argc-1, &argv[1], TK_CONFIG_ARGV_ONLY);
            break;

        case DDB_Cget:
            if (argc < 2) {
                Tcl_AppendResult(interp, "too few args", 0);
                return TCL_ERROR;
            }
            return Tcl_ConfigureValue(interp, 0, getConfigSpec(),
                    (char *)this, argv[1], 0);
            break;

        case DDB_Glob:
            tclCmdGlob(argc-1, &argv[1]); 
        break;

        case DDB_Scan:
            scanDesign();
        break;

        default:
            fprintf(stderr, "ERROR: Unknown ddb command!!\n");
        break;
    }
 
   return TCL_OK;
}

/********************************************************************
 * TclCmd_DdbInstCmd()
 ********************************************************************/
static int TclCmd_DdbInstCmd(ClientData cdata, Tcl_Interp *interp,
                      int argc, char **argv)
{
    DesignDB   *ddb = (DesignDB *)cdata;
    return ddb->InstCmd(argc-1, &argv[1]);
}

/*********************************************************
 * get_sim()
 *********************************************************/
IviSim *DesignDB::get_sim()
{
    return sim;
}

/*********************************************************
 * get_dfio()
 *********************************************************/
DFIO *DesignDB::get_dfio()
{
    return dfio;
}

/*********************************************************
 * DesignDB_ScanDesign()
 *********************************************************/
static int DesignDB_ScanDesign(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        Tcl_Obj          *const objv[])
{
    DesignDB *ddb = (DesignDB *)clientData;
    ddb->scanDesign();
    return TCL_OK;
}

/*********************************************************
 * TclCmd_ddb()
 *********************************************************/
static int TclCmd_ddb(ClientData cdata, Tcl_Interp *interp,
                      int argc, char **argv)
{
    ClientData   cbId;
    Char        *dfioName = 0;
    Char        *simName  = 0;

    if (argc < 2) {
        Tcl_AppendResult(interp,
            "wrong # args", NULL);
        return TCL_ERROR;
    }

    DesignDB *ddb = new DesignDB(interp, argc, argv);
 
    return TCL_OK; 
}

/*********************************************************
 * DesignDB_Register()
 *
 * This function is called during the registration stage.
 * it must register callbacks for certain events...
 *********************************************************/
int DesignDB_Register(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        char            **argv)
{
    return -1;
}


/*********************************************************
 * DesignDB_Init()
 *********************************************************/
int DesignDB_Init(Tcl_Interp *interp)
{
    WidgetMgr_AddType(interp, WIDGET_TYPE_DDB);
    Tcl_CreateCommand(interp, "ddb", (Tcl_CmdProc *)TclCmd_ddb, 0, NULL);
    CallbackMgr_AddCbType(CBTYPE_DDB_UPDATE);
    return TCL_OK;
}


