/* im-protocol-version.c */


#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>

#include <iiimp-data.h>
#include <iiimp-opcode.h>

#include "iiimp-dataP.h"

/*
  IM_PROTOCOL_VERSION / V3

  im-id:	CARD16
  version:	CARD8
  UNUSED:	Pad(3) = 1
*/

IIIMP_message *
iiimp_protocol_version_new(
    IIIMP_data_s *	data_s,
    IIIMP_card16	im_id,
    int			protocol_version
)
{
    IIIMP_message *	m;

    REQUIRE_PROTOCOL_VERSION(data_s, 3);

    m = (IIIMP_message *)malloc(sizeof (IIIMP_message));
    if (NULL == m) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }
    m->opcode = IM_PROTOCOL_VERSION;
    m->im_id = im_id;
    m->ic_id = -1;

    if (data_s->protocol_version < protocol_version) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    m->v.protocol_version.number = protocol_version;

    return m;
}


void
iiimp_protocol_version_delete(
    IIIMP_data_s * data_s,
    IIIMP_message * m
)
{
    if (NULL == m) return;
    free(m);
    return;
}


uchar_t *
iiimp_protocol_version_pack(
    IIIMP_data_s *	data_s,
    IIIMP_card16	im_id,
    int			protocol_version,
    size_t *		buf_size
)
{
    size_t		nbyte;
    int			length;
    uchar_t *		buf;
    size_t		rest;
    uchar_t *		p;

    nbyte = 4; /* im-id(2) + version(1) + UNUSED(1) */

    length = (nbyte >> 2);
    *buf_size = (1 + 3 + nbyte);

    buf = (uchar_t *) malloc(1 + 3 + nbyte);
    if (NULL == buf) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    PUT_PACKET_HEADER(buf, IM_PROTOCOL_VERSION, length);

    rest = nbyte;
    p = (buf + 4);

    PUTU16(im_id, rest, p, data_s->byte_swap);
    PUTU8(protocol_version, rest, p, data_s->byte_swap);
    PUTU8(0, rest, p, data_s->byte_swap);

    return buf;
}


IIIMP_message *
iiimp_protocol_version_unpack(
    IIIMP_data_s *	data_s,
    IIIMP_card7		opcode,
    size_t *		nbyte,
    const uchar_t **	ptr)
{
    IIIMP_message *		m;
    IIIMP_protocol_version_v *	v;
    size_t			rest;
    const uchar_t *		p;

    rest = *nbyte;
    p = *ptr;

    if (rest < (2 + 1 + 1)) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    m = (IIIMP_message *)malloc(sizeof (IIIMP_message));
    if (NULL == m) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }
    v = &(m->v.protocol_version);

    m->opcode = opcode;
    GETU16(m->im_id, rest, p, data_s->byte_swap);
    m->ic_id = -1;
    GETU8(v->number, rest, p, data_s->byte_swap);
    SKIP8(rest, p);

    *nbyte = rest;
    *ptr = p;

    return m;
}


void
iiimp_protocol_version_print(
    IIIMP_data_s *	data_s,
    IIIMP_message *	m)
{
    IIIMP_protocol_version_v *	v;

    v = &(m->v.protocol_version);

    iiimp_message_header_print(data_s, m->opcode, m->im_id, -1);
    (void)fprintf(data_s->print_fp, "\tprotocol_version = 0x%02x\n",
		  v->number);
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
