/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
                                                                                
/*
 * Modified by the GTK+ Team and others 1997-2000.  See the AUTHORS
 * file for a list of people on the GTK+ Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GTK+ at ftp://ftp.gtk.org/pub/gtk/.
 */
                                                                                
/******************************************************
*Imported from gtk+-2.0 source by phill.zhang@sun.com
******************************************************/

#include <math.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <pango/pango.h>

#include <gtk/gtkmain.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkclipboard.h>
#include <gtk/gtkimagemenuitem.h>
#include <gtk/gtkseparatormenuitem.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkstock.h>
#include <gtk/gtkbindings.h>

//#include "gtkmarshalers.h"

#include "imlabel.h"

#define _(a)  (a)

struct _ImLabelSelectionInfo
{
  GdkWindow *window;
  gint selection_anchor;
  gint selection_end;
  GtkWidget *popup_menu;
};

enum {
  MOVE_CURSOR,
  COPY_CLIPBOARD,
  POPULATE_POPUP,
  LAST_SIGNAL
};

enum {
  PROP_0,
  PROP_LABEL,
  PROP_ATTRIBUTES,
  PROP_USE_MARKUP,
  PROP_USE_UNDERLINE,
  PROP_JUSTIFY,
  PROP_PATTERN,
  PROP_WRAP,
  PROP_SELECTABLE,
  PROP_MNEMONIC_KEYVAL,
  PROP_MNEMONIC_WIDGET,
  PROP_CURSOR_POSITION,
  PROP_SELECTION_BOUND
};

static guint signals[LAST_SIGNAL] = { 0 };

static void im_label_class_init        (ImLabelClass    *klass);
static void im_label_init              (ImLabel         *label);
static void im_label_set_property      (GObject          *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void im_label_get_property      (GObject          *object, guint prop_id, GValue *value, GParamSpec *pspec);
static void im_label_destroy           (GtkObject        *object);
static void im_label_finalize          (GObject          *object);
static void im_label_size_request      (GtkWidget        *widget, GtkRequisition *requisition);
static void im_label_size_allocate     (GtkWidget        *widget, GtkAllocation *allocation);
static void im_label_state_changed     (GtkWidget        *widget, GtkStateType state);
static void im_label_style_set         (GtkWidget        *widget, GtkStyle *previous_style);
static void im_label_direction_changed (GtkWidget        *widget, GtkTextDirection  previous_dir);
static gint im_label_expose            (GtkWidget        *widget,
                                         GdkEventExpose   *event);

static void im_label_realize           (GtkWidget        *widget);
static void im_label_unrealize         (GtkWidget        *widget);
static void im_label_map               (GtkWidget        *widget);
static void im_label_unmap             (GtkWidget        *widget);
static gint im_label_button_press      (GtkWidget        *widget, GdkEventButton *event);
static gint im_label_button_release    (GtkWidget        *widget, GdkEventButton *event);
static gint im_label_motion            (GtkWidget        *widget, GdkEventMotion *event);


static void im_label_set_text_internal          (ImLabel      *label, gchar *str);
static void im_label_set_label_internal         (ImLabel      *label, gchar *str);
static void im_label_set_attributes_internal    (ImLabel      *label, PangoAttrList *attrs);
static void im_label_set_pattern_internal       (ImLabel      *label, const gchar *pattern);
static void set_markup                          (ImLabel      *label, const gchar *str, gboolean with_uline);
static void im_label_recalculate                (ImLabel      *label);

static void im_label_hierarchy_changed          (GtkWidget     *widget, GtkWidget *old_toplevel);
static void im_label_screen_changed             (GtkWidget     *widget, GdkScreen *old_screen);
static gboolean im_label_focus                  (GtkWidget     *widget, GtkDirectionType direction);

static void im_label_create_window       (ImLabel *label);
static void im_label_destroy_window      (ImLabel *label);
static void im_label_clear_layout        (ImLabel *label);
static void im_label_ensure_layout       (ImLabel *label);
static void im_label_select_region_index (ImLabel *label, gint anchor_index, gint end_index);



/* For selectable lables: */
static void im_label_move_cursor        (ImLabel        *label, GtkMovementStep  step,  gint count, gboolean extend_selection);
static void im_label_copy_clipboard     (ImLabel        *label);
static void im_label_select_all         (ImLabel        *label);
static void im_label_do_popup           (ImLabel        *label, GdkEventButton  *event);
static gint im_label_move_forward_word  (ImLabel        *label, gint start);
static gint im_label_move_backward_word (ImLabel        *label, gint start);

static GtkMiscClass *parent_class = NULL;

GType
im_label_get_type (void)
{
  static GType label_type = 0;

  if (!label_type)
    {
      static const GTypeInfo label_info =
      {
        sizeof (ImLabelClass),
        NULL,           /* base_init */
        NULL,           /* base_finalize */
        (GClassInitFunc) im_label_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (ImLabel),
        32,             /* n_preallocs */
        (GInstanceInitFunc) im_label_init,
      };

      label_type = g_type_register_static (GTK_TYPE_MISC, "ImLabel",
                                           &label_info, 0);
    }

  return label_type;
}

static void
im_label_class_init (ImLabelClass *class)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (class);
  GtkObjectClass *object_class = GTK_OBJECT_CLASS (class);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (class);
  GtkBindingSet *binding_set;

  parent_class = g_type_class_peek_parent (class);

  gobject_class->set_property = im_label_set_property;
  gobject_class->get_property = im_label_get_property;
  gobject_class->finalize = im_label_finalize;
  object_class->destroy = im_label_destroy;

  widget_class->size_request = im_label_size_request;
  widget_class->size_allocate = im_label_size_allocate;
  widget_class->state_changed = im_label_state_changed;
  widget_class->style_set = im_label_style_set;
  widget_class->direction_changed = im_label_direction_changed;
  widget_class->expose_event = im_label_expose;
  widget_class->realize = im_label_realize;
  widget_class->unrealize = im_label_unrealize;
  widget_class->map = im_label_map;
  widget_class->unmap = im_label_unmap;
  widget_class->button_press_event = im_label_button_press;
  widget_class->button_release_event = im_label_button_release;
  widget_class->motion_notify_event = im_label_motion;
  widget_class->screen_changed = im_label_screen_changed;
  widget_class->focus = im_label_focus;
  widget_class->hierarchy_changed = NULL;//im_label_hierarchy_changed;
  widget_class->mnemonic_activate = NULL;//im_label_mnemonic_activate;

  //class->move_cursor = im_label_move_cursor;
  //class->copy_clipboard = im_label_copy_clipboard;

/*
  signals[MOVE_CURSOR] =
    g_signal_new ("move_cursor",
                  G_OBJECT_CLASS_TYPE (gobject_class),
                  G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                  G_STRUCT_OFFSET (ImLabelClass, move_cursor),
                  NULL, NULL,
                  _gtk_marshal_VOID__ENUM_INT_BOOLEAN,
                  G_TYPE_NONE, 3,
                  GTK_TYPE_MOVEMENT_STEP,
                  G_TYPE_INT,
                  G_TYPE_BOOLEAN);

*/

  g_object_class_install_property (gobject_class,
                                   PROP_LABEL,
                                   g_param_spec_string ("label",
                                                        _("Label"),
                                                        _("The text of the label"),
                                                        NULL,
                                                        G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class,
                                   PROP_ATTRIBUTES,
                                   g_param_spec_boxed ("attributes",
                                                       _("Attributes"),
                                                       _("A list of style attributes to apply to the text of the label"),
                                                       PANGO_TYPE_ATTR_LIST,
                                                       G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class,
                                   PROP_USE_MARKUP,
                                   g_param_spec_boolean ("use_markup",
                                                         _("Use markup"),
                                                         _("The text of the label includes XML markup. See pango_parse_markup()"),
                                                        FALSE,
                                                        G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class,
                                   PROP_USE_UNDERLINE,
                                   g_param_spec_boolean ("use_underline",
                                                         _("Use underline"),
                                                         _("If set, an underline in the text indicates the next character should be used for the mnemonic accelerator key"),
                                                        FALSE,
                                                        G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
                                   PROP_JUSTIFY,
                                   g_param_spec_enum ("justify",
                                                      _("Justification"),
                                                      _("The alignment of the lines in the text of the label relative to each other. This does NOT affect the alignment of the label within its allocation. See GtkMisc::xalign for that"),
                                                      GTK_TYPE_JUSTIFICATION,
                                                      GTK_JUSTIFY_LEFT,
                                                      G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
                                   PROP_PATTERN,
                                   g_param_spec_string ("pattern",
                                                        _("Pattern"),
                                                        _("A string with _ characters in positions correspond to characters in the text to underline"),
                                                        NULL,
                                                        G_PARAM_WRITABLE));

  g_object_class_install_property (gobject_class,
                                   PROP_SELECTABLE,
                                   g_param_spec_boolean ("selectable",
                                                        _("Selectable"),
                                                        _("Whether the label text can be selected with the mouse"),
                                                        FALSE,
                                                        G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
                                   PROP_CURSOR_POSITION,
                                   g_param_spec_int ("cursor_position",
                                                     _("Cursor Position"),
                                                     _("The current position of the insertion cursor in chars"),
                                                     0,
                                                     G_MAXINT,
                                                     0,
                                                     G_PARAM_READABLE));

  g_object_class_install_property (gobject_class,
                                   PROP_SELECTION_BOUND,
                                   g_param_spec_int ("selection_bound",
                                                     _("Selection Bound"),
                                                     _("The position of the opposite end of the selection from the cursor in chars"),
                                                     0,
                                                     G_MAXINT,
                                                     0,
                                                     G_PARAM_READABLE));

}

void
im_label_set_caret_index(ImLabel *label, gint caret)
{
    g_return_if_fail (IM_IS_LABEL (label));

    if (caret >= 0)
        caret = g_utf8_offset_to_pointer (label->text, caret) - label->text;

    if (label->caret_index == caret)
        return;

    label->caret_index = caret;
    label->showing_caret = label->delay_hide_caret = 1;
    gtk_widget_queue_draw(GTK_WIDGET (label));
}

/*
gint
im_label_get_caret_index(ImLabel *label)
{
    return label->caret_index;
}
*/

static void
im_label_set_property (GObject      *object,
                        guint         prop_id,
                        const GValue *value,
                        GParamSpec   *pspec)
{
  ImLabel *label;

  label = IM_LABEL (object);

  switch (prop_id)
    {
    case PROP_LABEL:
      im_label_set_label (label, g_value_get_string (value));
      break;
    case PROP_ATTRIBUTES:
      im_label_set_attributes (label, g_value_get_boxed (value));
      break;
    case PROP_JUSTIFY:
      im_label_set_justify (label, g_value_get_enum (value));
      break;
    case PROP_PATTERN:
      im_label_set_pattern (label, g_value_get_string (value));
      break;
    case PROP_SELECTABLE:
      im_label_set_selectable (label, g_value_get_boolean (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
im_label_get_property (GObject     *object,
                        guint        prop_id,
                        GValue      *value,
                        GParamSpec  *pspec)
{
  ImLabel *label;

  label = IM_LABEL (object);

  switch (prop_id)
    {
    case PROP_LABEL:
      g_value_set_string (value, label->label);
      break;
    case PROP_ATTRIBUTES:
      g_value_set_boxed (value, label->attrs);
      break;
    case PROP_USE_MARKUP:
      g_value_set_boolean (value, label->use_markup);
      break;
    case PROP_USE_UNDERLINE:
      break;
    case PROP_JUSTIFY:
      g_value_set_enum (value, label->jtype);
      break;
    case PROP_SELECTABLE:
      g_value_set_boolean (value, im_label_get_selectable (label));
      break;
    case PROP_MNEMONIC_WIDGET:
      break;
    case PROP_CURSOR_POSITION:
      if (label->select_info)
        {
          gint offset = g_utf8_pointer_to_offset (label->text,
                                                  label->text + label->select_info->selection_end);
          g_value_set_int (value, offset);
        }
      else
        g_value_set_int (value, 0);
      break;
    case PROP_SELECTION_BOUND:
      if (label->select_info)
        {
          gint offset = g_utf8_pointer_to_offset (label->text,
                                                  label->text + label->select_info->selection_anchor);
          g_value_set_int (value, offset);
        }
      else
        g_value_set_int (value, 0);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
im_label_init (ImLabel *label)
{
  GTK_WIDGET_SET_FLAGS (label, GTK_NO_WINDOW);

  label->label = NULL;

  label->jtype = GTK_JUSTIFY_LEFT;

  label->use_markup = TRUE;

  label->caret_index = -1;
  label->timer_id = -1;
  label->showing_caret = 1;
  label->delay_hide_caret = 0;

  label->layout = NULL;
  label->text = NULL;
  label->attrs = NULL;

  im_label_set_text (label, "");
}

static gint
ImLabelCaretBlink(gpointer data)
{

    g_return_if_fail (IM_IS_LABEL (data));
    ImLabel* label = IM_LABEL(data);

    if (label->showing_caret != 0) {
        if (label->delay_hide_caret == 0)
            label->showing_caret = 0;
        else {
            label->delay_hide_caret = 0;
            return;
        }
    } else {
        label->showing_caret = 1;
    }
    label->delay_hide_caret = 0;
    gtk_widget_queue_draw(GTK_WIDGET(label));
    return 1;
}

/**
 * im_label_new:
 * @str: The text of the label
 *
 * Creates a new label with the given text inside it. You can
 * pass %NULL to get an empty label widget.
 *
 * Return value: the new #ImLabel
 **/
GtkWidget*
im_label_new (const gchar *str)
{
  ImLabel *label;

  label = g_object_new (IM_TYPE_LABEL, NULL);

  label->caret_index = -1;
  label->timer_id = -1;
  label->showing_caret = 1;
  label->delay_hide_caret = 0;

  if (str && *str)
      im_label_set_text (label, str);

  label->timer_id = gtk_timeout_add(600, (GtkFunction)ImLabelCaretBlink, label);

  return GTK_WIDGET (label);
}

static void
im_label_screen_changed (GtkWidget *widget,
                          GdkScreen *old_screen)
{
  im_label_clear_layout (IM_LABEL (widget));
}

static void
im_label_set_text_internal (ImLabel *label,
                             gchar    *str)
{
  g_free (label->text);

  label->text = str;

  im_label_select_region_index (label, 0, 0);
}

static void
im_label_set_label_internal (ImLabel *label,
                              gchar    *str)
{
  g_free (label->label);

  label->label = str;

  g_object_notify (G_OBJECT (label), "label");
}

static void
im_label_set_attributes_internal (ImLabel      *label,
                                   PangoAttrList *attrs)
{
  if (attrs)
    pango_attr_list_ref (attrs);

  if (label->attrs)
    pango_attr_list_unref (label->attrs);

  if (!label->use_markup)
    {
      pango_attr_list_ref (attrs);
      if (label->effective_attrs)
        pango_attr_list_unref (label->effective_attrs);
      label->effective_attrs = attrs;
    }

  label->attrs = attrs;
  g_object_notify (G_OBJECT (label), "attributes");
}


/* Calculates text, attrs and mnemonic_keyval from
 * label, use_underline and use_markup
 */
static void
im_label_recalculate (ImLabel *label)
{
  if (label->use_markup)
    set_markup (label, label->label, 0);
  else {
    im_label_set_text_internal (label, g_strdup (label->label));
    if (label->attrs)
      pango_attr_list_ref (label->attrs);
    if (label->effective_attrs)
      pango_attr_list_unref (label->effective_attrs);
    label->effective_attrs = label->attrs;
  }

  im_label_clear_layout (label);
  gtk_widget_queue_resize (GTK_WIDGET (label));
}

/**
 * im_label_set_text:
 * @label: a #ImLabel
 * @str: The text you want to set.
 *
 * Sets the text within the #ImLabel widget.  It overwrites any text that
 * was there before.
 *
 * This will also clear any previously set mnemonic accelerators.
 **/
void
im_label_set_text (ImLabel    *label,
                    const gchar *str)
{
  g_return_if_fail (IM_IS_LABEL (label));

  g_object_freeze_notify (G_OBJECT (label));

  im_label_set_label_internal (label, g_strdup (str ? str : ""));

  im_label_recalculate (label);

  g_object_thaw_notify (G_OBJECT (label));
}

/**
 * im_label_set_attributes:
 * @label: a #ImLabel
 * @attrs: a #PangoAttrList
 *
 * Sets a #PangoAttrList; the attributes in the list are applied to the
 * label text. The attributes set with this function will be ignored
 * if the "use_underline" property or the "use_markup" property
 * is %TRUE.
 **/
void
im_label_set_attributes (ImLabel         *label,
                          PangoAttrList    *attrs)
{
  g_return_if_fail (IM_IS_LABEL (label));

  im_label_set_attributes_internal (label, attrs);

  im_label_clear_layout (label);
  gtk_widget_queue_resize (GTK_WIDGET (label));
}

/**
 * im_label_get_attributes:
 * @label: a #ImLabel
 *
 * Gets the attribute list that was set on the label using
 * im_label_set_attributes(), if any. This function does
 * not reflect attributes that come from the labels markup
 * (see im_label_set_markup()). If you want to get the
 * effective attributes for the label, use
 * pango_layout_get_attribute (im_label_get_layout (label)).
 *
 * Return value: the attribute list, or %NULL if none was set.
 **/
PangoAttrList *
im_label_get_attributes (ImLabel *label)
{
  g_return_val_if_fail (IM_IS_LABEL (label), NULL);

  return label->attrs;
}

/**
 * im_label_set_label:
 * @label: a #ImLabel
 * @str: the new text to set for the label
 *
 * Sets the text of the label. The label is interpreted as
 * including embedded underlines and/or Pango markup depending
 * on the values of label->use_underline and label->use_markup.
 **/
void
im_label_set_label (ImLabel    *label,
                     const gchar *str)
{
  g_return_if_fail (IM_IS_LABEL (label));
  g_return_if_fail (str != NULL);

  im_label_set_label_internal (label, g_strdup (str));
  im_label_recalculate (label);
}

/**
 * im_label_get_label:
 * @label: a #ImLabel
 *
 * Fetches the text from a label widget including any embedded
 * underlines indicating mnemonics and Pango markup. (See
 * im_label_get_text ()).
 *
 * Return value: the text of the label widget. This string is
 *   owned by the widget and must not be modified or freed.
 **/
G_CONST_RETURN gchar *
im_label_get_label (ImLabel *label)
{
  g_return_val_if_fail (IM_IS_LABEL (label), NULL);

  return label->label;
}

static void
set_markup (ImLabel    *label,
            const gchar *str,
            gboolean     with_uline)
{
  gchar *text = NULL;
  GError *error = NULL;
  PangoAttrList *attrs = NULL;
  gunichar accel_char = 0;

  if (!pango_parse_markup (str,
                           -1,
                           with_uline ? '_' : 0,
                           &attrs,
                           &text,
                           with_uline ? &accel_char : NULL,
                           &error))
    {
      g_warning ("Failed to set label from markup due to error parsing markup: %s",
                 error->message);
      g_error_free (error);
      return;
    }

  if (text)
    im_label_set_text_internal (label, text);

  if (attrs)
    {
      if (label->effective_attrs)
        pango_attr_list_unref (label->effective_attrs);
      label->effective_attrs = attrs;
    }
}

/**
 * im_label_set_markup:
 * @label: a #ImLabel
 * @str: a markup string (see <link linkend="PangoMarkupFormat">Pango markup format</link>)
 *
 * Parses @str which is marked up with the <link linkend="PangoMarkupFormat">Pango text markup language</link>,
 * setting the label's text and attribute list based on the parse results.
 **/
void
im_label_set_markup (ImLabel *label, const gchar *str)
{
  g_return_if_fail (IM_IS_LABEL (label));

  im_label_set_label_internal (label, g_strdup (str ? str : ""));
  im_label_recalculate (label);
}

/**
 * im_label_get_text:
 * @label: a #ImLabel
 *
 * Fetches the text from a label widget, as displayed on the
 * screen. This does not include any embedded underlines
 * indicating mnemonics or Pango markup. (See im_label_get_label())
 *
 * Return value: the text in the label widget. This is the internal
 *   string used by the label, and must not be modified.
 **/
G_CONST_RETURN gchar *
im_label_get_text (ImLabel *label)
{
  g_return_val_if_fail (IM_IS_LABEL (label), NULL);

  return label->text;
}

static PangoAttrList *
im_label_pattern_to_attrs (ImLabel      *label,
                            const gchar   *pattern)
{
  const char *start;
  const char *p = label->text;
  const char *q = pattern;
  PangoAttrList *attrs;

  attrs = pango_attr_list_new ();

  while (1)
    {
      while (*p && *q && *q != '_')
        {
          p = g_utf8_next_char (p);
          q++;
        }
      start = p;
      while (*p && *q && *q == '_')
        {
          p = g_utf8_next_char (p);
          q++;
        }

      if (p > start)
        {
          PangoAttribute *attr = pango_attr_underline_new (PANGO_UNDERLINE_LOW);
          attr->start_index = start - label->text;
          attr->end_index = p - label->text;

          pango_attr_list_insert (attrs, attr);
        }
      else
        break;
    }

  return attrs;
}

static void
im_label_set_pattern_internal (ImLabel    *label,
                                const gchar *pattern)
{
  PangoAttrList *attrs;
  g_return_if_fail (IM_IS_LABEL (label));

  attrs = im_label_pattern_to_attrs (label, pattern);

  if (label->effective_attrs)
    pango_attr_list_unref (label->effective_attrs);
  label->effective_attrs = attrs;
}

void
im_label_set_pattern (ImLabel	   *label,
                       const gchar *pattern)
{
  g_return_if_fail (IM_IS_LABEL (label));

  im_label_set_pattern_internal (label, pattern);

  im_label_clear_layout (label);
  gtk_widget_queue_resize (GTK_WIDGET (label));
}


/**
 * im_label_set_justify:
 * @label: a #ImLabel
 * @jtype: a #GtkJustification
 *
 * Sets the alignment of the lines in the text of the label relative to
 * each other.  %GTK_JUSTIFY_LEFT is the default value when the
 * widget is first created with im_label_new(). If you instead want
 * to set the alignment of the label as a whole, use
 * gtk_misc_set_alignment() instead. im_label_set_justify() has no
 * effect on labels containing only a single line.
 **/
void
im_label_set_justify (ImLabel        *label,
                       GtkJustification jtype)
{
  g_return_if_fail (IM_IS_LABEL (label));
  g_return_if_fail (jtype >= GTK_JUSTIFY_LEFT && jtype <= GTK_JUSTIFY_FILL);

  if ((GtkJustification) label->jtype != jtype)
    {
      label->jtype = jtype;

      /* No real need to be this drastic, but easier than duplicating the code */
      im_label_clear_layout (label);

      g_object_notify (G_OBJECT (label), "justify");
      gtk_widget_queue_resize (GTK_WIDGET (label));
    }
}

/**
 * im_label_get_justify:
 * @label: a #ImLabel
 *
 * Returns the justification of the label. See im_label_set_justify ().
 *
 * Return value: #GtkJustification
 **/
GtkJustification
im_label_get_justify (ImLabel *label)
{
  g_return_val_if_fail (IM_IS_LABEL (label), 0);

  return label->jtype;
}

static void
im_label_destroy (GtkObject *object)
{
  ImLabel *label = IM_LABEL (object);
  GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
im_label_finalize (GObject *object)
{
  ImLabel *label;

  g_return_if_fail (IM_IS_LABEL (object));

  label = IM_LABEL (object);

  if (label->timer_id > 0)
      gtk_timeout_remove(label->timer_id);

  g_free (label->label);
  g_free (label->text);

  if (label->layout)
    g_object_unref (label->layout);

  if (label->attrs)
    pango_attr_list_unref (label->attrs);

  if (label->effective_attrs)
    pango_attr_list_unref (label->effective_attrs);

  g_free (label->select_info);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
im_label_clear_layout (ImLabel *label)
{
  if (label->layout)
    {
      g_object_unref (label->layout);
      label->layout = NULL;
    }
}

typedef struct _ImLabelWrapWidth ImLabelWrapWidth;
struct _ImLabelWrapWidth
{
  gint width;
  PangoFontDescription *font_desc;
};

static void
label_wrap_width_free (gpointer data)
{
  ImLabelWrapWidth *wrap_width = data;
  pango_font_description_free (wrap_width->font_desc);
  g_free (wrap_width);
}

static gint
get_label_wrap_width (ImLabel *label)
{
  PangoLayout *layout;
  GtkStyle *style = GTK_WIDGET (label)->style;

  ImLabelWrapWidth *wrap_width = g_object_get_data (G_OBJECT (style), "gtk-label-wrap-width");
  if (!wrap_width)
    {
      wrap_width = g_new0 (ImLabelWrapWidth, 1);
      g_object_set_data_full (G_OBJECT (style), "gtk-label-wrap-width",
                              wrap_width, label_wrap_width_free);
    }

  if (wrap_width->font_desc && pango_font_description_equal (wrap_width->font_desc, style->font_desc))
    return wrap_width->width;

  if (wrap_width->font_desc)
    pango_font_description_free (wrap_width->font_desc);

  wrap_width->font_desc = pango_font_description_copy (style->font_desc);

  layout = gtk_widget_create_pango_layout (GTK_WIDGET (label),
                                           "This long string gives a good enough length for any line to have.");
  pango_layout_get_size (layout, &wrap_width->width, NULL);
  g_object_unref (layout);

  return wrap_width->width;
}

static void
im_label_ensure_layout (ImLabel *label)
{
  GtkWidget *widget;
  PangoRectangle logical_rect;
  gint rwidth, rheight;

  widget = GTK_WIDGET (label);

  rwidth = label->misc.xpad * 2;
  rheight = label->misc.ypad * 2;

  if (!label->layout)
    {
      PangoAlignment align = PANGO_ALIGN_LEFT; /* Quiet gcc */

      label->layout = gtk_widget_create_pango_layout (widget, label->text);

      if (label->effective_attrs)
        pango_layout_set_attributes (label->layout, label->effective_attrs);

      switch (label->jtype)
        {
        case GTK_JUSTIFY_LEFT:
          align = PANGO_ALIGN_LEFT;
          break;
        case GTK_JUSTIFY_RIGHT:
          align = PANGO_ALIGN_RIGHT;
          break;
        case GTK_JUSTIFY_CENTER:
          align = PANGO_ALIGN_CENTER;
          break;
        case GTK_JUSTIFY_FILL:
          /* FIXME: This just doesn't work to do this */
          align = PANGO_ALIGN_LEFT;
          pango_layout_set_justify (label->layout, TRUE);
          break;
        default:
          g_assert_not_reached();
        }

      pango_layout_set_alignment (label->layout, align);
      pango_layout_set_width (label->layout, -1);
    }
}

static void
im_label_size_request (GtkWidget      *widget,
                        GtkRequisition *requisition)
{
  ImLabel *label;
  gint width, height;
  PangoRectangle logical_rect;
  GtkWidgetAuxInfo *aux_info;

  g_return_if_fail (IM_IS_LABEL (widget));
  g_return_if_fail (requisition != NULL);

  label = IM_LABEL (widget);

  im_label_ensure_layout (label);

  width = label->misc.xpad * 2;
  height = label->misc.ypad * 2;

  pango_layout_get_extents (label->layout, NULL, &logical_rect);

  aux_info = _gtk_widget_get_aux_info (widget, FALSE);
  width += PANGO_PIXELS (logical_rect.width);

  height += PANGO_PIXELS (logical_rect.height);

  if (width < 180)
      width = 180;
  requisition->width = width;
  requisition->height = height;
}

static void
im_label_size_allocate (GtkWidget     *widget,
                         GtkAllocation *allocation)
{
  ImLabel *label;

  label = IM_LABEL (widget);

  (* GTK_WIDGET_CLASS (parent_class)->size_allocate) (widget, allocation);

  if (label->select_info && label->select_info->window)
    {
      gdk_window_move_resize (label->select_info->window,
                              allocation->x,
                              allocation->y,
                              allocation->width,
                              allocation->height);
    }
}

static void
im_label_state_changed (GtkWidget   *widget,
                         GtkStateType prev_state)
{
  ImLabel *label;

  label = IM_LABEL (widget);

  if (label->select_info)
    im_label_select_region (label, 0, 0);

  if (GTK_WIDGET_CLASS (parent_class)->state_changed)
    GTK_WIDGET_CLASS (parent_class)->state_changed (widget, prev_state);
}

static void
im_label_style_set (GtkWidget *widget,
                     GtkStyle  *previous_style)
{
  ImLabel *label;

  g_return_if_fail (IM_IS_LABEL (widget));

  label = IM_LABEL (widget);

  /* We have to clear the layout, fonts etc. may have changed */
  im_label_clear_layout (label);
}

static void
im_label_direction_changed (GtkWidget        *widget,
                             GtkTextDirection previous_dir)
{
  ImLabel *label = IM_LABEL (widget);

  if (label->layout)
    pango_layout_context_changed (label->layout);

  GTK_WIDGET_CLASS (parent_class)->direction_changed (widget, previous_dir);
}

#if 0
static void
im_label_paint_word (ImLabel     *label,
                      gint          x,
                      gint          y,
                      ImLabelWord *word,
                      GdkRectangle *area)
{
  GtkWidget *widget = GTK_WIDGET (label);
  ImLabelULine *uline;
  gchar *tmp_str;

  tmp_str = gdk_wcstombs (word->beginning);
  if (tmp_str)
    {
      gtk_paint_string (widget->style, widget->window, widget->state,
                        area, widget, "label",
                        x + word->x,
                        y + word->y,
                        tmp_str);
      g_free (tmp_str);
    }

  for (uline = word->uline; uline; uline = uline->next)
    gtk_paint_hline (widget->style, widget->window,
                     widget->state, area,
                     widget, "label",
                     x + uline->x1, x + uline->x2, y + uline->y);
}
#endif

static void
get_layout_location (ImLabel  *label,
                     gint      *xp,
                     gint      *yp)
{
  GtkMisc *misc;
  GtkWidget *widget;
  gfloat xalign;
  gint x, y;

  misc = GTK_MISC (label);
  widget = GTK_WIDGET (label);

  if (gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR)
    xalign = misc->xalign;
  else
    xalign = 1.0 - misc->xalign;

  x = floor (widget->allocation.x + (gint)misc->xpad +
             xalign * (widget->allocation.width - widget->requisition.width)
             + 0.5);

  if (gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR)
    x = MAX (x, widget->allocation.x + misc->xpad);
  else
    x = MIN (x,
             widget->allocation.x + widget->allocation.width -
             widget->requisition.width - misc->xpad);

  y = floor (widget->allocation.y + (gint)misc->ypad
             + MAX (((widget->allocation.height - widget->requisition.height) * misc->yalign)
                    + 0.5, 0));

  if (xp)
    *xp = x;

  if (yp)
    *yp = y;
}

static void
im_label_draw_cursor (ImLabel *label, gint xoffset, gint yoffset)
{
    //if (label->select_info == NULL) return;

    if (GTK_WIDGET_DRAWABLE (label)) {
        GtkWidget *widget = GTK_WIDGET (label);

        GtkTextDirection keymap_direction;
        GtkTextDirection widget_direction;
        PangoRectangle strong_pos, weak_pos;
        gboolean split_cursor;
        PangoRectangle *cursor1 = NULL;
        PangoRectangle *cursor2 = NULL;
        GdkRectangle cursor_location;
        GtkTextDirection dir1 = GTK_TEXT_DIR_NONE;
        GtkTextDirection dir2 = GTK_TEXT_DIR_NONE;
        GdkGC *gc;


        keymap_direction = GTK_TEXT_DIR_RTL;
        if (gdk_keymap_get_direction(gdk_keymap_get_for_display(gtk_widget_get_display (widget))) == PANGO_DIRECTION_LTR)
            keymap_direction = GTK_TEXT_DIR_LTR;

        widget_direction = gtk_widget_get_direction (widget);
        im_label_ensure_layout (label);

        //pango_layout_get_cursor_pos (label->layout, label->select_info->selection_end, &strong_pos, &weak_pos);
        pango_layout_get_cursor_pos (label->layout, label->caret_index, &strong_pos, &weak_pos);

        g_object_get (gtk_widget_get_settings (widget), "gtk-split-cursor", &split_cursor, NULL);

        dir1 = widget_direction;

        if (split_cursor) {
            cursor1 = &strong_pos;

            if (strong_pos.x != weak_pos.x || strong_pos.y != weak_pos.y) {
                dir2 = (widget_direction == GTK_TEXT_DIR_LTR) ? GTK_TEXT_DIR_RTL : GTK_TEXT_DIR_LTR;
                cursor2 = &weak_pos;
            }

        } else {
            if (keymap_direction == widget_direction)
              cursor1 = &strong_pos;
            else
              cursor1 = &weak_pos;
        }

        cursor_location.x = xoffset + PANGO_PIXELS (cursor1->x);
        cursor_location.y = yoffset + PANGO_PIXELS (cursor1->y);
        cursor_location.width = 0;
        cursor_location.height = PANGO_PIXELS (cursor1->height);

#if (GTK_MAJOR_VERSION > 2 || (GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION >= 4))
            gtk_draw_insertion_cursor(widget, widget->window, NULL, &cursor_location, FALSE, dir1, dir2 != GTK_TEXT_DIR_NONE);
#else
            gc = _gtk_get_insertion_cursor_gc (widget, TRUE);
            _gtk_draw_insertion_cursor (widget, widget->window, gc, &cursor_location, dir1, dir2 != GTK_TEXT_DIR_NONE);
            g_object_unref (gc);


            if (dir2 != GTK_TEXT_DIR_NONE) {
                cursor_location.x = xoffset + PANGO_PIXELS (cursor2->x);
                cursor_location.y = yoffset + PANGO_PIXELS (cursor2->y);
                cursor_location.width = 0;
                cursor_location.height = PANGO_PIXELS (cursor2->height);

                gc = _gtk_get_insertion_cursor_gc (widget, FALSE);
                _gtk_draw_insertion_cursor (widget, widget->window, gc,
                                            &cursor_location, dir2, TRUE);
                g_object_unref (gc);
            }
#endif
    }
}


static gint
im_label_expose (GtkWidget *widget, GdkEventExpose *event)
{
    ImLabel *label;
    gint x, y;

    g_return_val_if_fail (IM_IS_LABEL (widget), FALSE);
    g_return_val_if_fail (event != NULL, FALSE);

    label = IM_LABEL (widget);

    im_label_ensure_layout (label);

    if (GTK_WIDGET_VISIBLE (widget) && GTK_WIDGET_MAPPED (widget) && label->text && (*label->text != '\0')) {
        get_layout_location (label, &x, &y);
        gtk_paint_layout (widget->style, widget->window, GTK_WIDGET_STATE (widget),
                          FALSE, &event->area, widget, "label", x, y, label->layout);

        if (label->select_info && (label->select_info->selection_anchor != label->select_info->selection_end)) {
            gint range[2];
            GdkRegion *clip;
            GtkStateType state;

            range[0] = label->select_info->selection_anchor;
            range[1] = label->select_info->selection_end;

            if (range[0] > range[1]) {
                gint tmp = range[0];
                range[0] = range[1];
                range[1] = tmp;
            }

            clip = gdk_pango_layout_get_clip_region (label->layout, x, y, range, 1);

            /* FIXME should use gtk_paint, but it can't use a clip region */
            gdk_gc_set_clip_region (widget->style->black_gc, clip);


            state = GTK_STATE_SELECTED;
            if (!GTK_WIDGET_HAS_FOCUS (widget))
              state = GTK_STATE_ACTIVE;

            gdk_draw_layout_with_colors (widget->window, widget->style->black_gc, x, y, label->layout,
                                         &widget->style->text[state],&widget->style->base[state]);

            gdk_gc_set_clip_region (widget->style->black_gc, NULL);
            gdk_region_destroy (clip);

        } else if (label->caret_index >= 0 && label->showing_caret != 0) {
          im_label_draw_cursor(label, x, y);
        }
    }

    return FALSE;
}


static void
im_label_realize (GtkWidget *widget)
{
  ImLabel *label;

  label = IM_LABEL (widget);

  (* GTK_WIDGET_CLASS (parent_class)->realize) (widget);

  if (label->select_info)
    im_label_create_window (label);
}

static void
im_label_unrealize (GtkWidget *widget)
{
  ImLabel *label;

  label = IM_LABEL (widget);

  if (label->select_info)
    im_label_destroy_window (label);

  (* GTK_WIDGET_CLASS (parent_class)->unrealize) (widget);
}

static void
im_label_map (GtkWidget *widget)
{
  ImLabel *label;

  label = IM_LABEL (widget);

  (* GTK_WIDGET_CLASS (parent_class)->map) (widget);

  if (label->select_info)
    gdk_window_show (label->select_info->window);
}

static void
im_label_unmap (GtkWidget *widget)
{
  ImLabel *label;

  label = IM_LABEL (widget);

  if (label->select_info)
    gdk_window_hide (label->select_info->window);

  (* GTK_WIDGET_CLASS (parent_class)->unmap) (widget);
}

static void
window_to_layout_coords (ImLabel *label,
                         gint     *x,
                         gint     *y)
{
  gint lx, ly;
  GtkWidget *widget;

  widget = GTK_WIDGET (label);

  /* get layout location in widget->window coords */
  get_layout_location (label, &lx, &ly);

  if (x)
    {
      *x += widget->allocation.x; /* go to widget->window */
      *x -= lx;                   /* go to layout */
    }

  if (y)
    {
      *y += widget->allocation.y; /* go to widget->window */
      *y -= ly;                   /* go to layout */
    }
}

#if 0
static void
layout_to_window_coords (ImLabel *label,
                         gint     *x,
                         gint     *y)
{
  gint lx, ly;
  GtkWidget *widget;

  widget = GTK_WIDGET (label);

  /* get layout location in widget->window coords */
  get_layout_location (label, &lx, &ly);

  if (x)
    {
      *x += lx;                   /* go to widget->window */
      *x -= widget->allocation.x; /* go to selection window */
    }

  if (y)
    {
      *y += ly;                   /* go to widget->window */
      *y -= widget->allocation.y; /* go to selection window */
    }
}
#endif

static void
get_layout_index (ImLabel *label,
                  gint      x,
                  gint      y,
                  gint     *index)
{
  gint trailing = 0;
  const gchar *cluster;
  const gchar *cluster_end;

  *index = 0;

  im_label_ensure_layout (label);

  window_to_layout_coords (label, &x, &y);

  x *= PANGO_SCALE;
  y *= PANGO_SCALE;

  pango_layout_xy_to_index (label->layout,
                            x, y,
                            index, &trailing);


  cluster = label->text + *index;
  cluster_end = cluster;
  while (trailing)
    {
      cluster_end = g_utf8_next_char (cluster_end);
      --trailing;
    }

  *index += (cluster_end - cluster);
}

static void
im_label_select_word (ImLabel *label)
{
  gint min, max;

  gint start_index = im_label_move_backward_word (label, label->select_info->selection_end);
  gint end_index = im_label_move_forward_word (label, label->select_info->selection_end);

  min = MIN (label->select_info->selection_anchor,
             label->select_info->selection_end);
  max = MAX (label->select_info->selection_anchor,
             label->select_info->selection_end);

  min = MIN (min, start_index);
  max = MAX (max, end_index);

  im_label_select_region_index (label, min, max);
}

static gint
im_label_button_press (GtkWidget      *widget,
                        GdkEventButton *event)
{
  ImLabel *label;
  gint index = 0;

  label = IM_LABEL (widget);

  if (label->select_info == NULL)
    return FALSE;

  if (event->button == 1)
    {
      if (!GTK_WIDGET_HAS_FOCUS (widget))
        gtk_widget_grab_focus (widget);

      if (event->type == GDK_3BUTTON_PRESS)
        {
          im_label_select_region_index (label, 0, strlen (label->text));
          return TRUE;
        }

      if (event->type == GDK_2BUTTON_PRESS)
        {
          im_label_select_word (label);
          return TRUE;
        }

      get_layout_index (label, event->x, event->y, &index);

      if ((label->select_info->selection_anchor !=
           label->select_info->selection_end) &&
          (event->state & GDK_SHIFT_MASK))
        {
          gint min, max;

          /* extend (same as motion) */
          min = MIN (label->select_info->selection_anchor,
                     label->select_info->selection_end);
          max = MAX (label->select_info->selection_anchor,
                     label->select_info->selection_end);

          min = MIN (min, index);
          max = MAX (max, index);

          /* ensure the anchor is opposite index */
          if (index == min)
            {
              gint tmp = min;
              min = max;
              max = tmp;
            }

          im_label_select_region_index (label, min, max);
        }
      else
        {
          if (event->type == GDK_3BUTTON_PRESS)
              im_label_select_region_index (label, 0, strlen (label->text));
          else if (event->type == GDK_2BUTTON_PRESS)
              im_label_select_word (label);
          else
            /* start a replacement */
            im_label_select_region_index (label, index, index);
        }

      return TRUE;
    }
  else if (event->button == 3 && event->type == GDK_BUTTON_PRESS)
    {
      //im_label_do_popup (label, event);

      return TRUE;

    }
  return FALSE;
}

static gint
im_label_button_release (GtkWidget      *widget,
                          GdkEventButton *event)

{
  ImLabel *label;

  label = IM_LABEL (widget);

  if (label->select_info == NULL)
    return FALSE;

  if (event->button != 1)
    return FALSE;

  /* The goal here is to return TRUE iff we ate the
   * button press to start selecting.
   */

  return TRUE;
}

static gint
im_label_motion (GtkWidget      *widget,
                  GdkEventMotion *event)
{
  ImLabel *label;
  gint index;
  gint x, y;

  label = IM_LABEL (widget);

  if (label->select_info == NULL)
    return FALSE;

  if ((event->state & GDK_BUTTON1_MASK) == 0)
    return FALSE;

  gdk_window_get_pointer (label->select_info->window,
                          &x, &y, NULL);

  get_layout_index (label, x, y, &index);

  im_label_select_region_index (label,
                                 label->select_info->selection_anchor,
                                 index);

  return TRUE;
}

static void
im_label_create_window (ImLabel *label)
{
  GtkWidget *widget;
  GdkWindowAttr attributes;
  gint attributes_mask;

  g_assert (label->select_info);
  g_assert (GTK_WIDGET_REALIZED (label));

  if (label->select_info->window)
    return;

  widget = GTK_WIDGET (label);

  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.window_type = GDK_WINDOW_TEMP;
  attributes.wclass = GDK_INPUT_ONLY;
  attributes.override_redirect = TRUE;
  attributes.cursor = gdk_cursor_new_for_display (gtk_widget_get_display (widget),
                                                  GDK_XTERM);
  attributes.event_mask = gtk_widget_get_events (widget) |
    GDK_BUTTON_PRESS_MASK        |
    GDK_BUTTON_RELEASE_MASK      |
    GDK_BUTTON_MOTION_MASK;

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_NOREDIR | GDK_WA_CURSOR;

  label->select_info->window = gdk_window_new (widget->window,
                                               &attributes, attributes_mask);
  gdk_window_set_user_data (label->select_info->window, widget);

  gdk_cursor_unref (attributes.cursor);
}

static void
im_label_destroy_window (ImLabel *label)
{
  g_assert (label->select_info);

  if (label->select_info->window == NULL)
    return;

  gdk_window_set_user_data (label->select_info->window, NULL);
  gdk_window_destroy (label->select_info->window);
  label->select_info->window = NULL;
}

/**
 * im_label_set_selectable:
 * @label: a #ImLabel
 * @setting: %TRUE to allow selecting text in the label
 *
 * Selectable labels allow the user to select text from the label, for
 * copy-and-paste.
 *
 **/
void
im_label_set_selectable (ImLabel *label,
                          gboolean  setting)
{
  gboolean old_setting;

  g_return_if_fail (IM_IS_LABEL (label));

  setting = setting != FALSE;
  old_setting = label->select_info != NULL;

  if (setting)
    {
      if (label->select_info == NULL)
        {
          label->select_info = g_new0 (ImLabelSelectionInfo, 1);

          GTK_WIDGET_SET_FLAGS (label, GTK_CAN_FOCUS);

          if (GTK_WIDGET_REALIZED (label))
            im_label_create_window (label);

          if (GTK_WIDGET_MAPPED (label))
            gdk_window_show (label->select_info->window);
        }
    }
  else
    {
      if (label->select_info)
        {
          /* unselect, to give up the selection */
          im_label_select_region (label, 0, 0);

          if (label->select_info->window)
            {
              im_label_destroy_window (label);
            }

          g_free (label->select_info);

          label->select_info = NULL;

          GTK_WIDGET_UNSET_FLAGS (label, GTK_CAN_FOCUS);
        }
    }
  if (setting != old_setting)
    {
      g_object_freeze_notify (G_OBJECT (label));
      g_object_notify (G_OBJECT (label), "selectable");
      g_object_notify (G_OBJECT (label), "cursor_position");
      g_object_notify (G_OBJECT (label), "selection_bound");
      g_object_thaw_notify (G_OBJECT (label));
      gtk_widget_queue_draw (GTK_WIDGET (label));
    }
}

/**
 * im_label_get_selectable:
 * @label: a #ImLabel
 *
 * Gets the value set by im_label_set_selectable().
 *
 * Return value: %TRUE if the user can copy text from the label
 **/
gboolean
im_label_get_selectable (ImLabel *label)
{
  g_return_val_if_fail (IM_IS_LABEL (label), FALSE);

  return label->select_info != NULL;
}

static void
get_text_callback (GtkClipboard     *clipboard,
                   GtkSelectionData *selection_data,
                   guint             info,
                   gpointer          user_data_or_owner)
{
  ImLabel *label;

  label = IM_LABEL (user_data_or_owner);

  if ((label->select_info->selection_anchor !=
       label->select_info->selection_end) &&
      label->text)
    {
      gint start, end;
      gint len;

      start = MIN (label->select_info->selection_anchor,
                   label->select_info->selection_end);
      end = MAX (label->select_info->selection_anchor,
                 label->select_info->selection_end);

      len = strlen (label->text);

      if (end > len)
        end = len;

      if (start > len)
        start = len;

      gtk_selection_data_set_text (selection_data,
                                   label->text + start,
                                   end - start);
    }
}

static void
clear_text_callback (GtkClipboard     *clipboard,
                     gpointer          user_data_or_owner)
{
  ImLabel *label;

  label = IM_LABEL (user_data_or_owner);

  if (label->select_info)
    {
      label->select_info->selection_anchor = label->select_info->selection_end;

      gtk_widget_queue_draw (GTK_WIDGET (label));
    }
}

static void
im_label_select_region_index (ImLabel *label,
                               gint      anchor_index,
                               gint      end_index)
{
  static const GtkTargetEntry targets[] = {
    { "STRING", 0, 0 },
    { "TEXT",   0, 0 },
    { "COMPOUND_TEXT", 0, 0 },
    { "UTF8_STRING", 0, 0 }
  };

  g_return_if_fail (IM_IS_LABEL (label));

  if (label->select_info)
    {
      GtkClipboard *clipboard;

      if (label->select_info->selection_anchor == anchor_index &&
          label->select_info->selection_end == end_index)
        return;

      label->select_info->selection_anchor = anchor_index;
      label->select_info->selection_end = end_index;

      clipboard = gtk_widget_get_clipboard (GTK_WIDGET (label),
                                            GDK_SELECTION_PRIMARY);

      if (anchor_index != end_index)
        {
          gtk_clipboard_set_with_owner (clipboard,
                                        targets,
                                        G_N_ELEMENTS (targets),
                                        get_text_callback,
                                        clear_text_callback,
                                        G_OBJECT (label));
        }
      else
        {
          if (gtk_clipboard_get_owner (clipboard) == G_OBJECT (label))
            gtk_clipboard_clear (clipboard);
        }

      gtk_widget_queue_draw (GTK_WIDGET (label));

      g_object_freeze_notify (G_OBJECT (label));
      g_object_notify (G_OBJECT (label), "cursor_position");
      g_object_notify (G_OBJECT (label), "selection_bound");
      g_object_thaw_notify (G_OBJECT (label));
    }
}

/**
 * im_label_select_region:
 * @label: a #ImLabel
 * @start_offset: start offset (in characters not bytes)
 * @end_offset: end offset (in characters not bytes)
 *
 * Selects a range of characters in the label, if the label is selectable.
 * See im_label_set_selectable(). If the label is not selectable,
 * this function has no effect. If @start_offset or
 * @end_offset are -1, then the end of the label will be substituted.
 *
 **/
void
im_label_select_region  (ImLabel *label,
                          gint      start_offset,
                          gint      end_offset)
{
  g_return_if_fail (IM_IS_LABEL (label));

  if (label->text && label->select_info)
    {
      if (start_offset < 0)
        start_offset = g_utf8_strlen (label->text, -1);

      if (end_offset < 0)
        end_offset = g_utf8_strlen (label->text, -1);

      im_label_select_region_index (label,
                                     g_utf8_offset_to_pointer (label->text, start_offset) - label->text,
                                     g_utf8_offset_to_pointer (label->text, end_offset) - label->text);
    }
}

/**
 * im_label_get_selection_bounds:
 * @label: a #ImLabel
 * @start: return location for start of selection, as a character offset
 * @end: return location for end of selection, as a character offset
 *
 * Gets the selected range of characters in the label, returning %TRUE
 * if there's a selection.
 *
 * Return value: %TRUE if selection is non-empty
 **/
gboolean
im_label_get_selection_bounds (ImLabel  *label,
                                gint      *start,
                                gint      *end)
{
  g_return_val_if_fail (IM_IS_LABEL (label), FALSE);

  if (label->select_info == NULL)
    {
      /* not a selectable label */
      if (start)
        *start = 0;
      if (end)
        *end = 0;

      return FALSE;
    }
  else
    {
      gint start_index, end_index;
      gint start_offset, end_offset;
      gint len;

      start_index = MIN (label->select_info->selection_anchor,
                   label->select_info->selection_end);
      end_index = MAX (label->select_info->selection_anchor,
                 label->select_info->selection_end);

      len = strlen (label->text);

      if (end_index > len)
        end_index = len;

      if (start_index > len)
        start_index = len;

      start_offset = g_utf8_strlen (label->text, start_index);
      end_offset = g_utf8_strlen (label->text, end_index);

      if (start_offset > end_offset)
        {
          gint tmp = start_offset;
          start_offset = end_offset;
          end_offset = tmp;
        }

      if (start)
        *start = start_offset;

      if (end)
        *end = end_offset;

      return start_offset != end_offset;
    }
}


/**
 * im_label_get_layout:
 * @label: a #ImLabel
 *
 * Gets the #PangoLayout used to display the label.
 * The layout is useful to e.g. convert text positions to
 * pixel positions, in combination with im_label_get_layout_offsets().
 * The returned layout is owned by the label so need not be
 * freed by the caller.
 *
 * Return value: the #PangoLayout for this label
 **/
PangoLayout*
im_label_get_layout (ImLabel *label)
{
  g_return_val_if_fail (IM_IS_LABEL (label), NULL);

  im_label_ensure_layout (label);

  return label->layout;
}

/**
 * im_label_get_layout_offsets:
 * @label: a #ImLabel
 * @x: location to store X offset of layout, or %NULL
 * @y: location to store Y offset of layout, or %NULL
 *
 * Obtains the coordinates where the label will draw the #PangoLayout
 * representing the text in the label; useful to convert mouse events
 * into coordinates inside the #PangoLayout, e.g. to take some action
 * if some part of the label is clicked. Of course you will need to
 * create a #GtkEventBox to receive the events, and pack the label
 * inside it, since labels are a #GTK_NO_WINDOW widget. Remember
 * when using the #PangoLayout functions you need to convert to
 * and from pixels using PANGO_PIXELS() or #PANGO_SCALE.
 *
 **/
void
im_label_get_layout_offsets (ImLabel *label, gint *x, gint *y)
{
  g_return_if_fail (IM_IS_LABEL (label));
  get_layout_location (label, x, y);
}

static gboolean
im_label_focus (GtkWidget *widget,  GtkDirectionType   direction)
{
  /* We never want to be in the tab chain */
  return FALSE;
}

/* Compute the X position for an offset that corresponds to the "more important
 * cursor position for that offset. We use this when trying to guess to which
 * end of the selection we should go to when the user hits the left or
 * right arrow key.
 */
static void
get_better_cursor (ImLabel *label, gint index, gint *x, gint *y)
{
  GdkKeymap *keymap = gdk_keymap_get_for_display (gtk_widget_get_display (GTK_WIDGET (label)));
  GtkTextDirection keymap_direction =
    (gdk_keymap_get_direction (keymap) == PANGO_DIRECTION_LTR) ?
    GTK_TEXT_DIR_LTR : GTK_TEXT_DIR_RTL;
  GtkTextDirection widget_direction = gtk_widget_get_direction (GTK_WIDGET (label));
  gboolean split_cursor;
  PangoRectangle strong_pos, weak_pos;

  g_object_get (gtk_widget_get_settings (GTK_WIDGET (label)),
                "gtk-split-cursor", &split_cursor,
                NULL);

  im_label_ensure_layout (label);

  pango_layout_get_cursor_pos (label->layout, index,
                               &strong_pos, &weak_pos);

  if (split_cursor)
    {
      *x = strong_pos.x / PANGO_SCALE;
      *y = strong_pos.y / PANGO_SCALE;
    }
  else
    {
      if (keymap_direction == widget_direction)
        {
          *x = strong_pos.x / PANGO_SCALE;
          *y = strong_pos.y / PANGO_SCALE;
        }
      else
        {
          *x = weak_pos.x / PANGO_SCALE;
          *y = weak_pos.y / PANGO_SCALE;
        }
    }
}


static gint
im_label_move_logically (ImLabel *label,
                          gint      start,
                          gint      count)
{
  gint offset = g_utf8_pointer_to_offset (label->text,
                                          label->text + start);

  if (label->text)
    {
      PangoLogAttr *log_attrs;
      gint n_attrs;
      gint length;

      im_label_ensure_layout (label);

      length = g_utf8_strlen (label->text, -1);

      pango_layout_get_log_attrs (label->layout, &log_attrs, &n_attrs);

      while (count > 0 && offset < length)
        {
          do
            offset++;
          while (offset < length && !log_attrs[offset].is_cursor_position);

          count--;
        }
      while (count < 0 && offset > 0)
        {
          do
            offset--;
          while (offset > 0 && !log_attrs[offset].is_cursor_position);

          count++;
        }

      g_free (log_attrs);
    }

  return g_utf8_offset_to_pointer (label->text, offset) - label->text;
}

static gint
im_label_move_visually (ImLabel *label,
                         gint      start,
                         gint      count)
{
  gint index;

  index = start;

  while (count != 0)
    {
      int new_index, new_trailing;
      gboolean split_cursor;
      gboolean strong;

      im_label_ensure_layout (label);

      g_object_get (gtk_widget_get_settings (GTK_WIDGET (label)),
                    "gtk-split-cursor", &split_cursor,
                    NULL);

      if (split_cursor)
        strong = TRUE;
      else
        {
          GdkKeymap *keymap = gdk_keymap_get_for_display (gtk_widget_get_display (GTK_WIDGET (label)));
          GtkTextDirection keymap_direction =
            (gdk_keymap_get_direction (keymap) == PANGO_DIRECTION_LTR) ?
            GTK_TEXT_DIR_LTR : GTK_TEXT_DIR_RTL;

          strong = keymap_direction == gtk_widget_get_direction (GTK_WIDGET (label));
        }

      if (count > 0)
        {
          pango_layout_move_cursor_visually (label->layout, strong, index, 0, 1, &new_index, &new_trailing);
          count--;
        }
      else
        {
          pango_layout_move_cursor_visually (label->layout, strong, index, 0, -1, &new_index, &new_trailing);
          count++;
        }

      if (new_index < 0 || new_index == G_MAXINT)
        break;

      index = new_index;

      while (new_trailing--)
        index = g_utf8_next_char (label->text + new_index) - label->text;
    }

  return index;
}

static gint
im_label_move_forward_word (ImLabel *label,
                             gint      start)
{
  gint new_pos = g_utf8_pointer_to_offset (label->text,
                                           label->text + start);
  gint length;

  length = g_utf8_strlen (label->text, -1);
  if (new_pos < length)
    {
      PangoLogAttr *log_attrs;
      gint n_attrs;

      im_label_ensure_layout (label);

      pango_layout_get_log_attrs (label->layout, &log_attrs, &n_attrs);

      /* Find the next word end */
      new_pos++;
      while (new_pos < n_attrs && !log_attrs[new_pos].is_word_end)
        new_pos++;

      g_free (log_attrs);
    }

  return g_utf8_offset_to_pointer (label->text, new_pos) - label->text;
}


static gint
im_label_move_backward_word (ImLabel *label,
                              gint      start)
{
  gint new_pos = g_utf8_pointer_to_offset (label->text,
                                           label->text + start);
  gint length;

  length = g_utf8_strlen (label->text, -1);

  if (new_pos > 0)
    {
      PangoLogAttr *log_attrs;
      gint n_attrs;

      im_label_ensure_layout (label);

      pango_layout_get_log_attrs (label->layout, &log_attrs, &n_attrs);

      new_pos -= 1;

      /* Find the previous word beginning */
      while (new_pos > 0 && !log_attrs[new_pos].is_word_start)
        new_pos--;

      g_free (log_attrs);
    }

  return g_utf8_offset_to_pointer (label->text, new_pos) - label->text;
}

static void
im_label_move_cursor (ImLabel       *label,
                       GtkMovementStep step,
                       gint            count,
                       gboolean        extend_selection)
{
  gint new_pos;

  if (label->select_info == NULL)
    return;

  new_pos = label->select_info->selection_end;

  if (label->select_info->selection_end != label->select_info->selection_anchor &&
      !extend_selection)
    {
      /* If we have a current selection and aren't extending it, move to the
       * start/or end of the selection as appropriate
       */
      switch (step)
        {
        case GTK_MOVEMENT_VISUAL_POSITIONS:
          {
            gint end_x, end_y;
            gint anchor_x, anchor_y;
            gboolean end_is_left;

            get_better_cursor (label, label->select_info->selection_end, &end_x, &end_y);
            get_better_cursor (label, label->select_info->selection_anchor, &anchor_x, &anchor_y);

            end_is_left = (end_y < anchor_y) || (end_y == anchor_y && end_x < anchor_x);

            if (count < 0)
              new_pos = end_is_left ? label->select_info->selection_end : label->select_info->selection_anchor;
            else
              new_pos = !end_is_left ? label->select_info->selection_end : label->select_info->selection_anchor;

            break;
          }
        case GTK_MOVEMENT_LOGICAL_POSITIONS:
        case GTK_MOVEMENT_WORDS:
          if (count < 0)
            new_pos = MIN (label->select_info->selection_end, label->select_info->selection_anchor);
          else
            new_pos = MAX (label->select_info->selection_end, label->select_info->selection_anchor);
          break;
        case GTK_MOVEMENT_DISPLAY_LINE_ENDS:
        case GTK_MOVEMENT_PARAGRAPH_ENDS:
        case GTK_MOVEMENT_BUFFER_ENDS:
          /* FIXME: Can do better here */
          new_pos = count < 0 ? 0 : strlen (label->text);
          break;
        case GTK_MOVEMENT_DISPLAY_LINES:
        case GTK_MOVEMENT_PARAGRAPHS:
        case GTK_MOVEMENT_PAGES:
          break;
        }
    }
  else
    {
      switch (step)
        {
        case GTK_MOVEMENT_LOGICAL_POSITIONS:
          new_pos = im_label_move_logically (label, new_pos, count);
          break;
        case GTK_MOVEMENT_VISUAL_POSITIONS:
          new_pos = im_label_move_visually (label, new_pos, count);
          break;
        case GTK_MOVEMENT_WORDS:
          while (count > 0)
            {
              new_pos = im_label_move_forward_word (label, new_pos);
              count--;
            }
          while (count < 0)
            {
              new_pos = im_label_move_backward_word (label, new_pos);
              count++;
            }
          break;
        case GTK_MOVEMENT_DISPLAY_LINE_ENDS:
        case GTK_MOVEMENT_PARAGRAPH_ENDS:
        case GTK_MOVEMENT_BUFFER_ENDS:
          /* FIXME: Can do better here */
          new_pos = count < 0 ? 0 : strlen (label->text);
          break;
        case GTK_MOVEMENT_DISPLAY_LINES:
        case GTK_MOVEMENT_PARAGRAPHS:
        case GTK_MOVEMENT_PAGES:
          break;
        }
    }

  if (extend_selection)
    im_label_select_region_index (label,
                                   label->select_info->selection_anchor,
                                   new_pos);
  else
    im_label_select_region_index (label, new_pos, new_pos);
}

static void
im_label_copy_clipboard (ImLabel *label)
{
  if (label->text && label->select_info)
    {
      gint start, end;
      gint len;

      start = MIN (label->select_info->selection_anchor,
                   label->select_info->selection_end);
      end = MAX (label->select_info->selection_anchor,
                 label->select_info->selection_end);

      len = strlen (label->text);

      if (end > len)
        end = len;

      if (start > len)
        start = len;

      if (start != end)
        gtk_clipboard_set_text (gtk_widget_get_clipboard (GTK_WIDGET (label),
                                                          GDK_SELECTION_CLIPBOARD),
                                label->text + start, end - start);
    }
}

static void
im_label_select_all (ImLabel *label)
{
  im_label_select_region_index (label, 0, strlen (label->text));
}

