module Foreign.Storable
  ( Storable ( sizeOf         -- :: a -> Int
             , alignment      -- :: a -> Int
             , peekElemOff    -- :: Ptr a -> Int      -> IO a
             , pokeElemOff    -- :: Ptr a -> Int -> a -> IO ()
             , peekByteOff    -- :: Ptr b -> Int      -> IO a
             , pokeByteOff    -- :: Ptr b -> Int -> a -> IO ()
             , peek           -- :: Ptr a             -> IO a
             , poke           -- :: Ptr a        -> a -> IO ()
             )
  ) where

import Control.Monad (liftM)
import PreludeBuiltinTypes
import Foreign.BuiltinTypes
import Foreign.Ptr
import qualified TraceOrigForeign.Storable
import qualified TraceOrigForeign.StablePtr

class Storable a where
    sizeOf      :: a -> Int
    alignment   :: a -> Int
    peekElemOff :: Ptr a -> Int      -> IO a
    pokeElemOff :: Ptr a -> Int -> a -> IO ()
    peekByteOff :: Ptr b -> Int      -> IO a
    pokeByteOff :: Ptr b -> Int -> a -> IO ()
    peek        :: Ptr a      -> IO a
    poke        :: Ptr a -> a -> IO ()

    peekElemOff ptr off = peekByteOff ptr (off * sizeOfP ptr undefined)
        where sizeOfP :: Storable b => Ptr b -> b -> Int
              sizeOfP px x = sizeOf x
    pokeElemOff ptr off val = pokeByteOff ptr (off * sizeOf val) val

    peekByteOff ptr off = peek (ptr `plusPtr` off)
    pokeByteOff ptr off = poke (ptr `plusPtr` off)

    peek ptr = peekElemOff ptr 0
    poke ptr = pokeElemOff ptr 0



foreign import haskell "Foreign.Storable.sizeOf"
  sizeOfBool :: Bool -> Int
foreign import haskell "Foreign.Storable.alignment"
  alignmentBool :: Bool -> Int
foreign import haskell "Foreign.Storable.peekElemOff"
  peekElemOffBool :: Ptr Bool -> Int -> IO Bool
foreign import haskell "Foreign.Storable.pokeElemOff"
  pokeElemOffBool :: Ptr Bool -> Int -> Bool -> IO ()
foreign import haskell "Foreign.Storable.peekByteOff"
  peekByteOffBool :: Ptr b -> Int -> IO Bool
foreign import haskell "Foreign.Storable.pokeByteOff"
  pokeByteOffBool :: Ptr b -> Int -> Bool -> IO ()
foreign import haskell "Foreign.Storable.peek"
  peekBool :: Ptr Bool -> IO Bool
foreign import haskell "Foreign.Storable.poke"
  pokeBool :: Ptr Bool -> Bool -> IO ()

instance Storable Bool where
    sizeOf = sizeOfBool
    alignment = alignmentBool
    peekElemOff = peekElemOffBool
    pokeElemOff = pokeElemOffBool
    peekByteOff = peekByteOffBool
    pokeByteOff = pokeByteOffBool
    peek = peekBool
    poke = pokeBool

foreign import haskell "Foreign.Storable.sizeOf"
  sizeOfChar :: Char -> Int
foreign import haskell "Foreign.Storable.alignment"
  alignmentChar :: Char -> Int
foreign import haskell "Foreign.Storable.peekElemOff"
  peekElemOffChar :: Ptr Char -> Int -> IO Char
foreign import haskell "Foreign.Storable.pokeElemOff"
  pokeElemOffChar :: Ptr Char -> Int -> Char -> IO ()
foreign import haskell "Foreign.Storable.peekByteOff"
  peekByteOffChar :: Ptr b -> Int -> IO Char
foreign import haskell "Foreign.Storable.pokeByteOff"
  pokeByteOffChar :: Ptr b -> Int -> Char -> IO ()
foreign import haskell "Foreign.Storable.peek"
  peekChar :: Ptr Char -> IO Char
foreign import haskell "Foreign.Storable.poke"
  pokeChar :: Ptr Char -> Char -> IO ()

instance Storable Char where
    sizeOf = sizeOfChar
    alignment = alignmentChar
    peekElemOff = peekElemOffChar
    pokeElemOff = pokeElemOffChar
    peekByteOff = peekByteOffChar
    pokeByteOff = pokeByteOffChar
    peek = peekChar
    poke = pokeChar

foreign import haskell "Foreign.Storable.sizeOf"
  sizeOfInt :: Int -> Int
foreign import haskell "Foreign.Storable.alignment"
  alignmentInt :: Int -> Int
foreign import haskell "Foreign.Storable.peekElemOff"
  peekElemOffInt :: Ptr Int -> Int -> IO Int
foreign import haskell "Foreign.Storable.pokeElemOff"
  pokeElemOffInt :: Ptr Int -> Int -> Int -> IO ()
foreign import haskell "Foreign.Storable.peekByteOff"
  peekByteOffInt :: Ptr b -> Int -> IO Int
foreign import haskell "Foreign.Storable.pokeByteOff"
  pokeByteOffInt :: Ptr b -> Int -> Int -> IO ()
foreign import haskell "Foreign.Storable.peek"
  peekInt :: Ptr Int -> IO Int
foreign import haskell "Foreign.Storable.poke"
  pokeInt :: Ptr Int -> Int -> IO ()

instance Storable Int where
    sizeOf = sizeOfInt
    alignment = alignmentInt
    peekElemOff = peekElemOffInt
    pokeElemOff = pokeElemOffInt
    peekByteOff = peekByteOffInt
    pokeByteOff = pokeByteOffInt
    peek = peekInt
    poke = pokeInt

foreign import haskell "Foreign.Storable.sizeOf"
  sizeOfPtr :: Ptr a -> Int
foreign import haskell "Foreign.Storable.alignment"
  alignmentPtr :: Ptr a -> Int
foreign import haskell "Foreign.Storable.peekElemOff"
  peekElemOffPtr :: Ptr (Ptr a) -> Int -> IO (Ptr a)
foreign import haskell "Foreign.Storable.pokeElemOff"
  pokeElemOffPtr :: Ptr (Ptr a) -> Int -> (Ptr a) -> IO ()
foreign import haskell "Foreign.Storable.peekByteOff"
  peekByteOffPtr :: Ptr b -> Int -> IO (Ptr a)
foreign import haskell "Foreign.Storable.pokeByteOff"
  pokeByteOffPtr :: Ptr b -> Int -> (Ptr a) -> IO ()
foreign import haskell "Foreign.Storable.peek"
  peekPtr :: Ptr (Ptr a) -> IO (Ptr a)
foreign import haskell "Foreign.Storable.poke"
  pokePtr :: Ptr (Ptr a) -> (Ptr a) -> IO ()

instance Storable (Ptr a) where
    sizeOf = sizeOfPtr
    alignment = alignmentPtr
    peekElemOff = peekElemOffPtr
    pokeElemOff = pokeElemOffPtr
    peekByteOff = peekByteOffPtr
    pokeByteOff = pokeByteOffPtr
    peek = peekPtr
    poke = pokePtr

foreign import haskell "Foreign.Storable.sizeOf"
  sizeOfFunPtr :: FunPtr a -> Int
foreign import haskell "Foreign.Storable.alignment"
  alignmentFunPtr :: FunPtr a -> Int
foreign import haskell "Foreign.Storable.peekElemOff"
  peekElemOffFunPtr :: Ptr (FunPtr a) -> Int -> IO (FunPtr a)
foreign import haskell "Foreign.Storable.pokeElemOff"
  pokeElemOffFunPtr :: Ptr (FunPtr a) -> Int -> (FunPtr a) -> IO ()
foreign import haskell "Foreign.Storable.peekByteOff"
  peekByteOffFunPtr :: Ptr b -> Int -> IO (FunPtr a)
foreign import haskell "Foreign.Storable.pokeByteOff"
  pokeByteOffFunPtr :: Ptr b -> Int -> (FunPtr a) -> IO ()
foreign import haskell "Foreign.Storable.peek"
  peekFunPtr :: Ptr (FunPtr a) -> IO (FunPtr a)
foreign import haskell "Foreign.Storable.poke"
  pokeFunPtr :: Ptr (FunPtr a) -> (FunPtr a) -> IO ()

instance Storable (FunPtr a) where
    sizeOf = sizeOfFunPtr
    alignment = alignmentFunPtr
    peekElemOff = peekElemOffFunPtr
    pokeElemOff = pokeElemOffFunPtr
    peekByteOff = peekByteOffFunPtr
    pokeByteOff = pokeByteOffFunPtr
    peek = peekFunPtr
    poke = pokeFunPtr

foreign import haskell "Foreign.Storable.sizeOf"
  sizeOfStablePtr :: StablePtr a -> Int
foreign import haskell "Foreign.Storable.alignment"
  alignmentStablePtr :: StablePtr a -> Int
foreign import haskell "Foreign.Storable.peekElemOff"
  peekElemOffStablePtr :: Ptr (StablePtr a) -> Int -> IO (StablePtr a)
foreign import haskell "Foreign.Storable.pokeElemOff"
  pokeElemOffStablePtr :: Ptr (StablePtr a) -> Int -> (StablePtr a) -> IO ()
foreign import haskell "Foreign.Storable.peekByteOff"
  peekByteOffStablePtr :: Ptr b -> Int -> IO (StablePtr a)
foreign import haskell "Foreign.Storable.pokeByteOff"
  pokeByteOffStablePtr :: Ptr b -> Int -> (StablePtr a) -> IO ()
foreign import haskell "Foreign.Storable.peek"
  peekStablePtr :: Ptr (StablePtr a) -> IO (StablePtr a)
foreign import haskell "Foreign.Storable.poke"
  pokeStablePtr :: Ptr (StablePtr a) -> (StablePtr a) -> IO ()

instance Storable (StablePtr a) where
    sizeOf = sizeOfStablePtr
    alignment = alignmentStablePtr
    peekElemOff = peekElemOffStablePtr
    pokeElemOff = pokeElemOffStablePtr
    peekByteOff = peekByteOffStablePtr
    pokeByteOff = pokeByteOffStablePtr
    peek = peekStablePtr
    poke = pokeStablePtr

foreign import haskell "Foreign.Storable.sizeOf"
  sizeOfFloat :: Float -> Int
foreign import haskell "Foreign.Storable.alignment"
  alignmentFloat :: Float -> Int
foreign import haskell "Foreign.Storable.peekElemOff"
  peekElemOffFloat :: Ptr Float -> Int -> IO Float
foreign import haskell "Foreign.Storable.pokeElemOff"
  pokeElemOffFloat :: Ptr Float -> Int -> Float -> IO ()
foreign import haskell "Foreign.Storable.peekByteOff"
  peekByteOffFloat :: Ptr b -> Int -> IO Float
foreign import haskell "Foreign.Storable.pokeByteOff"
  pokeByteOffFloat :: Ptr b -> Int -> Float -> IO ()
foreign import haskell "Foreign.Storable.peek"
  peekFloat :: Ptr Float -> IO Float
foreign import haskell "Foreign.Storable.poke"
  pokeFloat :: Ptr Float -> Float -> IO ()

instance Storable Float where
    sizeOf = sizeOfFloat
    alignment = alignmentFloat
    peekElemOff = peekElemOffFloat
    pokeElemOff = pokeElemOffFloat
    peekByteOff = peekByteOffFloat
    pokeByteOff = pokeByteOffFloat
    peek = peekFloat
    poke = pokeFloat

foreign import haskell "Foreign.Storable.sizeOf"
  sizeOfDouble :: Double -> Int
foreign import haskell "Foreign.Storable.alignment"
  alignmentDouble :: Double -> Int
foreign import haskell "Foreign.Storable.peekElemOff"
  peekElemOffDouble :: Ptr Double -> Int -> IO Double
foreign import haskell "Foreign.Storable.pokeElemOff"
  pokeElemOffDouble :: Ptr Double -> Int -> Double -> IO ()
foreign import haskell "Foreign.Storable.peekByteOff"
  peekByteOffDouble :: Ptr b -> Int -> IO Double
foreign import haskell "Foreign.Storable.pokeByteOff"
  pokeByteOffDouble :: Ptr b -> Int -> Double -> IO ()
foreign import haskell "Foreign.Storable.peek"
  peekDouble :: Ptr Double -> IO Double
foreign import haskell "Foreign.Storable.poke"
  pokeDouble :: Ptr Double -> Double -> IO ()

instance Storable Double where
    sizeOf = sizeOfDouble
    alignment = alignmentDouble
    peekElemOff = peekElemOffDouble
    pokeElemOff = pokeElemOffDouble
    peekByteOff = peekByteOffDouble
    pokeByteOff = pokeByteOffDouble
    peek = peekDouble
    poke = pokeDouble

