/* This file defines the setup for the filetypes supported by
 * stdfiletrack.c as well as the filetype support database itself. */

#ifndef FILETYPES_H
#define FILETYPES_H

#include <gtk/gtk.h>

typedef struct 
{
	GtkWidget *sampler;       /* this is a hbox containing all the widgets below */

	GtkWidget *w_suffix;
	GtkWidget *w_tracktype;   /* currently one of data,audio */
   	GtkWidget *w_precache;    /* precache file before writing ? */   
	GtkWidget *w_filter;      /* currently one of bigEndian,littleEndian */
	GtkWidget *w_filtername;  /* filter program like "sox [...]" */
	GtkWidget *w_detectsize;  /* program returning nothing but the tracksize in bytes on stdout */
	GtkWidget *w_sizealg;     /* function calculating the track size */
	
	char suffix[32];          /* the suffix identifying this filetype */
	char tracktype[32];       /* the fields containing the real values after apply */
        int  precache;            /* precache file ? */
	char filter[32];
	char filtername[1024];
	char detectsize[1024];
	char sizealg[1024];
   
        char encoder[1024];  /* this is the encoder part where */
        int  enc_lendian;    /* encoder is the command to call for encoding */
        int  enc_default;    /* enc_lendian is true if encoder expects stream to be little endian */
                             /* and enc_default is true if this is the default encoder for the specified track type */
   	char settitle[1024]; /* settitle is the command to execute to set a files tags to the correct values */
} filetypes_filetypeentry;      /* maybe detectsize should also provide some magics to Gnometoaster internal functions */

void filetypes_init();
/* create a new filetype registry entry */
filetypes_filetypeentry *filetypes_entrycreate(char *suffix,
					       char *tracktype,
					       int  precache,
					       char *filter,
					       char *filtername,
					       char *detectsize,
					       char *sizealg,
					       
					       char *encoder,
					       int  enc_lendian,
					       int  enc_default,
					       char *settitle
					       );
/* register a filetype */
void filetypes_register(filetypes_filetypeentry*);
/* unregister a filetype entry from the database */
void filetypes_unregister(filetypes_filetypeentry*);

/* Note that it is only safe to store references to filetypes_filetypeentry
 * as long as the user doesn't have access to the filetypes registry's
 * edit functions. 
 * Instead of storing direct references to filetypes_filetypeentry
 * I suggest storing the suffix associated with this entry.
 * The entry can later be looked up based on this suffix.
 * This way, a lookup failure will still leave you in charge of the
 * situation.
 * FIXME: maybe introduce a reference counter for this as it was done
 * for the tracks_trackinfo structure */

/* get a registry entry for a given suffix */
filetypes_filetypeentry *filetypes_getfiletypehandler(char *suffix);
/* get default encoder for tracktype */
filetypes_filetypeentry *filetypes_getdefaultencoder(char *tracktype);
/* get number of encoders available */
int filetypes_getenccount(char *tracktype);
/* get a list of possible filetypes (->data is of filetypes_filetypeentry*)
 * This list must be freed using g_list_free(result) */
GList *filetypes_getencoders(char *tracktype);

/* this function block is responsible for defining temporary track->file
 * mappings as occurring e.g. during a stream drag and drop with
 * the user deciding what to make of a given tracktype.
 * Maplists are normal GLists and must be initialized with NULL */
void filetypes_addmapping(GList **maplist,
			  char *tracktype,
			  char *suffix);
void filetypes_removemapping(GList **maplist,
			     char *tracktype);
char *filetypes_getmapping(GList *maplist,
			   char *tracktype);
void filetypes_clearmapping(GList **maplist);


/* update a filetype entry externally */
void filetypes_modify(filetypes_filetypeentry *entry,
		      char *suffix,     /* NULL == don't modify */
		      char *tracktype,
		      int  precache,    /* -1   == don't modify */
		      char *filter,
		      char *filtername,
		      char *detectsize,
		      char *sizealg,
		      char *encoder,
		      int  enc_lendian,
		      int  enc_default,
		      char *settitle
		      );
/* the following defines are convenience calls to filetypes_getfiletypehandler,
 * referring to specific parts of the filetypeentry structure */
#define filetypes_gettracktype(suffix)  (char*)&filetypes_getfiletypehandler(suffix)->tracktype
#define filetypes_getfilter(suffix)     (char*)&filetypes_getfiletypehandler(suffix)->filter
#define filetypes_getfiltername(suffix) (char*)&filetypes_getfiletypehandler(suffix)->filtername
#define filetypes_getdetectsize(suffix) (char*)&filetypes_getfiletypehandler(suffix)->detectsize
#define filetypes_getsizealg(suffix)    (char*)&filetypes_getfiletypehandler(suffix)->sizealg
#define filetypes_getprecache(suffix)   filetypes_getfiletypehandler(suffix)->precache
#endif // FILETYPES_H
