/* routines to look up cd and track title based on cddb number and track number
 * stored in the track's data structure
 * Once a cddb lookup has been completed the track is assigned it's final name
 * automatically.
 * The cddb_lookup() function is getting called by the track itself as soon
 * as it's cddb field contains a number !=0 */

/* FIXME: cddb_cache doesn't get freed by the program.
 * Once an entry gets in there it's gonna stay until the program is terminated */

/* uncomment for debugging */
// #define DEBUG
//
#define cddb_server varman_getvar(global_defs,"cddbserver")

/* we're running all communications in a separate thread to prevent any
 * slow connections from blocking gnometoaster.
 * we could just put gethostbyname() into a child process and
 * do all I/O access in O_NONBLOCK mode but this would require major efforts
 * so we just do it this way for a start.
 *  => pthreads support is required for cddb lookup to work */

#include <gtk/gtk.h>
#include <pwd.h>
#include <sys/types.h>
#include <unistd.h>
#include <time.h>

#include "config.h"
#include "int.h"
#include "preferences.h"

#include "cddb.h"
#include "tracks.h"
#include "helpings.h"
#include "varman.h"
#include "varmanwidgets.h"

#ifdef HAVE_PTHREADS
# include <pthread.h>
# include <signal.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/types.h>
#ifndef __FreeBSD__
# include <arpa/inet.h>
#else
# include <netinet/in.h>
#endif
#include <netdb.h>
#include <string.h>
#include <unistd.h>

/* this is a GList containing all cddb requests until they get either
 * killed or executed */
GList *cddb_requests;
#ifdef HAVE_PTHREADS
/* the thread identifier of our comm thread */
pthread_t *cddb_thread=NULL;
volatile int commthread_exit_requested=0;
/* this controls access to the cddb_requests list */
pthread_mutex_t cddb_sem = PTHREAD_MUTEX_INITIALIZER;
#else
/* some dummy definitions */
# define pthread_mutex_lock(sem)
# define pthread_mutex_unlock(sem)
#endif

/* the id referencing our timeout handler in the main thread */
int thid=0;

/* The ->data field of this GList is a pointer to yet another GList,
 * this time with the ->data field pointing to a single line of the
 * cddb server response with the cddb key of the query prepended as
 * the ->data field casted to int */
GList *cddb_cache=NULL;
FILE *cddb_sfd=NULL;

gint cddb_timeouthandler(gpointer data)
{
   GList *current;
   GList *next;
   cddb_infostruct *info;

   /* block lookup thread */
   pthread_mutex_lock(&cddb_sem);
   /* step through our lookup list and remove completed jobs */
   for (current=cddb_requests;
	current!=NULL;
	current=next)
     {
	/* save this value */
	next=current->next;

	if (current->data!=NULL)
	  {
	     info=(cddb_infostruct*)current->data;
	     if (info->complete)
	       {
		  if (info->callback!=NULL)
		    ((cddb_datahandler)info->callback)(info,
						       info->data);
		  cddb_requests=g_list_remove(cddb_requests,info);
		  /* free memory used by request */
		  free(info);
	       };
	  }
	else
	  {
	     printf ("cddb_timeouthandler: invalid entry in cddb request queue.\n");
	     printf ("cddb_timeouthandler: Please send a bugreport to A.Eckleder@bigfoot.com\n");
	  };
     };
   pthread_mutex_unlock(&cddb_sem);
   /* keep running */
   return 1;
};

/* send a command to the cddb database.
 * (command mustn't be CR or LF terminated)
 * this function returns the result code of the operation and,
 * if char *error !=NULL the error description as returned by the server.
 * the buffer size of char *error has to be MAXSTRINGSIZE+1 bytes */
int cddb_sendcmd(FILE *sfd,char *cmd,char *error)
{
   char buffer[MAXSTRINGSIZE+1];
   int  result=0;

#ifdef DEBUG
   printf ("cddb_sendcmd: sending command '%s'\n",
	   cmd);
#endif
   /* send command plus terminating CR */
   fprintf(sfd,"%s\r\n",cmd);
   /* read result */
   fgets(buffer,MAXSTRINGSIZE,sfd);
   if (error!=NULL)
     strcpy(error,buffer);
#ifdef DEBUG
   printf ("cddb_sendcmd: got result '%s'\n",
	   buffer);
#endif
   /* terminate behind the 3 digit result code */
   buffer[3]=0;
   sscanf(buffer,"%d",&result);

   return result;
};

/* reads the next line of output into buffer.
 * returns 0 if end of file or termination mark ( '.' ) has been reached,
 * 1 otherwise
 * if result is 0, buffer contains an empty string when the
 * connection got closed and '.' as it's first character when the end of the
 * current command's output has been reached */
int cddb_poll(FILE *sfd,char *buffer)
{
   int result=0;

   buffer[0]=0;
   if (!feof(sfd))
     {
	result=1;
	fgets(buffer,MAXSTRINGSIZE,sfd);
#ifdef DEBUG
	printf ("cddb_poll: server message '%s'\n",
		buffer);
#endif
	if (buffer[0]=='.')
	  result=0;
     };
   return result;
};

void cddb_closeconnection(FILE *connection)
{
#ifdef DEBUG
   printf("cddb_closeconnection: closing connection\n");
#endif
   cddb_sendcmd(connection,"quit",NULL);
   fclose(connection);
};

/* connect to server and negotiate connection.
 * returns a stream identifier or NULL if connection couldn't be established */
FILE *cddb_connect(gchar *s)
{
   /* make a local copy of our string */
   gchar *server=g_strdup((gchar*)s);
   char  *portstr;
   int port=888;
   struct hostent *address;
   struct sockaddr_in destination;

   int clientsocket;
   FILE *sfd=NULL;

   /* parse server description into server name in *server and port number
    * in port. */
   portstr=strchr(server,':');
   if (portstr!=NULL)
     {
	*portstr=0;
	portstr++;
	sscanf(portstr,"%d",&port);
     };
#ifdef DEBUG
   printf ("cddb_connect: resolving '%s'\n",
	   server);
#endif
   /* lookup hostname */
   address=NULL;
   address=gethostbyname(server);
   if (!address)
     {
#ifdef DEBUG
	perror ("cddb_connect: error resolving hostname");
#endif
     }
   else
     {
#ifdef DEBUG
	printf ("cddb_connect: creating socket\n");
#endif
	/* create a TCP/IP socket */
	clientsocket=socket(PF_INET,
			    SOCK_STREAM,
			    6);
	/* only continue if socket could be created */
	if (clientsocket!=-1)
	  {
#ifdef DEBUG
	     printf ("cddb_connect: connecting (%u.%u.%u.%u:%u)...\n",
		     (unsigned char)address->h_addr[0],
		     (unsigned char)address->h_addr[1],
		     (unsigned char)address->h_addr[2],
		     (unsigned char)address->h_addr[3],
		     port);
#endif
	     destination.sin_family=AF_INET;
	     destination.sin_port=htons(port);
	     memcpy(&destination.sin_addr.s_addr,
		    address->h_addr,
		    4);
	     /* if connection with socket failed, set clientsocket
	      * to -1, the socket itself has already been freed
	      * at this time */
	     if (connect(clientsocket,
			 &destination,
			 sizeof(destination))==-1)
	       clientsocket=-1;
	     if (clientsocket==-1)
	       {
#ifdef DEBUG
		  perror ("cddb_connect: error connecting to remote ");
#endif
	       }
	     /* if everything went fine */
	     else
	       {
		  sfd=fdopen(clientsocket,"r+");
		  if (sfd!=NULL)
		    {
		       struct passwd *userentry=getpwuid(getuid());
		       char hostname[65];
		       char *cmd;
		       int result;
		       char *username;

		       hostname[0]=0;
		       gethostname(hostname,64);

#ifdef DEBUG
		       printf ("cddb_connect: connected.\n");
#endif
		       /* get login message */
		       /* ignore any result this might yield */
		       result=cddb_sendcmd(sfd,
					   "",
					   NULL);

		       /* go through the handshake procedure */
		       if (strlen(hostname)==0)
			 strcpy(hostname,"unknown");
		       if (userentry==NULL)
			 username="unknown";
		       else
			 username=userentry->pw_name;
		       varman_setvar(dynamic_defs,"$username",username);
		       varman_setvar(dynamic_defs,"$hostname",hostname);
		       varman_setvar(dynamic_defs,"$version",VERSION);
		       cmd=varman_replacevars_copy(
						   dynamic_defs,
						   "cddb hello \"$username\" \"$hostname\" gnometoaster $version"
						   );
		       result=cddb_sendcmd(sfd,
					   cmd,
					   NULL);
		       /* if handshake wasn't successful,
			* close connection and return NULL */
		       if (((result<200)||(result>209))&&
			   (result!=402))
			 {
			    fclose(sfd);
			    sfd=NULL;
			 };
		       free(cmd);
		    };
	       };
	  };
     };
   /* free our local copy of the server location (hostname+port) */
   free(server);
   /* return stream identifier or NULL (the default value) */
   return sfd;
};

/* this gets the cddb server output for es or NULL if the request failed */
GList *cddb_serverrequest(gchar *server,cddb_essentials *es)
{
   GList *entry=NULL;
#ifdef DEBUG
   printf("cddb_serverrequest doing request\n");
#endif
   if ((cddb_sfd!=NULL)&&(feof(cddb_sfd)))
     {
#ifdef DEBUG
	printf ("cddb_serverrequest: closing invalid file descriptor\n");
#endif
	fclose(cddb_sfd);
	cddb_sfd=NULL;
     };
   /* connect to cddb database if necessary */
   if (cddb_sfd==NULL)
     cddb_sfd=cddb_connect(server);
   /* continue only if connection could be established */
   if (cddb_sfd!=NULL)
     {
	char *query;
	int  count;
	int  result;
	char *resstr;
	/* contains the closest match
	 * of our disc key query */
	char *discfound;

	/* allocate memory for query */
	query=(char*)malloc(CDDB_MAXQUERYSIZE);
	resstr=(char*)malloc(MAXSTRINGSIZE);

	sprintf(query,"cddb query %x %u ",
		es->cddb_key,
		es->tracks);
	for (count=0;count<es->tracks;count++)
	  sprintf((char*)&query[strlen(query)],
		  "%u ",
		  es->offsets[count]
		  );
	sprintf((char*)&query[strlen(query)],
		"%u",
		es->playingtime);
	result=cddb_sendcmd(cddb_sfd,
			    query,
			    resstr);
	/* found something,continue */
	if ((result==200)||(result==211))
	  {
	     char *category;
	     char *discid;
	     char *cd_title;

	     /* if only one disc was found the informations
	      * about this disc are returned together with
	      * the result code */
	     discfound=NULL;
	     if (result==200)
	       discfound=g_strdup((gchar*)&resstr[4]);
	     /* otherwise,we have to poll the list */
	     if (result==211)
	       {
		  discfound=(char*)malloc(MAXSTRINGSIZE);
		  /* take the first entry in the list
		   * and discard the other ones.
		   * This is not the suggested behavior
		   * for applications using the cddb
		   * protocol but because of the transparent
		   * nature of gnometoaster's cddb support
		   * I think it's the best alternative
		   * ==> the CDDB protocol definition
		   *     says you have to let the user
		   *     choose which entry to take
		   * */
		  if (cddb_poll(cddb_sfd,discfound))
		    while (cddb_poll(cddb_sfd,resstr));
	       };
#ifdef DEBUG
	     printf ("cddb_serverrequest: found cddb entry '%s'\n",
		     discfound);
#endif
	     /* parse the server result */
	     category=discfound;
	     discid=discfound?strchr(discfound,' '):NULL;
	     if (discid!=NULL)
	       {
		  *discid=0;
		  ++discid;
		  cd_title=strchr(discid,' ');
		  if (cd_title!=NULL)
		    {
		       *cd_title=0;
		       cd_title++;
		    };
	       };

	     /* now that we know everything we need and have satisfied the
	      * server's curiosity about our request we can actually
	      * read out the database entry */
	     if ((discid!=NULL)&&(cd_title!=NULL))
	       {
		  sprintf(query,"cddb read %s %s",
			  category,
			  discid);
		  result=cddb_sendcmd(cddb_sfd,
				      query,
				      NULL);
		  /* if everything went ok get track description
		   * for requested track */
		  if (result==210)
		    {
		       /* the first line is the corresponding cddb_key */
		       entry=g_list_append(entry,(gpointer)es->cddb_key);
		       /* store a response line */
		       while (cddb_poll(cddb_sfd,resstr))
			 entry=g_list_append(entry,g_strdup(resstr));
		    };
	       };
	     free(discfound);

	  }

	free(query);free(resstr);
     }
   else
     /* wait 10 seconds before trying to reconnect */
     usleep(10000000);
   return entry;
};

/* look up a cddb result in the cache */
GList *cddb_cachelookup(cddb_essentials *es)
{
   GList *current=cddb_cache;
   GList *result=NULL;

   while ((current!=NULL)&&(result==NULL))
     {
	GList *entry=(GList*)current->data;
	if ((int)entry->data==es->cddb_key)
	  result=entry;
	current=current->next;
     };
   return result;
};

/* returns the smallest timeout count left for es */
int cddb_getminimumtimeout(cddb_essentials *es)
{
   GList *current=NULL;
   int stimeout=0;
   pthread_mutex_lock(&cddb_sem); // Lock the request queue
   current=cddb_requests;
#ifdef DEBUG
   printf("cddb_getminimumtimeout scanning for cddb key 0x%x\n",es->cddb_key);
#endif
   while (current!=NULL)
     {
	cddb_infostruct *ce=(cddb_infostruct*)current->data;
	/* Request may not yet be complete,
	 * cddb keys have to match and a timeout must have been set
	 * for the request to be compared */
	if ((!ce->complete)&&(ce->es.cddb_key==es->cddb_key)&&(ce->timeout))
	  {
	     int thistimeout=
	       ce->timeout-
	       ((int)difftime(time(NULL),ce->last_lookup));
	     if (stimeout)
	       stimeout=MIN(stimeout,thistimeout);
	     else
	       stimeout=thistimeout;
	  };
	current=current->next;
     };
   pthread_mutex_unlock(&cddb_sem);
#ifdef DEBUG
   printf("cddb_getminimumtimeout result=%i\n",stimeout);
#endif
   return stimeout;
};

#ifdef HAVE_PTHREADS
/* this is the communication part running in a separate thread.
 * although not strictly necessary, we pass it the cddb_server as argument. */
void *cddb_commthread(void *s)
{
   gchar *server=(gchar*)s;
   GList *current;
   gpointer listentry; /* points to the data section of the list
			* entry currently processed.
			* used to find the request once it has been
			* looked up */

   int close_connect=0;

# ifdef DEBUG
   printf ("cddb_commthread: communication thread started.\n");
# endif

   /* an endless loop in a separate thread. until the SIGTERM is taking
    * our life */
   while (!commthread_exit_requested)
     {
	/* grab the requests database and check for requests again */
	pthread_mutex_lock(&cddb_sem);

	current=cddb_requests;
	while ((current!=NULL)&&
	       ((((cddb_infostruct*)current->data)->complete)||
		((int)difftime(time(NULL),((cddb_infostruct*)current->data)->last_lookup)<((cddb_infostruct*)current->data)->timeout)))
	  current=current->next;
	if (current!=NULL)
	  {
	     cddb_infostruct *info;

	     /* A request was found. do not close connection */
	     close_connect=0;

	     listentry=current->data;
	     if (listentry==NULL)
	       printf("cddb_commthread: invalid entry in cddb request queue. Please send a bug report to a.eckleder@bigfoot.com\n");
	     else
	       {
		  /* now that we have reserved ourselves exclusive access
		   * to the database, we copy out the next lookup entry
		   * for processing and release it again (we don't want
		   * to block the main app) */
		  info=(cddb_infostruct*)malloc(sizeof(cddb_infostruct));
		  memcpy(info,listentry,sizeof(cddb_infostruct));
		  /* let main thread continue it's work */
		  pthread_mutex_unlock(&cddb_sem);
# ifdef DEBUG
		  printf ("cddb_commthread: found cddb lookup request (cddb number %x,job id: %i).\n",
			  info->es.cddb_key,
			  (int)info);
# endif
		  /* do some sort of sanity check on the request we got */
		  if (info->es.tracks<=99)
		    {
		       GList *entry;

		       /* first try to satisfy our request using the cache,
			* if this fails, ask the server */
		       entry=cddb_cachelookup(&info->es);
		       if (entry==NULL)
			 {
# ifdef DEBUG
			    printf ("cddb_commthread: couldn't find entry in cache, looking up\n");
# endif
			    /* Timeouts are bound to their respective requests.
			     * So we'll scan through our request database
			     * if the same cddb essentials have already
			     * been requested and their request couldn't
			     * be fulfilled. This way, we honour a timeout
			     * even if it doesn't affect us directly */
			    if (cddb_getminimumtimeout(&info->es)<=0)
			      entry=cddb_serverrequest(server,&info->es);
			    /* if we got a valid result from our server,
			     * add it to the cache */
			    if (entry!=NULL)
			      cddb_cache=g_list_prepend(cddb_cache,entry);
			 };
		       if (entry!=NULL)
			 {
			    char query[MAXSTRINGSIZE];
			    /* skip the first entry which is the cddb key */
			    GList *current=entry->next;

# ifdef DEBUG
			    printf ("cddb_commthread: reading database entry\n");
# endif

			    strcpy(info->cd_title,"");
			    strcpy(info->track_title,"");
			    strcpy(info->artist,"");

			    while (current!=NULL)
			      {
				 strcpy(query,"DTITLE=");
				 if (!strncmp(query,
					      (gchar*)current->data,
					      strlen(query)))
				   strcpy(info->cd_title,
					  &((gchar*)current->data)[strlen(query)]);
				 sprintf(query,"TTITLE%i=",info->tracknum-1);
				 if (!strncmp(query,
					      (gchar*)current->data,
					      strlen(query)))
				   strcpy(info->track_title,
					  &((gchar*)current->data)[strlen(query)]);
				 sprintf(query,"EXTT%i=",info->tracknum-1);
				 if (!strncmp(query,
					      (gchar*)current->data,
					      strlen(query)))
				   strcat(info->artist,
					  &((gchar*)current->data)[strlen(query)]);
				 current=current->next;
			      };

			    helpings_cutcr(info->cd_title);
			    helpings_cutcr(info->track_title);
			    helpings_cutcr(info->artist);

			    /* Let's see if we can separate artist from
			     * disc title
			     * parsing the 'by convention' syntax
			     * of the disc-title field */
			    if (strstr(info->cd_title," / "))
			      {
				 /* If we don't have a credits field yet,
				  * this is it */
				 if (!strlen(info->artist))
				   {
				      strcpy(info->artist, info->cd_title);
				      *strstr(info->artist," / ")=0;
				   };
				 /* The disc title should, by definition,
				  * contain the disc-title only, without
				  * the artist */
				 strcpy(info->cd_title,strstr(info->cd_title," / ")+3);
			      };

			    info->complete=1;
# ifdef DEBUG
			    printf ("cddb_commthread: lookup completed for %s,%s,%s\n",
				    info->cd_title,
				    info->track_title,
				    info->artist);
# endif

			    /* replace the request queue entry with
			     * our local copy which contains lookup
			     * data from the cddb database */
			    pthread_mutex_lock(&cddb_sem);
			    current=cddb_requests;
			    /* our listentry might of course not even exist any
			     * more. But it certainly won't change memory adress
			     * so if we can find this adress still to be
			     * present in our request list we can be sure
			     * that the main thread is still expecting our answer */
			    while ((current!=NULL)&&
				   (((cddb_infostruct*)current->data)!=listentry))
			      current=current->next;
			    if (current!=NULL)
			      {
				 memcpy(current->data,info,sizeof(cddb_infostruct));
			      };
			    pthread_mutex_unlock(&cddb_sem);
			 }
		       else
			 {
# ifdef DEBUG
			    printf("cddb_commthread: couldn't find track, setting 3600 sec. retry timeout\n");
# endif
			    pthread_mutex_lock(&cddb_sem);
			    /* move failing entries to the end of our request
			     * list.
			     * We have to make sure that our request is
			     * still valid as it may have been deleted by
			     * the main thread.
			     * So we have to scan through the request list and
			     * try to find "our" request */
			    current=cddb_requests;
			    while ((current!=NULL)&&
				   (((cddb_infostruct*)current->data)!=listentry))
			      current=current->next;
			    if (current!=NULL)
			      {
				 cddb_requests=g_list_remove(cddb_requests,listentry);
				 /* pause for at least 1 hour before
				  * our next attempt to lookup this entry */
				 ((cddb_infostruct*)listentry)->last_lookup=time(NULL);
				 ((cddb_infostruct*)listentry)->timeout=3600;

				 cddb_requests=g_list_prepend(cddb_requests,listentry);
			      };
			    pthread_mutex_unlock(&cddb_sem);
			 };
		    };
		  free(info);
	       };
	  }
	else
	  {
	     pthread_mutex_unlock(&cddb_sem);
	     /* sleep for 3 seconds before checking for valid lookup requests again */
# ifdef DEBUG
	     printf ("cddb_commthread: couldn't find any valid requests, pausing for 3 seconds\n");
# endif
	     sleep(3);
	  };
# ifdef DEBUG
	printf("cddb_commthread: close connection flag is %i\n",close_connect);
# endif
	/* Close the connection if no more requests are in the queue
	 * and our timeout has passed */
	if (close_connect&&cddb_sfd)
	  {
	     cddb_closeconnection(cddb_sfd);
	     cddb_sfd=NULL;
	  };
	/* Close connection if we couldn't find any requests.
	 * If the next loop run will provide some more,
	 * it's gonna reset the close_connect flag */
	if (!current)
	  {
# ifdef DEBUG
	     printf("cddb_commthread: no pending requests found, terminating connection\n");
# endif
	     close_connect=1;
	  };
     };
   /* this only gets executed if the cddb thread was requested to exit */
# ifdef DEBUG
   printf ("cddb_commthread: exiting.\n");
# endif
   if (cddb_sfd!=NULL)
     {
	cddb_closeconnection(cddb_sfd);
	cddb_sfd=NULL;
     };
   commthread_exit_requested=0;
   return NULL;
};
#endif

/* the following function is rather lowlevel. It looks up an entry
 * based on the cddb_essentials struct passed to it
 * (cddb_lookup is making a copy of that struct so the original is no
 * longer needed and can safely be freed)
 * and calls a handler with the result as it's parameter.
 * This function also returns a job identifier with which the job can be
 * cancelled.
 * data can point to additional informations passed to the callback
 * but may otherwise safely be set to NULL */
int cddb_lookup(cddb_essentials *es,
		int tracknum,
		cddb_datahandler callback,
		gpointer data)
{
   cddb_infostruct *info;

   info=(cddb_infostruct*)malloc(sizeof(cddb_infostruct));
   if (info!=NULL)
     {
	strcpy(info->cd_title,"");
	strcpy(info->track_title,"");
	strcpy(info->artist,"");

	info->complete=0;

	info->callback=(gpointer)callback;
	info->data=data;

	memcpy(&info->es,es,sizeof(cddb_essentials));
	info->tracknum=tracknum;

	/* start looking up track immediately */
	info->last_lookup=time(NULL);
	info->timeout=0;

	/* append our request to the requestlist */
	pthread_mutex_lock(&cddb_sem);
	cddb_requests=g_list_prepend(cddb_requests,info);
	pthread_mutex_unlock(&cddb_sem);

     };
   return (int)info;
};

/* kill a cddb lookup job. this is done e.g. in tracks.c when the last reference
 * to a track has been cleared and the track is about to be deleted */
void cddb_killjob(int jobid)
{
   if (jobid)
     {
	pthread_mutex_lock(&cddb_sem);
	cddb_requests=g_list_remove(cddb_requests,(gpointer)jobid);
	pthread_mutex_unlock(&cddb_sem);
	/* free memory owned by this request */
	free((gpointer)jobid);
     }
   else
     printf ("Attempt to cancell job with illegal job id. Please send a bug report to A.Eckleder@bigfoot.com\n");
};

/* The following two functions are primarily used to restart the
 * cddb lookup process in case the server adress got changed in
 * the preferences setup */
int cddb_start_lookup_exec(gpointer data)
{
   int keepgoing=0;
#ifdef HAVE_PTHREADS
   keepgoing=1;
   /* wait for an eventually running cddb comm process to exit.
    * the comm process resets the commthread_exit_requested just
    * before it is actually leaving the stage, so we can rely on
    * this value as an indication on wether an older version of the
    * comm thread is still running */
   if (!commthread_exit_requested)
     {
	keepgoing=0;
	cddb_thread=(pthread_t*)malloc(sizeof(pthread_t));
	/* if thread couldn't be initialized, set thread id to NULL
	 * == not present */
	if (pthread_create(cddb_thread,
			   NULL,
			   cddb_commthread,
			   (void *)cddb_server))
	  {
	     free(cddb_thread);
	     cddb_thread=NULL;
	  }
	else
	  {
	     thid=gtk_timeout_add(500,
				  cddb_timeouthandler,
				  NULL);
	  };
     };
#endif
   return keepgoing;
};

/* wait in a timeout until the previous commthread has left the stage */
void cddb_start_lookup()
{
   gtk_timeout_add(250,
		   cddb_start_lookup_exec,
		   NULL);
};

void cddb_kill_lookup()
{
#ifdef HAVE_PTHREADS
   /* quit the communication thread. */
   if (cddb_thread!=NULL)
     {
# ifdef DEBUG
	printf ("cddb_kill_lookup: stopping lookup thread as soon as possible.\n");
# endif
	commthread_exit_requested=1;
	free(cddb_thread);
	cddb_thread=NULL;
     };
#endif
   if (thid)
     {
	gtk_timeout_remove(thid);
	thid=0;
     };
};

void cddb_serverchange()
{
   cddb_kill_lookup();
   cddb_start_lookup();
};

/* control functions called by main.c to bring up/down cddb interface */
void cddb_init()
{
   varmanwidgets_widget *w;

   cddb_requests=NULL;
#ifdef HAVE_PTHREADS
   pthread_mutex_init(&cddb_sem,NULL);
#endif
   cddb_start_lookup();

   w=varmanwidgets_entry_new(_("CDDB server"),
			     "cddbserver",
			     global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(preferences_common),w->visual,0,0,0);
   varman_install_handler(global_defs,
			  "cddbserver",
			  cddb_serverchange,NULL);
};

void cddb_destroy()
{
   cddb_kill_lookup();
};

