/*  
  Copyright 2002, Andreas Rottmann

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public  License
  along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

// Much of this code was taken and adapted from the PostgreSQL DB
// terminal psql.  See README for more information.

#include <ctype.h>

#include "app.h"

namespace GQLShell
{

using namespace std;

// different ways for scan_option to handle parameter words
enum OptionType
{
  OT_NORMAL,
  OT_SQLID,
  OT_SQLIDHACK,
  OT_FILEPIPE
};

namespace
{

string scan_option(App *app, const string& s, OptionType type, 
                   string::size_type& end_of_opt, char *quote, bool semicolon);
string unescape(const string& s);

}

/*----------
 * handle_slash_cmd:
 *
 * Handles all the different commands that start with '\', ordinarily
 * called by App::run().
 *
 * 'line' is the current input line, which should not start with a '\'
 * but with the actual command name.  (that is taken care of by
 * App::run())
 *
 * 'query' contains the query-so-far, which may be modified by
 * execution of the backslash command (for example, \r clears it)
 *
 * Returns a status code indicating what action is desired, see app.h.
 *----------
 */
App::CmdStatus App::handle_slash_cmd(const std::string& line, 
                                     string& query, 
                                     string::size_type& end_of_cmd)
{
  CmdStatus status = CMD_STATUS_SKIP_LINE;
  string::size_type blank_loc, end_of_opts;
  string cmd, options_str;
  /*
   * Find the first whitespace. line[blank_loc] will now be the
   * whitespace character (maybe the \n at the end)
   *
   * Also look for a backslash, so stuff like \p\g works.
   */
  blank_loc = line.find_first_of(" \t\n\r\\");
  
  if (line[blank_loc] == '\\')
  {
    end_of_cmd = blank_loc;
    cmd = line.substr(0, blank_loc);
    // If it's a double backslash, we skip it.
    if (line[blank_loc + 1] == '\\')
      end_of_cmd += 2;
  }
  else if (line[blank_loc] != '\n')
  {
    cmd = line.substr(0, blank_loc);
    options_str = line.substr(blank_loc + 1);
    end_of_cmd = blank_loc + 1;
  }
  else
  {
    cmd = line.substr(0, line.length() - 1);
    end_of_cmd = line.length() - 1;
    options_str = "\n";
  }

  status = exec_command(cmd, options_str, end_of_opts, query);
  if (end_of_opts == string::npos)
    end_of_cmd += options_str.length();
  else
    end_of_cmd += end_of_opts;
  
  if (status == CMD_STATUS_UNKNOWN)
  {
    if (interactive())
      fprintf(stderr, "Invalid command \\%s\n", cmd.c_str());
    status = CMD_STATUS_ERROR;
  }
  
  if (end_of_cmd != string::npos && line[end_of_cmd] != '\n' &&
      line[end_of_cmd] == '\\')
    end_of_cmd += 2;
  
  return status;
}

App::CmdStatus App::exec_command(const string& cmd, 
                                 const string& options_str,
                                 string::size_type& end_of_opts,
                                 string& query)
{
  bool success = true;
  const char *cmd_cp = cmd.c_str();
  CmdStatus status = CMD_STATUS_SKIP_LINE;
  
  end_of_opts = 0;
  
  // \copyright
  if (cmd == "copyright")
    print_copyright();
  // \d* commands
  else if (cmd_cp[0] == 'd')
  {
    string name = scan_option(this, options_str, OT_SQLID, end_of_opts, 
                              NULL, true);
    bool show_verbose = (cmd.find('+') != string::npos);
    
    switch (cmd_cp[1])
    {
      case '\0':
      case '+':
        if (!name.empty())
          success = describe_table_details(name, show_verbose);
        else
          // standard listing of interesting things
          success = list_tables("tvs", "", show_verbose);
        break;
      case 't':
      case 'v':
      case 'i':
      case 's':
      case 'S':
        success = list_tables(cmd_cp + 1, name, show_verbose);
        break;
      default:
        status = CMD_STATUS_UNKNOWN;
    }
  }
  // \g means send query
  else if (cmd == "g")
  {
    gfname_ = scan_option(this, options_str, OT_FILEPIPE, end_of_opts,
                          NULL, false);
    status = CMD_STATUS_SEND;
  }
  // help
  else if (cmd == "h" || cmd == "help")
  {
    string::size_type pos = options_str.find_first_not_of(" \t\n\r");
    if (pos != string::npos)
      help_sql(options_str.substr(pos));
    else
      help_sql();
    end_of_opts = options_str.length() - 1;
  }
  // \q or \quit
  else if (cmd == "q" || cmd == "quit")
    status = CMD_STATUS_TERMINATE;
  else if (cmd == "r" || cmd == "reset")
  {
    query = "";
    if (!quiet())
      puts("Query buffer reset (cleared).");
  }
  // \? -- slash command help
  else if (cmd == "?")
    slash_usage();
  else
    status = CMD_STATUS_UNKNOWN;
  
  if (!success)
    status = CMD_STATUS_ERROR;
  
  // TODO: warn about remaining args
  
  return status;
}


namespace
{

string scan_option(App *		app, 
                   const string& 	s, 
                   OptionType 		type, 
                   string::size_type& 	opt_pos, 
                   char *		quote, 
                   bool 		semicolon)
{
  string::size_type pos;
  string retval;

  if (quote)
    *quote = 0;
  
  if (opt_pos >= s.length() || opt_pos == string::npos)
    return retval;
  
  if ((pos = s.find_first_not_of(" \t\n\r", opt_pos)) == string::npos)
  {
    opt_pos = string::npos;
    return retval;
  }
  
  switch (s[pos])
  {
    case '"':     // double-quoted string
    {
      unsigned bslash_count = 0;
      string::size_type jj;
      
      // scan for end of quote
      for (jj = pos + 1; jj < s.length(); jj++)
      {
        if (s[jj] == '"' && bslash_count % 2 == 0)
          break;
        if (s[jj] == '\\')
          bslash_count++;
        else
          bslash_count = 0;
      }
      if (jj >= s.length())
      {
        app->error("parse error at the end of line");
        opt_pos = string::npos;
        return retval;
      }
      
      
      // If this is expected to be an SQL identifier like option then
      // we strip out the double quotes
      if (type == OT_SQLID || type == OT_SQLIDHACK)
      {
        unsigned int k, cc;
        
        bslash_count = 0;
        cc = 0;
        for (k = pos + 1; k < s.length(); k++)
        {
          if (s[k] == '"' && bslash_count % 2 == 0)
            break;
          if (s[k] == '\\')
            bslash_count++;
          else
            bslash_count = 0;
          
          retval += s[k];
        }
      }
      else
        return s.substr(pos, jj - pos + 1);
      
      opt_pos = jj + 1;
      
      if (quote)
        *quote = '"';
      
      return retval;
    }

    // A single quote has a gql-shell internal meaning, such as for
    // delimiting file names, and it also allows for such escape
    // sequences as \t
    case '\'':
    {
      unsigned int jj;
      unsigned short bslash_count = 0;
      
      for (jj = pos + 1; jj < s.length(); jj++)
      {
        if (s[jj] == '\'' && bslash_count % 2 == 0)
          break;

        if (s[jj] == '\\')
          bslash_count++;
        else
          bslash_count = 0;
      }

      if (jj >= s.length())
      {
        app->error("parse error at the end of line");
        opt_pos = string::npos;
        return retval;
      }

      retval = unescape(s.substr(pos + 1, jj - (pos + 1)));
      opt_pos = jj + 1;
      if (quote)
        *quote = '\'';
      return retval;
    }
    
    // Next command
    case '\\':
      opt_pos = pos;
      return retval;
      
    // '|' could be the beginning of a pipe if so, take rest of line
    // as option.
    case '|':
      if (type == OT_FILEPIPE)
      {
        opt_pos = string::npos;
        return s.substr(pos);
      }
      // fallthrough for other option types
      
    // A normal word
    default:
    {
      string::size_type token_end = s.find_first_of(" \t\n\r");
      
      if (token_end == string::npos)
        token_end = s.length();
      retval = s.substr(pos, token_end - pos);
      
      // Strip any trailing semi-colons for some types
      if (semicolon)
      {
        string::size_type last_nosemi = retval.find_last_not_of(';');
        
        if (last_nosemi != string::npos)
        {
          if (last_nosemi < retval.length() - 1)
            retval.erase(last_nosemi + 1);
        }
        else
          retval = "";
      }
      
      if (type == OT_SQLID)
        for (string::iterator cp = retval.begin(); cp != retval.end(); ++cp)
          if (isupper(*cp))
            *cp = tolower(*cp);
      
      opt_pos = token_end;
    }
  }
  
  return retval;
}

string unescape(const string& str)
{
  bool esc = false; // Last character we saw was the escape character
  string retval;
  
  for (const char *cp = str.c_str(); *cp; ++cp)
  {
    char c;
    
    if (esc)
    {
      switch (*cp)
      {
        case 'n':
          c = '\n';
          break;
        case 't':
          c = '\t';
          break;
        case 'b':
          c = '\b';
          break;
        case 'r':
          c = '\r';
          break;
        case 'f':
          c = '\f';
          break;
        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
        {
          long int l = 0;
          char *end;
          
          l = strtol(cp, &end, 0);
          c = l;
          cp = end - 1;
          break;
        }
        default:
          c = *cp;
      }
      retval += c;
      esc = false;
    }
    else if (*cp == '\\')
      esc = true;
    else
      retval += *cp;
  }
  
  return retval;
}

}

}
